"""
1. Loads tool information from 'tool_config.json' upon initialization.
2. Constructs the main user views with a tree view (`QTreeWidget`) for tool categories
   and a stacked widget (`QStackedWidget`) to display tool details using `ToolCard` instances.
3. Allows selection of tools and displays their details dynamically.
4. Provides functionality to run selected tools in Docker containers with user-defined flags.
5. Implements themes for the views and displays a credits dialog with information about the application.
"""

import json
from dataclasses import dataclass
from PySide6.QtWidgets import (
    QWidget, QVBoxLayout, QHBoxLayout, QLabel, QTreeWidget, QStackedWidget
)
from src.controller.docker_controller import DockerController
from src.controller.switch_validate_controller import SwitchValidateController
from src.path_resolver import PathResolver
from src.views.info_dialog import InfoDialog
from src.views.tool_list import ToolListManager
from src.controller.container_monitor_controller import ContainerStatusController


@dataclass
class UIState:
    """
    Groups UI-related widgets and state.
    This class is used to reduce the complexity of the main window's layout (App class).
    """
    tools_list: QTreeWidget
    tool_details: QStackedWidget
    docker_status_label: QLabel
    docker_status_icon: QLabel
    tool_cards: list


class App(QWidget):
    """
    Main window for the application.
    """
    def __init__(self):
        """
        Set up the user views for the main window.
        """
        super().__init__()
        self.tools = self._load_tools()

        self.ui = UIState(
            tools_list=QTreeWidget(),
            tool_details=QStackedWidget(),
            docker_status_label=QLabel('', self),
            docker_status_icon=QLabel(self),
            tool_cards=[]
        )

        self.tool_list_manager = ToolListManager(self.ui, self.tools)

        self._setup_window()
        self._setup_layout()


    def _setup_window(self) -> tuple[None, None]:
        return self.setWindowTitle('Rocket'), self.setGeometry(100, 100, 600, 500)

    def _setup_layout(self) -> None:
        main_layout = QVBoxLayout()
        content_layout = QHBoxLayout()

        self.ui.tools_list.setHeaderHidden(True)
        content_layout.addWidget(self.ui.tools_list)
        content_layout.addWidget(self.ui.tool_details)

        main_layout.addLayout(content_layout)

        self.switch_validator = SwitchValidateController()
        self.docker_controller = DockerController(self)
        self.container_status = ContainerStatusController(
            self.ui.docker_status_label,
            self.ui.docker_status_icon,
            self
        )
        self.container_status.init_docker_status_update()

        InfoDialog.create_info_layout(self, main_layout)

        return self.setLayout(main_layout)

    @staticmethod
    def _load_tools() -> list[dict]:
        """
        Load tool information from resources/tool_config.json
        """
        tools_path = PathResolver.resource_path('resources/tool_config.json')
        try:
            with open(tools_path, encoding='utf-8') as file:
                return json.load(file)
        except (FileNotFoundError, json.JSONDecodeError) as e:
            print(f"Unable to load tool_config.json: {e}")
            return []
