module cubeio_desc
  use cubetools_parameters
  use cubeio_messaging
  !
  ! This string sequence can be changed in the future: the data
  ! intrinsic orders are always built and compared between themselves,
  ! not with this original string.
  character(len=maxdim), parameter :: default_iorder='ABCDEFG'
  !
  type cubeio_desc_t
    ! --- Elements intrinsic to the data---
    logical                        :: iscplx = .false.  ! R*4 or C*4?
    character(len=maxdim), public  :: iorder = ''       ! Intrinsic data order (compared to reference)
    integer(kind=ndim_k)           :: ndim = 0          ! Number of intrinsic dimensions
    ! Because of possible degenerate dimensions, used dimensions may not be 1, 2, 3:
    integer(kind=ndim_k)           :: i1 = 1            ! 1st used dimension
    integer(kind=ndim_k)           :: i2 = 2            ! 2nd used dimension
    integer(kind=ndim_k)           :: i3 = 3            ! 3rd used dimension
    integer(kind=data_k)           :: n1 = 0            ! Size of 1st used dimension
    integer(kind=data_k)           :: n2 = 0            ! Size of 2nd used dimension
    integer(kind=data_k)           :: n3 = 0            ! Size of 3rd used dimension
    ! --- Mapped information ---
    ! These are informations external to the data, redefinable before any access to the data
    integer(kind=code_k), public  :: order = code_null  ! Mapped order (imaset/speset/unkset)
    integer(kind=ndim_k), public  :: bakx = code_unk    ! X back pointer to intrinsic dimension
    integer(kind=ndim_k), public  :: baky = code_unk    ! Y back pointer to intrinsic dimension
    integer(kind=ndim_k), public  :: bakc = code_unk    ! C back pointer to intrinsic dimension
    integer(kind=pixe_k)          :: nx = 0             ! Number of X pixels
    integer(kind=pixe_k)          :: ny = 0             ! Number of Y pixels
    integer(kind=chan_k)          :: nc = 0             ! Number of channels
    ! --- Desired access ---
    integer(kind=code_k) :: access = code_null           ! Data access mode (e.g. image/spectrum)
    integer(kind=4)      :: buffered = code_buffer_none  ! Data buffering kind
    integer(kind=4)      :: action = code_null           ! Data action mode (read/write/update)
    integer(kind=4)      :: unblank = code_null          ! Data unblank mode at read time (none/error/patch)
    logical              :: reblank = .false.            ! Data reblank mode at write time
    real(kind=4)         :: bval,eval                    ! Blanking values if reblank is .true.
  contains
    procedure, public :: default_iorder => cubeio_desc_default_intrinsic_order
    procedure, public :: set_iorder     => cubeio_desc_set_intrinsic_order
    ! final :: cubeio_desc_final  ! Can not be done implicitly as other components rely on the descriptor
  end type cubeio_desc_t

  public :: cubeio_desc_t
  public :: cubeio_desc_reset,cubeio_desc_transpose,cubeio_desc_final
  private

contains

  subroutine cubeio_desc_reset(desc,error)
    type(cubeio_desc_t), intent(out)   :: desc
    logical,             intent(inout) :: error
    ! All done by intent(out)
    return
  end subroutine cubeio_desc_reset

  subroutine cubeio_desc_default_intrinsic_order(desc,error)
    !-------------------------------------------------------------------
    ! Set the intrinsic order string to default
    !-------------------------------------------------------------------
    class(cubeio_desc_t), intent(inout) :: desc
    logical,              intent(inout) :: error
    !
    integer(kind=ndim_k) :: idim
    character(len=*), parameter :: rname='DEFAULT>INTRINSIC>ORDER'
    !
    if (desc%ndim.le.0) then
      call cubeio_message(seve%e,rname,'Internal error: ndim is not set')
      error = .true.
      return
    endif
    !
    desc%iorder = ''
    do idim=1,desc%ndim
      desc%iorder(idim:idim) = default_iorder(idim:idim)
    enddo
  end subroutine cubeio_desc_default_intrinsic_order

  subroutine cubeio_desc_set_intrinsic_order(desc,iorder,error)
    !-------------------------------------------------------------------
    ! Set the intrinsic order string
    !-------------------------------------------------------------------
    class(cubeio_desc_t), intent(inout) :: desc
    character(len=*),     intent(in)    :: iorder
    logical,              intent(inout) :: error
    !
    desc%iorder = iorder
  end subroutine cubeio_desc_set_intrinsic_order

  subroutine cubeio_desc_transpose(in,out,oaccess,error)
    use cubetools_access_types
    type(cubeio_desc_t),  intent(in)    :: in
    type(cubeio_desc_t),  intent(inout) :: out
    integer(kind=code_k), intent(in)    :: oaccess
    logical,              intent(inout) :: error
    !
    integer(kind=ndim_k) :: idim,odim
    !
    ! This is generic, even if already in correct access
    call cubeio_desc_copy(in,out,error)
    if (error)  return
    out%access = oaccess
    out%order = cubetools_access2order(oaccess)
    if (in%order.eq.out%order)  return
    !
    ! For an actual transposition, need to reorder the intrinsic
    ! order code. Put the relevant dimensions first, keep the
    ! others as trailing, unreordered.
    if (out%order.eq.code_cube_imaset) then
      ! Need X then Y as leading dimensions
      if (in%bakx.ne.code_unk) then
        out%bakx = 1
        out%iorder(1:1) = in%iorder(in%bakx:in%bakx)
      endif
      if (in%baky.ne.code_unk) then
        out%baky = 2
        out%iorder(2:2) = in%iorder(in%baky:in%baky)
      endif
      odim = 2
      do idim=1,out%ndim
        if (in%bakx.eq.idim)  cycle
        if (in%baky.eq.idim)  cycle
        odim = odim+1
        out%iorder(odim:odim) = in%iorder(idim:idim)
        if (in%bakc.eq.idim)  out%bakc = odim
      enddo
      !
    elseif (out%order.eq.code_cube_speset) then
      ! Need C as leading dimension
      if (in%bakc.ne.code_unk) then
        out%bakc = 1
        out%iorder(1:1) = in%iorder(in%bakc:in%bakc)
      endif
      odim = 1
      do idim=1,out%ndim
        if (in%bakc.eq.idim)  cycle
        odim = odim+1
        out%iorder(odim:odim) = in%iorder(idim:idim)
        if (in%bakx.eq.idim)  out%bakx = odim
        if (in%baky.eq.idim)  out%baky = odim
      enddo
      !
    else
      ! Can this happen?
      ! Everything copied "as is"
    endif
  end subroutine cubeio_desc_transpose

  subroutine cubeio_desc_copy(in,out,error)
    type(cubeio_desc_t), intent(in)    :: in
    type(cubeio_desc_t), intent(inout) :: out
    logical,             intent(inout) :: error
    out = in
  end subroutine cubeio_desc_copy

  subroutine cubeio_desc_final(desc)
    !---------------------------------------------------------------------
    ! Finalize a 'cubeio_desc_t' instance
    !---------------------------------------------------------------------
    type(cubeio_desc_t), intent(inout) :: desc
    !
    ! Nothing to be freed
  end subroutine cubeio_desc_final

end module cubeio_desc
