# SPDX-License-Identifier: GPL-3.0-only
# Copyright (c) 2022 Caleb La Grange <thonkpeasant@protonmail.com>
# Copyright (c) 2022 Ferass El Hafidi <vitali64pmemail@protonmail.com>
# Copyright (c) 2023-2025 Leah Rowe <leah@libreboot.org>

cbcfgsdir="config/coreboot"
hashfiles="vendorhashes blobhashes" # used to detect and error out
			# if running on an archive that needs vendor files
dontflash="!!! AN ERROR OCCURED! Do NOT flash these images !!!"
tmpromdel="$xbmkpwd/tmp/DO_NOT_FLASH"
nvm="util/nvmutil/nvm"
ifdtool="elf/ifdtool/default/ifdtool"

cv="CONFIG_GBE_BIN_PATH CONFIG_IFD_BIN_PATH"

eval "`setvars "" tree new_mac archive boarddir rom cbdir xchanged \
    tmpromdir IFD_platform ifdprefix xromsize $cv`"

fail_inject()
{
	[ -L "$tmpromdel" ] || [ ! -d "$tmpromdel" ] || \
	    rm -Rf "$tmpromdel" || :
	printf "\n\n%s\n\n" "$dontflash" 1>&2
	printf "WARNING: File '%s' was NOT modified.\n\n" "$archive" 1>&2
	err_ "$1"
}

inject()
{
	err="fail_inject"
	remkdir "$tmpromdel"

	set +u +e
	[ $# -lt 1 ] && $err "No options specified. - $dontflash"
	eval "`setvars "" new_mac xchanged`"

	archive="$1";
	new_mac="??:??:??:??:??:??"

	[ $# -gt 1 ] && case "$2" in
	setmac)
		[ $# -gt 2 ] && new_mac="$3" && \
		    [ -z "$new_mac" ] && $err "Empty MAC address specified" ;;
	*) $err "Unrecognised inject mode: '$2'"
	esac

	check_release "$archive" || $err "'$archive' is not a release archive"

	if readcfg && readkconfig; then
		patch_release_roms
	fi

	[ "$xchanged" != "y" ] && printf "\n'%s' *NOT* modified.\n" "$archive"
	[ "$xchanged" = "y" ] && printf "\n'%s' %s.\n" "$archive" "$xtype"; :
}

check_release()
{
	[ -L "$archive" ] && $err "'$archive' is a symlink. $dontflash"
	e "$archive" f missing && return 1

	archivename="`basename "$archive"`"
	[ -z "$archivename" ] && $err "Can't determine archive name. $dontflash"

	case "$archivename" in
	*_src.tar.xz)
		$err "'$archive' is a src archive, silly!" ;;
	grub_*|seagrub_*|custom_*|seauboot_*|seabios_withgrub_*)
		return 1 ;;
	*.tar.xz) _stripped_prefix="${archivename#*_}"
		board="${_stripped_prefix%.tar.xz}" ;;
	*) $err "'$archive': could not detect board type - $dontflash"
	esac; :
}

readcfg()
{
	if [ "$board" = "serprog_rp2040" ] || [ "$board" = "serprog_stm32" ] \
	    || [ "$board" = "serprog_pico" ]; then
		printf "'%s' is a serprog firmware archive.\n" "$archive" 1>&2
		return 1
	fi
	boarddir="$cbcfgsdir/$board"

	eval "`setcfg "$boarddir/target.cfg"`"
	chkvars tree && x_ ./mk -d coreboot "$tree"

	cbdir="src/coreboot/$tree"
	cbfstool="elf/cbfstool/$tree/cbfstool"
	rmodtool="elf/cbfstool/$tree/rmodtool"
	cbfstool="elf/cbfstool/$tree/cbfstool"
	ifdtool="elf/ifdtool/$tree/ifdtool"
	[ -n "$IFD_platform" ] && ifdprefix="-p $IFD_platform"; :
}

readkconfig()
{
	x_ rm -f "$xbmktmp/cbcfg"
	cbcfg="`check_defconfig "$boarddir"`" || for cbc in $cv; do
		grep "$cbc" "$cbcfg" 1>>"$xbmktmp/cbcfg" 2>/dev/null || :
	done
	eval "`setcfg "$xbmktmp/cbcfg" 1`"
}

patch_release_roms()
{
	tmpromdir="tmp/DO_NOT_FLASH/bin/$board"

	remkdir "${tmpromdir%"/bin/$board"}"
	x_ tar -xf "$archive" -C "${tmpromdir%"/bin/$board"}"

	for _hashes in $hashfiles; do
		e "$tmpromdir/$_hashes" f && $err \
		    "'$archive': vendorfile insertion unsupported; $dontflash"
	done

	[ -z "$new_mac" ] || modify_mac || printf "\nGbE not defined\n" 1>&2

	[ "$xchanged" = "y" ] || rm -Rf "$tmpromdel" || :
	[ "$xchanged" = "y" ] || return 0
	(
	x_ cd "${tmpromdir%"/bin/$board"}"
	mkrom_tarball "bin/$board"
	) || $err "Cannot re-generate '$archive' - $dontflash"

	mv "${tmpromdir%"/bin/$board"}/bin/${relname}_${board}.tar.xz" \
	    "$archive" || $err "'$archive' -> Can't overwrite - $dontflash"; :
}

modify_mac()
{
	[ -n "$CONFIG_GBE_BIN_PATH" ] || return 1

	x_ mkdir -p tmp && x_ cp "${CONFIG_GBE_BIN_PATH##*../}" tmp/gbe

	if [ "$new_mac" != "restore" ]; then
		x_ make -C util/nvmutil
		x_ "$nvm" tmp/gbe setmac "$new_mac"
	fi

	find "$tmpromdir" -maxdepth 1 -type f -name "*.rom" > "tmp/rom.list" \
	    || $err "'$archive' -> Can't make tmp/rom.list - $dontflash"

	while read -r _xrom; do
		e "$_xrom" f && xchanged="y" && x_ \
		    "$ifdtool" $ifdprefix -i GbE:tmp/gbe "$_xrom" -O "$_xrom"
	done < "tmp/rom.list"

	printf "\nGbE NVM written to '%s':\n" "$archive"
	x_ "$nvm" tmp/gbe dump | grep -v "bytes read from file" || :

	[ "$new_mac" = "restore" ] && \
		printf "\nDefault GbE file '%s' written, unmodified.\n" \
		    "${CONFIG_GBE_BIN_PATH##*../}"; :
}
