/*
 * Copyright (C) 2014 Aaron Seigo <aseigo@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) version 3, or any
 * later version accepted by the membership of KDE e.V. (or its
 * successor approved by the membership of KDE e.V.), which shall
 * act as a proxy defined in Section 6 of version 3 of the license.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "resource.h"

#include <QCoreApplication>
#include <QDir>
#include <QPluginLoader>
#include <QPointer>

#include "facadefactory.h"
#include "adaptorfactoryregistry.h"

namespace Sink {

Resource::Resource() : QObject(), d(nullptr)
{
    Q_UNUSED(d);
}

Resource::~Resource()
{
    // delete d;
}

void Resource::processCommand(int commandId, const QByteArray &data)
{
    Q_UNUSED(commandId)
    Q_UNUSED(data)
}

void Resource::setLowerBoundRevision(qint64 revision)
{
    Q_UNUSED(revision)
}

void Resource::setSecret(const QString &s)
{
    Q_UNUSED(s)
}

bool Resource::checkForUpgrade()
{
    return false;
}


class ResourceFactory::Private
{
public:
    QByteArrayList capabilities;
};

typedef QHash<QString, QPointer<ResourceFactory>> FactoryRegistry;
Q_GLOBAL_STATIC(FactoryRegistry, s_loadedFactories);

ResourceFactory::ResourceFactory(QObject *parent, const QByteArrayList &capabilities) : QObject(parent), d(new ResourceFactory::Private)
{
    d->capabilities = capabilities;
}

ResourceFactory::~ResourceFactory()
{
    delete d;
}

ResourceFactory *ResourceFactory::load(const QByteArray &resourceName)
{
    ResourceFactory *factory = s_loadedFactories->value(resourceName);
    if (factory) {
        return factory;
    }

    for (auto const &path : QCoreApplication::instance()->libraryPaths()) {
        QDir pluginDir(path);
        // TODO: centralize this so that it is easy to change centrally
        //      also ref'd in cmake as ${SINK_RESOURCE_PLUGINS_PATH}
        if (!pluginDir.cd(QStringLiteral("sink")) || !pluginDir.cd(QStringLiteral("resources"))) {
            continue;
        }

        for (const QString &fileName : pluginDir.entryList(QDir::Files)) {
            const QString path = pluginDir.absoluteFilePath(fileName);
            QPluginLoader loader(path);

            const QString id = loader.metaData()[QStringLiteral("IID")].toString();
            if (id == resourceName) {
                QObject *object = loader.instance();
                if (object) {
                    factory = qobject_cast<ResourceFactory *>(object);
                    if (factory) {
                        s_loadedFactories->insert(resourceName, factory);
                        //TODO: Instead of always loading both facades and adaptorfactories into the respective singletons, we could also leave this up to the caller. (ResourceFactory::loadFacades(...))
                        factory->registerFacades(resourceName, FacadeFactory::instance());
                        factory->registerAdaptorFactories(resourceName, AdaptorFactoryRegistry::instance());
                        // TODO: if we need more data on it const QJsonObject json = loader.metaData()[QStringLiteral("MetaData")].toObject();
                        return factory;
                    } else {
                        qWarning() << "Plugin for" << resourceName << "from plugin" << loader.fileName() << "produced the wrong object type:" << object;
                        delete object;
                    }
                } else {
                    qWarning() << "Could not load factory for" << resourceName << "from plugin" << loader.fileName() << "due to the following error:" << loader.errorString();
                }
            }
        }
    }

    qWarning() << "Failed to find factory for resource:" << resourceName;
    qWarning() << "Looked into the following directories: " << QCoreApplication::instance()->libraryPaths();
    return nullptr;
}

QByteArrayList ResourceFactory::capabilities() const
{
    return d->capabilities;
}

} // namespace Sink

// Ignore warning I don't know how to fix in a moc file
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wundefined-reinterpret-cast"
#include "moc_resource.cpp"
#pragma clang diagnostic pop
