/* vim: set ts=2 et sw=2 tw=80: */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/. */

#ifndef kw_vectors_h__
#define kw_vectors_h__

#include <string>
#include <vector>
#include <map>

/* The result struct adds some granularity to what we are testing.
 * With the single "valid" boolean, we don't know whether the function under
 * test itself is expected to fail (i.e. return SECFailure), or if we simply
 * expect the output to differ from what's in the test vector. Therefore,
 * each vector contains a map of Actions (i.e. NSS capabilities that we wish to
 * test) mapping to expected Result values (both in terms of return code as
 * well as whether or not the output should match the test vector).
 *
 * - If |output_match| is true, the function under test MUST return the same
 *   data (length and contents).
 * - If |output_match| is false, the function under test MUST NOT return the
 *   same data (length or contents must be different).
 */

typedef struct Result {
  SECStatus expect_rv;
  bool output_match;
} Result;

enum Action { WRAP, UNWRAP };

typedef struct keywrap_vector {
  uint32_t test_id;
  std::vector<uint8_t> key;
  std::vector<uint8_t> msg;
  std::vector<uint8_t> ct;
  std::map<Action, Result> tests;
} keywrap_vector;

const keywrap_vector kWycheproofAesKWVectors[] = {
    {1,
     {0x6f, 0x67, 0x48, 0x6d, 0x1e, 0x91, 0x44, 0x19, 0xcb, 0x43, 0xc2, 0x85,
      0x09, 0xc7, 0xc1, 0xea},
     {0x8d, 0xc0, 0x63, 0x2d, 0x92, 0xee, 0x0b, 0xe4, 0xf7, 0x40, 0x02, 0x84,
      0x10, 0xb0, 0x82, 0x70},
     {0x9d, 0xe4, 0x53, 0xce, 0xd5, 0xd4, 0xab, 0x46, 0xa5, 0x60, 0x17, 0x08,
      0xee, 0xef, 0xef, 0xb5, 0xe5, 0x93, 0xe6, 0xae, 0x8e, 0x86, 0xb2, 0x6b},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {2,
     {0xa0, 0xb1, 0x71, 0x72, 0xbb, 0x29, 0x6d, 0xb7, 0xf5, 0xc8, 0x69, 0xe9,
      0xa3, 0x6b, 0x5c, 0xe3},
     {0x61, 0x5d, 0xd0, 0x22, 0xd6, 0x07, 0xc9, 0x10, 0xf2, 0x01, 0x78, 0xcb,
      0xdf, 0x42, 0x06, 0x0f},
     {0x8c, 0x3a, 0xba, 0x85, 0xcc, 0x0a, 0xe1, 0xae, 0x10, 0xb3, 0x66, 0x58,
      0xb0, 0x68, 0xf5, 0x95, 0xba, 0xf8, 0xca, 0xaf, 0xb7, 0x45, 0xef, 0x3c},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {3,
     {0x0e, 0x49, 0xd5, 0x71, 0xc1, 0x9b, 0x52, 0x50, 0xef, 0xfd, 0x41, 0xd9,
      0x4b, 0xde, 0x39, 0xd6},
     {0xf2, 0x5e, 0x4d, 0xe8, 0xca, 0xca, 0x36, 0x3f, 0xd5, 0xf2, 0x94, 0x42,
      0xeb, 0x14, 0x7b, 0x55},
     {0x1d, 0xe0, 0x93, 0x65, 0x48, 0x26, 0xf1, 0x8f, 0xcd, 0x0f, 0x3f, 0xd4,
      0x99, 0x41, 0x6f, 0xf2, 0x2e, 0xd7, 0x5e, 0xe1, 0x2f, 0xe0, 0xb6, 0x24},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {4,  // wrapped key is longer than wrapping key
     {0xe0, 0xe1, 0x29, 0x59, 0x10, 0x91, 0x03, 0xe3, 0x0a, 0xe8, 0xb5, 0x68,
      0x4a, 0x22, 0xe6, 0x62},
     {0xdb, 0xb0, 0xf2, 0xbb, 0x2b, 0xe9, 0x12, 0xa2, 0x04, 0x30, 0x97, 0x2d,
      0x98, 0x42, 0xce, 0x3f, 0xd3, 0xb9, 0x28, 0xe5, 0x73, 0xe1, 0xac, 0x8e},
     {0x9c, 0x3d, 0xdc, 0x23, 0x82, 0x7b, 0x7b, 0x3c, 0x13, 0x10, 0x5f,
      0x9e, 0x8b, 0x11, 0x52, 0x3b, 0xac, 0xcd, 0xfb, 0x6c, 0x8b, 0x7e,
      0x78, 0x25, 0x49, 0x6e, 0x7a, 0x84, 0x0b, 0xd3, 0x2a, 0xec},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {5,  // wrapped key is longer than wrapping key
     {0xdd, 0x58, 0x3d, 0x9f, 0x10, 0x59, 0x86, 0x14, 0x30, 0xec, 0x8b, 0x5d,
      0x8a, 0x18, 0x0e, 0x9b},
     {0xf2, 0xe3, 0x4f, 0x35, 0x63, 0x62, 0xa3, 0x1b, 0x51, 0xd6, 0xe0, 0x2b,
      0xcd, 0x33, 0x3c, 0x9e, 0x61, 0x70, 0x49, 0x4c, 0xa5, 0xff, 0x54, 0x87},
     {0xaf, 0xb7, 0x44, 0xaa, 0xf7, 0x46, 0xdc, 0xc0, 0xb5, 0x7f, 0x8b,
      0x37, 0x8c, 0x40, 0x4c, 0xbe, 0x87, 0x7f, 0x44, 0xcf, 0x3d, 0x45,
      0x14, 0x0d, 0x60, 0x81, 0x4e, 0xda, 0x3f, 0x54, 0x1f, 0x01},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {6,  // wrapped key is longer than wrapping key
     {0xfa, 0xf5, 0xcc, 0xfa, 0xe4, 0x2b, 0x43, 0xce, 0xe2, 0xc5, 0xf0, 0xf3,
      0x17, 0x7a, 0x7c, 0x5d},
     {0x4e, 0x02, 0x08, 0x48, 0x33, 0x66, 0x0c, 0x46, 0x38, 0x30, 0x48, 0x3b,
      0x36, 0xda, 0xb8, 0x66, 0xc6, 0x4c, 0x8c, 0xf7, 0x42, 0x9c, 0xac, 0x3d},
     {0xcf, 0xf9, 0x8c, 0xd6, 0x4c, 0xb5, 0x1a, 0xb9, 0x9b, 0x81, 0xae,
      0xe8, 0x2c, 0xee, 0x42, 0x74, 0xd0, 0xdf, 0x3e, 0x1b, 0x6a, 0x49,
      0x43, 0xd3, 0x92, 0x36, 0xea, 0x98, 0x98, 0x46, 0xd0, 0xcc},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {7,  // wrapped key is longer than wrapping key
     {0xc2, 0xb9, 0xd2, 0x3f, 0x28, 0x31, 0xdd, 0xcd, 0xeb, 0x45, 0x68, 0x53,
      0xd4, 0x01, 0x4d, 0xb9},
     {0xf4, 0xcf, 0xea, 0x98, 0xe5, 0x8b, 0x93, 0x9c, 0xc8, 0x59, 0x55,
      0x43, 0x85, 0xcf, 0x3a, 0x6c, 0x7f, 0x82, 0x17, 0xf7, 0x28, 0xef,
      0xb4, 0x31, 0xc9, 0x64, 0x78, 0x6d, 0xe8, 0x27, 0x49, 0x07},
     {0x58, 0xdc, 0xfb, 0x0e, 0x7e, 0xc4, 0xd3, 0xbc, 0x80, 0x03,
      0x41, 0x8d, 0x86, 0x5f, 0xbd, 0x52, 0x0c, 0x6b, 0x24, 0xb2,
      0xbd, 0xe3, 0x5b, 0x1b, 0xe5, 0xb1, 0xc5, 0xff, 0x32, 0xa1,
      0x30, 0xf3, 0x3d, 0x03, 0x5e, 0x59, 0x32, 0x61, 0x60, 0x83},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {8,  // wrapped key is longer than wrapping key
     {0x62, 0x0a, 0x08, 0xf3, 0x20, 0xcd, 0xed, 0xbf, 0x7a, 0xe5, 0x51, 0xad,
      0xd3, 0x48, 0xd9, 0x5e},
     {0xce, 0xc3, 0x4e, 0xaf, 0x8e, 0x67, 0xe1, 0xce, 0x61, 0x9d, 0xdf,
      0xc3, 0x09, 0x53, 0x1c, 0x42, 0xf1, 0x60, 0x33, 0xa7, 0xe2, 0xcb,
      0xc4, 0xf5, 0xeb, 0x3a, 0x54, 0x81, 0x64, 0xe9, 0xb2, 0x91},
     {0x4e, 0xe4, 0x7b, 0xd6, 0x8d, 0x41, 0x85, 0x86, 0xc4, 0x47,
      0xa3, 0x91, 0x11, 0xe2, 0xec, 0x15, 0x02, 0xff, 0x0f, 0x17,
      0x26, 0xea, 0x91, 0xc5, 0xd9, 0x73, 0x70, 0x40, 0x9d, 0x89,
      0xb8, 0xe6, 0x6e, 0x88, 0x9b, 0x63, 0x8a, 0xc4, 0x0c, 0xed},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {9,  // wrapped key is longer than wrapping key
     {0xed, 0x08, 0x9a, 0xc2, 0x74, 0xf8, 0xc7, 0xce, 0xa2, 0x41, 0x56, 0x71,
      0xa9, 0x4b, 0x5e, 0x53},
     {0x60, 0x65, 0xe4, 0x1d, 0xf1, 0x4d, 0xae, 0xee, 0xfa, 0xca, 0xc5,
      0xda, 0xeb, 0x76, 0x74, 0xcd, 0xc9, 0xc1, 0xf6, 0x86, 0x01, 0x3b,
      0x79, 0x71, 0x53, 0xe8, 0x0e, 0xf2, 0x15, 0x89, 0x32, 0x99},
     {0xd3, 0xb0, 0x93, 0xfd, 0x82, 0x2c, 0xe4, 0x54, 0xeb, 0xc2,
      0x51, 0xc6, 0xf2, 0x1f, 0xa7, 0x1c, 0x38, 0x58, 0xee, 0x7e,
      0x62, 0x3e, 0xcb, 0xfb, 0xbf, 0x88, 0x73, 0x98, 0xa3, 0x0b,
      0x40, 0xc5, 0x5d, 0x05, 0x65, 0xc7, 0xa1, 0x5e, 0x40, 0x15},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {10,  // Round counter overflows 256
     {0x31, 0xca, 0xcb, 0xb1, 0x7d, 0x6d, 0xbb, 0xec, 0xae, 0x40, 0x72, 0x7c,
      0x50, 0x48, 0xfe, 0x0c},
     {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
     {0x22, 0x2d, 0xea, 0xdd, 0xe6, 0xef, 0xb7, 0x60, 0xca, 0xe4, 0x2f, 0xa1,
      0x88, 0x31, 0x0e, 0x0c, 0x07, 0xe7, 0xd5, 0x57, 0x52, 0x97, 0x66, 0x44,
      0x4a, 0x9e, 0xfb, 0x33, 0x09, 0x07, 0xd4, 0x2f, 0x0d, 0xd8, 0xf3, 0xd1,
      0x7b, 0x3a, 0x38, 0xbf, 0x40, 0xd6, 0x8c, 0x09, 0x5a, 0x9c, 0xce, 0x19,
      0xda, 0xf9, 0x07, 0xbf, 0x2c, 0x92, 0xf1, 0xe5, 0x9b, 0x18, 0xb2, 0x77,
      0xff, 0x03, 0x97, 0xfc, 0x50, 0xf4, 0x5f, 0x58, 0x2d, 0xb9, 0x36, 0xaa,
      0x8a, 0xfb, 0x94, 0x3d, 0xe0, 0x1b, 0x58, 0xab, 0xfd, 0xc8, 0x1d, 0xae,
      0xf4, 0xe0, 0x38, 0xc9, 0x9e, 0x4c, 0x1e, 0xb3, 0xee, 0x44, 0x74, 0x64,
      0xbb, 0x8f, 0x89, 0xa4, 0xea, 0x81, 0xe5, 0x65, 0x56, 0xcc, 0x26, 0xc7,
      0x28, 0x83, 0xc0, 0x6a, 0x7f, 0xe8, 0x50, 0xd0, 0x43, 0x47, 0xd6, 0x8b,
      0xba, 0xdc, 0x4a, 0x06, 0x77, 0x50, 0x30, 0x67, 0x6d, 0xb8, 0xcc, 0x34,
      0xae, 0xb0, 0x7e, 0x39, 0xc7, 0xf0, 0x59, 0xc2, 0xbf, 0xf7, 0x6a, 0x7f,
      0x2b, 0xaf, 0x07, 0x67, 0x49, 0x00, 0x4e, 0xf7, 0x18, 0x9f, 0x88, 0x7f,
      0x89, 0x02, 0x9f, 0x88, 0xc5, 0xc1, 0xd0, 0xf5, 0xee, 0x62, 0x32, 0x0b,
      0x42, 0x30, 0x48, 0xe2, 0xad, 0x81, 0x86, 0xe6, 0x3b, 0xe2, 0x3c, 0x55,
      0x3f, 0x55, 0x76, 0xa4, 0x0d, 0x96, 0x7e, 0x8b, 0x52, 0x7c, 0xcd, 0x78,
      0x3c, 0x41, 0xab, 0xc6, 0x4b, 0xea, 0x1a, 0xec, 0x8a, 0x76, 0xde, 0xb3,
      0xe9, 0xc9, 0xa6, 0x65, 0x67, 0x56, 0xe1, 0xda, 0xc3, 0x8b, 0xd2, 0x5f,
      0xf0, 0x08, 0x88, 0x8a, 0x55, 0x91, 0xaf, 0x76, 0x3f, 0xd7, 0x3f, 0x5a,
      0xdd, 0xd7, 0x49, 0x79, 0x48, 0x17, 0x07, 0x09, 0x90, 0x48, 0x46, 0x54,
      0xa4, 0x6e, 0xf4, 0x42, 0xad, 0xae, 0xac, 0xab, 0x14, 0xd1, 0x2d, 0xaa,
      0xf8, 0x7b, 0xcb, 0x91, 0x66, 0x1a, 0x21, 0x59, 0x37, 0x17, 0xd9, 0xb9,
      0x35, 0x29, 0xd8, 0x13, 0xea, 0x5f, 0xc8, 0x12, 0xd7, 0x08, 0xc0, 0xd8,
      0xe9, 0xb6, 0x81, 0x22, 0xd9, 0xf5, 0xe6, 0x26, 0x7c, 0xde, 0x36, 0x37,
      0x80, 0xa4, 0x5d, 0x07, 0xe4, 0xca, 0xa5, 0xbf, 0x14, 0xf2, 0x33, 0x4f,
      0x25, 0xb9, 0xd1, 0x77, 0x63, 0x2b, 0xb8, 0x0a, 0x82, 0x89, 0x4a, 0xa4,
      0x7b, 0x6f, 0x36, 0x0b, 0x2d, 0xa1, 0x13, 0x8d, 0xed, 0xd4, 0x33, 0x1f,
      0x07, 0x7c, 0x90, 0x05, 0x54, 0xc7, 0xa6, 0x8b, 0x5d, 0x15, 0x49, 0x80,
      0xbb, 0x35, 0x17, 0xac, 0x20, 0xa7, 0x8a, 0x51, 0xf6, 0xf2, 0x1e, 0x42,
      0xb2, 0xc4, 0xc9, 0x60, 0xd3, 0x1c, 0xbd, 0x22, 0xbd, 0x61, 0x08, 0x19,
      0x18, 0x2c, 0x5e, 0x45, 0x6b, 0xa7, 0xd7, 0xe9, 0x03, 0xf5, 0xe6, 0x0f,
      0x6e, 0x0c, 0x4c, 0xc0, 0x14, 0xfe, 0xb4, 0xf8, 0x19, 0x73, 0xce, 0x48,
      0xad, 0x6f, 0x4d, 0xab, 0x8d, 0xa5, 0x1e, 0xb7},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {11,  // empty keys cannot be wrapped
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7, 0xa9,
      0xa7, 0xda, 0x64, 0x95},
     {},
     {0xa6, 0xa6, 0xa6, 0xa6, 0xa6, 0xa6, 0xa6, 0xa6},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {12,  // keys of size 8 byte cannot be wrapped
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7, 0xa9,
      0xa7, 0xda, 0x64, 0x95},
     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07},
     {0xdc, 0x26, 0xfb, 0x69, 0x11, 0xd7, 0x19, 0x71, 0xdf, 0x03, 0x56, 0xd6,
      0xbb, 0x9e, 0xd6, 0xe6},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {13,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7, 0xa9,
      0xa7, 0xda, 0x64, 0x95},
     {0x00},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {14,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7, 0xa9,
      0xa7, 0xda, 0x64, 0x95},
     {0x00, 0x01},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {15,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7, 0xa9,
      0xa7, 0xda, 0x64, 0x95},
     {0x00, 0x01, 0x02},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {16,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7, 0xa9,
      0xa7, 0xda, 0x64, 0x95},
     {0x00, 0x01, 0x02, 0x03},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {17,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7, 0xa9,
      0xa7, 0xda, 0x64, 0x95},
     {0x00, 0x01, 0x02, 0x03, 0x04},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {18,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7, 0xa9,
      0xa7, 0xda, 0x64, 0x95},
     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05},
     {},
     {{Action::WRAP, {SECFailure, false}}}},

    {19,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7, 0xa9,
      0xa7, 0xda, 0x64, 0x95},
     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {20,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7, 0xa9,
      0xa7, 0xda, 0x64, 0x95},
     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
      0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {21,  // invalid size of wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c, 0xa4,
      0xe8, 0x21, 0x82, 0xf7},
     {},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {22,  // invalid size of wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c, 0xa4,
      0xe8, 0x21, 0x82, 0xf7},
     {},
     {0x9f},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {23,  // invalid size of wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c, 0xa4,
      0xe8, 0x21, 0x82, 0xf7},
     {},
     {0xdc, 0x9e, 0x95, 0x80},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {24,  // invalid size of wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c, 0xa4,
      0xe8, 0x21, 0x82, 0xf7},
     {},
     {0xb9, 0xb2, 0x82, 0xd1, 0x38, 0x69, 0x30, 0x00},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {25,  // invalid size of wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c, 0xa4,
      0xe8, 0x21, 0x82, 0xf7},
     {},
     {0x0e, 0xfc, 0x63, 0x5b, 0x2d, 0x61, 0xe2, 0x44, 0x05, 0x6b, 0x9d, 0x45,
      0x91, 0xca, 0x6b},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {26,  // invalid size of wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c, 0xa4,
      0xe8, 0x21, 0x82, 0xf7},
     {},
     {0x4a, 0x30, 0x5d, 0xae, 0x08, 0x7b, 0x0d, 0x24, 0xd6, 0x2a, 0xf4, 0x18,
      0x31, 0x33, 0x8f, 0x33, 0xae},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {27,  // invalid size of wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c, 0xa4,
      0xe8, 0x21, 0x82, 0xf7},
     {},
     {0x82, 0xcb, 0x92, 0x70, 0x97, 0xcf, 0x31, 0xea, 0x4a, 0xff,
      0xea, 0x44, 0x0b, 0x0d, 0x8c, 0xa6, 0xa2, 0x40, 0xb9, 0x00},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {28,  // bytes appended to wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c, 0xa4,
      0xe8, 0x21, 0x82, 0xf7},
     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
      0x0c, 0x0d, 0x0e, 0x0f},
     {0x97, 0x90, 0xab, 0x51, 0xfb, 0xcb, 0x85, 0x0d, 0xf6, 0x76, 0x4e, 0x01,
      0x1a, 0xe9, 0x7c, 0x85, 0x78, 0x5b, 0xed, 0x26, 0x33, 0xae, 0xa6, 0x65},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {28,  // Bytes appended to wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c, 0xa4,
      0xe8, 0x21, 0x82, 0xf7},
     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
      0x0c, 0x0d, 0x0e, 0x0f},
     {0x97, 0x90, 0xab, 0x51, 0xfb, 0xcb, 0x85, 0x0d, 0xf6,
      0x76, 0x4e, 0x01, 0x1a, 0xe9, 0x7c, 0x85, 0x78, 0x5b,
      0xed, 0x26, 0x33, 0xae, 0xa6, 0x65, 0x00},
     {{Action::UNWRAP, {SECFailure, false}}}},

    {29,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x0a, 0xac, 0x32, 0x9c, 0xcd, 0x51, 0x3e, 0xdb, 0xdd, 0x63, 0x67, 0xdf,
      0x67, 0x99, 0x9e, 0xaa, 0xc9, 0xe7, 0xb5, 0x19, 0x84, 0xc4, 0xd3, 0x8d},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {30,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x5a, 0x55, 0xdc, 0x42, 0x97, 0x49, 0xca, 0x49, 0xbb, 0x4a, 0xb0, 0x1d,
      0x96, 0x6b, 0x19, 0xea, 0x9a, 0x9e, 0x14, 0x02, 0xe6, 0xab, 0x29, 0x62},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {31,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x45, 0xf5, 0x33, 0xf6, 0x07, 0x2f, 0x64, 0x0e, 0xb7, 0xe1, 0xe5, 0x12,
      0xd5, 0x60, 0x72, 0x08, 0x55, 0x67, 0xf4, 0xad, 0x60, 0x12, 0xa9, 0x7a},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {32,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x84, 0xf2, 0x84, 0x56, 0x5d, 0xf4, 0x7c, 0x40, 0x91, 0x07, 0xf7, 0xa0,
      0xa7, 0x1b, 0xc3, 0x70, 0xa8, 0xed, 0x44, 0x89, 0xd4, 0x14, 0xb9, 0xe9},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {33,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x39, 0x41, 0xc3, 0x66, 0x55, 0x4f, 0xc8, 0x96, 0xe9, 0xfe, 0x52, 0xf0,
      0x24, 0x93, 0xca, 0x03, 0xd4, 0x39, 0xeb, 0x17, 0xc2, 0x36, 0x14, 0x6d},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {34,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x45, 0xc9, 0xd4, 0x23, 0x63, 0xd9, 0x81, 0xd0, 0x86, 0xa9, 0x72, 0x72,
      0x8e, 0x13, 0x0a, 0x42, 0xf5, 0xdd, 0x90, 0xbd, 0xa5, 0x62, 0xa8, 0x5a},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {35,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x03, 0x7d, 0x17, 0x85, 0x95, 0x19, 0xd6, 0xc0, 0x72, 0x8a, 0x9e, 0xb6,
      0xe6, 0x41, 0x13, 0xe8, 0x69, 0x19, 0xde, 0xca, 0xbd, 0x3b, 0xbb, 0x88},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {36,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x1a, 0xd1, 0x0a, 0xf7, 0xf6, 0xc0, 0x42, 0xb2, 0x67, 0xa0, 0xc7, 0xbc,
      0x4d, 0x25, 0xd2, 0x7c, 0x00, 0x3d, 0xeb, 0x50, 0xe2, 0xcc, 0x56, 0x6a},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {37,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x63, 0x0c, 0x57, 0x1b, 0x7f, 0xb8, 0x64, 0x7a, 0xc5, 0x36, 0x0a, 0x25,
      0x5f, 0x9f, 0x5d, 0x36, 0x45, 0x79, 0x5a, 0xc4, 0x52, 0x85, 0xcb, 0xaa},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {38,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x16, 0xdb, 0x55, 0x3e, 0x46, 0x7d, 0x40, 0x29, 0xd0, 0xfe, 0xa6, 0x2b,
      0x2c, 0x44, 0x0e, 0x5d, 0xf6, 0xc6, 0x59, 0x1f, 0x04, 0x97, 0xa9, 0x9d},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {39,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x09, 0x79, 0x91, 0x09, 0x0a, 0x15, 0x60, 0x47, 0xd4, 0x78, 0x4b, 0x75,
      0x7f, 0x26, 0x2e, 0x12, 0xce, 0x57, 0xe1, 0x3a, 0x3d, 0x5d, 0x28, 0x6c},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {40,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x39, 0x57, 0xc3, 0x38, 0xb7, 0x50, 0xa3, 0x28, 0x5e, 0xb7, 0xb6, 0x5c,
      0x9c, 0xfe, 0x77, 0x05, 0x3d, 0xd7, 0xd8, 0x14, 0x9f, 0x42, 0xca, 0xa1},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {41,  // RFC 3394
     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
      0x0c, 0x0d, 0x0e, 0x0f},
     {0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xaa, 0xbb,
      0xcc, 0xdd, 0xee, 0xff},
     {0x1f, 0xa6, 0x8b, 0x0a, 0x81, 0x12, 0xb4, 0x47, 0xae, 0xf3, 0x4b, 0xd8,
      0xfb, 0x5a, 0x7b, 0x82, 0x9d, 0x3e, 0x86, 0x23, 0x71, 0xd2, 0xcf, 0xe5},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {42,
     {0xf7, 0x5a, 0x2f, 0x49, 0xa6, 0x30, 0xc7, 0xdc, 0x91, 0x62, 0x6b, 0x00,
      0xce, 0x02, 0x9f, 0x0b, 0xd2, 0x98, 0x1d, 0x7c, 0x74, 0xa9, 0x3e, 0xbe},
     {0x9a, 0xdb, 0xc0, 0x0c, 0x71, 0x0b, 0x11, 0x01, 0xbd, 0xf6, 0xa4, 0xed,
      0x65, 0xb3, 0x2d, 0x72},
     {0x00, 0xbe, 0x1c, 0xad, 0xdf, 0xd5, 0xad, 0x76, 0x97, 0x87, 0x70, 0x17,
      0x79, 0x5f, 0x9c, 0xee, 0x4b, 0xce, 0x5a, 0x61, 0x68, 0x7a, 0x61, 0x26},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {43,
     {0xb7, 0x13, 0xf6, 0xb7, 0x81, 0x4f, 0x98, 0x89, 0x4d, 0x7b, 0x15, 0x39,
      0x74, 0x68, 0x43, 0x59, 0xf1, 0x46, 0x02, 0x13, 0xeb, 0x74, 0xbe, 0x68},
     {0x78, 0x58, 0x5f, 0x0c, 0x49, 0x92, 0x2e, 0x82, 0xca, 0xf1, 0x7e, 0xbc,
      0x37, 0x21, 0xb4, 0xdb},
     {0x54, 0xe7, 0xf2, 0x78, 0xb5, 0xa1, 0xfb, 0x4c, 0x31, 0xa0, 0xd7, 0x9a,
      0xc1, 0xf6, 0x15, 0xed, 0xd9, 0x10, 0xbf, 0x22, 0x01, 0x5a, 0x06, 0x68},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {44,
     {0x13, 0xec, 0xf4, 0x23, 0x21, 0x1c, 0xaa, 0x33, 0x4b, 0xa6, 0xdb, 0x37,
      0x25, 0x9a, 0x53, 0x5c, 0x20, 0xde, 0x8a, 0xd1, 0x0f, 0xc8, 0xc4, 0x32},
     {0x4f, 0xc7, 0x5d, 0x0f, 0x22, 0x1e, 0x22, 0x40, 0x8a, 0x37, 0xe1, 0x12,
      0x65, 0xd4, 0x9a, 0x05},
     {0x51, 0x04, 0x55, 0xbd, 0x9c, 0x07, 0x8a, 0xc1, 0xf0, 0x7b, 0xb3, 0x75,
      0x2c, 0xbd, 0x04, 0xe4, 0x21, 0xb0, 0xdd, 0x63, 0x51, 0x90, 0xfa, 0x62},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {45,
     {0x44, 0x17, 0xfb, 0xbe, 0xa5, 0x1b, 0xdd, 0x91, 0x81, 0x8d, 0x74, 0x05,
      0x19, 0x57, 0xdd, 0x70, 0xe1, 0x35, 0xc5, 0xcf, 0x37, 0x32, 0xbd, 0xf1},
     {0xf5, 0x35, 0x7d, 0xa9, 0xf8, 0xfd, 0x4a, 0x11, 0x90, 0xf3, 0x6e, 0x9f,
      0xa0, 0x9a, 0x90, 0xfc, 0xf1, 0x4d, 0x87, 0xd6, 0x23, 0x32, 0xf1, 0xa5},
     {0x88, 0x0d, 0xa5, 0xb4, 0x10, 0xf9, 0x13, 0xad, 0x72, 0xcc, 0x93,
      0xf4, 0x63, 0x44, 0xf1, 0x15, 0x21, 0x65, 0xbd, 0xea, 0x14, 0x66,
      0x4f, 0xd2, 0xd3, 0xaf, 0xbd, 0x87, 0xb8, 0xcc, 0x5c, 0xfd},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {46,
     {0xb3, 0xf2, 0x6d, 0x8a, 0x22, 0xfd, 0xd6, 0x1f, 0x70, 0x98, 0x41, 0x23,
      0x1f, 0xbd, 0xe6, 0x95, 0xb3, 0xf2, 0x8d, 0xdd, 0xce, 0xd6, 0xd4, 0x1e},
     {0x0d, 0x0a, 0xf9, 0x55, 0xd2, 0xe3, 0x82, 0x9c, 0xc3, 0xd6, 0x43, 0x21,
      0x9b, 0x30, 0x1e, 0x64, 0xe0, 0x51, 0x0d, 0xfb, 0xc4, 0x28, 0x11, 0x9a},
     {0x27, 0x65, 0x4c, 0xf6, 0xa6, 0x3d, 0x60, 0x04, 0xae, 0x83, 0xda,
      0x54, 0xc2, 0xe5, 0xd7, 0xb5, 0xfa, 0xd2, 0x08, 0x78, 0xf3, 0x50,
      0x08, 0x7d, 0xdd, 0x17, 0xac, 0x44, 0xa2, 0xbe, 0x86, 0x8f},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {47,
     {0xf7, 0x0c, 0xfb, 0x26, 0x2c, 0x72, 0x9a, 0x18, 0x20, 0x6c, 0x8a, 0xfd,
      0x74, 0x35, 0x6e, 0xc7, 0xe0, 0x49, 0xd1, 0x0b, 0x44, 0xa6, 0xe0, 0x00},
     {0x24, 0x1c, 0xed, 0xfa, 0x64, 0xc4, 0xe7, 0xbe, 0xc5, 0x41, 0xa2, 0xeb,
      0x4c, 0x36, 0x82, 0x69, 0xe0, 0xf0, 0xdd, 0xeb, 0xc5, 0x82, 0x67, 0xea},
     {0xad, 0x7c, 0xa6, 0x6a, 0xd4, 0x66, 0x4f, 0x43, 0xe4, 0xdd, 0x09,
      0x29, 0x6a, 0x6e, 0x6f, 0x02, 0xd5, 0xaf, 0x44, 0x08, 0xf2, 0x25,
      0xc0, 0xab, 0xeb, 0x0d, 0x9b, 0x76, 0xc8, 0xd1, 0xe9, 0x82},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {48,  // wrapped key is longer than wrapping key
     {0x16, 0x39, 0xf9, 0xf8, 0x1e, 0x53, 0xe2, 0xee, 0xb6, 0x77, 0xa2, 0x49,
      0xe5, 0xec, 0xed, 0x3a, 0xf1, 0x08, 0x97, 0x13, 0x01, 0x60, 0x1a, 0x7b},
     {0xec, 0x3c, 0x6a, 0x1f, 0x1a, 0x95, 0x85, 0x32, 0x7f, 0xe6, 0x58,
      0x49, 0x0c, 0x74, 0x63, 0x5e, 0x53, 0x00, 0x87, 0x6d, 0xa5, 0x84,
      0x6a, 0x62, 0x93, 0x98, 0x98, 0x4f, 0xb5, 0x51, 0xd6, 0x91},
     {0xe2, 0x45, 0xc9, 0x0a, 0x6b, 0x46, 0xca, 0xec, 0xe9, 0x4f,
      0x47, 0x11, 0x7d, 0x60, 0x83, 0x31, 0x95, 0x8c, 0x8f, 0x75,
      0xf5, 0x31, 0xeb, 0xcd, 0xc9, 0x02, 0xc0, 0x21, 0x3d, 0x91,
      0x05, 0xf2, 0x15, 0x5a, 0xf0, 0x7d, 0xaa, 0x62, 0xd1, 0x32},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {49,  // wrapped key is longer than wrapping key
     {0x1f, 0x22, 0xd5, 0x65, 0x8a, 0xa6, 0x85, 0xb8, 0xba, 0x86, 0x59, 0xdc,
      0x34, 0x28, 0x80, 0xd5, 0xb2, 0x39, 0x9e, 0x6a, 0x81, 0x50, 0x05, 0xb0},
     {0x50, 0xbe, 0x4c, 0x1b, 0x2f, 0x29, 0xa6, 0x3f, 0x44, 0xd7, 0xfc,
      0x63, 0x73, 0x7f, 0x60, 0x0f, 0x01, 0x94, 0xea, 0x3f, 0xb3, 0x6e,
      0x17, 0x3d, 0x2d, 0xdd, 0x19, 0xf2, 0x18, 0x65, 0x63, 0x80},
     {0x8a, 0x32, 0xb9, 0xf2, 0x07, 0xae, 0x5a, 0xae, 0xdb, 0x7e,
      0x8a, 0x0d, 0x94, 0x51, 0x07, 0x41, 0x2c, 0x1b, 0xd0, 0x69,
      0x99, 0xbc, 0x5a, 0xc8, 0x3c, 0x1f, 0x95, 0x8d, 0xfb, 0x77,
      0xeb, 0xdc, 0xf9, 0xd9, 0x8c, 0x60, 0xdb, 0xd4, 0x65, 0x0a},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {50,  // wrapped key is longer than wrapping key
     {0x3a, 0x2f, 0x4a, 0xa5, 0x04, 0x41, 0x95, 0x4b, 0xba, 0x5a, 0x18, 0x36,
      0x29, 0x4c, 0xe0, 0x71, 0xf9, 0x29, 0x6b, 0x23, 0xdb, 0xed, 0x67, 0x71},
     {0x65, 0xda, 0x02, 0xff, 0x21, 0xb4, 0x83, 0xa1, 0xe3, 0x95, 0x75,
      0x49, 0x0b, 0x43, 0x19, 0xe8, 0x4a, 0xe0, 0x29, 0x9f, 0x1f, 0x00,
      0xb3, 0x85, 0x9f, 0xbe, 0x2e, 0x74, 0xb3, 0xec, 0x2a, 0xaf},
     {0x4a, 0x58, 0x42, 0xb1, 0x0d, 0x2d, 0xb9, 0x6e, 0xa1, 0x03,
      0x9e, 0xf4, 0x78, 0x5c, 0xe7, 0x22, 0x55, 0x5b, 0x37, 0x51,
      0xa9, 0xb6, 0xdd, 0x39, 0x12, 0x6a, 0xd3, 0x63, 0x37, 0x8c,
      0x72, 0x32, 0x0d, 0x83, 0xea, 0x7a, 0xdb, 0x81, 0x61, 0x5a},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {51,  // Round counter overflows 256
     {0x31, 0xca, 0xcb, 0xb1, 0x7d, 0x6d, 0xbb, 0xec, 0xae, 0x40, 0x72, 0x7c,
      0x50, 0x48, 0xfe, 0x0c, 0x01, 0xbc, 0x53, 0xb2, 0x3a, 0xb6, 0x35, 0x02},
     {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
     {0xe2, 0x19, 0x25, 0x98, 0xe6, 0x12, 0x4f, 0x27, 0x91, 0xb2, 0x75, 0x1f,
      0x93, 0x09, 0x58, 0x43, 0x5b, 0xb1, 0xd0, 0x2e, 0x98, 0xaa, 0x1e, 0x09,
      0x78, 0x1b, 0xba, 0x0b, 0x15, 0x94, 0x35, 0xdb, 0x65, 0x9f, 0xa7, 0x3f,
      0xa3, 0x10, 0x11, 0x17, 0x04, 0x69, 0x2c, 0x68, 0xe1, 0x6d, 0xdf, 0x4b,
      0xe0, 0x60, 0x22, 0xc5, 0x2f, 0xe9, 0xdb, 0xa6, 0x27, 0x9a, 0xad, 0x1a,
      0xeb, 0x81, 0x41, 0x25, 0xd0, 0xdd, 0xf3, 0x3f, 0x31, 0xe5, 0x8e, 0x62,
      0x5a, 0xf3, 0x23, 0x05, 0x05, 0x0c, 0xfe, 0xa3, 0x90, 0xd8, 0x78, 0x2d,
      0x32, 0xca, 0xac, 0x55, 0x88, 0x89, 0xca, 0x8e, 0x64, 0x19, 0x08, 0x20,
      0x8d, 0xa6, 0x97, 0x65, 0x42, 0xb4, 0x0d, 0xbd, 0x09, 0x01, 0x78, 0xe2,
      0xa6, 0x81, 0x2a, 0x43, 0x6c, 0x18, 0xa5, 0xe8, 0x91, 0xac, 0x80, 0x83,
      0x17, 0x6a, 0xce, 0x1d, 0xdf, 0xee, 0x4d, 0x38, 0x28, 0x56, 0xa5, 0x9c,
      0x80, 0xc6, 0x43, 0xec, 0xd5, 0xc1, 0xab, 0x68, 0xc6, 0x6b, 0x2b, 0x89,
      0x84, 0xce, 0x6e, 0x0e, 0x38, 0x6f, 0x65, 0x54, 0xa9, 0xcb, 0x91, 0xd3,
      0x63, 0xbb, 0x4a, 0xcc, 0xf0, 0x28, 0x87, 0x8e, 0xc2, 0x0b, 0x8b, 0x2e,
      0x37, 0x21, 0x4f, 0x7b, 0x12, 0xdb, 0xcd, 0x78, 0xfb, 0x38, 0xf7, 0x11,
      0xa9, 0x0f, 0xe2, 0x62, 0xc7, 0x84, 0x91, 0xb9, 0x05, 0x83, 0x54, 0xe2,
      0x7b, 0x34, 0xfc, 0x92, 0xef, 0x0d, 0x70, 0x28, 0x59, 0x4c, 0xb0, 0x82,
      0x59, 0xf8, 0x6b, 0x54, 0xcb, 0x1d, 0x31, 0x7e, 0xc5, 0x5f, 0x2e, 0xf2,
      0xab, 0x7e, 0x8b, 0x14, 0x16, 0x71, 0xf8, 0xbc, 0xb1, 0xa9, 0x0b, 0xb7,
      0xd8, 0x2b, 0xc8, 0xcb, 0x4f, 0xb0, 0x2c, 0x9c, 0xe4, 0x30, 0xef, 0x4a,
      0xe0, 0xdc, 0x84, 0x7e, 0x91, 0xe7, 0xd4, 0xfb, 0x46, 0x3e, 0xb9, 0xdd,
      0x87, 0xfc, 0x9d, 0xc9, 0x56, 0x8f, 0x3a, 0x4e, 0xf5, 0x04, 0xd7, 0x4c,
      0x13, 0x4d, 0xcb, 0x60, 0xca, 0x01, 0xb3, 0x6c, 0xe1, 0x0c, 0xb4, 0x67,
      0x26, 0x8e, 0xa2, 0x97, 0xc0, 0x51, 0x8a, 0x50, 0xd1, 0x2c, 0xc0, 0x25,
      0xcf, 0xee, 0xa1, 0x38, 0x1d, 0xdd, 0x7d, 0xd6, 0x3e, 0x4a, 0xc9, 0x48,
      0x90, 0xa0, 0xea, 0xe9, 0xdb, 0xd8, 0xdb, 0x24, 0x46, 0x25, 0xa3, 0xc8,
      0xaf, 0x2e, 0x1a, 0xff, 0x6a, 0x81, 0x12, 0xc4, 0xd3, 0xd9, 0x8e, 0x82,
      0x63, 0x58, 0x8e, 0xe1, 0xdd, 0x90, 0x63, 0xb7, 0x09, 0xb8, 0xec, 0x00,
      0x47, 0x4c, 0x4e, 0xa4, 0x13, 0xa8, 0x02, 0xb8, 0xcd, 0xe8, 0x14, 0xe7,
      0xb3, 0x73, 0x14, 0x10, 0x80, 0x9e, 0xd0, 0x00, 0x06, 0x0f, 0xb7, 0xb9,
      0xb0, 0xbf, 0x85, 0x09, 0xef, 0x17, 0x51, 0x5f, 0x33, 0x3e, 0x86, 0x8b,
      0x18, 0x8e, 0xa1, 0x64, 0x45, 0xf3, 0x80, 0xb3, 0xa7, 0xd4, 0x27, 0x74,
      0xf6, 0x45, 0x6c, 0xdf, 0x72, 0x42, 0x46, 0xfa},
     {{Action::WRAP,
       {SECSuccess, true}},  // Round counter overflows - acceptable
      {Action::UNWRAP, {SECSuccess, true}}}},

    {52,  // empty keys cannot be wrapped
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7, 0xa9,
      0xa7, 0xda, 0x64, 0x95, 0x39, 0x8f, 0x18, 0xea, 0x0d, 0x8e, 0xed, 0x76},
     {},
     {0xa6, 0xa6, 0xa6, 0xa6, 0xa6, 0xa6, 0xa6, 0xa6},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {53,  // keys of size 8 byte cannot be wrapped
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7, 0xa9,
      0xa7, 0xda, 0x64, 0x95, 0x39, 0x8f, 0x18, 0xea, 0x0d, 0x8e, 0xed, 0x76},
     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07},
     {0x38, 0xd8, 0x23, 0x8c, 0xdb, 0x0d, 0x9a, 0x2d, 0xa2, 0x8d, 0x6d, 0x56,
      0x19, 0x4f, 0x2e, 0x78},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {54,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7, 0xa9,
      0xa7, 0xda, 0x64, 0x95, 0x39, 0x8f, 0x18, 0xea, 0x0d, 0x8e, 0xed, 0x76},
     {0x00},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {55,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7, 0xa9,
      0xa7, 0xda, 0x64, 0x95, 0x39, 0x8f, 0x18, 0xea, 0x0d, 0x8e, 0xed, 0x76},
     {0x00, 0x01},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {56,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7, 0xa9,
      0xa7, 0xda, 0x64, 0x95, 0x39, 0x8f, 0x18, 0xea, 0x0d, 0x8e, 0xed, 0x76},
     {0x00, 0x01, 0x02},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {57,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7, 0xa9,
      0xa7, 0xda, 0x64, 0x95, 0x39, 0x8f, 0x18, 0xea, 0x0d, 0x8e, 0xed, 0x76},
     {0x00, 0x01, 0x02, 0x03},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {58,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7, 0xa9,
      0xa7, 0xda, 0x64, 0x95, 0x39, 0x8f, 0x18, 0xea, 0x0d, 0x8e, 0xed, 0x76},
     {0x00, 0x01, 0x02, 0x03, 0x04},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {59,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7, 0xa9,
      0xa7, 0xda, 0x64, 0x95, 0x39, 0x8f, 0x18, 0xea, 0x0d, 0x8e, 0xed, 0x76},
     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {60,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7, 0xa9,
      0xa7, 0xda, 0x64, 0x95, 0x39, 0x8f, 0x18, 0xea, 0x0d, 0x8e, 0xed, 0x76},
     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {61,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7, 0xa9,
      0xa7, 0xda, 0x64, 0x95, 0x39, 0x8f, 0x18, 0xea, 0x0d, 0x8e, 0xed, 0x76},
     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
      0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {62,  // invalid size of wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c, 0xa4,
      0xe8, 0x21, 0x82, 0xf7, 0xe9, 0x92, 0x3a, 0x74, 0x61, 0x10, 0xfd, 0x97},
     {},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {63,  // invalid size of wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c, 0xa4,
      0xe8, 0x21, 0x82, 0xf7, 0xe9, 0x92, 0x3a, 0x74, 0x61, 0x10, 0xfd, 0x97},
     {},
     {0x9f},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {64,  // invalid size of wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c, 0xa4,
      0xe8, 0x21, 0x82, 0xf7, 0xe9, 0x92, 0x3a, 0x74, 0x61, 0x10, 0xfd, 0x97},
     {},
     {0xdc, 0x9e, 0x95, 0x80},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {65,  // invalid size of wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c, 0xa4,
      0xe8, 0x21, 0x82, 0xf7, 0xe9, 0x92, 0x3a, 0x74, 0x61, 0x10, 0xfd, 0x97},
     {},
     {0xb9, 0xb2, 0x82, 0xd1, 0x38, 0x69, 0x30, 0x00},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {66,  // invalid size of wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c, 0xa4,
      0xe8, 0x21, 0x82, 0xf7, 0xe9, 0x92, 0x3a, 0x74, 0x61, 0x10, 0xfd, 0x97},
     {},
     {0x0e, 0xfc, 0x63, 0x5b, 0x2d, 0x61, 0xe2, 0x44, 0x05, 0x6b, 0x9d, 0x45,
      0x91, 0xca, 0x6b},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {67,  // invalid size of wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c, 0xa4,
      0xe8, 0x21, 0x82, 0xf7, 0xe9, 0x92, 0x3a, 0x74, 0x61, 0x10, 0xfd, 0x97},
     {},
     {0x4a, 0x30, 0x5d, 0xae, 0x08, 0x7b, 0x0d, 0x24, 0xd6, 0x2a, 0xf4, 0x18,
      0x31, 0x33, 0x8f, 0x33, 0xae},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {68,  // invalid size of wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c, 0xa4,
      0xe8, 0x21, 0x82, 0xf7, 0xe9, 0x92, 0x3a, 0x74, 0x61, 0x10, 0xfd, 0x97},
     {},
     {0x82, 0xcb, 0x92, 0x70, 0x97, 0xcf, 0x31, 0xea, 0x4a, 0xff,
      0xea, 0x44, 0x0b, 0x0d, 0x8c, 0xa6, 0xa2, 0x40, 0xb9, 0x00},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {69,  // bytes appended to wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c, 0xa4,
      0xe8, 0x21, 0x82, 0xf7, 0xe9, 0x92, 0x3a, 0x74, 0x61, 0x10, 0xfd, 0x97},
     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
      0x0c, 0x0d, 0x0e, 0x0f},
     {0x55, 0xdf, 0xb2, 0xf7, 0xe0, 0xc1, 0xea, 0x04, 0xfe,
      0xad, 0x89, 0x7c, 0x45, 0x1c, 0x05, 0x05, 0x92, 0x1d,
      0xc4, 0x7f, 0x30, 0x8c, 0x49, 0x17, 0x00},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {70,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x9b, 0x15, 0x93, 0xfd, 0x7d, 0x4f, 0xe2, 0x5a, 0x66, 0x0b, 0xbc, 0x19,
      0x76, 0xea, 0x4a, 0xb6, 0x8b, 0xcc, 0x53, 0xf8, 0x48, 0xa8, 0xeb, 0x9d},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {71,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0xe7, 0xed, 0xb8, 0x47, 0xfa, 0x91, 0xe2, 0xde, 0xde, 0xd7, 0x26, 0xed,
      0xf3, 0xab, 0x93, 0xda, 0x91, 0x15, 0x16, 0x97, 0x42, 0x5f, 0xee, 0x28},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {72,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x1b, 0x51, 0xa7, 0xc0, 0x33, 0xc1, 0xef, 0xb5, 0xee, 0x29, 0x94, 0x25,
      0x9c, 0x40, 0xf0, 0x3b, 0xb5, 0x7d, 0x8c, 0xc0, 0x9e, 0x50, 0x7e, 0x6e},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {73,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0xc4, 0x0b, 0x61, 0x4a, 0x50, 0x62, 0xf5, 0xfd, 0x04, 0x9c, 0x53, 0x79,
      0xb3, 0xe8, 0x14, 0x16, 0x14, 0xc2, 0xda, 0x97, 0x89, 0x35, 0x89, 0xfb},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {74,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0xd7, 0x57, 0x5e, 0xf0, 0x2d, 0xf5, 0x4b, 0x30, 0x86, 0xeb, 0x49, 0x03,
      0x5e, 0xea, 0xfb, 0xce, 0x0e, 0x08, 0x33, 0x6e, 0x89, 0xb3, 0x5a, 0xb0},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {75,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x0d, 0x61, 0x7f, 0x1c, 0x12, 0x48, 0x5a, 0x35, 0x91, 0x7d, 0x2a, 0x94,
      0x1e, 0x94, 0x9d, 0x2f, 0xdb, 0xf0, 0x3a, 0x34, 0x68, 0x89, 0xb8, 0x50},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {76,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0xc7, 0xdf, 0x34, 0x72, 0x91, 0x74, 0xdb, 0x2e, 0x83, 0xee, 0x16, 0xc6,
      0xde, 0x74, 0xd5, 0xeb, 0x97, 0x66, 0x71, 0x5f, 0xad, 0x04, 0x9b, 0x40},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {77,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0xc3, 0xed, 0x38, 0xd6, 0xf9, 0xcc, 0xb9, 0xbf, 0x3c, 0x56, 0xbb, 0x31,
      0x76, 0xf0, 0x0d, 0x3c, 0xe9, 0x88, 0x75, 0x21, 0xf8, 0xd4, 0xc7, 0x0b},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {78,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x65, 0x82, 0xb5, 0xf4, 0x65, 0x27, 0x44, 0xb0, 0x53, 0x7e, 0x97, 0xe7,
      0xcd, 0xae, 0x0f, 0x44, 0x31, 0x30, 0x14, 0x0d, 0xba, 0xea, 0x60, 0x4c},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {79,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0xfc, 0x9b, 0x9b, 0xdf, 0x25, 0xf2, 0xb4, 0x8a, 0xd7, 0x99, 0x34, 0xc8,
      0xd3, 0x48, 0x97, 0xcd, 0xbf, 0x4c, 0x84, 0x6f, 0x8c, 0xb4, 0xb1, 0x1d},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {80,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0xfc, 0x23, 0xc7, 0xf4, 0xfe, 0x20, 0xaa, 0x81, 0x10, 0x5e, 0xfc, 0x1a,
      0x71, 0x05, 0xa5, 0x31, 0x6b, 0x23, 0x39, 0x9a, 0xc7, 0x92, 0xc8, 0x24},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {81,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x37, 0xb4, 0xa2, 0x61, 0xb9, 0x6b, 0xce, 0xc9, 0xcc, 0x93, 0xee, 0xf5,
      0xb2, 0xfb, 0xbb, 0xe8, 0x46, 0x34, 0xf9, 0x78, 0xc5, 0x89, 0x3d, 0xda},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {82,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0x18, 0x31, 0x10, 0x98, 0x47, 0xd1, 0x7d, 0x01, 0x0b, 0xfc, 0xd9,
      0x3f, 0xb4, 0x6f, 0x31, 0x50, 0xcd, 0xaf, 0xd5, 0x27, 0x33, 0xdb,
      0x74, 0xc2, 0x21, 0xb0, 0x34, 0xfe, 0x8e, 0x15, 0x52, 0xc0},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {83,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0x2b, 0x2b, 0x89, 0xc3, 0xc6, 0xb2, 0xdb, 0x49, 0x03, 0x87, 0x7a,
      0xd4, 0x62, 0x2c, 0xa3, 0x3f, 0x3a, 0x3c, 0xb7, 0xe6, 0x70, 0x1d,
      0x13, 0x40, 0xe6, 0xaf, 0xc0, 0xfd, 0xab, 0x7d, 0xbd, 0x72},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {84,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0x3c, 0x69, 0x03, 0xf3, 0x7d, 0xa5, 0x71, 0x61, 0xaf, 0x67, 0x06,
      0x05, 0x0a, 0x2e, 0xd7, 0x47, 0xcd, 0x55, 0x34, 0x4d, 0x86, 0x91,
      0x89, 0xfd, 0x49, 0xd2, 0x53, 0x6f, 0xf9, 0x48, 0x12, 0x9d},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {85,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0x8a, 0x8d, 0xc8, 0xb1, 0x31, 0xc6, 0xb9, 0x68, 0xb6, 0x0c, 0x1d,
      0xd8, 0x19, 0xa6, 0x55, 0x39, 0x2d, 0x1a, 0x96, 0xd6, 0xca, 0xfa,
      0x48, 0xe3, 0x0f, 0xb1, 0x14, 0x6f, 0x09, 0x62, 0x29, 0xc6},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {86,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0x30, 0x3e, 0x0b, 0x36, 0x02, 0xd5, 0x14, 0xa7, 0xd5, 0x2e, 0xdb,
      0xa3, 0x30, 0x6d, 0x73, 0x83, 0xe8, 0x99, 0x9e, 0x7c, 0x65, 0x2a,
      0x51, 0x03, 0x35, 0xa8, 0x94, 0x9e, 0xfb, 0x42, 0xeb, 0x66},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {87,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0x31, 0x9b, 0xef, 0xbf, 0x2d, 0xdb, 0xb4, 0x75, 0x72, 0x3f, 0xb2,
      0xfa, 0x30, 0xf2, 0xae, 0x7f, 0xc1, 0xce, 0xb1, 0xe6, 0xf3, 0x61,
      0x71, 0x5e, 0xca, 0x72, 0x09, 0x60, 0x88, 0x73, 0xc7, 0xfc},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {88,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0x9b, 0x3e, 0x44, 0x03, 0x41, 0xc5, 0xda, 0x13, 0x15, 0x59, 0x95,
      0x9d, 0xb6, 0xb3, 0x55, 0x3a, 0x53, 0x46, 0x91, 0x16, 0x2f, 0x4f,
      0x00, 0x93, 0x27, 0xbf, 0x2c, 0x21, 0xd6, 0xfe, 0x5a, 0xda},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {89,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0xeb, 0xa6, 0xcc, 0x09, 0x59, 0xe6, 0xa5, 0x63, 0x39, 0xb1, 0x41,
      0x62, 0x98, 0x40, 0xad, 0xd8, 0x0f, 0x45, 0x65, 0x65, 0x6d, 0xc6,
      0x87, 0xa3, 0xb9, 0x96, 0x96, 0x0c, 0x99, 0x4d, 0xfd, 0x26},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {90,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0xb7, 0x03, 0xb6, 0xcf, 0x45, 0x87, 0x70, 0x93, 0x53, 0xc7, 0xe4,
      0x00, 0x4d, 0x3d, 0xa6, 0x1c, 0xe5, 0xf5, 0xde, 0xaf, 0x71, 0x63,
      0xca, 0x9d, 0x61, 0x58, 0xdd, 0xe9, 0x19, 0xe0, 0xac, 0x34},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {91,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0x72, 0x54, 0x9d, 0x52, 0xd6, 0xf4, 0xff, 0x91, 0x2d, 0x83, 0x3c,
      0x74, 0x13, 0x6d, 0x90, 0x63, 0x4c, 0xe8, 0xaf, 0xa4, 0xf8, 0x44,
      0x12, 0xbb, 0xee, 0x80, 0x74, 0x08, 0x4d, 0x4c, 0xec, 0xff},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {92,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0x13, 0x37, 0xc8, 0xbd, 0x6c, 0x8a, 0x5d, 0xd4, 0x3a, 0xba, 0x8d,
      0x29, 0x88, 0x64, 0xff, 0xe7, 0x6a, 0xd6, 0xea, 0x90, 0x9f, 0x34,
      0x88, 0x15, 0x7a, 0x15, 0xe6, 0xc4, 0x6a, 0xcf, 0x22, 0x14},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {93,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3, 0xdc,
      0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a, 0x84, 0xb4},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0x4a, 0xaf, 0xfa, 0xca, 0x5f, 0xe8, 0x58, 0x14, 0xd0, 0x40, 0xaa,
      0x2a, 0x30, 0x6b, 0xa4, 0xd1, 0xd4, 0x47, 0x46, 0xcf, 0xe4, 0x6c,
      0x97, 0x8a, 0xa0, 0x57, 0xb5, 0x3f, 0xd2, 0x53, 0x16, 0xc1},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {94,  // RFC 3394
     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
      0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17},
     {0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xaa, 0xbb,
      0xcc, 0xdd, 0xee, 0xff},
     {0x96, 0x77, 0x8b, 0x25, 0xae, 0x6c, 0xa4, 0x35, 0xf9, 0x2b, 0x5b, 0x97,
      0xc0, 0x50, 0xae, 0xd2, 0x46, 0x8a, 0xb8, 0xa1, 0x7a, 0xd8, 0x4e, 0x5d},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {95,  // RFC 3394
     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
      0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17},
     {0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xaa, 0xbb,
      0xcc, 0xdd, 0xee, 0xff, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07},
     {0x03, 0x1d, 0x33, 0x26, 0x4e, 0x15, 0xd3, 0x32, 0x68, 0xf2, 0x4e,
      0xc2, 0x60, 0x74, 0x3e, 0xdc, 0xe1, 0xc6, 0xc7, 0xdd, 0xee, 0x72,
      0x5a, 0x93, 0x6b, 0xa8, 0x14, 0x91, 0x5c, 0x67, 0x62, 0xd2},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {96,
     {0xfc, 0xe0, 0x42, 0x9c, 0x61, 0x06, 0x58, 0xef, 0x8e, 0x7c, 0xfb,
      0x01, 0x54, 0xc5, 0x1d, 0xe2, 0x23, 0x9a, 0x8a, 0x31, 0x7f, 0x5a,
      0xf5, 0xb6, 0x71, 0x4f, 0x98, 0x5f, 0xb5, 0xc4, 0xd7, 0x5c},
     {0x28, 0x73, 0x26, 0xb5, 0xed, 0x00, 0x78, 0xe7, 0xca, 0x01, 0x64, 0xd7,
      0x48, 0xf6, 0x67, 0xe7},
     {0x94, 0x0b, 0x1c, 0x58, 0x0e, 0x0c, 0x72, 0x33, 0xa7, 0x91, 0xb0, 0xf1,
      0x92, 0x43, 0x8d, 0x2e, 0xac, 0xe1, 0x42, 0x14, 0xce, 0xe4, 0x55, 0xb7},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {97,
     {0x0d, 0xda, 0x6d, 0xa5, 0x12, 0x3e, 0x2c, 0x37, 0xc6, 0xfa, 0x16,
      0xba, 0x0d, 0x33, 0x4c, 0xd0, 0x1a, 0xcd, 0x65, 0x2f, 0x89, 0x94,
      0x21, 0x17, 0x51, 0xdf, 0xab, 0x4f, 0xaa, 0xc2, 0xfc, 0x22},
     {0xb4, 0x0b, 0x68, 0x28, 0x72, 0x9b, 0x45, 0x63, 0x22, 0xa8, 0xd0, 0x65,
      0xab, 0xc0, 0xd0, 0x81},
     {0x93, 0x9b, 0x33, 0x89, 0x33, 0x6f, 0xea, 0x4a, 0x97, 0x51, 0xbf, 0x01,
      0x4e, 0xf1, 0x80, 0x11, 0x32, 0x30, 0x90, 0xe8, 0xa0, 0x50, 0x0b, 0xc4},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {98,
     {0xd6, 0x92, 0x59, 0x14, 0xcd, 0x06, 0x30, 0x8f, 0x81, 0xad, 0x91,
      0xe2, 0x30, 0x73, 0x59, 0x3d, 0x99, 0xd4, 0xe5, 0x03, 0x51, 0xb2,
      0x0e, 0xb2, 0xa8, 0xd1, 0xa1, 0xac, 0x4c, 0xed, 0x65, 0x88},
     {0x03, 0x7b, 0x27, 0xb3, 0xdc, 0x95, 0xb1, 0x9d, 0x15, 0xbd, 0x40, 0x91,
      0xe3, 0x20, 0xbf, 0xe1},
     {0x59, 0xee, 0x8e, 0x51, 0x98, 0x86, 0x12, 0x37, 0xf6, 0x82, 0xed, 0xec,
      0x6b, 0xa9, 0x06, 0x52, 0x6c, 0x01, 0x6d, 0x4d, 0x93, 0x59, 0x42, 0xbd},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {99,
     {0x07, 0x51, 0x8a, 0x82, 0xcb, 0xc8, 0xda, 0x1d, 0xce, 0xc5, 0x5f,
      0x37, 0x63, 0xa2, 0x06, 0xd2, 0x77, 0x48, 0x7a, 0xbd, 0x03, 0xce,
      0xdd, 0x0b, 0x8b, 0xef, 0x9e, 0xe2, 0xfb, 0x15, 0x71, 0x21},
     {0xfa, 0xa4, 0x66, 0x4d, 0x79, 0xfc, 0xe3, 0xc7, 0xd2, 0xfd, 0xd4, 0x62,
      0xf6, 0xc1, 0xc4, 0x23, 0xc2, 0xf8, 0xe6, 0xb6, 0x9b, 0xe2, 0xe0, 0x71},
     {0x1a, 0xb5, 0x3a, 0x06, 0x5d, 0x8f, 0x77, 0x6a, 0x08, 0xb3, 0x3e,
      0x51, 0x38, 0x30, 0x71, 0xb6, 0xf1, 0x54, 0x61, 0x21, 0x16, 0x65,
      0x51, 0x37, 0xbd, 0x3b, 0x7e, 0xc2, 0x9b, 0x70, 0xfd, 0x56},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {100,
     {0xea, 0x46, 0x99, 0x1d, 0x4e, 0x71, 0xf5, 0x3d, 0xd6, 0x24, 0xe7,
      0xfe, 0x7f, 0xde, 0x11, 0x94, 0x4a, 0x7c, 0x59, 0x42, 0xd2, 0x32,
      0x36, 0x9b, 0x80, 0x65, 0xd4, 0x2b, 0x8c, 0xd2, 0xdd, 0xe1},
     {0xdf, 0xfc, 0x5c, 0xf1, 0xdd, 0x54, 0x11, 0xd0, 0x15, 0xd8, 0x46, 0x01,
      0xfa, 0x38, 0xdf, 0x5e, 0xff, 0xe8, 0x85, 0xc7, 0xf2, 0x6a, 0x48, 0x25},
     {0xa1, 0xbf, 0x8e, 0x73, 0xe3, 0xfa, 0x1d, 0xb7, 0x59, 0xf0, 0xab,
      0x2a, 0xb0, 0xb1, 0xca, 0x6f, 0x2c, 0x85, 0xb6, 0x3d, 0x83, 0xe2,
      0x5f, 0x7a, 0x0b, 0x52, 0x93, 0xd0, 0xa2, 0x16, 0xa2, 0xb7},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {101,
     {0xfd, 0xcf, 0xa9, 0x02, 0xc6, 0xf2, 0x22, 0xf5, 0x27, 0xaf, 0x84,
      0xda, 0x53, 0x3b, 0x14, 0xb5, 0x2e, 0x26, 0x15, 0xda, 0x3a, 0x89,
      0xd1, 0xd3, 0x57, 0x08, 0xb0, 0xcd, 0x49, 0xf6, 0x0d, 0x87},
     {0x96, 0x6b, 0x07, 0x04, 0x73, 0x54, 0x96, 0x6a, 0x70, 0x3e, 0x79, 0x60,
      0x7b, 0x55, 0x60, 0x32, 0xf4, 0xf5, 0x96, 0xb7, 0xf9, 0x20, 0x6f, 0x05},
     {0x27, 0x30, 0x8a, 0x0e, 0x1a, 0x6c, 0x0a, 0x1d, 0x15, 0xd6, 0x17,
      0x4a, 0xb7, 0xd6, 0x86, 0x75, 0x20, 0x7b, 0x61, 0x5d, 0xf1, 0x6f,
      0xcf, 0x7a, 0x3c, 0x69, 0xb2, 0x5f, 0x55, 0x1c, 0xca, 0x9f},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {102,
     {0x38, 0xe1, 0xb1, 0xd0, 0x75, 0xd9, 0xd8, 0x52, 0xb9, 0xa6, 0xc0,
      0x1c, 0x8f, 0xf6, 0x96, 0x5a, 0xf0, 0x1b, 0xac, 0x45, 0x7a, 0x4e,
      0x33, 0x9a, 0xe3, 0xe1, 0xd7, 0xb2, 0xff, 0xac, 0xc0, 0xcd},
     {0x80, 0xad, 0x68, 0x20, 0xf1, 0xc9, 0x09, 0x81, 0xe2, 0xca, 0x42,
      0xb8, 0x17, 0xa3, 0x45, 0xc1, 0x17, 0x9d, 0x0a, 0x11, 0xd8, 0xe2,
      0x3a, 0x8a, 0xdc, 0x05, 0x05, 0xe1, 0x3d, 0x87, 0x29, 0x5a},
     {0x71, 0x55, 0xee, 0x93, 0x2b, 0x03, 0x58, 0xd9, 0x81, 0x82,
      0xa2, 0x3f, 0x7f, 0x42, 0x7c, 0x77, 0x4a, 0xb3, 0x40, 0xa4,
      0x75, 0x7d, 0x0b, 0x6a, 0x63, 0xfa, 0xcd, 0x3d, 0xe9, 0x05,
      0x78, 0x43, 0x8c, 0xf0, 0x32, 0x01, 0xc3, 0xf8, 0x80, 0x57},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {103,
     {0xc6, 0x41, 0xf1, 0x68, 0x9d, 0x81, 0xca, 0xa8, 0xba, 0x37, 0xd8,
      0x95, 0x27, 0x22, 0x40, 0x66, 0x40, 0x54, 0xed, 0x97, 0x4c, 0xff,
      0xfc, 0x40, 0xe6, 0xc5, 0xc0, 0xca, 0xd1, 0xb9, 0x16, 0xc7},
     {0x3f, 0xd0, 0xba, 0x19, 0x95, 0x5e, 0x46, 0x74, 0x9f, 0x54, 0xd8,
      0x8e, 0x99, 0xd0, 0x80, 0xb7, 0x33, 0x9d, 0x58, 0x8f, 0xe6, 0x12,
      0xec, 0x0f, 0x40, 0x21, 0xca, 0x3c, 0xa2, 0x10, 0x42, 0x70},
     {0xf2, 0x0b, 0x9b, 0x55, 0x3b, 0xc0, 0x84, 0x75, 0x29, 0xf8,
      0xd4, 0x37, 0x9f, 0xa9, 0x09, 0x21, 0x1e, 0x68, 0x24, 0x9b,
      0xd4, 0x29, 0xf4, 0x36, 0xc6, 0x15, 0xc4, 0xc4, 0xa5, 0xd9,
      0xf1, 0xa1, 0x96, 0x8f, 0x0b, 0x89, 0xc5, 0x23, 0x7b, 0x30},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {104,
     {0xaa, 0x0a, 0xb9, 0xd6, 0x8e, 0xd4, 0xa0, 0x4e, 0x72, 0x3f, 0x81,
      0xb4, 0x4c, 0x0c, 0x88, 0xd0, 0xbc, 0xde, 0x7a, 0x80, 0xcf, 0xd4,
      0x76, 0xeb, 0x4b, 0x88, 0x36, 0xd9, 0xaa, 0x01, 0xec, 0x4c},
     {0x57, 0xfa, 0xa8, 0x76, 0x6f, 0x6d, 0x6a, 0x0a, 0xa1, 0xcf, 0x64,
      0x3f, 0x85, 0x7c, 0x15, 0x0d, 0xf5, 0xb3, 0x13, 0x03, 0xb5, 0x0a,
      0xf4, 0x80, 0xe2, 0x1c, 0x4b, 0x5e, 0x8c, 0x8a, 0x15, 0xd5},
     {0x28, 0x11, 0x71, 0x68, 0x54, 0xa2, 0x14, 0xae, 0xcd, 0xd3,
      0x18, 0xc6, 0x67, 0x0b, 0x9b, 0x2a, 0x75, 0x11, 0x71, 0x3c,
      0x9a, 0x0c, 0x0f, 0xa8, 0x05, 0x23, 0x0f, 0xf0, 0x5c, 0xf8,
      0x4a, 0xf7, 0x95, 0xdd, 0x72, 0xf6, 0xc1, 0xa4, 0x45, 0x12},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {105,  // Round counter overflows 256
     {0x31, 0xca, 0xcb, 0xb1, 0x7d, 0x6d, 0xbb, 0xec, 0xae, 0x40, 0x72,
      0x7c, 0x50, 0x48, 0xfe, 0x0c, 0x01, 0xbc, 0x53, 0xb2, 0x3a, 0xb6,
      0x35, 0x02, 0x5c, 0xba, 0xc1, 0xec, 0xf5, 0x2c, 0xa4, 0x95},
     {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
     {0x53, 0x22, 0xbc, 0x62, 0xbd, 0x83, 0x79, 0xab, 0xbb, 0x75, 0xf6, 0x96,
      0x88, 0xb8, 0xf0, 0x0e, 0x31, 0x96, 0x2b, 0x8f, 0x9a, 0xe1, 0xe8, 0x97,
      0x71, 0x95, 0x2d, 0x8a, 0x2a, 0x74, 0x9e, 0x13, 0x52, 0xec, 0x33, 0xb5,
      0x43, 0x5a, 0x67, 0x42, 0x71, 0xb3, 0x14, 0x76, 0x0a, 0xb9, 0xf7, 0x53,
      0xb1, 0x67, 0x26, 0x18, 0x5e, 0xc7, 0xd3, 0x19, 0xac, 0x60, 0x53, 0x13,
      0x55, 0x34, 0x4c, 0x1d, 0x53, 0xc7, 0x90, 0x30, 0x00, 0xa8, 0x03, 0x9e,
      0xb4, 0x0f, 0x70, 0x56, 0x4a, 0x0a, 0xd3, 0xf4, 0x1b, 0x23, 0x55, 0xea,
      0x5d, 0xfa, 0x6a, 0x1d, 0x46, 0x39, 0x2f, 0x1d, 0x10, 0x6a, 0x5d, 0xa5,
      0x69, 0xbf, 0xb0, 0x49, 0x3b, 0xf7, 0xc1, 0x2d, 0xff, 0x04, 0xba, 0xfe,
      0xae, 0x37, 0x7d, 0xf4, 0xbb, 0x47, 0xcd, 0x44, 0x0b, 0x6f, 0x60, 0xfb,
      0xab, 0x2a, 0x54, 0xa9, 0x85, 0x51, 0xb7, 0x62, 0x73, 0xe0, 0x2c, 0xac,
      0x8d, 0x7b, 0xe9, 0xf2, 0x34, 0x3d, 0x3a, 0xbb, 0x2a, 0x23, 0xaf, 0x1c,
      0x91, 0xa7, 0x01, 0x1c, 0x67, 0xa7, 0x90, 0x72, 0x60, 0x11, 0x6b, 0x67,
      0xd5, 0x10, 0xaa, 0xbe, 0x5c, 0x7c, 0xa4, 0x6c, 0x1c, 0x20, 0xf7, 0x71,
      0x06, 0xc4, 0x59, 0x66, 0x58, 0x3b, 0x3e, 0x4e, 0xd7, 0x99, 0xa4, 0x7e,
      0xe1, 0x9b, 0xb4, 0x22, 0x32, 0x09, 0x26, 0x5c, 0x2b, 0x1a, 0xc8, 0x18,
      0x3a, 0x67, 0x8f, 0xf4, 0x3b, 0xc9, 0xa3, 0x47, 0x0a, 0x50, 0xb9, 0xee,
      0x4d, 0x10, 0xf6, 0x06, 0x74, 0x26, 0x8d, 0x72, 0xc6, 0x8b, 0xe0, 0x03,
      0xa0, 0xd9, 0xdd, 0x68, 0x49, 0xab, 0xa8, 0x36, 0xfe, 0x8a, 0x17, 0x92,
      0xb8, 0x1c, 0x90, 0xe1, 0x29, 0x45, 0xbf, 0xe1, 0xf2, 0x7a, 0x9c, 0x6b,
      0x2a, 0xf6, 0x6a, 0x06, 0x3c, 0xdb, 0x77, 0x21, 0xa7, 0xf2, 0x3f, 0xb3,
      0x0a, 0xfd, 0xbd, 0x8b, 0x18, 0xdb, 0x37, 0x7d, 0xd1, 0x69, 0x7f, 0x15,
      0x7b, 0xe8, 0x1c, 0x15, 0x78, 0xd0, 0x8b, 0x06, 0x6c, 0x71, 0xb0, 0xc5,
      0xca, 0x7f, 0xee, 0xca, 0xc1, 0xcd, 0xd9, 0x38, 0xcf, 0x9a, 0xd5, 0x25,
      0xf7, 0x48, 0x44, 0x66, 0x09, 0x08, 0x19, 0x4b, 0x28, 0xe5, 0x4e, 0xdd,
      0xb1, 0x58, 0xf6, 0xe4, 0xc9, 0x21, 0xb8, 0x11, 0xd9, 0x0d, 0xb7, 0x1a,
      0x92, 0xff, 0x87, 0x2d, 0xb2, 0x25, 0x0d, 0xcc, 0x3f, 0x84, 0x7f, 0xc7,
      0x52, 0xb6, 0x6d, 0x77, 0xe3, 0x3c, 0x85, 0x08, 0x4d, 0x3f, 0xa5, 0x3b,
      0x4f, 0x30, 0x50, 0x3d, 0x2a, 0x06, 0xe6, 0xca, 0xbb, 0x90, 0x33, 0xd5,
      0x9f, 0xc1, 0xdc, 0x15, 0xe3, 0x2f, 0x19, 0x84, 0x6a, 0x03, 0x9b, 0x91,
      0x59, 0x7b, 0x0b, 0xa3, 0x14, 0x1a, 0x7d, 0x97, 0x17, 0x63, 0x00, 0x62,
      0x17, 0x01, 0x17, 0xb2, 0x13, 0x1d, 0xab, 0x77, 0x40, 0x6e, 0x9e, 0xd7,
      0xd6, 0x6b, 0x2c, 0x75, 0x4d, 0x84, 0x9c, 0x60},
     {{Action::WRAP,
       {SECSuccess, true}},  // Round counter overflows - acceptable
      {Action::UNWRAP, {SECSuccess, true}}}},

    {106,  // empty keys cannot be wrapped
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7,
      0xa9, 0xa7, 0xda, 0x64, 0x95, 0x39, 0x8f, 0x18, 0xea, 0x0d, 0x8e,
      0xed, 0x76, 0xa5, 0x1a, 0xac, 0x96, 0x03, 0x8a, 0xd6, 0x92},
     {},
     {0xa6, 0xa6, 0xa6, 0xa6, 0xa6, 0xa6, 0xa6, 0xa6},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {107,  // keys of size 8 byte cannot be wrapped
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7,
      0xa9, 0xa7, 0xda, 0x64, 0x95, 0x39, 0x8f, 0x18, 0xea, 0x0d, 0x8e,
      0xed, 0x76, 0xa5, 0x1a, 0xac, 0x96, 0x03, 0x8a, 0xd6, 0x92},
     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07},
     {0x18, 0x1b, 0xa6, 0xa3, 0xa4, 0x39, 0x24, 0x69, 0xe3, 0xde, 0x98, 0xdd,
      0xbb, 0xdd, 0x24, 0x32},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {108,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7,
      0xa9, 0xa7, 0xda, 0x64, 0x95, 0x39, 0x8f, 0x18, 0xea, 0x0d, 0x8e,
      0xed, 0x76, 0xa5, 0x1a, 0xac, 0x96, 0x03, 0x8a, 0xd6, 0x92},
     {0x00},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {109,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7,
      0xa9, 0xa7, 0xda, 0x64, 0x95, 0x39, 0x8f, 0x18, 0xea, 0x0d, 0x8e,
      0xed, 0x76, 0xa5, 0x1a, 0xac, 0x96, 0x03, 0x8a, 0xd6, 0x92},
     {0x00, 0x01},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {110,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7,
      0xa9, 0xa7, 0xda, 0x64, 0x95, 0x39, 0x8f, 0x18, 0xea, 0x0d, 0x8e,
      0xed, 0x76, 0xa5, 0x1a, 0xac, 0x96, 0x03, 0x8a, 0xd6, 0x92},
     {0x00, 0x01, 0x02},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {111,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7,
      0xa9, 0xa7, 0xda, 0x64, 0x95, 0x39, 0x8f, 0x18, 0xea, 0x0d, 0x8e,
      0xed, 0x76, 0xa5, 0x1a, 0xac, 0x96, 0x03, 0x8a, 0xd6, 0x92},
     {0x00, 0x01, 0x02, 0x03},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {112,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7,
      0xa9, 0xa7, 0xda, 0x64, 0x95, 0x39, 0x8f, 0x18, 0xea, 0x0d, 0x8e,
      0xed, 0x76, 0xa5, 0x1a, 0xac, 0x96, 0x03, 0x8a, 0xd6, 0x92},
     {0x00, 0x01, 0x02, 0x03, 0x04},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {113,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7,
      0xa9, 0xa7, 0xda, 0x64, 0x95, 0x39, 0x8f, 0x18, 0xea, 0x0d, 0x8e,
      0xed, 0x76, 0xa5, 0x1a, 0xac, 0x96, 0x03, 0x8a, 0xd6, 0x92},
     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {114,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7,
      0xa9, 0xa7, 0xda, 0x64, 0x95, 0x39, 0x8f, 0x18, 0xea, 0x0d, 0x8e,
      0xed, 0x76, 0xa5, 0x1a, 0xac, 0x96, 0x03, 0x8a, 0xd6, 0x92},
     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {115,  // wrapped key size must be divisible by 8
     {0x57, 0x49, 0x57, 0x15, 0x1f, 0xc2, 0xaf, 0xe0, 0xfa, 0x3d, 0xc7,
      0xa9, 0xa7, 0xda, 0x64, 0x95, 0x39, 0x8f, 0x18, 0xea, 0x0d, 0x8e,
      0xed, 0x76, 0xa5, 0x1a, 0xac, 0x96, 0x03, 0x8a, 0xd6, 0x92},
     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
      0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {116,  // invalid size of wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c,
      0xa4, 0xe8, 0x21, 0x82, 0xf7, 0xe9, 0x92, 0x3a, 0x74, 0x61, 0x10,
      0xfd, 0x97, 0x8e, 0x3b, 0xd2, 0xde, 0xfc, 0x1c, 0x10, 0xd7},
     {},
     {},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {117,  // invalid size of wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c,
      0xa4, 0xe8, 0x21, 0x82, 0xf7, 0xe9, 0x92, 0x3a, 0x74, 0x61, 0x10,
      0xfd, 0x97, 0x8e, 0x3b, 0xd2, 0xde, 0xfc, 0x1c, 0x10, 0xd7},
     {},
     {0x9f},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {118,  // invalid size of wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c,
      0xa4, 0xe8, 0x21, 0x82, 0xf7, 0xe9, 0x92, 0x3a, 0x74, 0x61, 0x10,
      0xfd, 0x97, 0x8e, 0x3b, 0xd2, 0xde, 0xfc, 0x1c, 0x10, 0xd7},
     {},
     {0xdc, 0x9e, 0x95, 0x80},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {119,  // invalid size of wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c,
      0xa4, 0xe8, 0x21, 0x82, 0xf7, 0xe9, 0x92, 0x3a, 0x74, 0x61, 0x10,
      0xfd, 0x97, 0x8e, 0x3b, 0xd2, 0xde, 0xfc, 0x1c, 0x10, 0xd7},
     {},
     {0xb9, 0xb2, 0x82, 0xd1, 0x38, 0x69, 0x30, 0x00},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {120,  // invalid size of wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c,
      0xa4, 0xe8, 0x21, 0x82, 0xf7, 0xe9, 0x92, 0x3a, 0x74, 0x61, 0x10,
      0xfd, 0x97, 0x8e, 0x3b, 0xd2, 0xde, 0xfc, 0x1c, 0x10, 0xd7},
     {},
     {0x0e, 0xfc, 0x63, 0x5b, 0x2d, 0x61, 0xe2, 0x44, 0x05, 0x6b, 0x9d, 0x45,
      0x91, 0xca, 0x6b},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {121,  // invalid size of wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c,
      0xa4, 0xe8, 0x21, 0x82, 0xf7, 0xe9, 0x92, 0x3a, 0x74, 0x61, 0x10,
      0xfd, 0x97, 0x8e, 0x3b, 0xd2, 0xde, 0xfc, 0x1c, 0x10, 0xd7},
     {},
     {0x4a, 0x30, 0x5d, 0xae, 0x08, 0x7b, 0x0d, 0x24, 0xd6, 0x2a, 0xf4, 0x18,
      0x31, 0x33, 0x8f, 0x33, 0xae},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {122,  // invalid size of wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c,
      0xa4, 0xe8, 0x21, 0x82, 0xf7, 0xe9, 0x92, 0x3a, 0x74, 0x61, 0x10,
      0xfd, 0x97, 0x8e, 0x3b, 0xd2, 0xde, 0xfc, 0x1c, 0x10, 0xd7},
     {},
     {0x82, 0xcb, 0x92, 0x70, 0x97, 0xcf, 0x31, 0xea, 0x4a, 0xff,
      0xea, 0x44, 0x0b, 0x0d, 0x8c, 0xa6, 0xa2, 0x40, 0xb9, 0x00},
     {{Action::WRAP, {SECFailure, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {123,  // bytes appended to wrapped key
     {0xfe, 0x60, 0xfc, 0x8d, 0xf7, 0xd9, 0xf4, 0xeb, 0xb5, 0x41, 0x6c,
      0xa4, 0xe8, 0x21, 0x82, 0xf7, 0xe9, 0x92, 0x3a, 0x74, 0x61, 0x10,
      0xfd, 0x97, 0x8e, 0x3b, 0xd2, 0xde, 0xfc, 0x1c, 0x10, 0xd7},
     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
      0x0c, 0x0d, 0x0e, 0x0f},
     {0x7d, 0xfb, 0xd7, 0xcf, 0x61, 0x58, 0xd7, 0x5b, 0xb5,
      0x90, 0x0b, 0x3b, 0xf1, 0xe3, 0x87, 0x10, 0x03, 0x40,
      0x2a, 0x65, 0x08, 0xb1, 0x91, 0x28, 0x00},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {124,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0xa4, 0x17, 0x67, 0x1b, 0xc6, 0x2a, 0x23, 0xc7, 0xa6, 0x55, 0x43, 0x09,
      0x21, 0x24, 0x02, 0x4d, 0xf7, 0x2c, 0x04, 0x8d, 0x8d, 0xb3, 0x30, 0xc7},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {125,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x95, 0x18, 0xd0, 0xf9, 0x9d, 0x7a, 0x73, 0xed, 0x4a, 0x50, 0x2b, 0x44,
      0x9c, 0x14, 0xc2, 0x85, 0x97, 0x1b, 0x0e, 0x61, 0x77, 0xce, 0x0e, 0xca},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {126,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0xf3, 0x51, 0x1f, 0x04, 0x91, 0xbd, 0x74, 0xae, 0x1d, 0xef, 0xb5, 0x30,
      0x7f, 0x0e, 0x18, 0xdb, 0x86, 0x4b, 0x57, 0xb5, 0xc4, 0x04, 0xd4, 0x28},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {127,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x6c, 0x03, 0xce, 0x77, 0x92, 0x59, 0x66, 0x1c, 0x43, 0xd4, 0x1d, 0x5d,
      0x0e, 0x45, 0x68, 0x7f, 0x87, 0x43, 0x53, 0xbb, 0xa5, 0x16, 0xc7, 0x3e},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {128,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x8d, 0xf0, 0x19, 0x69, 0xa1, 0x1c, 0x87, 0x02, 0x65, 0x35, 0xbf, 0xcc,
      0xf7, 0x2b, 0x1d, 0x06, 0x4c, 0x86, 0xec, 0xc7, 0xe5, 0x22, 0x71, 0x57},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {129,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x8e, 0xd1, 0xcd, 0xe2, 0x28, 0xd9, 0xc8, 0xd0, 0x46, 0xdc, 0xa6, 0x5c,
      0x7a, 0x27, 0xae, 0xf2, 0xed, 0xf8, 0xae, 0x90, 0xc7, 0x05, 0xd1, 0xe9},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {130,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0xd6, 0x9b, 0x3e, 0x34, 0xe9, 0xde, 0x38, 0xd4, 0x4d, 0xe1, 0x99, 0x89,
      0x92, 0x36, 0x2a, 0x6f, 0xa1, 0xf6, 0x96, 0xb5, 0xac, 0xab, 0x3f, 0x10},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {131,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x67, 0x86, 0x51, 0x22, 0xaf, 0x32, 0x94, 0xb8, 0xda, 0x05, 0x88, 0x77,
      0x51, 0x25, 0xcb, 0xd6, 0xdc, 0x19, 0xd5, 0xe5, 0xca, 0xb9, 0x7b, 0x6d},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {132,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x9f, 0x0f, 0xa5, 0x23, 0x63, 0xdd, 0x55, 0xdf, 0x47, 0x2d, 0x86, 0x7e,
      0x6f, 0xaf, 0x5d, 0xa8, 0xeb, 0x20, 0x4a, 0x1d, 0x6d, 0x49, 0x70, 0x30},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {133,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0xc3, 0x99, 0xf9, 0x99, 0xc9, 0x6a, 0x42, 0x04, 0x32, 0x5e, 0x7f, 0x08,
      0xd6, 0xa4, 0xde, 0x25, 0x6f, 0xaf, 0x21, 0xec, 0x2c, 0x00, 0x7d, 0xdf},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {134,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0x28, 0x20, 0x82, 0x26, 0x4a, 0x87, 0xdc, 0x35, 0xce, 0x1c, 0xc5, 0xb9,
      0x93, 0x1b, 0x77, 0xd8, 0x0d, 0x82, 0xfc, 0xac, 0xc0, 0x92, 0x7f, 0x85},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {135,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5},
     {0xc1, 0x92, 0xc9, 0x0b, 0x83, 0x00, 0x3c, 0xa9, 0x67, 0x44, 0x49, 0x80,
      0x14, 0xb6, 0xad, 0x6b, 0xed, 0xda, 0x83, 0x79, 0x55, 0x26, 0x98, 0x19},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {136,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0x30, 0xa9, 0x83, 0xcd, 0x9e, 0x69, 0xd5, 0x61, 0xac, 0xc9, 0x5c,
      0x42, 0xb2, 0x52, 0xab, 0xa4, 0x18, 0x5f, 0x83, 0x92, 0xf2, 0xe6,
      0xc9, 0x35, 0xc8, 0xeb, 0x10, 0x5a, 0xf8, 0x08, 0x2e, 0x34},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {137,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0x4d, 0xe9, 0xa6, 0x39, 0xb7, 0x99, 0x63, 0x0b, 0x45, 0xb4, 0x9e,
      0x28, 0xdb, 0xfc, 0x44, 0xda, 0xbb, 0x98, 0x43, 0xee, 0x58, 0x8a,
      0x8c, 0xff, 0x28, 0x6b, 0x8d, 0x5f, 0xbd, 0x7b, 0x32, 0xee},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {138,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0xd9, 0x15, 0xb2, 0xcd, 0xfb, 0x76, 0x9d, 0x9d, 0x82, 0x25, 0x9d,
      0xc3, 0xd1, 0x24, 0x64, 0x6b, 0xbf, 0x97, 0x2b, 0x83, 0xef, 0xd4,
      0xc2, 0xea, 0xe9, 0xb9, 0xf7, 0x51, 0x07, 0x3f, 0x78, 0xd6},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {139,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0x11, 0x7d, 0x65, 0x3f, 0x48, 0x0b, 0x69, 0xfc, 0xe5, 0x64, 0xf1,
      0xfe, 0x99, 0x57, 0x24, 0x92, 0x94, 0x51, 0x89, 0xed, 0x5a, 0xf7,
      0x89, 0xce, 0x05, 0xa2, 0x65, 0x1b, 0xaf, 0x90, 0xbb, 0x5e},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {140,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0x82, 0x26, 0xd0, 0x7a, 0x2f, 0x91, 0x9e, 0x24, 0xad, 0xa1, 0x08,
      0x1c, 0x69, 0xa7, 0x55, 0x20, 0xbe, 0x89, 0x5e, 0x3a, 0x2b, 0xda,
      0x9b, 0x80, 0x5d, 0x97, 0x47, 0x77, 0x3d, 0xde, 0xaa, 0x38},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {141,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0xb0, 0xa7, 0x43, 0x45, 0xbe, 0xdf, 0x88, 0x65, 0x34, 0x8d, 0xaf,
      0x45, 0xd0, 0x54, 0xb9, 0x9c, 0xe5, 0x15, 0xea, 0x8b, 0xe1, 0x36,
      0x27, 0x0d, 0x1c, 0xf7, 0x1e, 0x1c, 0xfa, 0x7a, 0xa4, 0xa2},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {142,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0xa2, 0x61, 0xdb, 0x77, 0xf1, 0x7f, 0x7e, 0xc7, 0x36, 0xd1, 0xa8,
      0xbe, 0x16, 0xe5, 0xf9, 0xae, 0x43, 0x2f, 0xe2, 0xa1, 0x70, 0x12,
      0xe5, 0xa6, 0xf0, 0x7c, 0x54, 0x26, 0xa9, 0xf0, 0xca, 0x59},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {143,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0xc5, 0x3a, 0xcb, 0x5e, 0x09, 0x6b, 0x54, 0x54, 0x8e, 0x13, 0x85,
      0xb2, 0xff, 0x18, 0xea, 0xef, 0x68, 0xd2, 0x35, 0xc9, 0x5b, 0x01,
      0x94, 0xe7, 0x4a, 0x23, 0x83, 0xd3, 0xa7, 0xa5, 0x30, 0xdc},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {144,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0xfa, 0xef, 0x48, 0x2d, 0x99, 0xeb, 0xb1, 0x80, 0xe5, 0xbc, 0x5e,
      0x3c, 0xf7, 0x75, 0xba, 0x29, 0x2c, 0x2a, 0x3b, 0x6c, 0x44, 0xaa,
      0x4a, 0x21, 0xad, 0x40, 0x09, 0x06, 0xf1, 0x1a, 0xf3, 0x92},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {145,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0x73, 0xa4, 0x50, 0xb6, 0x3b, 0x07, 0xb3, 0xae, 0xce, 0x9d, 0x1a,
      0xe5, 0xbf, 0x09, 0x7a, 0x3d, 0xd3, 0xfc, 0xf7, 0x3e, 0x3e, 0xc2,
      0xf1, 0xbd, 0x8f, 0xc3, 0xb5, 0x58, 0x6c, 0xb9, 0xbd, 0x73},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {146,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0xd3, 0x4b, 0x6e, 0xe1, 0x84, 0xd3, 0x87, 0xc9, 0xaa, 0x4b, 0x2d,
      0x18, 0x0a, 0xe0, 0xa8, 0x94, 0x98, 0x01, 0x4e, 0x55, 0xfe, 0x8e,
      0x41, 0x6b, 0xe4, 0xf8, 0x26, 0xfc, 0xf7, 0xd5, 0x65, 0x22},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {147,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84, 0xfa,
      0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b, 0xe4, 0xf1},
     {0x2a, 0xf8, 0x23, 0xd1, 0x60, 0x28, 0x03, 0x74, 0x0b, 0xfa, 0x90,
      0x40, 0xc2, 0xc4, 0xe7, 0x69, 0xa5, 0xb6, 0xde, 0x91, 0x9d, 0x40,
      0x3c, 0xfb, 0xa9, 0xad, 0x36, 0x0f, 0x63, 0xaf, 0x11, 0x13},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {148,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84,
      0xfa, 0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b,
      0xe4, 0xf1, 0xac, 0xce, 0x34, 0x97, 0x35, 0x26, 0x90, 0x8c},
     {0xdd, 0x78, 0xeb, 0xd3, 0x09, 0x1c, 0x55, 0xa5, 0xda, 0x5b,
      0x24, 0x50, 0x42, 0x00, 0xf7, 0xfa, 0xdd, 0x1b, 0x3a, 0xc6,
      0xad, 0x35, 0xf8, 0x14, 0xf7, 0x33, 0xe6, 0x03, 0xc1, 0x39,
      0x36, 0x24, 0x5d, 0x69, 0xd8, 0x3f, 0x26, 0x2f, 0x6b, 0x1e},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {149,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84,
      0xfa, 0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b,
      0xe4, 0xf1, 0xac, 0xce, 0x34, 0x97, 0x35, 0x26, 0x90, 0x8c},
     {0x3d, 0x83, 0x38, 0xea, 0xe7, 0xde, 0x32, 0x23, 0x99, 0xe1,
      0xd1, 0xb4, 0xa3, 0xdf, 0x54, 0x32, 0x6b, 0x24, 0x2b, 0x56,
      0x36, 0x12, 0xea, 0x4b, 0x27, 0xda, 0x22, 0xa0, 0x41, 0xd3,
      0xc8, 0x09, 0x66, 0x91, 0x1b, 0xc0, 0x09, 0x91, 0x17, 0x61},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {150,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84,
      0xfa, 0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b,
      0xe4, 0xf1, 0xac, 0xce, 0x34, 0x97, 0x35, 0x26, 0x90, 0x8c},
     {0xd0, 0x4b, 0xf7, 0x5c, 0xad, 0xd3, 0xb5, 0xf0, 0x99, 0xc3,
      0x4b, 0x27, 0xa9, 0x1e, 0x64, 0xa8, 0xf2, 0xdb, 0xcf, 0x08,
      0xe8, 0xc5, 0xc1, 0xc9, 0xf0, 0x7a, 0x77, 0x7e, 0xeb, 0x80,
      0x5d, 0x5d, 0x0e, 0x8c, 0x5c, 0x01, 0xaf, 0xc4, 0x39, 0x44},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {151,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84,
      0xfa, 0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b,
      0xe4, 0xf1, 0xac, 0xce, 0x34, 0x97, 0x35, 0x26, 0x90, 0x8c},
     {0xbe, 0x0c, 0x5d, 0x19, 0x3b, 0x61, 0xc5, 0x13, 0x7a, 0x8f,
      0xd8, 0xa6, 0xd7, 0xd1, 0xed, 0x8f, 0x0f, 0xa2, 0x8c, 0xec,
      0x51, 0x6f, 0x54, 0x46, 0x97, 0xc1, 0x2a, 0xdd, 0x4f, 0x8f,
      0x4d, 0x5c, 0xfc, 0xa6, 0x5e, 0xde, 0xb1, 0x01, 0x99, 0x74},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {152,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84,
      0xfa, 0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b,
      0xe4, 0xf1, 0xac, 0xce, 0x34, 0x97, 0x35, 0x26, 0x90, 0x8c},
     {0xa8, 0x3e, 0xbc, 0xbe, 0xb2, 0xbe, 0x9d, 0x68, 0x07, 0xb5,
      0xcf, 0xc3, 0x1c, 0x89, 0x84, 0x9d, 0x13, 0x43, 0xdd, 0x4e,
      0xb2, 0x2e, 0x5b, 0xfe, 0x9e, 0x2b, 0x2b, 0x37, 0x90, 0xad,
      0x89, 0x00, 0x60, 0x1f, 0x1f, 0x5d, 0x54, 0xfd, 0x47, 0x2f},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {153,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84,
      0xfa, 0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b,
      0xe4, 0xf1, 0xac, 0xce, 0x34, 0x97, 0x35, 0x26, 0x90, 0x8c},
     {0x41, 0x62, 0x21, 0x48, 0x5a, 0x6c, 0xb9, 0x8a, 0xd1, 0x34,
      0x2e, 0xa9, 0xa1, 0x29, 0x26, 0xa9, 0xa1, 0x33, 0xea, 0xd8,
      0xbd, 0x91, 0x93, 0x23, 0xfe, 0x78, 0x9b, 0xb8, 0xf8, 0x9a,
      0x4f, 0xca, 0xf8, 0x1e, 0x1b, 0xe5, 0x4f, 0x9d, 0x35, 0x8e},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {154,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84,
      0xfa, 0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b,
      0xe4, 0xf1, 0xac, 0xce, 0x34, 0x97, 0x35, 0x26, 0x90, 0x8c},
     {0xbf, 0x6a, 0x53, 0x28, 0x6f, 0xac, 0x48, 0xe7, 0xf2, 0x5d,
      0x89, 0xb7, 0x05, 0x6b, 0x27, 0xaa, 0x91, 0x7d, 0x5b, 0x54,
      0xc0, 0xd3, 0x17, 0x1d, 0xff, 0x36, 0x9f, 0x72, 0x49, 0x15,
      0x3b, 0xf0, 0x9d, 0xa5, 0x89, 0x1e, 0xb4, 0xdc, 0x2d, 0x88},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {155,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84,
      0xfa, 0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b,
      0xe4, 0xf1, 0xac, 0xce, 0x34, 0x97, 0x35, 0x26, 0x90, 0x8c},
     {0x99, 0xd5, 0x17, 0xa1, 0x32, 0x1b, 0xb6, 0x33, 0xb0, 0xd5,
      0xf3, 0xaf, 0xda, 0x23, 0x72, 0xd3, 0xab, 0xf6, 0x8b, 0x41,
      0xd1, 0x3c, 0xbf, 0xdf, 0xfc, 0x78, 0xf1, 0x73, 0xb8, 0x8b,
      0xc4, 0xb9, 0x7e, 0xfc, 0xab, 0x2b, 0x29, 0x04, 0x78, 0x8d},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {156,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84,
      0xfa, 0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b,
      0xe4, 0xf1, 0xac, 0xce, 0x34, 0x97, 0x35, 0x26, 0x90, 0x8c},
     {0xd9, 0x24, 0x56, 0xbc, 0x77, 0xa2, 0x68, 0xef, 0x71, 0xcb,
      0xa7, 0x60, 0x64, 0xa1, 0xb7, 0x72, 0xd1, 0xfe, 0xe2, 0xae,
      0x4f, 0x0e, 0xe3, 0xbb, 0x93, 0x2a, 0x2a, 0xdb, 0x2b, 0x03,
      0x17, 0x96, 0xb9, 0xea, 0xdb, 0x51, 0x75, 0x3f, 0x28, 0x68},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {157,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84,
      0xfa, 0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b,
      0xe4, 0xf1, 0xac, 0xce, 0x34, 0x97, 0x35, 0x26, 0x90, 0x8c},
     {0x5b, 0xb5, 0x46, 0x30, 0xab, 0x8d, 0x73, 0xa0, 0x40, 0xf0,
      0xf8, 0x7e, 0x70, 0xe2, 0x63, 0xd1, 0xae, 0xb2, 0x35, 0x8b,
      0xcd, 0xc0, 0xdc, 0xe6, 0x99, 0x4d, 0x0d, 0x87, 0x44, 0x52,
      0xbb, 0xd8, 0x74, 0x1b, 0x7e, 0xc1, 0xd5, 0x9d, 0x82, 0x98},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {158,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84,
      0xfa, 0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b,
      0xe4, 0xf1, 0xac, 0xce, 0x34, 0x97, 0x35, 0x26, 0x90, 0x8c},
     {0x45, 0x81, 0xd6, 0x53, 0x60, 0x39, 0xdb, 0x1b, 0x23, 0xda,
      0x50, 0xc6, 0x48, 0x77, 0x7e, 0x90, 0xc8, 0x2d, 0x61, 0x28,
      0xbb, 0x92, 0xe2, 0x8b, 0x29, 0x74, 0xba, 0xe1, 0x14, 0x15,
      0x43, 0xa1, 0x9a, 0x15, 0x92, 0xfd, 0xa1, 0xfb, 0xd6, 0x1f},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {159,  // Incorrect IV
     {0x4f, 0x71, 0x0e, 0xb6, 0xb5, 0xe2, 0x87, 0x03, 0xbe, 0xcf, 0xc3,
      0xdc, 0x52, 0xfa, 0x8b, 0xc1, 0xdd, 0x44, 0xa4, 0xa6, 0xd3, 0x8a,
      0x84, 0xb4, 0xf9, 0x4e, 0x89, 0xac, 0x32, 0xd9, 0x87, 0xe7},
     {0xa8, 0x28, 0xcb, 0xda, 0x9b, 0x5f, 0xf0, 0xae, 0x37, 0x4f, 0x84,
      0xfa, 0x01, 0xd0, 0x70, 0xa5, 0xf0, 0xa1, 0x7a, 0x0c, 0x46, 0x2b,
      0xe4, 0xf1, 0xac, 0xce, 0x34, 0x97, 0x35, 0x26, 0x90, 0x8c},
     {0xd3, 0x5b, 0xc6, 0x7e, 0x62, 0x06, 0x4c, 0x34, 0xf4, 0x81,
      0x50, 0x99, 0x9b, 0xa3, 0x0d, 0xed, 0x47, 0x5d, 0x8c, 0x75,
      0x97, 0x8f, 0x45, 0x73, 0x73, 0x20, 0xf2, 0x3e, 0xda, 0xaa,
      0x7a, 0x40, 0xd7, 0x80, 0x3f, 0xc6, 0x1a, 0xdd, 0x34, 0xa4},
     {{Action::WRAP, {SECSuccess, false}},
      {Action::UNWRAP, {SECFailure, false}}}},

    {160,  // RFC 3394
     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a,
      0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15,
      0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f},
     {0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xaa, 0xbb,
      0xcc, 0xdd, 0xee, 0xff},
     {0x64, 0xe8, 0xc3, 0xf9, 0xce, 0x0f, 0x5b, 0xa2, 0x63, 0xe9, 0x77, 0x79,
      0x05, 0x81, 0x8a, 0x2a, 0x93, 0xc8, 0x19, 0x1e, 0x7d, 0x6e, 0x8a, 0xe7},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {161,  // RFC 3394
     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a,
      0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15,
      0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f},
     {0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xaa, 0xbb,
      0xcc, 0xdd, 0xee, 0xff, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07},
     {0xa8, 0xf9, 0xbc, 0x16, 0x12, 0xc6, 0x8b, 0x3f, 0xf6, 0xe6, 0xf4,
      0xfb, 0xe3, 0x0e, 0x71, 0xe4, 0x76, 0x9c, 0x8b, 0x80, 0xa3, 0x2c,
      0xb8, 0x95, 0x8c, 0xd5, 0xd1, 0x7d, 0x6b, 0x25, 0x4d, 0xa1},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},

    {162,  // RFC 3394
     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a,
      0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15,
      0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f},
     {0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xaa,
      0xbb, 0xcc, 0xdd, 0xee, 0xff, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
      0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f},
     {0x28, 0xc9, 0xf4, 0x04, 0xc4, 0xb8, 0x10, 0xf4, 0xcb, 0xcc,
      0xb3, 0x5c, 0xfb, 0x87, 0xf8, 0x26, 0x3f, 0x57, 0x86, 0xe2,
      0xd8, 0x0e, 0xd3, 0x26, 0xcb, 0xc7, 0xf0, 0xe7, 0x1a, 0x99,
      0xf4, 0x3b, 0xfb, 0x98, 0x8b, 0x9b, 0x7a, 0x02, 0xdd, 0x21},
     {{Action::WRAP, {SECSuccess, true}},
      {Action::UNWRAP, {SECSuccess, true}}}},
};
#endif  // kw_vectors_h__