/* Clapper Enhancer MPRIS
 * Copyright (C) 2025 Rafał Dzięgiel <rafostar.github@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, see
 * <https://www.gnu.org/licenses/>.
 */

/**
 * ClapperMpris:
 *
 * An enhancer that adds `MPRIS` support to the player.
 *
 * In order for this enhancer to work with a player instance,
 * application must set "app-id", "own-name", "identity"
 * and "desktop-entry" (this one can be set to %NULL).
 */

#include "config.h"

#include <glib.h>
#include <glib-object.h>
#include <gio/gio.h>
#include <gmodule.h>
#include <libpeas.h>
#include <clapper/clapper.h>

#include <gst/gst.h>
#include <gst/tag/tag.h>

#include "clapper-mpris-gdbus.h"

#define CLAPPER_MPRIS_SECONDS_TO_USECONDS(seconds) ((gint64) (seconds * G_GINT64_CONSTANT (1000000)))
#define CLAPPER_MPRIS_USECONDS_TO_SECONDS(useconds) ((gdouble) useconds / G_GINT64_CONSTANT (1000000))

#define CLAPPER_MPRIS_COMPARE(a,b) (strcmp (a,b) == 0)

#define CLAPPER_MPRIS_NO_TRACK "/org/mpris/MediaPlayer2/TrackList/NoTrack"

#define CLAPPER_MPRIS_PLAYBACK_STATUS_PLAYING "Playing"
#define CLAPPER_MPRIS_PLAYBACK_STATUS_PAUSED "Paused"
#define CLAPPER_MPRIS_PLAYBACK_STATUS_STOPPED "Stopped"

#define CLAPPER_MPRIS_LOOP_NONE "None"
#define CLAPPER_MPRIS_LOOP_TRACK "Track"
#define CLAPPER_MPRIS_LOOP_PLAYLIST "Playlist"

#define DEFAULT_QUEUE_CONTROLLABLE FALSE

/* Compat */
/* FIXME: 1.0: Remove and rename back to ClapperMpris in meson */
#define clapper_mpris_debug      clapper_enhancer_mpris_debug
#define clapper_mpris_get_type   clapper_enhancer_mpris_get_type
#define clapper_mpris_class_init clapper_enhancer_mpris_class_init
#define clapper_mpris_init       clapper_enhancer_mpris_init

#define clapper_mpris_set_queue_controllable clapper_enhancer_mpris_set_queue_controllable
#define clapper_mpris_set_fallback_art_url   clapper_enhancer_mpris_set_fallback_art_url

#define clapper_mpris_media_player2_set_identity                 clapper_enhancer_mpris_media_player2_set_identity
#define clapper_mpris_media_player2_set_desktop_entry            clapper_enhancer_mpris_media_player2_set_desktop_entry
#define clapper_mpris_media_player2_set_supported_mime_types     clapper_enhancer_mpris_media_player2_set_supported_mime_types
#define clapper_mpris_media_player2_set_supported_uri_schemes    clapper_enhancer_mpris_media_player2_set_supported_uri_schemes
#define clapper_mpris_media_player2_set_has_track_list           clapper_enhancer_mpris_media_player2_set_has_track_list

#define clapper_mpris_media_player2_player_set_can_play          clapper_enhancer_mpris_media_player2_player_set_can_play
#define clapper_mpris_media_player2_player_set_can_pause         clapper_enhancer_mpris_media_player2_player_set_can_pause
#define clapper_mpris_media_player2_player_set_can_seek          clapper_enhancer_mpris_media_player2_player_set_can_seek
#define clapper_mpris_media_player2_player_set_can_control       clapper_enhancer_mpris_media_player2_player_set_can_control
#define clapper_mpris_media_player2_player_set_can_go_previous   clapper_enhancer_mpris_media_player2_player_set_can_go_previous
#define clapper_mpris_media_player2_player_set_can_go_next       clapper_enhancer_mpris_media_player2_player_set_can_go_next
#define clapper_mpris_media_player2_player_set_playback_status   clapper_enhancer_mpris_media_player2_player_set_playback_status
#define clapper_mpris_media_player2_player_set_position          clapper_enhancer_mpris_media_player2_player_set_position
#define clapper_mpris_media_player2_player_set_metadata          clapper_enhancer_mpris_media_player2_player_set_metadata
#define clapper_mpris_media_player2_player_set_minimum_rate      clapper_enhancer_mpris_media_player2_player_set_minimum_rate
#define clapper_mpris_media_player2_player_set_maximum_rate      clapper_enhancer_mpris_media_player2_player_set_maximum_rate

#define clapper_mpris_media_player2_player_set_volume            clapper_enhancer_mpris_media_player2_player_set_volume
#define clapper_mpris_media_player2_player_get_volume            clapper_enhancer_mpris_media_player2_player_get_volume

#define clapper_mpris_media_player2_player_set_rate              clapper_enhancer_mpris_media_player2_player_set_rate
#define clapper_mpris_media_player2_player_get_rate              clapper_enhancer_mpris_media_player2_player_get_rate

#define clapper_mpris_media_player2_player_set_loop_status       clapper_enhancer_mpris_media_player2_player_set_loop_status
#define clapper_mpris_media_player2_player_get_loop_status       clapper_enhancer_mpris_media_player2_player_get_loop_status

#define clapper_mpris_media_player2_player_set_shuffle           clapper_enhancer_mpris_media_player2_player_set_shuffle
#define clapper_mpris_media_player2_player_get_shuffle           clapper_enhancer_mpris_media_player2_player_get_shuffle

#define clapper_mpris_media_player2_player_complete_play_pause   clapper_enhancer_mpris_media_player2_player_complete_play_pause
#define clapper_mpris_media_player2_player_complete_play         clapper_enhancer_mpris_media_player2_player_complete_play
#define clapper_mpris_media_player2_player_complete_pause        clapper_enhancer_mpris_media_player2_player_complete_pause
#define clapper_mpris_media_player2_player_complete_stop         clapper_enhancer_mpris_media_player2_player_complete_stop
#define clapper_mpris_media_player2_player_complete_seek         clapper_enhancer_mpris_media_player2_player_complete_seek
#define clapper_mpris_media_player2_player_complete_set_position clapper_enhancer_mpris_media_player2_player_complete_set_position
#define clapper_mpris_media_player2_player_complete_previous     clapper_enhancer_mpris_media_player2_player_complete_previous
#define clapper_mpris_media_player2_player_complete_next         clapper_enhancer_mpris_media_player2_player_complete_next
#define clapper_mpris_media_player2_player_complete_open_uri     clapper_enhancer_mpris_media_player2_player_complete_open_uri

#define clapper_mpris_media_player2_track_list_emit_track_added             clapper_enhancer_mpris_media_player2_track_list_emit_track_added
#define clapper_mpris_media_player2_track_list_emit_track_removed           clapper_enhancer_mpris_media_player2_track_list_emit_track_removed
#define clapper_mpris_media_player2_track_list_emit_track_list_replaced     clapper_enhancer_mpris_media_player2_track_list_emit_track_list_replaced
#define clapper_mpris_media_player2_track_list_emit_track_metadata_changed  clapper_enhancer_mpris_media_player2_track_list_emit_track_metadata_changed

#define clapper_mpris_media_player2_track_list_complete_get_tracks_metadata clapper_enhancer_mpris_media_player2_track_list_complete_get_tracks_metadata
#define clapper_mpris_media_player2_track_list_complete_go_to               clapper_enhancer_mpris_media_player2_track_list_complete_go_to
#define clapper_mpris_media_player2_track_list_complete_add_track           clapper_enhancer_mpris_media_player2_track_list_complete_add_track
#define clapper_mpris_media_player2_track_list_complete_remove_track        clapper_enhancer_mpris_media_player2_track_list_complete_remove_track

#define clapper_mpris_media_player2_track_list_set_tracks                   clapper_enhancer_mpris_media_player2_track_list_set_tracks
#define clapper_mpris_media_player2_track_list_set_can_edit_tracks          clapper_enhancer_mpris_media_player2_track_list_set_can_edit_tracks

#define clapper_mpris_media_player2_skeleton_new            clapper_enhancer_mpris_media_player2_skeleton_new
#define clapper_mpris_media_player2_player_skeleton_new     clapper_enhancer_mpris_media_player2_player_skeleton_new
#define clapper_mpris_media_player2_track_list_skeleton_new clapper_enhancer_mpris_media_player2_track_list_skeleton_new

#define ClapperMprisMediaPlayer2          ClapperEnhancerMprisMediaPlayer2
#define ClapperMprisMediaPlayer2Player    ClapperEnhancerMprisMediaPlayer2Player
#define ClapperMprisMediaPlayer2TrackList ClapperEnhancerMprisMediaPlayer2TrackList
/* Compat End */

#define GST_CAT_DEFAULT clapper_mpris_debug
GST_DEBUG_CATEGORY_STATIC (GST_CAT_DEFAULT);

/* FIXME: 1.0: Uncomment (rename back to ClapperMpris) */
//#define CLAPPER_TYPE_MPRIS (clapper_mpris_get_type())
//#define CLAPPER_MPRIS_CAST(obj) ((ClapperMpris *)(obj))
//G_DECLARE_FINAL_TYPE (ClapperMpris, clapper_mpris, CLAPPER, MPRIS, GstObject);

/* Compat */
/* FIXME: 1.0: Remove (these are defined in feature header) */
#ifndef CLAPPER_TYPE_MPRIS
#define CLAPPER_TYPE_MPRIS (clapper_mpris_get_type())
#endif
#ifndef CLAPPER_MPRIS_CAST
#define CLAPPER_MPRIS_CAST(obj) ((ClapperMpris *)(obj))
#endif
G_DECLARE_FINAL_TYPE (ClapperEnhancerMpris, clapper_enhancer_mpris, CLAPPER, ENHANCER_MPRIS, GstObject);
#define _ClapperMpris _ClapperEnhancerMpris
#define ClapperMpris ClapperEnhancerMpris
#define ClapperMprisClass ClapperEnhancerMprisClass
/* Compat End */

G_MODULE_EXPORT void peas_register_types (PeasObjectModule *module);

typedef struct
{
  gchar *id;
  ClapperMediaItem *item;
  GstSample *art_sample;
  gchar *art_url;
  gboolean art_from_data;
} ClapperMprisTrack;

struct _ClapperMpris
{
  GstObject parent;

  ClapperMprisMediaPlayer2 *base_skeleton;
  ClapperMprisMediaPlayer2Player *player_skeleton;
  ClapperMprisMediaPlayer2TrackList *tracks_skeleton;

  gboolean base_exported;
  gboolean player_exported;
  gboolean tracks_exported;

  guint name_id;
  gboolean registered;

  GMainLoop *loop;

  GPtrArray *tracks;
  ClapperMprisTrack *current_track;

  ClapperQueueProgressionMode default_mode;
  ClapperQueueProgressionMode non_shuffle_mode;

  gchar *app_id;
  gchar *own_name;
  gchar *identity;

  gchar *desktop_entry;
  gboolean desktop_entry_set;

  gboolean queue_controllable;
  gchar *fallback_art_url;
};

enum
{
  PROP_0,
  PROP_APP_ID,
  PROP_OWN_NAME,
  PROP_IDENTITY,
  PROP_DESKTOP_ENTRY,
  PROP_QUEUE_CONTROLLABLE,
  PROP_FALLBACK_ART_URL,
  PROP_LAST
};

static const gchar *const empty_tracklist[] = { NULL, };
static GParamSpec *param_specs[PROP_LAST] = { NULL, };

static ClapperMprisTrack *
clapper_mpris_track_new (ClapperMediaItem *item)
{
  ClapperMprisTrack *track = g_new (ClapperMprisTrack, 1);

  /* MPRIS docs: "Media players may not use any paths starting with /org/mpris
   * unless explicitly allowed by this specification. Such paths are intended to
   * have special meaning, such as /org/mpris/MediaPlayer2/TrackList/NoTrack" */
  track->id = g_strdup_printf ("/org/clapper/MediaItem%u",
      clapper_media_item_get_id (item));

  track->item = gst_object_ref (item);
  track->art_sample = NULL;
  track->art_url = NULL;
  track->art_from_data = FALSE;

  GST_TRACE ("Created track: %s", track->id);

  return track;
}

static inline void
_delete_art_data (ClapperMprisTrack *track)
{
  GFile *file = g_file_new_for_uri (track->art_url);
  g_file_delete (file, NULL, NULL);
  g_object_unref (file);
}

static void
clapper_mpris_track_free (ClapperMprisTrack *track)
{
  GST_TRACE ("Freeing track: %s", track->id);

  /* XXX: Must call before freeing art url */
  if (track->art_from_data)
    _delete_art_data (track);

  g_free (track->id);
  gst_object_unref (track->item);
  gst_clear_sample (&track->art_sample);
  g_free (track->art_url);

  g_free (track);
}

static gboolean
_mpris_find_track_by_item (ClapperMpris *self, ClapperMediaItem *search_item, guint *index)
{
  guint i;

  for (i = 0; i < self->tracks->len; ++i) {
    ClapperMprisTrack *track = (ClapperMprisTrack *) g_ptr_array_index (self->tracks, i);

    if (search_item == track->item) {
      if (index)
        *index = i;

      return TRUE;
    }
  }

  return FALSE;
}

static gboolean
_mpris_find_track_by_id (ClapperMpris *self, const gchar *search_id, guint *index)
{
  guint i;

  for (i = 0; i < self->tracks->len; ++i) {
    ClapperMprisTrack *track = (ClapperMprisTrack *) g_ptr_array_index (self->tracks, i);

    if (CLAPPER_MPRIS_COMPARE (track->id, search_id)) {
      if (index)
        *index = i;

      return TRUE;
    }
  }

  return FALSE;
}

static gchar **
_make_tag_strv (GstTagList *tags, const gchar *tag)
{
  guint n_tags = gst_tag_list_get_tag_size (tags, tag);
  gchar **strv = NULL;

  if (n_tags > 0) {
    GStrvBuilder *builder = g_strv_builder_new ();
    const gchar *value;
    guint i;

    for (i = 0; i < n_tags; ++i) {
      /* Amount of tags was checked, so success is expected */
      if (G_LIKELY (gst_tag_list_peek_string_index (tags, tag, i, &value)))
        g_strv_builder_add (builder, value);
    }

    strv = g_strv_builder_end (builder);
    g_strv_builder_unref (builder);
  }

  return strv;
}

static GstSample *
_get_image_sample (GstTagList *tags, const gchar *tag)
{
  GstSample *sample = NULL;
  guint i, n_tags = gst_tag_list_get_tag_size (tags, tag);

  for (i = 0; i < n_tags; ++i) {
    GstSample *next_sample;
    GstStructure *next_structure;
    const gchar *next_type;

    /* Amount of tags was checked, so success is expected */
    if (G_UNLIKELY (!gst_tag_list_get_sample_index (tags, tag, i, &next_sample)))
      continue;

    next_structure = gst_caps_get_structure (gst_sample_get_caps (next_sample), 0);
    next_type = gst_structure_get_name (next_structure);

    /* Check for compatible media type */
    if (!g_str_has_prefix (next_type, "image/") && strcmp (next_type, "text/uri-list") != 0) {
      gst_sample_unref (next_sample);
      continue;
    }

    /* If no compatible sample yet */
    if (!sample) {
      sample = next_sample;
    } else {
      GstStructure *structure = gst_caps_get_structure (gst_sample_get_caps (sample), 0);
      gint width = 0, height = 0, next_width = 0, next_height = 0;

      gst_structure_get (structure,
          "width", G_TYPE_INT, &width,
          "height", G_TYPE_INT, &height, NULL);
      gst_structure_get (next_structure,
          "width", G_TYPE_INT, &next_width,
          "height", G_TYPE_INT, &next_height, NULL);

      /* Better if higher resolution. Note that in URI case, resolution is usually
       * not present, thus image data is selected (which is what we prefer) */
      if (next_width * next_height > width * height) {
        gst_sample_unref (sample);
        sample = next_sample;
      } else {
        gst_sample_unref (next_sample);
      }
    }
  }

  return sample;
}

static void
_track_take_art_sample (ClapperMpris *self, ClapperMprisTrack *track, GstSample *sample)
{
  GstStructure *structure;
  const gchar *media_type;

  /* Delete outdated artwork data first,
   * then free the art url */
  if (track->art_from_data) {
    _delete_art_data (track);
    track->art_from_data = FALSE;
  }
  g_clear_pointer (&track->art_url, g_free);

  /* Replace stored sample */
  gst_clear_sample (&track->art_sample);
  track->art_sample = sample;

  /* After above cleanup, regenerate art url */
  GST_DEBUG_OBJECT (self, "Unpacking art sample...");

  structure = gst_caps_get_structure (gst_sample_get_caps (track->art_sample), 0);
  media_type = gst_structure_get_name (structure);

  if (g_str_has_prefix (media_type, "image/")) {
    GFile *data_dir;
    GError *error = NULL;

    GST_DEBUG_OBJECT (self, "Sample stores image data");

    /* XXX: When item is moved between queues, item added message may arrive on
     * 2nd bus before removed in the first one. Separate directory for each
     * own name is thus needed, so we do not remove file after its created. */
    data_dir = g_file_new_build_filename (g_get_user_runtime_dir (),
        "app", self->app_id, CLAPPER_API_NAME, "enhancers", "clapper-mpris",
        self->own_name, NULL);

    if (!g_file_make_directory_with_parents (data_dir, NULL, &error)) {
      if (error->domain != G_IO_ERROR || error->code != G_IO_ERROR_EXISTS) {
        GST_ERROR_OBJECT (self, "Failed to create directory for data: %s", error->message);
        g_clear_object (&data_dir);
      }
      g_clear_error (&error);
    }

    /* When dir was present or created */
    if (data_dir) {
      GFile *art_file;
      GOutputStream *ostream;
      gchar name[22]; // 2 * uint + "_" + NULL

      /* Some clients (e.g. GNOME Shell) cache generated artwork even
       * after app is closed, so each file must have an unique name
       * (unless it can be considered to be the exact same media item) */
      g_snprintf (name, sizeof (name), "%u_%u", clapper_media_item_get_id (track->item),
          g_str_hash (clapper_media_item_get_uri (track->item))); // no need to use redirect
      art_file = g_file_get_child (data_dir, name);

      if ((ostream = G_OUTPUT_STREAM (g_file_replace (art_file,
          NULL, FALSE, G_FILE_CREATE_NONE, NULL, &error)))) {
        GstBuffer *buffer = gst_sample_get_buffer (track->art_sample);
        GstMemory *mem = gst_buffer_peek_memory (buffer, 0);
        GstMapInfo map_info;

        if (G_LIKELY (mem && gst_memory_map (mem, &map_info, GST_MAP_READ))) {
          if (g_output_stream_write_all (ostream, map_info.data, map_info.size,
              NULL, NULL, &error)) {
            track->art_url = g_file_get_uri (art_file);
            if (G_LIKELY (track->art_url != NULL))
              track->art_from_data = TRUE;
          } else if (error) {
            GST_ERROR_OBJECT (self, "Could write art image file, reason: %s",
                GST_STR_NULL (error->message));
            g_clear_error (&error);
          }
          gst_memory_unmap (mem, &map_info);
        } else {
          GST_ERROR_OBJECT (self, "Could not map image sample buffer for reading");
        }

        if (G_UNLIKELY (!g_output_stream_close (ostream, NULL, &error))) {
          GST_ERROR_OBJECT (self, "Could not close file output stream, reason: %s",
              GST_STR_NULL (error->message));
          g_clear_error (&error);
        }
        g_object_unref (ostream);
      } else if (error) {
        GST_ERROR_OBJECT (self, "Could not open file for writing, reason: %s",
            GST_STR_NULL (error->message));
        g_clear_error (&error);
      }

      g_object_unref (art_file);
      g_object_unref (data_dir);
    }
  } else if (strcmp (media_type, "text/uri-list") == 0) {
    GstBuffer *buffer = gst_sample_get_buffer (track->art_sample);
    GstMemory *mem = gst_buffer_peek_memory (buffer, 0);
    GstMapInfo map_info;

    if (G_LIKELY (mem && gst_memory_map (mem, &map_info, GST_MAP_READ))) {
      /* Fast path for only single URI (usually) */
      if (!memchr (map_info.data, '\n', map_info.size)
          && !memchr (map_info.data, '\r', map_info.size)) {
        GST_DEBUG_OBJECT (self, "Sample stores single image URI");
        track->art_url = g_strndup ((const gchar *) map_info.data, map_info.size);
      } else {
        gchar **uris;

        GST_DEBUG_OBJECT (self, "Sample stores one or more image URIs");

        /* Safety - check for NULL termination within data */
        if (memchr (map_info.data, '\0', map_info.size)) {
          uris = g_uri_list_extract_uris ((const gchar *) map_info.data);
        } else {
          gchar *text = g_strndup ((const gchar *) map_info.data, map_info.size);
          uris = g_uri_list_extract_uris (text);
          g_free (text);
        }

        if (uris && uris[0])
          track->art_url = g_strdup (uris[0]);

        g_strfreev (uris);
      }
      gst_memory_unmap (mem, &map_info);
    }
  }

  GST_DEBUG_OBJECT (self, "Updated art sample, URI: \"%s\"",
      GST_STR_NULL (track->art_url));
}

static GVariant *
_mpris_build_track_metadata (ClapperMpris *self, ClapperMprisTrack *track)
{
  GVariantBuilder builder;
  GVariant *variant;
  GstTagList *tags;
  GstSample *sample = NULL;
  GstDateTime *dt_val;
  GDate *date_val;
  gchar **strv_val;
  gdouble dbl_val;
  guint uint_val;
  gchar *str_val;

  g_variant_builder_init (&builder, G_VARIANT_TYPE_ARRAY);

  g_variant_builder_add (&builder, "{sv}", "mpris:trackid",
      g_variant_new_string (track->id));
  g_variant_builder_add (&builder, "{sv}", "mpris:length",
      g_variant_new_int64 (CLAPPER_MPRIS_SECONDS_TO_USECONDS (
      clapper_media_item_get_duration (track->item))));
  if ((str_val = clapper_media_item_get_title (track->item))) {
    g_variant_builder_add (&builder, "{sv}", "xesam:title",
        g_variant_new_take_string (str_val));
  }

  if ((str_val = clapper_media_item_get_redirect_uri (track->item))) {
    g_variant_builder_add (&builder, "{sv}", "xesam:url",
        g_variant_new_take_string (str_val));
  } else {
    g_variant_builder_add (&builder, "{sv}", "xesam:url",
        g_variant_new_string (clapper_media_item_get_uri (track->item)));
  }

  tags = clapper_media_item_get_tags (track->item);

  if (gst_tag_list_get_string (tags, GST_TAG_ALBUM, &str_val)) {
    g_variant_builder_add (&builder, "{sv}", "xesam:album",
        g_variant_new_take_string (str_val));
  }
  if ((strv_val = _make_tag_strv (tags, GST_TAG_ALBUM_ARTIST))) {
    g_variant_builder_add (&builder, "{sv}", "xesam:albumArtist",
        g_variant_new_strv ((const gchar * const *) strv_val, -1));
    g_strfreev (strv_val);
  }
  if ((strv_val = _make_tag_strv (tags, GST_TAG_ARTIST))) {
    g_variant_builder_add (&builder, "{sv}", "xesam:artist",
        g_variant_new_strv ((const gchar * const *) strv_val, -1));
    g_strfreev (strv_val);
  }
  if (gst_tag_list_get_string (tags, GST_TAG_LYRICS, &str_val)) {
    g_variant_builder_add (&builder, "{sv}", "xesam:asText",
        g_variant_new_take_string (str_val));
  }
  if (gst_tag_list_get_double (tags, GST_TAG_BEATS_PER_MINUTE, &dbl_val)) {
    g_variant_builder_add (&builder, "{sv}", "xesam:audioBPM",
        g_variant_new_int32 ((gint) dbl_val));
  }
  if ((strv_val = _make_tag_strv (tags, GST_TAG_COMMENT))) {
    g_variant_builder_add (&builder, "{sv}", "xesam:comment",
        g_variant_new_strv ((const gchar * const *) strv_val, -1));
    g_strfreev (strv_val);
  }
  if ((strv_val = _make_tag_strv (tags, GST_TAG_COMPOSER))) {
    g_variant_builder_add (&builder, "{sv}", "xesam:composer",
        g_variant_new_strv ((const gchar * const *) strv_val, -1));
    g_strfreev (strv_val);
  }

  /* Either date with time or date alone can be used for content created,
   * as MPRIS clients are expected to mostly care for a year only. */
  if (gst_tag_list_get_date_time (tags, GST_TAG_DATE_TIME, &dt_val)) {
    if ((str_val = gst_date_time_to_iso8601_string (dt_val))) {
      g_variant_builder_add (&builder, "{sv}", "xesam:contentCreated",
          g_variant_new_take_string (str_val));
    }
    gst_date_time_unref (dt_val);
  } else if (gst_tag_list_get_date (tags, GST_TAG_DATE, &date_val)) {
    gchar buf[11]; // "YYYY-MM-DD" + NULL

    g_date_strftime (buf, sizeof (buf), "%Y-%m-%d", date_val);
    g_variant_builder_add (&builder, "{sv}", "xesam:contentCreated",
        g_variant_new_string (buf));

    g_date_free (date_val);
  }

  if (gst_tag_list_get_uint (tags, GST_TAG_ALBUM_VOLUME_NUMBER, &uint_val)) {
    g_variant_builder_add (&builder, "{sv}", "xesam:discNumber",
        g_variant_new_int32 ((gint) uint_val));
  }
  if ((strv_val = _make_tag_strv (tags, GST_TAG_GENRE))) {
    g_variant_builder_add (&builder, "{sv}", "xesam:genre",
        g_variant_new_strv ((const gchar * const *) strv_val, -1));
    g_strfreev (strv_val);
  }
  if (gst_tag_list_get_uint (tags, GST_TAG_TRACK_NUMBER, &uint_val)) {
    g_variant_builder_add (&builder, "{sv}", "xesam:trackNumber",
        g_variant_new_int32 ((gint) uint_val));
  }
  if (gst_tag_list_get_uint (tags, GST_TAG_USER_RATING, &uint_val)) {
    g_variant_builder_add (&builder, "{sv}", "xesam:userRating",
        g_variant_new_double ((gdouble) uint_val / 100));
  }

  /* When no art url yet, check image tag again. Use preview image as fallback.
   * This refs sample, so we can unref tags and prepare URI afterwards. */
  if (!(sample = _get_image_sample (tags, GST_TAG_IMAGE)))
    sample = _get_image_sample (tags, GST_TAG_PREVIEW_IMAGE);

  gst_tag_list_unref (tags);

  if (sample) {
    if (track->art_sample != sample)
      _track_take_art_sample (self, track, sample);
    else
      gst_sample_unref (sample);
  }

  /* Use track art when available, otherwise user set art fallback */
  if (track->art_url) {
    g_variant_builder_add (&builder, "{sv}", "mpris:artUrl",
        g_variant_new_string (track->art_url));
  } else if (self->fallback_art_url) {
    g_variant_builder_add (&builder, "{sv}", "mpris:artUrl",
        g_variant_new_string (self->fallback_art_url));
  }

  variant = g_variant_builder_end (&builder);

  return variant;
}

static void
clapper_mpris_refresh_current_track (ClapperMpris *self, GVariant *variant)
{
  gboolean is_live = FALSE;

  GST_LOG_OBJECT (self, "Current track refresh");

  /* Set or clear metadata */
  clapper_mpris_media_player2_player_set_metadata (self->player_skeleton, variant);

  /* Properties related to media item availablity, not current state */
  clapper_mpris_media_player2_player_set_can_play (self->player_skeleton, self->current_track != NULL);
  clapper_mpris_media_player2_player_set_can_pause (self->player_skeleton, self->current_track != NULL);

  /* FIXME: Also disable for LIVE content */
  clapper_mpris_media_player2_player_set_can_seek (self->player_skeleton, self->current_track != NULL);
  clapper_mpris_media_player2_player_set_minimum_rate (self->player_skeleton, (is_live) ? 1.0 : G_MINDOUBLE);
  clapper_mpris_media_player2_player_set_maximum_rate (self->player_skeleton, (is_live) ? 1.0 : G_MAXDOUBLE);
}

static void
clapper_mpris_refresh_track (ClapperMpris *self, ClapperMprisTrack *track)
{
  GVariant *variant = g_variant_take_ref (_mpris_build_track_metadata (self, track));

  if (track == self->current_track)
    clapper_mpris_refresh_current_track (self, variant);

  clapper_mpris_media_player2_track_list_emit_track_metadata_changed (self->tracks_skeleton,
      track->id, variant);

  g_variant_unref (variant);
}

static void
clapper_mpris_refresh_all_tracks (ClapperMpris *self)
{
  guint i;

  for (i = 0; i < self->tracks->len; ++i) {
    ClapperMprisTrack *track = (ClapperMprisTrack *) g_ptr_array_index (self->tracks, i);
    clapper_mpris_refresh_track (self, track);
  }
}

static void
clapper_mpris_refresh_track_list (ClapperMpris *self)
{
  GStrvBuilder *builder;
  gchar **tracks_ids;
  guint i;

  GST_LOG_OBJECT (self, "Track list refresh");

  /* Track list is empty */
  if (self->tracks->len == 0) {
    clapper_mpris_media_player2_track_list_set_tracks (self->tracks_skeleton, empty_tracklist);
    return;
  }

  builder = g_strv_builder_new ();

  for (i = 0; i < self->tracks->len; ++i) {
    ClapperMprisTrack *track = (ClapperMprisTrack *) g_ptr_array_index (self->tracks, i);
    g_strv_builder_add (builder, track->id);
  }

  tracks_ids = g_strv_builder_end (builder);
  g_strv_builder_unref (builder);

  clapper_mpris_media_player2_track_list_set_tracks (self->tracks_skeleton, (const gchar *const *) tracks_ids);
  g_strfreev (tracks_ids);
}

static void
clapper_mpris_refresh_can_go_next_previous (ClapperMpris *self)
{
  gboolean can_previous = FALSE, can_next = FALSE;

  GST_LOG_OBJECT (self, "Next/Previous availability refresh");

  if (self->current_track && self->queue_controllable) {
    guint index = 0;

    if (_mpris_find_track_by_item (self, self->current_track->item, &index)) {
      can_previous = (index > 0);
      can_next = (index < self->tracks->len - 1);
    }
  }

  clapper_mpris_media_player2_player_set_can_go_previous (self->player_skeleton, can_previous);
  clapper_mpris_media_player2_player_set_can_go_next (self->player_skeleton, can_next);
}

static void
clapper_mpris_state_changed (ClapperReactable *reactable, ClapperPlayerState state)
{
  ClapperMpris *self = CLAPPER_MPRIS_CAST (reactable);
  const gchar *status_str = CLAPPER_MPRIS_PLAYBACK_STATUS_STOPPED;

  switch (state) {
    case CLAPPER_PLAYER_STATE_PLAYING:
      status_str = CLAPPER_MPRIS_PLAYBACK_STATUS_PLAYING;
      break;
    case CLAPPER_PLAYER_STATE_PAUSED:
    case CLAPPER_PLAYER_STATE_BUFFERING:
      status_str = CLAPPER_MPRIS_PLAYBACK_STATUS_PAUSED;
      break;
    default:
      break;
  }

  GST_DEBUG_OBJECT (self, "Playback status changed to: %s", status_str);
  clapper_mpris_media_player2_player_set_playback_status (self->player_skeleton, status_str);
}

static void
clapper_mpris_position_changed (ClapperReactable *reactable, gdouble position)
{
  ClapperMpris *self = CLAPPER_MPRIS_CAST (reactable);

  GST_LOG_OBJECT (self, "Position changed to: %lf", position);
  clapper_mpris_media_player2_player_set_position (self->player_skeleton,
      CLAPPER_MPRIS_SECONDS_TO_USECONDS (position));
}

static void
clapper_mpris_speed_changed (ClapperReactable *reactable, gdouble speed)
{
  ClapperMpris *self = CLAPPER_MPRIS_CAST (reactable);
  gdouble mpris_speed;

  mpris_speed = clapper_mpris_media_player2_player_get_rate (self->player_skeleton);

  if (!G_APPROX_VALUE (speed, mpris_speed, FLT_EPSILON)) {
    GST_LOG_OBJECT (self, "Speed changed to: %lf", speed);
    clapper_mpris_media_player2_player_set_rate (self->player_skeleton, speed);
  }
}

static void
clapper_mpris_volume_changed (ClapperReactable *reactable, gdouble volume)
{
  ClapperMpris *self = CLAPPER_MPRIS_CAST (reactable);
  gdouble mpris_volume;

  if (G_UNLIKELY (volume < 0))
    volume = 0;

  mpris_volume = clapper_mpris_media_player2_player_get_volume (self->player_skeleton);

  if (!G_APPROX_VALUE (volume, mpris_volume, FLT_EPSILON)) {
    GST_LOG_OBJECT (self, "Volume changed to: %lf", volume);
    clapper_mpris_media_player2_player_set_volume (self->player_skeleton, volume);
  }
}

static void
clapper_mpris_played_item_changed (ClapperReactable *reactable, ClapperMediaItem *item)
{
  ClapperMpris *self = CLAPPER_MPRIS_CAST (reactable);
  GVariant *variant = NULL;
  guint index = 0;

  GST_DEBUG_OBJECT (self, "Played item changed to: %" GST_PTR_FORMAT, item);

  if (G_LIKELY (_mpris_find_track_by_item (self, item, &index))) {
    self->current_track = (ClapperMprisTrack *) g_ptr_array_index (self->tracks, index);
    variant = _mpris_build_track_metadata (self, self->current_track);
  } else {
    self->current_track = NULL;
  }

  clapper_mpris_refresh_current_track (self, variant);
  clapper_mpris_refresh_can_go_next_previous (self);
}

static void
clapper_mpris_item_updated (ClapperReactable *reactable, ClapperMediaItem *item, ClapperReactableItemUpdatedFlags flags)
{
  ClapperMpris *self = CLAPPER_MPRIS_CAST (reactable);
  guint index = 0;

  GST_LOG_OBJECT (self, "Item updated: %" GST_PTR_FORMAT ", flags: %u", item, flags);

  /* Ignore if only timeline was updated */
  if (flags == CLAPPER_REACTABLE_ITEM_UPDATED_TIMELINE)
    return;

  if (_mpris_find_track_by_item (self, item, &index)) {
    ClapperMprisTrack *track = (ClapperMprisTrack *) g_ptr_array_index (self->tracks, index);
    clapper_mpris_refresh_track (self, track);
  }
}

static void
clapper_mpris_queue_item_added (ClapperReactable *reactable, ClapperMediaItem *item, guint index)
{
  ClapperMpris *self = CLAPPER_MPRIS_CAST (reactable);
  ClapperMprisTrack *track, *prev_track = NULL;
  GVariant *variant;

  /* Safety precaution for a case when someone adds MPRIS feature
   * in middle of altering playlist from another thread, since we
   * also read initial playlist after name is acquired. */
  if (G_UNLIKELY (_mpris_find_track_by_item (self, item, NULL)))
    return;

  GST_DEBUG_OBJECT (self, "Queue item added at position: %u", index);

  track = clapper_mpris_track_new (item);
  g_ptr_array_insert (self->tracks, index, track);

  clapper_mpris_refresh_track_list (self);
  clapper_mpris_refresh_can_go_next_previous (self);

  variant = g_variant_take_ref (_mpris_build_track_metadata (self, track));

  /* NoTrack when item is added at first position in queue */
  clapper_mpris_media_player2_track_list_emit_track_added (self->tracks_skeleton,
      variant, (prev_track != NULL) ? prev_track->id : CLAPPER_MPRIS_NO_TRACK);
  g_variant_unref (variant);
}

static void
clapper_mpris_queue_item_removed (ClapperReactable *reactable, ClapperMediaItem *item, guint index)
{
  ClapperMpris *self = CLAPPER_MPRIS_CAST (reactable);
  ClapperMprisTrack *track;

  GST_DEBUG_OBJECT (self, "Queue item removed");

  track = (ClapperMprisTrack *) g_ptr_array_steal_index (self->tracks, index);

  if (track == self->current_track) {
    self->current_track = NULL;
    clapper_mpris_refresh_current_track (self, NULL);
  }

  clapper_mpris_refresh_track_list (self);
  clapper_mpris_refresh_can_go_next_previous (self);
  clapper_mpris_media_player2_track_list_emit_track_removed (self->tracks_skeleton, track->id);

  clapper_mpris_track_free (track);
}

static void
clapper_mpris_queue_item_repositioned (ClapperReactable *reactable, guint before, guint after)
{
  ClapperMpris *self = CLAPPER_MPRIS_CAST (reactable);
  ClapperMprisTrack *track;

  GST_DEBUG_OBJECT (self, "Queue item repositioned: %u -> %u", before, after);

  track = (ClapperMprisTrack *) g_ptr_array_steal_index (self->tracks, before);
  g_ptr_array_insert (self->tracks, after, track);

  clapper_mpris_refresh_track_list (self);
  clapper_mpris_refresh_can_go_next_previous (self);
}

static void
clapper_mpris_queue_cleared (ClapperReactable *reactable)
{
  ClapperMpris *self = CLAPPER_MPRIS_CAST (reactable);
  guint n_items = self->tracks->len;

  if (n_items > 0)
    g_ptr_array_remove_range (self->tracks, 0, n_items);

  self->current_track = NULL;
  clapper_mpris_refresh_current_track (self, NULL);
  clapper_mpris_refresh_can_go_next_previous (self);
  clapper_mpris_refresh_track_list (self);

  clapper_mpris_media_player2_track_list_emit_track_list_replaced (self->tracks_skeleton,
      empty_tracklist, CLAPPER_MPRIS_NO_TRACK);
}

static void
clapper_mpris_queue_progression_changed (ClapperReactable *reactable, ClapperQueueProgressionMode mode)
{
  ClapperMpris *self = CLAPPER_MPRIS_CAST (reactable);
  const gchar *loop_status = CLAPPER_MPRIS_LOOP_NONE;
  gboolean shuffle = FALSE;

  GST_DEBUG_OBJECT (self, "Queue progression changed to: %i", mode);

  switch (mode) {
    case CLAPPER_QUEUE_PROGRESSION_REPEAT_ITEM:
      loop_status = CLAPPER_MPRIS_LOOP_TRACK;
      break;
    case CLAPPER_QUEUE_PROGRESSION_CAROUSEL:
      loop_status = CLAPPER_MPRIS_LOOP_PLAYLIST;
      break;
    case CLAPPER_QUEUE_PROGRESSION_SHUFFLE:
      shuffle = TRUE;
      break;
    case CLAPPER_QUEUE_PROGRESSION_NONE:
    case CLAPPER_QUEUE_PROGRESSION_CONSECUTIVE:
      self->default_mode = mode;
      break;
    default:
      break;
  }

  if (mode != CLAPPER_QUEUE_PROGRESSION_SHUFFLE)
    self->non_shuffle_mode = mode;

  clapper_mpris_media_player2_player_set_loop_status (self->player_skeleton, loop_status);
  clapper_mpris_media_player2_player_set_shuffle (self->player_skeleton, shuffle);
}

static void
clapper_mpris_reactable_iface_init (ClapperReactableInterface *iface)
{
  iface->state_changed = clapper_mpris_state_changed;
  iface->position_changed = clapper_mpris_position_changed;
  iface->speed_changed = clapper_mpris_speed_changed;
  iface->volume_changed = clapper_mpris_volume_changed;
  iface->played_item_changed = clapper_mpris_played_item_changed;
  iface->item_updated = clapper_mpris_item_updated;
  iface->queue_item_added = clapper_mpris_queue_item_added;
  iface->queue_item_removed = clapper_mpris_queue_item_removed;
  iface->queue_item_repositioned = clapper_mpris_queue_item_repositioned;
  iface->queue_cleared = clapper_mpris_queue_cleared;
  iface->queue_progression_changed = clapper_mpris_queue_progression_changed;
}

/* Compat */
/* FIXME: 1.0: Rename back to ClapperMpris */
#define parent_class clapper_enhancer_mpris_parent_class
G_DEFINE_TYPE_WITH_CODE (ClapperEnhancerMpris, clapper_enhancer_mpris, GST_TYPE_OBJECT,
    G_IMPLEMENT_INTERFACE (CLAPPER_TYPE_REACTABLE, clapper_mpris_reactable_iface_init));
/* Compat End */

static gchar **
_filter_names (const gchar *const *all_names)
{
  GStrvBuilder *builder;
  gchar **filtered_names;
  guint i;

  builder = g_strv_builder_new ();

  for (i = 0; all_names[i]; ++i) {
    const gchar *const *remaining_names = all_names + i + 1;

    if (*remaining_names && g_strv_contains (remaining_names, all_names[i]))
      continue;

    GST_LOG ("Found: %s", all_names[i]);
    g_strv_builder_add (builder, all_names[i]);
  }

  filtered_names = g_strv_builder_end (builder);
  g_strv_builder_unref (builder);

  return filtered_names;
}

static gchar **
clapper_mpris_get_supported_uri_schemes (ClapperMpris *self)
{
  GStrvBuilder *builder;
  gchar **all_schemes, **filtered_schemes;
  GList *elements, *el;
  guint i;

  GST_DEBUG_OBJECT (self, "Checking supported URI schemes");

  builder = g_strv_builder_new ();
  elements = gst_element_factory_list_get_elements (
      GST_ELEMENT_FACTORY_TYPE_SRC, GST_RANK_NONE);

  for (el = elements; el != NULL; el = el->next) {
    GstElementFactory *factory = GST_ELEMENT_FACTORY (el->data);
    const gchar *const *protocols;

    if (gst_element_factory_get_uri_type (factory) != GST_URI_SRC)
      continue;

    if (!(protocols = gst_element_factory_get_uri_protocols (factory)))
      continue;

    for (i = 0; protocols[i]; ++i)
      g_strv_builder_add (builder, protocols[i]);
  }

  all_schemes = g_strv_builder_end (builder);
  g_strv_builder_unref (builder);
  gst_plugin_feature_list_free (elements);

  filtered_schemes = _filter_names ((const gchar *const *) all_schemes);
  g_strfreev (all_schemes);

  return filtered_schemes;
}

static gchar **
clapper_mpris_get_supported_mime_types (ClapperMpris *self)
{
  GStrvBuilder *builder;
  gchar **all_types, **filtered_types;
  GList *elements, *el;

  GST_DEBUG_OBJECT (self, "Checking supported mime-types");

  builder = g_strv_builder_new ();
  elements = gst_element_factory_list_get_elements (
      GST_ELEMENT_FACTORY_TYPE_DEMUXER, GST_RANK_NONE);

  for (el = elements; el != NULL; el = el->next) {
    GstElementFactory *factory = GST_ELEMENT_FACTORY (el->data);
    const GList *pad_templates, *pt;

    pad_templates = gst_element_factory_get_static_pad_templates (factory);

    for (pt = pad_templates; pt != NULL; pt = pt->next) {
      GstStaticPadTemplate *template = (GstStaticPadTemplate *) pt->data;
      GstCaps *caps;
      guint i, size;

      if (template->direction != GST_PAD_SINK)
        continue;

      caps = gst_static_pad_template_get_caps (template);
      size = gst_caps_get_size (caps);

      for (i = 0; i < size; ++i) {
        GstStructure *structure = gst_caps_get_structure (caps, i);
        const gchar *name = gst_structure_get_name (structure);

        /* Skip GStreamer internal mime types */
        if (g_str_has_prefix (name, "application/x-gst-"))
          continue;

        /* GStreamer uses "video/quicktime" for MP4. If we can
         * handle it, then also add more generic ones. */
        if (strcmp (name, "video/quicktime") == 0) {
          g_strv_builder_add (builder, "video/mp4");
          g_strv_builder_add (builder, "audio/mp4");
        }

        g_strv_builder_add (builder, name);
      }

      gst_caps_unref (caps);
    }
  }

  all_types = g_strv_builder_end (builder);
  g_strv_builder_unref (builder);
  gst_plugin_feature_list_free (elements);

  filtered_types = _filter_names ((const gchar *const *) all_types);
  g_strfreev (all_types);

  return filtered_types;
}

static void
clapper_mpris_unregister (ClapperMpris *self)
{
  if (self->base_exported) {
    g_dbus_interface_skeleton_unexport (G_DBUS_INTERFACE_SKELETON (self->base_skeleton));
    self->base_exported = FALSE;
  }
  if (self->player_exported) {
    g_dbus_interface_skeleton_unexport (G_DBUS_INTERFACE_SKELETON (self->player_skeleton));
    self->player_exported = FALSE;
  }
  if (self->tracks_exported) {
    g_dbus_interface_skeleton_unexport (G_DBUS_INTERFACE_SKELETON (self->tracks_skeleton));
    self->tracks_exported = FALSE;
  }
  if (self->name_id > 0) {
    g_bus_unown_name (self->name_id);
    self->name_id = 0;
  }
  if (self->registered) {
    GST_DEBUG_OBJECT (self, "Unregistered");
    self->registered = FALSE;
  }
}

static gboolean
_handle_open_uri_cb (ClapperMprisMediaPlayer2Player *player_skeleton,
    GDBusMethodInvocation *invocation, const gchar *uri, ClapperMpris *self)
{
  ClapperPlayer *player;

  if (!self->queue_controllable)
    return G_DBUS_METHOD_INVOCATION_UNHANDLED;

  GST_DEBUG_OBJECT (self, "Handle open URI: %s", uri);

  if ((player = clapper_reactable_get_player (CLAPPER_REACTABLE_CAST (self)))) {
    ClapperQueue *queue = clapper_player_get_queue (player);
    ClapperMediaItem *item = clapper_media_item_new (uri);

    /* We can only alter ClapperQueue from main thread.
     * Adding items to it will trigger clapper_mpris_queue_item_added(),
     * then we will add this new item to our track list */
    clapper_reactable_queue_append_sync (CLAPPER_REACTABLE_CAST (self), item);

    if (clapper_queue_select_item (queue, item))
      clapper_player_play (player);

    gst_object_unref (item);
    gst_object_unref (player);
  }

  clapper_mpris_media_player2_player_complete_open_uri (player_skeleton, invocation);

  return G_DBUS_METHOD_INVOCATION_HANDLED;
}

static gboolean
_handle_play_cb (ClapperMprisMediaPlayer2Player *player_skeleton,
    GDBusMethodInvocation *invocation, ClapperMpris *self)
{
  ClapperPlayer *player;

  GST_DEBUG_OBJECT (self, "Handle play");

  if ((player = clapper_reactable_get_player (CLAPPER_REACTABLE_CAST (self)))) {
    clapper_player_play (player);
    gst_object_unref (player);
  }

  clapper_mpris_media_player2_player_complete_play (player_skeleton, invocation);

  return G_DBUS_METHOD_INVOCATION_HANDLED;
}

static gboolean
_handle_pause_cb (ClapperMprisMediaPlayer2Player *player_skeleton,
    GDBusMethodInvocation *invocation, ClapperMpris *self)
{
  ClapperPlayer *player;

  GST_DEBUG_OBJECT (self, "Handle pause");

  if ((player = clapper_reactable_get_player (CLAPPER_REACTABLE_CAST (self)))) {
    clapper_player_pause (player);
    gst_object_unref (player);
  }

  clapper_mpris_media_player2_player_complete_pause (player_skeleton, invocation);

  return G_DBUS_METHOD_INVOCATION_HANDLED;
}

static gboolean
_handle_play_pause_cb (ClapperMprisMediaPlayer2Player *player_skeleton,
    GDBusMethodInvocation *invocation, ClapperMpris *self)
{
  ClapperPlayer *player;

  GST_DEBUG_OBJECT (self, "Handle play/pause");

  if ((player = clapper_reactable_get_player (CLAPPER_REACTABLE_CAST (self)))) {
    ClapperPlayerState state = clapper_player_get_state (player);

    switch (state) {
      case CLAPPER_PLAYER_STATE_PLAYING:
        clapper_player_pause (player);
        break;
      case CLAPPER_PLAYER_STATE_PAUSED:
      case CLAPPER_PLAYER_STATE_STOPPED:
        clapper_player_play (player);
        break;
      default:
        break;
    }

    gst_object_unref (player);
  }

  clapper_mpris_media_player2_player_complete_play_pause (player_skeleton, invocation);

  return G_DBUS_METHOD_INVOCATION_HANDLED;
}

static gboolean
_handle_stop_cb (ClapperMprisMediaPlayer2Player *player_skeleton,
    GDBusMethodInvocation *invocation, ClapperMpris *self)
{
  ClapperPlayer *player;

  GST_DEBUG_OBJECT (self, "Handle stop");

  if ((player = clapper_reactable_get_player (CLAPPER_REACTABLE_CAST (self)))) {
    clapper_player_stop (player);
    gst_object_unref (player);
  }

  clapper_mpris_media_player2_player_complete_stop (player_skeleton, invocation);

  return G_DBUS_METHOD_INVOCATION_HANDLED;
}

static gboolean
_handle_next_cb (ClapperMprisMediaPlayer2Player *player_skeleton,
    GDBusMethodInvocation *invocation, ClapperMpris *self)
{
  ClapperPlayer *player;

  if (!self->queue_controllable)
    return G_DBUS_METHOD_INVOCATION_UNHANDLED;

  GST_DEBUG_OBJECT (self, "Handle next");

  if ((player = clapper_reactable_get_player (CLAPPER_REACTABLE_CAST (self)))) {
    ClapperQueue *queue = clapper_player_get_queue (player);
    clapper_queue_select_next_item (queue);
    gst_object_unref (player);
  }

  clapper_mpris_media_player2_player_complete_next (player_skeleton, invocation);

  return G_DBUS_METHOD_INVOCATION_HANDLED;
}

static gboolean
_handle_previous_cb (ClapperMprisMediaPlayer2Player *player_skeleton,
    GDBusMethodInvocation *invocation, ClapperMpris *self)
{
  ClapperPlayer *player;

  if (!self->queue_controllable)
    return G_DBUS_METHOD_INVOCATION_UNHANDLED;

  GST_DEBUG_OBJECT (self, "Handle previous");

  if ((player = clapper_reactable_get_player (CLAPPER_REACTABLE_CAST (self)))) {
    ClapperQueue *queue = clapper_player_get_queue (player);
    clapper_queue_select_previous_item (queue);
    gst_object_unref (player);
  }

  clapper_mpris_media_player2_player_complete_previous (player_skeleton, invocation);

  return G_DBUS_METHOD_INVOCATION_HANDLED;
}

static gboolean
_handle_seek_cb (ClapperMprisMediaPlayer2Player *player_skeleton,
    GDBusMethodInvocation *invocation, gint64 offset, ClapperMpris *self)
{
  ClapperPlayer *player;

  GST_DEBUG_OBJECT (self, "Handle seek");

  if (!self->current_track)
    goto finish;

  if ((player = clapper_reactable_get_player (CLAPPER_REACTABLE_CAST (self)))) {
    gdouble position, seek_position;

    position = clapper_player_get_position (player);
    seek_position = position + CLAPPER_MPRIS_USECONDS_TO_SECONDS (offset);

    if (seek_position <= 0) {
      clapper_player_seek (player, 0);
    } else {
      gdouble duration = clapper_media_item_get_duration (self->current_track->item);

      if (seek_position > duration) {
        ClapperQueue *queue = clapper_player_get_queue (player);
        clapper_queue_select_next_item (queue);
      } else {
        clapper_player_seek (player, seek_position);
      }
    }

    gst_object_unref (player);
  }

finish:
  clapper_mpris_media_player2_player_complete_seek (player_skeleton, invocation);

  return G_DBUS_METHOD_INVOCATION_HANDLED;
}

static gboolean
_handle_set_position_cb (ClapperMprisMediaPlayer2Player *player_skeleton,
    GDBusMethodInvocation *invocation, const gchar *track_id,
    gint64 position, ClapperMpris *self)
{
  ClapperPlayer *player;

  GST_DEBUG_OBJECT (self, "Handle set position");

  if (G_UNLIKELY (position < 0) || !self->current_track)
    goto finish;

  if ((player = clapper_reactable_get_player (CLAPPER_REACTABLE_CAST (self)))) {
    gdouble duration, position_dbl;

    duration = clapper_media_item_get_duration (self->current_track->item);
    position_dbl = CLAPPER_MPRIS_USECONDS_TO_SECONDS (position);

    if (position_dbl <= duration)
      clapper_player_seek (player, position_dbl);

    gst_object_unref (player);
  }

finish:
  clapper_mpris_media_player2_player_complete_set_position (player_skeleton, invocation);

  return G_DBUS_METHOD_INVOCATION_HANDLED;
}

static void
_handle_rate_notify_cb (ClapperMprisMediaPlayer2Player *player_skeleton,
    GParamSpec *pspec G_GNUC_UNUSED, ClapperMpris *self)
{
  ClapperPlayer *player;

  GST_DEBUG_OBJECT (self, "Handle rate notify");

  if ((player = clapper_reactable_get_player (CLAPPER_REACTABLE_CAST (self)))) {
    gdouble speed, player_speed;

    speed = clapper_mpris_media_player2_player_get_rate (player_skeleton);
    player_speed = clapper_player_get_speed (player);

    if (!G_APPROX_VALUE (speed, player_speed, FLT_EPSILON))
      clapper_player_set_speed (player, speed);

    gst_object_unref (player);
  }
}

static void
_handle_volume_notify_cb (ClapperMprisMediaPlayer2Player *player_skeleton,
    GParamSpec *pspec G_GNUC_UNUSED, ClapperMpris *self)
{
  ClapperPlayer *player;

  GST_DEBUG_OBJECT (self, "Handle volume notify");

  if ((player = clapper_reactable_get_player (CLAPPER_REACTABLE_CAST (self)))) {
    gdouble volume, player_volume;

    volume = clapper_mpris_media_player2_player_get_volume (player_skeleton);
    player_volume = clapper_player_get_volume (player);

    if (!G_APPROX_VALUE (volume, player_volume, FLT_EPSILON))
      clapper_player_set_volume (player, volume);

    gst_object_unref (player);
  }
}

static void
_handle_loop_status_notify_cb (ClapperMprisMediaPlayer2Player *player_skeleton,
    GParamSpec *pspec G_GNUC_UNUSED, ClapperMpris *self)
{
  ClapperPlayer *player;

  GST_DEBUG_OBJECT (self, "Handle loop status notify");

  if ((player = clapper_reactable_get_player (CLAPPER_REACTABLE_CAST (self)))) {
    ClapperQueue *queue = clapper_player_get_queue (player);
    ClapperQueueProgressionMode mode, player_mode;
    const gchar *loop_status;

    loop_status = clapper_mpris_media_player2_player_get_loop_status (player_skeleton);
    player_mode = clapper_queue_get_progression_mode (queue);

    /* When in shuffle and no loop, assume default mode (none or consecutive).
     * This prevents us from getting stuck constantly changing loop and shuffle. */
    if (player_mode == CLAPPER_QUEUE_PROGRESSION_SHUFFLE)
      player_mode = self->default_mode;

    mode = CLAPPER_MPRIS_COMPARE (loop_status, CLAPPER_MPRIS_LOOP_TRACK)
        ? CLAPPER_QUEUE_PROGRESSION_REPEAT_ITEM
        : CLAPPER_MPRIS_COMPARE (loop_status, CLAPPER_MPRIS_LOOP_PLAYLIST)
        ? CLAPPER_QUEUE_PROGRESSION_CAROUSEL
        : self->default_mode;

    if (mode != player_mode)
      clapper_queue_set_progression_mode (queue, mode);

    gst_object_unref (player);
  }
}

static void
_handle_shuffle_notify_cb (ClapperMprisMediaPlayer2Player *player_skeleton,
    GParamSpec *pspec G_GNUC_UNUSED, ClapperMpris *self)
{
  ClapperPlayer *player;

  GST_DEBUG_OBJECT (self, "Handle shuffle notify");

  if ((player = clapper_reactable_get_player (CLAPPER_REACTABLE_CAST (self)))) {
    ClapperQueue *queue = clapper_player_get_queue (player);
    ClapperQueueProgressionMode player_mode;
    gboolean shuffle, player_shuffle;

    player_mode = clapper_queue_get_progression_mode (queue);

    shuffle = clapper_mpris_media_player2_player_get_shuffle (player_skeleton);
    player_shuffle = (player_mode == CLAPPER_QUEUE_PROGRESSION_SHUFFLE);

    if (shuffle != player_shuffle) {
      clapper_queue_set_progression_mode (queue,
          (shuffle) ? CLAPPER_QUEUE_PROGRESSION_SHUFFLE : self->non_shuffle_mode);
    }

    gst_object_unref (player);
  }
}

static gboolean
_handle_get_tracks_metadata_cb (ClapperMprisMediaPlayer2TrackList *tracks_skeleton,
    GDBusMethodInvocation *invocation, const gchar *const *tracks_ids, ClapperMpris *self)
{
  GVariantBuilder builder;
  GVariant *tracks_variant = NULL;
  gboolean initialized = FALSE;
  guint i;

  GST_DEBUG_OBJECT (self, "Handle get tracks metadata");

  for (i = 0; tracks_ids[i]; ++i) {
    guint index = 0;

    if (_mpris_find_track_by_id (self, tracks_ids[i], &index)) {
      ClapperMprisTrack *track = (ClapperMprisTrack *) g_ptr_array_index (self->tracks, index);
      GVariant *variant = _mpris_build_track_metadata (self, track);

      if (!initialized) {
        g_variant_builder_init (&builder, G_VARIANT_TYPE_ARRAY);
        initialized = TRUE;
      }

      g_variant_builder_add_value (&builder, variant);
    }
  }

  if (initialized)
    tracks_variant = g_variant_builder_end (&builder);

  clapper_mpris_media_player2_track_list_complete_get_tracks_metadata (tracks_skeleton,
      invocation, tracks_variant);

  return G_DBUS_METHOD_INVOCATION_HANDLED;
}

static gboolean
_handle_add_track_cb (ClapperMprisMediaPlayer2TrackList *tracks_skeleton,
    GDBusMethodInvocation *invocation, const gchar *uri,
    const gchar *after_track, gboolean set_current, ClapperMpris *self)
{
  ClapperMediaItem *after_item = NULL;
  gboolean add;

  if (!self->queue_controllable)
    return G_DBUS_METHOD_INVOCATION_UNHANDLED;

  GST_DEBUG_OBJECT (self, "Handle add track, URI: %s, after_track: %s,"
      " set_current: %s", uri, after_track, set_current ? "yes" : "no");

  if ((add = CLAPPER_MPRIS_COMPARE (after_track, CLAPPER_MPRIS_NO_TRACK))) {
    GST_DEBUG_OBJECT (self, "Prepend, since requested after \"NoTrack\"");
  } else {
    guint index = 0;

    if ((add = _mpris_find_track_by_id (self, after_track, &index))) {
      ClapperMprisTrack *track = (ClapperMprisTrack *) g_ptr_array_index (self->tracks, index);

      GST_DEBUG_OBJECT (self, "Append after: %s", track->id);
      after_item = track->item;
    }
  }

  if (add) {
    ClapperReactable *reactable = CLAPPER_REACTABLE_CAST (self);
    ClapperMediaItem *item = clapper_media_item_new (uri);

    clapper_reactable_queue_insert_sync (reactable, item, after_item);

    if (set_current) {
      ClapperPlayer *player;

      if ((player = clapper_reactable_get_player (reactable))) {
        ClapperQueue *queue = clapper_player_get_queue (player);

        if (clapper_queue_select_item (queue, item))
          clapper_player_play (player);

        gst_object_unref (player);
      }
    }

    gst_object_unref (item);
  }

  clapper_mpris_media_player2_track_list_complete_add_track (tracks_skeleton, invocation);

  return G_DBUS_METHOD_INVOCATION_HANDLED;
}

static gboolean
_handle_remove_track_cb (ClapperMprisMediaPlayer2TrackList *tracks_skeleton,
    GDBusMethodInvocation *invocation, const gchar *track_id, ClapperMpris *self)
{
  ClapperMprisTrack *track;
  guint index = 0;

  if (!self->queue_controllable)
    return G_DBUS_METHOD_INVOCATION_UNHANDLED;

  GST_DEBUG_OBJECT (self, "Handle remove track");

  if (!_mpris_find_track_by_id (self, track_id, &index))
    goto finish;

  track = (ClapperMprisTrack *) g_ptr_array_index (self->tracks, index);
  clapper_reactable_queue_remove_sync (CLAPPER_REACTABLE_CAST (self), track->item);

finish:
  clapper_mpris_media_player2_track_list_complete_remove_track (tracks_skeleton, invocation);

  return G_DBUS_METHOD_INVOCATION_HANDLED;
}

static gboolean
_handle_go_to_cb (ClapperMprisMediaPlayer2TrackList *tracks_skeleton,
    GDBusMethodInvocation *invocation, const gchar *track_id, ClapperMpris *self)
{
  ClapperPlayer *player;
  guint index = 0;

  if (!self->queue_controllable)
    return G_DBUS_METHOD_INVOCATION_UNHANDLED;

  if (!_mpris_find_track_by_id (self, track_id, &index))
    goto finish;

  if ((player = clapper_reactable_get_player (CLAPPER_REACTABLE_CAST (self)))) {
    ClapperQueue *queue = clapper_player_get_queue (player);
    ClapperMprisTrack *track = (ClapperMprisTrack *) g_ptr_array_index (self->tracks, index);

    if (clapper_queue_select_item (queue, track->item))
      clapper_player_play (player);

    gst_object_unref (player);
  }

finish:
  clapper_mpris_media_player2_track_list_complete_go_to (tracks_skeleton, invocation);

  return G_DBUS_METHOD_INVOCATION_HANDLED;
}

static void
_name_acquired_cb (GDBusConnection *connection, const gchar *name, ClapperMpris *self)
{
  ClapperReactable *reactable = CLAPPER_REACTABLE (self);
  ClapperPlayer *player;
  GError *error = NULL;
  gchar **uri_schemes, **mime_types;

  GST_DEBUG_OBJECT (self, "Name acquired: %s", name);

  if (!(self->base_exported = g_dbus_interface_skeleton_export (
      G_DBUS_INTERFACE_SKELETON (self->base_skeleton),
      connection, "/org/mpris/MediaPlayer2", &error))) {
    goto finish;
  }
  if (!(self->player_exported = g_dbus_interface_skeleton_export (
      G_DBUS_INTERFACE_SKELETON (self->player_skeleton),
      connection, "/org/mpris/MediaPlayer2", &error))) {
    goto finish;
  }
  if (!(self->tracks_exported = g_dbus_interface_skeleton_export (
      G_DBUS_INTERFACE_SKELETON (self->tracks_skeleton),
      connection, "/org/mpris/MediaPlayer2", &error))) {
    goto finish;
  }

  self->registered = TRUE;

  clapper_mpris_media_player2_set_identity (self->base_skeleton, self->identity);
  clapper_mpris_media_player2_set_desktop_entry (self->base_skeleton, self->desktop_entry);

  uri_schemes = clapper_mpris_get_supported_uri_schemes (self);
  clapper_mpris_media_player2_set_supported_uri_schemes (self->base_skeleton,
      (const gchar *const *) uri_schemes);
  g_strfreev (uri_schemes);

  mime_types = clapper_mpris_get_supported_mime_types (self);
  clapper_mpris_media_player2_set_supported_mime_types (self->base_skeleton,
      (const gchar *const *) mime_types);
  g_strfreev (mime_types);

  /* As stated in MPRIS docs: "This property is not expected to change,
   * as it describes an intrinsic capability of the implementation." */
  clapper_mpris_media_player2_player_set_can_control (self->player_skeleton, TRUE);
  clapper_mpris_media_player2_set_has_track_list (self->base_skeleton, TRUE);
  clapper_mpris_media_player2_track_list_set_can_edit_tracks (self->tracks_skeleton, self->queue_controllable);

  if ((player = clapper_reactable_get_player (reactable))) {
    ClapperQueue *queue = clapper_player_get_queue (player);
    GVariant *variant = NULL;

    /* Update tracks IDs after reading initial tracks from queue */
    clapper_mpris_refresh_track_list (self);

    if (self->current_track)
      variant = _mpris_build_track_metadata (self, self->current_track);

    clapper_mpris_refresh_current_track (self, variant);
    clapper_mpris_refresh_can_go_next_previous (self);

    /* Set some initial default progressions to revert to and
     * try to update them in progression_changed call below */
    self->default_mode = CLAPPER_QUEUE_PROGRESSION_NONE;
    self->non_shuffle_mode = CLAPPER_QUEUE_PROGRESSION_NONE;

    /* Trigger update with current values */
    clapper_mpris_state_changed (reactable, clapper_player_get_state (player));
    clapper_mpris_position_changed (reactable, clapper_player_get_position (player));
    clapper_mpris_speed_changed (reactable, clapper_player_get_speed (player));
    clapper_mpris_volume_changed (reactable, clapper_player_get_volume (player));
    clapper_mpris_queue_progression_changed (reactable, clapper_queue_get_progression_mode (queue));

    gst_object_unref (player);
  }

finish:
  if (error) {
    GST_ERROR_OBJECT (self, "Error: %s", (error && error->message)
        ? error->message : "Unknown DBUS error occured");
    g_error_free (error);

    clapper_mpris_unregister (self);
  }

  if (self->loop && g_main_loop_is_running (self->loop))
    g_main_loop_quit (self->loop);
}

static void
_name_lost_cb (GDBusConnection *connection, const gchar *name, ClapperMpris *self)
{
  GST_DEBUG_OBJECT (self, "Name lost: %s", name);

  if (self->loop && g_main_loop_is_running (self->loop))
    g_main_loop_quit (self->loop);

  clapper_mpris_unregister (self);
}

static void
clapper_mpris_register (ClapperMpris *self)
{
  GDBusConnection *connection;
  gchar *address;

  if (self->registered)
    return;

  GST_DEBUG_OBJECT (self, "Register");

  if (!(address = g_dbus_address_get_for_bus_sync (G_BUS_TYPE_SESSION, NULL, NULL))) {
    GST_WARNING_OBJECT (self, "No MPRIS bus address");
    return;
  }

  GST_INFO_OBJECT (self, "Obtained MPRIS DBus address: %s", address);

  connection = g_dbus_connection_new_for_address_sync (address,
      G_DBUS_CONNECTION_FLAGS_AUTHENTICATION_CLIENT
      | G_DBUS_CONNECTION_FLAGS_MESSAGE_BUS_CONNECTION, NULL, NULL, NULL);
  g_free (address);

  if (!connection) {
    GST_WARNING_OBJECT (self, "No MPRIS bus connection");
    return;
  }

  GST_INFO_OBJECT (self, "Obtained MPRIS DBus connection");

  self->loop = g_main_loop_new (g_main_context_get_thread_default (), FALSE);

  self->name_id = g_bus_own_name_on_connection (connection, self->own_name,
      G_BUS_NAME_OWNER_FLAGS_NONE,
      (GBusNameAcquiredCallback) _name_acquired_cb,
      (GBusNameLostCallback) _name_lost_cb,
      self, NULL);
  g_object_unref (connection);

  /* Wait until connection is established */
  g_main_loop_run (self->loop);
  g_clear_pointer (&self->loop, g_main_loop_unref);

  if (self->registered) {
    GST_DEBUG_OBJECT (self, "Own name ID: %u", self->name_id);
  } else if (self->name_id > 0) {
    GST_ERROR_OBJECT (self, "Could not register MPRIS connection");
    g_bus_unown_name (self->name_id);
    self->name_id = 0;
  }
}

static void
_on_registration_prop_changed (ClapperMpris *self)
{
  clapper_mpris_unregister (self);
  if (self->app_id && self->own_name && self->identity && self->desktop_entry_set)
    clapper_mpris_register (self);
}

static void
clapper_mpris_set_queue_controllable (ClapperMpris *self, gboolean controllable)
{
  if (self->queue_controllable == controllable)
    return;

  self->queue_controllable = controllable;

  clapper_mpris_media_player2_track_list_set_can_edit_tracks (self->tracks_skeleton, self->queue_controllable);
  clapper_mpris_refresh_can_go_next_previous (self);
}

static void
clapper_mpris_set_fallback_art_url (ClapperMpris *self, const gchar *art_url)
{
  gboolean changed = g_set_str (&self->fallback_art_url, art_url);

  if (changed)
    clapper_mpris_refresh_all_tracks (self);
}

static void
clapper_mpris_init (ClapperMpris *self)
{
  self->base_skeleton = clapper_mpris_media_player2_skeleton_new ();
  self->player_skeleton = clapper_mpris_media_player2_player_skeleton_new ();
  self->tracks_skeleton = clapper_mpris_media_player2_track_list_skeleton_new ();

  self->tracks = g_ptr_array_new_with_free_func ((GDestroyNotify) clapper_mpris_track_free);

  self->queue_controllable = DEFAULT_QUEUE_CONTROLLABLE;

  g_signal_connect (self->player_skeleton, "handle-open-uri",
      G_CALLBACK (_handle_open_uri_cb), self);
  g_signal_connect (self->player_skeleton, "handle-play",
      G_CALLBACK (_handle_play_cb), self);
  g_signal_connect (self->player_skeleton, "handle-pause",
      G_CALLBACK (_handle_pause_cb), self);
  g_signal_connect (self->player_skeleton, "handle-play-pause",
      G_CALLBACK (_handle_play_pause_cb), self);
  g_signal_connect (self->player_skeleton, "handle-stop",
      G_CALLBACK (_handle_stop_cb), self);
  g_signal_connect (self->player_skeleton, "handle-next",
      G_CALLBACK (_handle_next_cb), self);
  g_signal_connect (self->player_skeleton, "handle-previous",
      G_CALLBACK (_handle_previous_cb), self);
  g_signal_connect (self->player_skeleton, "handle-seek",
      G_CALLBACK (_handle_seek_cb), self);
  g_signal_connect (self->player_skeleton, "handle-set-position",
      G_CALLBACK (_handle_set_position_cb), self);
  g_signal_connect (self->player_skeleton, "notify::rate",
      G_CALLBACK (_handle_rate_notify_cb), self);
  g_signal_connect (self->player_skeleton, "notify::volume",
      G_CALLBACK (_handle_volume_notify_cb), self);
  g_signal_connect (self->player_skeleton, "notify::loop-status",
      G_CALLBACK (_handle_loop_status_notify_cb), self);
  g_signal_connect (self->player_skeleton, "notify::shuffle",
      G_CALLBACK (_handle_shuffle_notify_cb), self);

  g_signal_connect (self->tracks_skeleton, "handle-get-tracks-metadata",
      G_CALLBACK (_handle_get_tracks_metadata_cb), self);
  g_signal_connect (self->tracks_skeleton, "handle-add-track",
      G_CALLBACK (_handle_add_track_cb), self);
  g_signal_connect (self->tracks_skeleton, "handle-remove-track",
      G_CALLBACK (_handle_remove_track_cb), self);
  g_signal_connect (self->tracks_skeleton, "handle-go-to",
      G_CALLBACK (_handle_go_to_cb), self);
}

static void
clapper_mpris_dispose (GObject *object)
{
  ClapperMpris *self = CLAPPER_MPRIS_CAST (object);

  clapper_mpris_unregister (self);

  G_OBJECT_CLASS (parent_class)->dispose (object);
}

static void
clapper_mpris_finalize (GObject *object)
{
  ClapperMpris *self = CLAPPER_MPRIS_CAST (object);

  GST_TRACE_OBJECT (self, "Finalize");

  g_object_unref (self->base_skeleton);
  g_object_unref (self->player_skeleton);
  g_object_unref (self->tracks_skeleton);

  self->current_track = NULL;
  g_ptr_array_unref (self->tracks);

  g_free (self->app_id);
  g_free (self->own_name);
  g_free (self->identity);
  g_free (self->desktop_entry);
  g_free (self->fallback_art_url);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
clapper_mpris_set_property (GObject *object, guint prop_id,
    const GValue *value, GParamSpec *pspec)
{
  ClapperMpris *self = CLAPPER_MPRIS_CAST (object);

  switch (prop_id) {
    case PROP_APP_ID:
      if (g_set_str (&self->app_id, g_value_get_string (value)))
        _on_registration_prop_changed (self);
      break;
    case PROP_OWN_NAME:
      if (g_set_str (&self->own_name, g_value_get_string (value)))
        _on_registration_prop_changed (self);
      break;
    case PROP_IDENTITY:
      if (g_set_str (&self->identity, g_value_get_string (value)))
        _on_registration_prop_changed (self);
      break;
    case PROP_DESKTOP_ENTRY:
      if (g_set_str (&self->desktop_entry, g_value_get_string (value))
          || !self->desktop_entry_set) { // On first set (can be NULL)
        self->desktop_entry_set = TRUE;
        _on_registration_prop_changed (self);
      }
      break;
    case PROP_QUEUE_CONTROLLABLE:
      clapper_mpris_set_queue_controllable (self, g_value_get_boolean (value));
      break;
    case PROP_FALLBACK_ART_URL:
      clapper_mpris_set_fallback_art_url (self, g_value_get_string (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
clapper_mpris_get_property (GObject *object, guint prop_id,
    GValue *value, GParamSpec *pspec)
{
  ClapperMpris *self = CLAPPER_MPRIS_CAST (object);

  switch (prop_id) {
    case PROP_APP_ID:
      g_value_set_string (value, self->app_id);
      break;
    case PROP_OWN_NAME:
      g_value_set_string (value, self->own_name);
      break;
    case PROP_IDENTITY:
      g_value_set_string (value, self->identity);
      break;
    case PROP_DESKTOP_ENTRY:
      g_value_set_string (value, self->desktop_entry);
      break;
    case PROP_QUEUE_CONTROLLABLE:
      g_value_set_boolean (value, self->queue_controllable);
      break;
    case PROP_FALLBACK_ART_URL:
      g_value_set_string (value, self->fallback_art_url);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
clapper_mpris_class_init (ClapperMprisClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  GST_DEBUG_CATEGORY_INIT (GST_CAT_DEFAULT, "clappermpris", 0,
      "Clapper Mpris");

  gobject_class->get_property = clapper_mpris_get_property;
  gobject_class->set_property = clapper_mpris_set_property;
  gobject_class->dispose = clapper_mpris_dispose;
  gobject_class->finalize = clapper_mpris_finalize;

  /**
   * ClapperMpris:app-id:
   *
   * The ID registered for your app (usually in reverse DNS format).
   *
   * This is used in order to store media artwork in an application
   * specific directory that external MPRIS clients can access.
   *
   * Example: "com.example.MyApp"
   */
  param_specs[PROP_APP_ID] = g_param_spec_string ("app-id",
      NULL, NULL, NULL,
      G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | CLAPPER_ENHANCER_PARAM_LOCAL);

  /**
   * ClapperMpris:own-name:
   *
   * DBus name to own on connection.
   *
   * Must be written as a reverse DNS format starting with "org.mpris.MediaPlayer2." prefix.
   * Each #ClapperMpris instance running on the same system must have an unique name.
   *
   * Example: "org.mpris.MediaPlayer2.MyPlayer.instance123"
   */
  param_specs[PROP_OWN_NAME] = g_param_spec_string ("own-name",
      NULL, NULL, NULL,
      G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | CLAPPER_ENHANCER_PARAM_LOCAL);

  /**
   * ClapperMpris:identity:
   *
   * A friendly name to identify the media player.
   *
   * Example: "My Player"
   */
  param_specs[PROP_IDENTITY] = g_param_spec_string ("identity",
      NULL, NULL, NULL,
      G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | CLAPPER_ENHANCER_PARAM_LOCAL);

  /**
   * ClapperMpris:desktop-entry:
   *
   * The basename of an installed .desktop file with the ".desktop" extension stripped.
   * Can be set to %NULL.
   */
  param_specs[PROP_DESKTOP_ENTRY] = g_param_spec_string ("desktop-entry",
      NULL, NULL, NULL,
      G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | CLAPPER_ENHANCER_PARAM_LOCAL);

  /**
   * ClapperMpris:queue-controllable:
   *
   * Whether remote MPRIS clients can control #ClapperQueue.
   *
   * This includes ability to open new URIs, adding/removing
   * items from the queue and selecting current item for
   * playback remotely using MPRIS interface.
   *
   * You probably want to keep this disabled if your application
   * is supposed to manage what is played now and not MPRIS client.
   */
  param_specs[PROP_QUEUE_CONTROLLABLE] = g_param_spec_boolean ("queue-controllable",
      NULL, NULL, DEFAULT_QUEUE_CONTROLLABLE,
      G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | CLAPPER_ENHANCER_PARAM_LOCAL);

  /**
   * ClapperMpris:fallback-art-url:
   *
   * Fallback artwork to show when media does not provide one.
   */
  param_specs[PROP_FALLBACK_ART_URL] = g_param_spec_string ("fallback-art-url",
      NULL, NULL, NULL,
      G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | CLAPPER_ENHANCER_PARAM_LOCAL);

  g_object_class_install_properties (gobject_class, PROP_LAST, param_specs);
}

void
peas_register_types (PeasObjectModule *module)
{
  peas_object_module_register_extension_type (module, CLAPPER_TYPE_REACTABLE, CLAPPER_TYPE_MPRIS);
}
