/***************************************************************************
    testqgslegendrenderer.cpp
    ---------------------
    begin                : July 2014
    copyright            : (C) 2014 by Martin Dobias
    email                : wonder dot sk at gmail dot com
 ***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "qgstest.h"
#include <QObject>
#include <QJsonArray>

#include "qgsapplication.h"
#include "qgscategorizedsymbolrenderer.h"
#include "qgsdatadefinedsizelegend.h"
#include "qgseffectstack.h"
#include "qgsfillsymbollayer.h"
#include "qgsfontutils.h"
#include "qgsgloweffect.h"
#include "qgslayertree.h"
#include "qgslayertreeutils.h"
#include "qgslayertreemodel.h"
#include "qgslayertreemodellegendnode.h"
#include "qgslayertreefiltersettings.h"
#include "qgslinesymbollayer.h"
#include "qgsmaplayerlegend.h"
#include "qgspainteffect.h"
#include "qgsproject.h"
#include "qgslayertreefilterproxymodel.h"
#include "qgslegendrenderer.h"
#include "qgsrasterlayer.h"
#include "qgsshadoweffect.h"
#include "qgssinglesymbolrenderer.h"
#include "qgsvectorlayer.h"
#include "qgsvectordataprovider.h"
#include "qgsgeometry.h"
#include "qgsdiagramrenderer.h"
#include "qgspalettedrasterrenderer.h"
#include "diagram/qgspiediagram.h"
#include "qgspropertytransformer.h"
#include "qgsrulebasedlabeling.h"
#include "qgslinesymbol.h"
#include "qgsmarkersymbol.h"
#include "qgsfillsymbol.h"
#include "qgsheatmaprenderer.h"
#include "qgsmeshlayer.h"

#include "gdal.h"

class TestRasterRenderer : public QgsPalettedRasterRenderer
{
  public:
    TestRasterRenderer( QgsRasterInterface *input, int bandNumber, const ClassData &classes )
      : QgsPalettedRasterRenderer( input, bandNumber, classes )
    {}

    // don't create the default legend nodes for this layer!
    QList<QgsLayerTreeModelLegendNode *> createLegendNodes( QgsLayerTreeLayer *nodeLayer ) override
    {
      QList<QgsLayerTreeModelLegendNode *> res;

      const QList<QPair<QString, QColor>> items = legendSymbologyItems();
      res.reserve( res.size() + items.size() );
      for ( const QPair<QString, QColor> &item : items )
      {
        res << new QgsRasterSymbolLegendNode( nodeLayer, item.second, item.first );
      }

      return res;
    }
};

class TestQgsLegendRenderer : public QgsTest
{
    Q_OBJECT

  public:
    TestQgsLegendRenderer()
      : QgsTest( QStringLiteral( "Legend Renderer Tests" ), QStringLiteral( "legend" ) ) {}

  private slots:
    void initTestCase();    // will be called before the first testfunction is executed.
    void cleanupTestCase(); // will be called after the last testfunction was executed.
    void init();            // will be called before each testfunction is executed.
    void cleanup();         // will be called after every testfunction.

    void testModel();

    void testBasic();
    void testMultiline();
    void testOverrideSize();
    void testOverrideSizeSmall();
    void testSpacing();
    void testEffects();
    void testBigMarker();
    void testBigMarkerMaxSize();
    void testOverrideSymbol();

    void testRightAlignText();
    void testCenterAlignText();
    void testLeftAlignTextRightAlignSymbol();
    void testCenterAlignTextRightAlignSymbol();
    void testRightAlignTextRightAlignSymbol();
    void testDataDefinedTextFormat();

    void testGroupHeadingSpacing();
    void testGroupIndentSetup();
    void testGroupIndentDefault();
    void testGroupIndentRS();
    void testGroupIndentRT();
    void testGroupIndentRSRT();

    void testMapUnits();
    void testTallSymbol();
    void testLineSpacing();
    void testLongSymbolText();
    void testThreeColumns();
    void testFilterByMap();
    void testFilterByMapSameSymbol();
    void testColumns_data();
    void testColumns();
    void testColumnBreaks();
    void testColumnBreaks2();
    void testColumnBreaks3();
    void testColumnBreaks4();
    void testColumnBreaks5();
    void testLayerColumnSplittingAlwaysAllow();
    void testLayerColumnSplittingAlwaysPrevent();
    void testRasterStroke();
    void testFilterByPolygon();
    void testFilterByExpression();
    void testFilterByExpressionWithContext();
    void testDiagramAttributeLegend();
    void testDiagramMeshLegend();
    void testDiagramSizeLegend();
    void testDataDefinedSizeCollapsed();
    void testDataDefinedSizeSeparated();
    void testDataDefinedSizeCollapsedFilterByMap();
    void testDataDefinedSizeSeparatedFilterByMap();
    void testTextOnSymbol();
    void testColumnsMixedSymbolSize();

    void testBasicJson();
    void testOpacityJson();
    void testBigMarkerJson();

    void testLabelLegend();
    void testHeatmap();

    void testFilteredVector();
    void testFilteredRaster();

  private:
    QgsLayerTree *mRoot = nullptr;
    QgsVectorLayer *mVL1 = nullptr; // line
    QgsVectorLayer *mVL2 = nullptr; // polygon
    QgsVectorLayer *mVL3 = nullptr; // point
    QgsRasterLayer *mRL = nullptr;
    bool _testLegendColumns( int itemCount, int columnCount, const QString &testName, double symbolSpacing );

    bool _verifyImage( const QImage &image, const QString &testName, int diff = 30, const QSize &sizeTolerance = QSize( 6, 10 ) )
    {
      return QGSIMAGECHECK( testName, testName, image, QString(), diff, sizeTolerance );
    }

    static void setStandardTestFont( QgsLegendSettings &settings, const QString &style = QStringLiteral( "Roman" ) )
    {
      for ( const QgsLegendStyle::Style st :
            {
              QgsLegendStyle::Title,
              QgsLegendStyle::Group,
              QgsLegendStyle::Subgroup,
              QgsLegendStyle::SymbolLabel
            } )
      {
        QFont font( QgsFontUtils::getStandardTestFont( style ) );
        QgsTextFormat f = settings.rstyle( st ).textFormat();
        f.setFont( font );
        settings.rstyle( st ).setTextFormat( f );
      }
    }

    static QImage base64ToImage( const QString &base64 )
    {
      const QByteArray bytearray = QByteArray::fromBase64( base64.toStdString().c_str() );
      return QImage::fromData( bytearray, "PNG" );
    }

    static QImage renderLegend( QgsLegendRenderer &legendRenderer )
    {
      const QSizeF size = legendRenderer.minimumSize();

      constexpr int dpi = 96;
      constexpr qreal dpmm = dpi / 25.4;
      const QSize s( static_cast<int>( size.width() * dpmm ), static_cast<int>( size.height() * dpmm ) );
      // qDebug() << QStringLiteral( "testName:%1 size=%2x%3 dpmm=%4 s=%5x%6" ).arg( testName ).arg( size.width() ).arg( size.height() ).arg( dpmm ).arg( s.width() ).arg( s.height() );
      QImage img( s, QImage::Format_ARGB32_Premultiplied );
      img.fill( Qt::white );

      QPainter painter( &img );
      painter.setRenderHint( QPainter::Antialiasing, true );
      QgsRenderContext context = QgsRenderContext::fromQPainter( &painter );
      context.setTextRenderFormat( Qgis::TextRenderFormat::AlwaysText );
      context.setFlag( Qgis::RenderContextFlag::ApplyScalingWorkaroundForTextRendering, true );

      {
        const QgsScopedRenderContextScaleToMm scaleToMm( context );
        context.setRendererScale( 1000 );
        context.setMapToPixel( QgsMapToPixel( 1 / ( 0.1 * context.scaleFactor() ) ) );

        legendRenderer.drawLegend( context );
      }
      painter.end();

      return img;
    }

    static QImage renderLegend( QgsLayerTreeModel *legendModel, QgsLegendSettings &settings )
    {
      settings.setTitle( QStringLiteral( "Legend" ) );
      QgsLegendRenderer legendRenderer( legendModel, settings );
      return renderLegend( legendRenderer );
    }

    static QJsonObject renderJsonLegend( QgsLayerTreeModel *legendModel, const QgsLegendSettings &settings )
    {
      QgsLegendRenderer legendRenderer( legendModel, settings );

      QgsRenderContext context;
      context.setFlag( Qgis::RenderContextFlag::Antialiasing, true );
      return legendRenderer.exportLegendToJson( context );
    }
};


void TestQgsLegendRenderer::initTestCase()
{
  QgsApplication::init();
  QgsApplication::initQgis();
}

void TestQgsLegendRenderer::cleanupTestCase()
{
  QgsApplication::exitQgis();
}

void TestQgsLegendRenderer::init()
{
  mVL1 = new QgsVectorLayer( QStringLiteral( "LineString" ), QStringLiteral( "Line Layer" ), QStringLiteral( "memory" ) );
  QgsProject::instance()->addMapLayer( mVL1 );

  QgsLineSymbol *sym1 = new QgsLineSymbol();
  sym1->setColor( Qt::magenta );
  mVL1->setRenderer( new QgsSingleSymbolRenderer( sym1 ) );

  mVL2 = new QgsVectorLayer( QStringLiteral( "Polygon" ), QStringLiteral( "Polygon Layer" ), QStringLiteral( "memory" ) );
  QgsProject::instance()->addMapLayer( mVL2 );

  QgsFillSymbol *sym2 = new QgsFillSymbol();
  sym2->setColor( Qt::cyan );
  mVL2->setRenderer( new QgsSingleSymbolRenderer( sym2 ) );

  mVL3 = new QgsVectorLayer( QStringLiteral( "Point" ), QStringLiteral( "Point Layer" ), QStringLiteral( "memory" ) );
  {
    QgsVectorDataProvider *pr = mVL3->dataProvider();
    QList<QgsField> attrs;
    attrs << QgsField( QStringLiteral( "test_attr" ), QMetaType::Type::Int );
    pr->addAttributes( attrs );

    QgsFields fields;
    fields.append( attrs.back() );

    QList<QgsFeature> features;
    QgsFeature f1( fields, 1 );
    f1.setAttribute( 0, 1 );
    const QgsGeometry f1G = QgsGeometry::fromPointXY( QgsPointXY( 1.0, 1.0 ) );
    f1.setGeometry( f1G );
    QgsFeature f2( fields, 2 );
    f2.setAttribute( 0, 2 );
    const QgsGeometry f2G = QgsGeometry::fromPointXY( QgsPointXY( 9.0, 1.0 ) );
    f2.setGeometry( f2G );
    QgsFeature f3( fields, 3 );
    f3.setAttribute( 0, 3 );
    const QgsGeometry f3G = QgsGeometry::fromPointXY( QgsPointXY( 5.0, 5.0 ) );
    f3.setGeometry( f3G );
    features << f1 << f2 << f3;
    pr->addFeatures( features );
    mVL3->updateFields();
  }
  QgsProject::instance()->addMapLayer( mVL3 );

  char RASTER_ARRAY[] = { 1, 2, 2, 1 };
  GDALDriverH hGTiffDrv = GDALGetDriverByName( "GTiff" );
  Q_ASSERT( hGTiffDrv );
  const char *tempFileName = "/vsimem/temp.tif";
  GDALDatasetH hDS = GDALCreate( hGTiffDrv, tempFileName, 2, 2, 1, GDT_Byte, NULL );
  Q_ASSERT( hDS );
  CPLErr eErr = GDALRasterIO( GDALGetRasterBand( hDS, 1 ), GF_Write, 0, 0, 2, 2, RASTER_ARRAY, 2, 2, GDT_Byte, 1, 2 );
  QVERIFY( eErr == CE_None );
  GDALClose( hDS );

  mRL = new QgsRasterLayer( QString( tempFileName ), QStringLiteral( "Raster Layer" ), QStringLiteral( "gdal" ) );

  std::unique_ptr<TestRasterRenderer> rasterRenderer( new TestRasterRenderer( mRL->dataProvider(), 1, { QgsPalettedRasterRenderer::Class( 1, QColor( 0, 0, 0 ), QStringLiteral( "1" ) ), QgsPalettedRasterRenderer::Class( 2, QColor( 255, 255, 255 ), QStringLiteral( "2" ) ) } ) );
  mRL->setRenderer( rasterRenderer.release() );

  QgsProject::instance()->addMapLayer( mRL );

  QgsCategoryList cats;
  QgsMarkerSymbol *sym3_1 = new QgsMarkerSymbol();
  sym3_1->setColor( Qt::red );
  cats << QgsRendererCategory( 1, sym3_1, QStringLiteral( "Red" ) );
  QgsMarkerSymbol *sym3_2 = new QgsMarkerSymbol();
  sym3_2->setColor( Qt::green );
  cats << QgsRendererCategory( 2, sym3_2, QStringLiteral( "Green" ) );
  QgsMarkerSymbol *sym3_3 = new QgsMarkerSymbol();
  sym3_3->setColor( Qt::blue );
  cats << QgsRendererCategory( 3, sym3_3, QStringLiteral( "Blue" ) );
  QgsCategorizedSymbolRenderer *r3 = new QgsCategorizedSymbolRenderer( QStringLiteral( "test_attr" ), cats );
  mVL3->setRenderer( r3 );

  mRoot = new QgsLayerTree();
  QgsLayerTreeGroup *grp1 = mRoot->addGroup( QStringLiteral( "Line + Polygon" ) );
  grp1->addLayer( mVL1 );
  grp1->addLayer( mVL2 );
  mRoot->addLayer( mVL3 );
  mRoot->addLayer( mRL );

  VSIUnlink( tempFileName );
}

void TestQgsLegendRenderer::cleanup()
{
  delete mRoot;
  mRoot = nullptr;

  QgsProject::instance()->removeAllMapLayers();
}


void TestQgsLegendRenderer::testModel()
{
  QgsLayerTreeModel legendModel( mRoot );

  QgsLayerTreeNode *nodeGroup0 = mRoot->children().at( 0 );
  QVERIFY( nodeGroup0 );
  QgsLayerTreeNode *nodeLayer0 = nodeGroup0->children().at( 0 );
  QVERIFY( QgsLayerTree::isLayer( nodeLayer0 ) );
  const QModelIndex idx = legendModel.node2index( nodeLayer0 );
  QVERIFY( idx.isValid() );
  QgsLayerTreeLayer *nodeVL1 = QgsLayerTree::toLayer( nodeLayer0 );
  QVERIFY( nodeVL1 );

  QList<QgsLayerTreeModelLegendNode *> lstNodes = legendModel.layerLegendNodes( nodeVL1 );
  QVERIFY( lstNodes.count() == 1 );
  QCOMPARE( lstNodes[0]->data( Qt::DisplayRole ).toString(), QString( "Line Layer" ) );

  // set user text
  QgsMapLayerLegendUtils::setLegendNodeUserLabel( nodeVL1, 0, QStringLiteral( "Hurray" ) );

  legendModel.refreshLayerLegend( nodeVL1 );

  QList<QgsLayerTreeModelLegendNode *> lstNodes2 = legendModel.layerLegendNodes( nodeVL1 );
  QCOMPARE( lstNodes2[0]->data( Qt::DisplayRole ).toString(), QString( "Hurray" ) );

  // reset user text
  QgsMapLayerLegendUtils::setLegendNodeUserLabel( nodeVL1, 0, QString() );
}


void TestQgsLegendRenderer::testBasic()
{
  const QString testName = QStringLiteral( "legend_basic" );

  QgsLayerTreeModel legendModel( mRoot );

  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  const QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );
}

void TestQgsLegendRenderer::testMultiline()
{
  const QString testName = QStringLiteral( "legend_multiline" );

  QgsLayerTreeModel legendModel( mRoot );

  legendModel.findLegendNode( mVL1->id(), QString() );

  QgsLayerTreeLayer *layer = legendModel.rootGroup()->findLayer( mVL1 );
  layer->setCustomProperty( QStringLiteral( "legend/title-label" ), QStringLiteral( "some legend text\nwith newline\ncharacters in it" ) );

  QgsLayerTreeModelLegendNode *embeddedNode = legendModel.legendNodeEmbeddedInParent( layer );
  embeddedNode->setUserLabel( QString() );

  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  const QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );
}

void TestQgsLegendRenderer::testOverrideSize()
{
  const QString testName = QStringLiteral( "legend_override_size" );

  QgsLayerTreeModel legendModel( mRoot );

  legendModel.findLegendNode( mVL1->id(), QString() );

  QgsLayerTreeLayer *layer = legendModel.rootGroup()->findLayer( mVL1 );
  layer->setPatchSize( QSizeF( 30, 0 ) );

  QgsLayerTreeModelLegendNode *embeddedNode = legendModel.legendNodeEmbeddedInParent( layer );
  embeddedNode->setUserLabel( QString() );

  layer = legendModel.rootGroup()->findLayer( mVL3 );
  QgsMapLayerLegendUtils::setLegendNodeSymbolSize( layer, 1, QSizeF( 0, 30 ) );
  legendModel.refreshLayerLegend( layer );

  layer = legendModel.rootGroup()->findLayer( mRL );
  QgsMapLayerLegendUtils::setLegendNodeSymbolSize( layer, 0, QSizeF( 50, 30 ) );
  legendModel.refreshLayerLegend( layer );

  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );

  const QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );
}

void TestQgsLegendRenderer::testOverrideSizeSmall()
{
  // Setting an explicit size for a legend node should override all other settings,
  // including the heights calculated from minimum/maximum symbol size.
  // This is because explicit fixed sizes are PER NODE, and can be used as a last-resort
  // for users to manually adjust the sizing of one particular legend node
  const QString testName = QStringLiteral( "legend_override_size_small" );

  QgsLayerTreeModel legendModel( mRoot );

  legendModel.findLegendNode( mVL1->id(), QString() );

  QgsLayerTreeLayer *layer = legendModel.rootGroup()->findLayer( mVL1 );
  layer->setPatchSize( QSizeF( 30, 0 ) );

  QgsLayerTreeModelLegendNode *embeddedNode = legendModel.legendNodeEmbeddedInParent( layer );
  embeddedNode->setUserLabel( QString() );

  layer = legendModel.rootGroup()->findLayer( mVL3 );
  QgsMapLayerLegendUtils::setLegendNodeSymbolSize( layer, 1, QSizeF( 0, 1 ) );
  legendModel.refreshLayerLegend( layer );

  layer = legendModel.rootGroup()->findLayer( mVL3 );
  QgsMapLayerLegendUtils::setLegendNodeSymbolSize( layer, 2, QSizeF( 0, 0.5 ) );
  legendModel.refreshLayerLegend( layer );

  QgsLegendSettings settings;
  settings.rstyle( QgsLegendStyle::Symbol ).setMargin( QgsLegendStyle::Top, 0 );
  settings.rstyle( QgsLegendStyle::Symbol ).setMargin( QgsLegendStyle::Bottom, 0 );
  settings.setMinimumSymbolSize( 5 );
  settings.setMaximumSymbolSize( 9 );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );

  const QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );
}

void TestQgsLegendRenderer::testSpacing()
{
  QgsMarkerSymbol *sym = new QgsMarkerSymbol();
  sym->setColor( Qt::red );
  sym->setSize( sym->size() * 6 );
  QgsCategorizedSymbolRenderer *catRenderer = dynamic_cast<QgsCategorizedSymbolRenderer *>( mVL3->renderer() );
  QVERIFY( catRenderer );
  catRenderer->updateCategorySymbol( 0, sym );

  QgsLayerTreeModel legendModel( mRoot );
  QgsLegendSettings settings;

  settings.rstyle( QgsLegendStyle::Group ).setMargin( QgsLegendStyle::Left, 7 );
  settings.rstyle( QgsLegendStyle::Subgroup ).setMargin( QgsLegendStyle::Left, 11 );
  settings.rstyle( QgsLegendStyle::Symbol ).setMargin( QgsLegendStyle::Left, 5 );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );

  settings.rstyle( QgsLegendStyle::Group ).setAlignment( Qt::AlignLeft );
  settings.rstyle( QgsLegendStyle::Subgroup ).setAlignment( Qt::AlignLeft );
  settings.rstyle( QgsLegendStyle::SymbolLabel ).setAlignment( Qt::AlignLeft );

  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, QStringLiteral( "legend_left_align_side_space" ) ) );

  settings.rstyle( QgsLegendStyle::Group ).setAlignment( Qt::AlignRight );
  settings.rstyle( QgsLegendStyle::Subgroup ).setAlignment( Qt::AlignRight );
  settings.setSymbolAlignment( Qt::AlignRight );

  res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, QStringLiteral( "legend_right_align_side_space" ) ) );
}

void TestQgsLegendRenderer::testEffects()
{
  const QString testName = QStringLiteral( "legend_effects" );

  QgsEffectStack *effect = new QgsEffectStack();
  QgsSingleSymbolRenderer *renderer;
  QgsSymbol *symbol;

  renderer = dynamic_cast<QgsSingleSymbolRenderer *>( mVL1->renderer() );
  QVERIFY( renderer );
  symbol = renderer->symbol();
  QgsSimpleLineSymbolLayer *lineLayer = dynamic_cast<QgsSimpleLineSymbolLayer *>( symbol->symbolLayer( 0 ) );
  QVERIFY( lineLayer );
  lineLayer->setWidth( 1.8 );
  lineLayer->setColor( Qt::cyan );
  effect = new QgsEffectStack();
  effect->appendEffect( new QgsDropShadowEffect() );
  effect->appendEffect( new QgsDrawSourceEffect() );
  lineLayer->setPaintEffect( effect );

  renderer = dynamic_cast<QgsSingleSymbolRenderer *>( mVL2->renderer() );
  symbol = renderer->symbol();
  QVERIFY( renderer );
  QgsSimpleFillSymbolLayer *fillLayer = dynamic_cast<QgsSimpleFillSymbolLayer *>( symbol->takeSymbolLayer( 0 ) );
  QVERIFY( fillLayer );
  fillLayer->setColor( Qt::blue );
  effect = new QgsEffectStack();
  effect->appendEffect( new QgsDrawSourceEffect() );
  effect->appendEffect( new QgsInnerGlowEffect() );
  fillLayer->setPaintEffect( effect );

  lineLayer = new QgsSimpleLineSymbolLayer();
  lineLayer->setColor( Qt::cyan );
  lineLayer->setWidth( 1.8 );
  effect = new QgsEffectStack();
  effect->appendEffect( new QgsDropShadowEffect() );
  effect->appendEffect( new QgsDrawSourceEffect() );
  lineLayer->setPaintEffect( effect );

  symbol->appendSymbolLayer( lineLayer );
  symbol->appendSymbolLayer( fillLayer );

  symbol = new QgsMarkerSymbol();
  symbol->setColor( Qt::black );
  effect = new QgsEffectStack();
  effect->appendEffect( new QgsDropShadowEffect() );
  effect->appendEffect( new QgsDrawSourceEffect() );
  symbol->symbolLayer( 0 )->setPaintEffect( effect );

  QgsCategorizedSymbolRenderer *catRenderer = dynamic_cast<QgsCategorizedSymbolRenderer *>( mVL3->renderer() );
  QVERIFY( catRenderer );
  catRenderer->updateCategorySymbol( 0, symbol );

  QgsLayerTreeModel legendModel( mRoot );

  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );

  const QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );
}

void TestQgsLegendRenderer::testBigMarker()
{
  const QString testName = QStringLiteral( "legend_big_marker" );

  QgsMarkerSymbol *sym = new QgsMarkerSymbol();
  sym->setColor( Qt::red );
  sym->setSize( sym->size() * 6 );
  QgsCategorizedSymbolRenderer *catRenderer = dynamic_cast<QgsCategorizedSymbolRenderer *>( mVL3->renderer() );
  QVERIFY( catRenderer );
  catRenderer->updateCategorySymbol( 0, sym );

  //dynamic_cast<QgsCategorizedSymbolRenderer*>( mVL3->renderer() )->updateCategoryLabel( 2, "This is a long symbol label" );

  QgsLayerTreeModel legendModel( mRoot );

  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );

  const QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );
}

void TestQgsLegendRenderer::testBigMarkerMaxSize()
{
  const QString testName = QStringLiteral( "legend_big_marker_max_size" );
  QgsMarkerSymbol *sym = new QgsMarkerSymbol();
  sym->setColor( Qt::red );
  sym->setSize( sym->size() * 6 );
  QgsCategorizedSymbolRenderer *catRenderer = dynamic_cast<QgsCategorizedSymbolRenderer *>( mVL3->renderer() );
  QVERIFY( catRenderer );
  catRenderer->updateCategorySymbol( 0, sym );

  QgsLayerTreeModel legendModel( mRoot );
  QgsLegendSettings settings;
  settings.setMaximumSymbolSize( 5 ); //restrict maximum size to 5 mm
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );

  const QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );
}

void TestQgsLegendRenderer::testOverrideSymbol()
{
  const QString testName = QStringLiteral( "legend_override_symbol" );

  QgsLayerTreeModel legendModel( mRoot );

  QgsLayerTreeLayer *layer = legendModel.rootGroup()->findLayer( mVL2 );

  std::unique_ptr<QgsFillSymbol> sym2 = std::make_unique<QgsFillSymbol>();
  sym2->setColor( Qt::red );

  QgsLayerTreeModelLegendNode *embeddedNode = legendModel.legendNodeEmbeddedInParent( layer );
  qgis::down_cast<QgsSymbolLegendNode *>( embeddedNode )->setCustomSymbol( sym2.release() );

  std::unique_ptr<QgsMarkerSymbol> sym3 = std::make_unique<QgsMarkerSymbol>();
  sym3->setColor( QColor( 0, 150, 0 ) );
  sym3->setSize( 6 );

  layer = legendModel.rootGroup()->findLayer( mVL3 );
  QgsMapLayerLegendUtils::setLegendNodeCustomSymbol( layer, 1, sym3.get() );
  legendModel.refreshLayerLegend( layer );

  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );

  const QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );
}

void TestQgsLegendRenderer::testCenterAlignText()
{
  QgsMarkerSymbol *sym = new QgsMarkerSymbol();
  sym->setColor( Qt::red );
  sym->setSize( sym->size() * 6 );
  QgsCategorizedSymbolRenderer *catRenderer = dynamic_cast<QgsCategorizedSymbolRenderer *>( mVL3->renderer() );
  QVERIFY( catRenderer );
  catRenderer->updateCategorySymbol( 0, sym );

  QgsLayerTreeModel legendModel( mRoot );
  QgsLegendSettings settings;
  settings.rstyle( QgsLegendStyle::Group ).setAlignment( Qt::AlignHCenter );
  settings.rstyle( QgsLegendStyle::Subgroup ).setAlignment( Qt::AlignHCenter );
  settings.rstyle( QgsLegendStyle::SymbolLabel ).setAlignment( Qt::AlignHCenter );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );

  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, QStringLiteral( "legend_center_align_text" ) ) );

  settings.setColumnCount( 2 );

  res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, QStringLiteral( "legend_two_cols_center_align_text" ) ) );
}

void TestQgsLegendRenderer::testLeftAlignTextRightAlignSymbol()
{
  QgsMarkerSymbol *sym = new QgsMarkerSymbol();
  sym->setColor( Qt::red );
  sym->setSize( sym->size() * 6 );
  QgsCategorizedSymbolRenderer *catRenderer = dynamic_cast<QgsCategorizedSymbolRenderer *>( mVL3->renderer() );
  QVERIFY( catRenderer );
  catRenderer->updateCategorySymbol( 0, sym );

  QgsLayerTreeModel legendModel( mRoot );
  QgsLegendSettings settings;
  settings.rstyle( QgsLegendStyle::Group ).setAlignment( Qt::AlignLeft );
  settings.rstyle( QgsLegendStyle::Subgroup ).setAlignment( Qt::AlignLeft );
  settings.rstyle( QgsLegendStyle::SymbolLabel ).setAlignment( Qt::AlignLeft );
  settings.setSymbolAlignment( Qt::AlignRight );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );

  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, QStringLiteral( "legend_right_symbol_left_align_text" ) ) );

  settings.setColumnCount( 2 );

  res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, QStringLiteral( "legend_two_cols_right_align_symbol_left_align_text" ) ) );
}

void TestQgsLegendRenderer::testCenterAlignTextRightAlignSymbol()
{
  QgsMarkerSymbol *sym = new QgsMarkerSymbol();
  sym->setColor( Qt::red );
  sym->setSize( sym->size() * 6 );
  QgsCategorizedSymbolRenderer *catRenderer = dynamic_cast<QgsCategorizedSymbolRenderer *>( mVL3->renderer() );
  QVERIFY( catRenderer );
  catRenderer->updateCategorySymbol( 0, sym );

  QgsLayerTreeModel legendModel( mRoot );
  QgsLegendSettings settings;
  settings.rstyle( QgsLegendStyle::Group ).setAlignment( Qt::AlignHCenter );
  settings.rstyle( QgsLegendStyle::Subgroup ).setAlignment( Qt::AlignHCenter );
  settings.rstyle( QgsLegendStyle::SymbolLabel ).setAlignment( Qt::AlignHCenter );
  settings.setSymbolAlignment( Qt::AlignRight );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );

  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, QStringLiteral( "legend_right_symbol_center_align_text" ) ) );

  settings.setColumnCount( 2 );

  res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, QStringLiteral( "legend_two_cols_right_align_symbol_center_align_text" ) ) );
}

void TestQgsLegendRenderer::testRightAlignTextRightAlignSymbol()
{
  QgsMarkerSymbol *sym = new QgsMarkerSymbol();
  sym->setColor( Qt::red );
  sym->setSize( sym->size() * 6 );
  QgsCategorizedSymbolRenderer *catRenderer = dynamic_cast<QgsCategorizedSymbolRenderer *>( mVL3->renderer() );
  QVERIFY( catRenderer );
  catRenderer->updateCategorySymbol( 0, sym );

  QgsLayerTreeModel legendModel( mRoot );
  QgsLegendSettings settings;
  settings.rstyle( QgsLegendStyle::Group ).setAlignment( Qt::AlignRight );
  settings.rstyle( QgsLegendStyle::Subgroup ).setAlignment( Qt::AlignRight );
  settings.rstyle( QgsLegendStyle::SymbolLabel ).setAlignment( Qt::AlignRight );
  settings.setSymbolAlignment( Qt::AlignRight );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );

  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, QStringLiteral( "legend_right_symbol_right_align_text" ) ) );

  settings.setColumnCount( 2 );

  res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, QStringLiteral( "legend_two_cols_right_align_symbol_right_align_text" ) ) );
}

void TestQgsLegendRenderer::testDataDefinedTextFormat()
{
  QgsMarkerSymbol *sym = new QgsMarkerSymbol();
  sym->setColor( Qt::red );
  sym->setSize( sym->size() * 6 );
  QgsCategorizedSymbolRenderer *catRenderer = dynamic_cast<QgsCategorizedSymbolRenderer *>( mVL3->renderer() );
  QVERIFY( catRenderer );
  catRenderer->updateCategorySymbol( 0, sym );

  QgsLayerTreeModel legendModel( mRoot );
  QgsLegendSettings settings;

  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QgsTextFormat format = settings.style( QgsLegendStyle::Group ).textFormat();
  format.dataDefinedProperties().setProperty( QgsPalLayerSettings::Property::Color, QgsProperty::fromExpression( "@text_color_group" ) );
  settings.rstyle( QgsLegendStyle::Group ).setTextFormat( format );

  format = settings.style( QgsLegendStyle::Subgroup ).textFormat();
  format.dataDefinedProperties().setProperty( QgsPalLayerSettings::Property::Color, QgsProperty::fromExpression( "@text_color_subgroup" ) );
  settings.rstyle( QgsLegendStyle::Subgroup ).setTextFormat( format );

  format = settings.style( QgsLegendStyle::SymbolLabel ).textFormat();
  format.dataDefinedProperties().setProperty( QgsPalLayerSettings::Property::Color, QgsProperty::fromExpression( "@text_color_symbol_label" ) );
  settings.rstyle( QgsLegendStyle::SymbolLabel ).setTextFormat( format );

  QgsExpressionContext context;
  QgsExpressionContextScope *scope = new QgsExpressionContextScope();
  scope->setVariable( QStringLiteral( "text_color_group" ), QStringLiteral( "255,0,0" ) );
  scope->setVariable( QStringLiteral( "text_color_subgroup" ), QStringLiteral( "0,255,255" ) );
  scope->setVariable( QStringLiteral( "text_color_symbol_label" ), QStringLiteral( "255,0,255" ) );
  context.appendScope( scope );

  QgsRenderContext rc;
  rc.setExpressionContext( context );
  settings.updateDataDefinedProperties( rc );

  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, QStringLiteral( "data_defined_text_format" ) ) );
}

void TestQgsLegendRenderer::testGroupHeadingSpacing()
{
  QgsMarkerSymbol *sym = new QgsMarkerSymbol();
  sym->setColor( Qt::red );
  sym->setSize( sym->size() * 6 );
  QgsCategorizedSymbolRenderer *catRenderer = dynamic_cast<QgsCategorizedSymbolRenderer *>( mVL3->renderer() );
  QVERIFY( catRenderer );
  catRenderer->updateCategorySymbol( 0, sym );

  QgsLayerTreeModel legendModel( mRoot );
  QgsLegendSettings settings;
  settings.rstyle( QgsLegendStyle::Group ).setMargin( QgsLegendStyle::Top, 5 );
  settings.rstyle( QgsLegendStyle::Group ).setMargin( QgsLegendStyle::Bottom, 17 );
  settings.rstyle( QgsLegendStyle::Subgroup ).setMargin( QgsLegendStyle::Top, 13 );
  settings.rstyle( QgsLegendStyle::Subgroup ).setMargin( QgsLegendStyle::Bottom, 9 );
  settings.setSymbolAlignment( Qt::AlignRight );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );

  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, QStringLiteral( "legend_group_heading_spacing" ) ) );
}

void TestQgsLegendRenderer::testGroupIndentSetup()
{
  QgsMarkerSymbol *sym = new QgsMarkerSymbol();
  sym->setColor( Qt::red );
  QgsLayerTreeGroup *grp2 = mRoot->addGroup( QStringLiteral( "Subgroup" ) );
  grp2->setCustomProperty( QStringLiteral( "legend/title-style" ), QLatin1String( "subgroup" ) );
  for ( int i = 1; i <= 4; ++i )
  {
    QgsVectorLayer *vl = new QgsVectorLayer( QStringLiteral( "Polygon" ), QStringLiteral( "Layer %1" ).arg( i ), QStringLiteral( "memory" ) );
    QgsProject::instance()->addMapLayer( vl );
    vl->setRenderer( new QgsSingleSymbolRenderer( sym->clone() ) );
    grp2->addLayer( vl );
  }
}


void TestQgsLegendRenderer::testGroupIndentDefault()
{
  testGroupIndentSetup();
  QgsMarkerSymbol *sym = new QgsMarkerSymbol();
  sym->setColor( Qt::red );
  sym->setSize( sym->size() * 6 );
  QgsCategorizedSymbolRenderer *catRenderer = dynamic_cast<QgsCategorizedSymbolRenderer *>( mVL3->renderer() );
  QVERIFY( catRenderer );
  catRenderer->updateCategorySymbol( 0, sym );
  QgsLayerTreeModel legendModel( mRoot );
  QgsLegendSettings settings;
  settings.rstyle( QgsLegendStyle::Group ).setIndent( 10 );
  settings.rstyle( QgsLegendStyle::Subgroup ).setIndent( 5 );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );

  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, QStringLiteral( "legend_group_indent" ) ) );
}

void TestQgsLegendRenderer::testGroupIndentRT()
{
  testGroupIndentSetup();
  QgsMarkerSymbol *sym = new QgsMarkerSymbol();
  sym->setColor( Qt::red );
  sym->setSize( sym->size() * 6 );
  QgsCategorizedSymbolRenderer *catRenderer = dynamic_cast<QgsCategorizedSymbolRenderer *>( mVL3->renderer() );
  QVERIFY( catRenderer );
  catRenderer->updateCategorySymbol( 0, sym );
  QgsLayerTreeModel legendModel( mRoot );
  QgsLegendSettings settings;
  settings.rstyle( QgsLegendStyle::Group ).setIndent( 10 );
  settings.rstyle( QgsLegendStyle::Subgroup ).setIndent( 5 );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  settings.rstyle( QgsLegendStyle::Group ).setAlignment( Qt::AlignRight );
  settings.rstyle( QgsLegendStyle::Subgroup ).setAlignment( Qt::AlignRight );
  settings.rstyle( QgsLegendStyle::SymbolLabel ).setAlignment( Qt::AlignRight );

  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, QStringLiteral( "legend_group_indent_right_align_text" ) ) );
}

void TestQgsLegendRenderer::testGroupIndentRS()
{
  testGroupIndentSetup();
  QgsMarkerSymbol *sym = new QgsMarkerSymbol();
  sym->setColor( Qt::red );
  sym->setSize( sym->size() * 6 );
  QgsCategorizedSymbolRenderer *catRenderer = dynamic_cast<QgsCategorizedSymbolRenderer *>( mVL3->renderer() );
  QVERIFY( catRenderer );
  catRenderer->updateCategorySymbol( 0, sym );
  QgsLayerTreeModel legendModel( mRoot );
  QgsLegendSettings settings;
  settings.rstyle( QgsLegendStyle::Group ).setIndent( 10 );
  settings.rstyle( QgsLegendStyle::Subgroup ).setIndent( 5 );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  settings.rstyle( QgsLegendStyle::Group ).setAlignment( Qt::AlignLeft );
  settings.rstyle( QgsLegendStyle::Subgroup ).setAlignment( Qt::AlignLeft );
  settings.rstyle( QgsLegendStyle::SymbolLabel ).setAlignment( Qt::AlignLeft );
  settings.setSymbolAlignment( Qt::AlignRight );

  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, QStringLiteral( "legend_group_indent_right_align_symbol" ) ) );
}

void TestQgsLegendRenderer::testGroupIndentRSRT()
{
  testGroupIndentSetup();
  QgsMarkerSymbol *sym = new QgsMarkerSymbol();
  sym->setColor( Qt::red );
  sym->setSize( sym->size() * 6 );
  QgsCategorizedSymbolRenderer *catRenderer = dynamic_cast<QgsCategorizedSymbolRenderer *>( mVL3->renderer() );
  QVERIFY( catRenderer );
  catRenderer->updateCategorySymbol( 0, sym );
  QgsLayerTreeModel legendModel( mRoot );
  QgsLegendSettings settings;
  settings.rstyle( QgsLegendStyle::Group ).setIndent( 10 );
  settings.rstyle( QgsLegendStyle::Subgroup ).setIndent( 5 );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  settings.rstyle( QgsLegendStyle::Group ).setAlignment( Qt::AlignRight );
  settings.rstyle( QgsLegendStyle::Subgroup ).setAlignment( Qt::AlignRight );
  settings.rstyle( QgsLegendStyle::SymbolLabel ).setAlignment( Qt::AlignRight );
  settings.setSymbolAlignment( Qt::AlignRight );

  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, QStringLiteral( "legend_group_indent_right_align_symbol_right_align_text" ) ) );
}

void TestQgsLegendRenderer::testRightAlignText()
{
  QgsMarkerSymbol *sym = new QgsMarkerSymbol();
  sym->setColor( Qt::red );
  sym->setSize( sym->size() * 6 );
  QgsCategorizedSymbolRenderer *catRenderer = dynamic_cast<QgsCategorizedSymbolRenderer *>( mVL3->renderer() );
  QVERIFY( catRenderer );
  catRenderer->updateCategorySymbol( 0, sym );

  QgsLayerTreeModel legendModel( mRoot );
  QgsLegendSettings settings;
  settings.rstyle( QgsLegendStyle::Group ).setAlignment( Qt::AlignRight );
  settings.rstyle( QgsLegendStyle::Subgroup ).setAlignment( Qt::AlignRight );
  settings.rstyle( QgsLegendStyle::SymbolLabel ).setAlignment( Qt::AlignRight );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );

  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, QStringLiteral( "legend_right_align_text" ) ) );

  settings.setColumnCount( 2 );
  res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, QStringLiteral( "legend_two_cols_right_align_text" ) ) );
}

void TestQgsLegendRenderer::testMapUnits()
{
  const QString testName = QStringLiteral( "legend_mapunits" );

  QgsMarkerSymbol *sym = new QgsMarkerSymbol();
  sym->setColor( Qt::red );
  sym->setSize( 100 );
  sym->setSizeUnit( Qgis::RenderUnit::MapUnits );
  QgsCategorizedSymbolRenderer *catRenderer = dynamic_cast<QgsCategorizedSymbolRenderer *>( mVL3->renderer() );
  QVERIFY( catRenderer );
  catRenderer->updateCategorySymbol( 0, sym );

  sym = new QgsMarkerSymbol();
  sym->setColor( Qt::green );
  sym->setSize( 300 );
  sym->setSizeUnit( Qgis::RenderUnit::MapUnits );
  catRenderer->updateCategorySymbol( 1, sym );

  sym = new QgsMarkerSymbol();
  sym->setColor( Qt::blue );
  sym->setSize( 5 );
  sym->setSizeUnit( Qgis::RenderUnit::Millimeters );
  catRenderer->updateCategorySymbol( 2, sym );

  std::unique_ptr<QgsLayerTree> root( new QgsLayerTree() );
  root->addLayer( mVL3 );
  QgsLayerTreeModel legendModel( root.get() );

  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );

  Q_NOWARN_DEPRECATED_PUSH
  // TODO QGIS 4.0 -- move these to parameters on _renderLegend, and set the render context to match
  settings.setMmPerMapUnit( 0.1 );
  settings.setMapScale( 1000 );
  Q_NOWARN_DEPRECATED_POP

  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );
}

void TestQgsLegendRenderer::testTallSymbol()
{
  const QString testName = QStringLiteral( "legend_tall_symbol" );

  QgsCategorizedSymbolRenderer *catRenderer = dynamic_cast<QgsCategorizedSymbolRenderer *>( mVL3->renderer() );
  QVERIFY( catRenderer );
  catRenderer->updateCategoryLabel( 1, QStringLiteral( "This is\nthree lines\nlong label" ) );

  mVL2->setName( QStringLiteral( "This is a two lines\nlong label" ) );

  QgsLayerTreeModel legendModel( mRoot );

  QgsLegendSettings settings;
  settings.setWrapChar( QStringLiteral( "\n" ) );
  settings.setSymbolSize( QSizeF( 10.0, 10.0 ) );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );

  mVL2->setName( QStringLiteral( "Polygon Layer" ) );
}

void TestQgsLegendRenderer::testLineSpacing()
{
  const QString testName = QStringLiteral( "legend_line_spacing" );

  QgsCategorizedSymbolRenderer *catRenderer = dynamic_cast<QgsCategorizedSymbolRenderer *>( mVL3->renderer() );
  QVERIFY( catRenderer );
  catRenderer->updateCategoryLabel( 1, QStringLiteral( "This is\nthree lines\nlong label" ) );

  mVL2->setName( QStringLiteral( "This is a two lines\nlong label" ) );

  QgsLayerTreeModel legendModel( mRoot );

  QgsLegendSettings settings;
  settings.setWrapChar( QStringLiteral( "\n" ) );
  Q_NOWARN_DEPRECATED_PUSH
  settings.setLineSpacing( 3 );
  Q_NOWARN_DEPRECATED_POP
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );

  mVL2->setName( QStringLiteral( "Polygon Layer" ) );
}

void TestQgsLegendRenderer::testLongSymbolText()
{
  const QString testName = QStringLiteral( "legend_long_symbol_text" );

  QgsCategorizedSymbolRenderer *catRenderer = dynamic_cast<QgsCategorizedSymbolRenderer *>( mVL3->renderer() );
  QVERIFY( catRenderer );
  catRenderer->updateCategoryLabel( 1, QStringLiteral( "This is\nthree lines\nlong label" ) );

  QgsLayerTreeModel legendModel( mRoot );

  QgsLegendSettings settings;
  settings.setWrapChar( QStringLiteral( "\n" ) );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );
}

void TestQgsLegendRenderer::testThreeColumns()
{
  const QString testName = QStringLiteral( "legend_three_columns" );

  QgsLayerTreeModel legendModel( mRoot );

  QgsLegendSettings settings;
  settings.setColumnCount( 3 );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );
}

void TestQgsLegendRenderer::testFilterByMap()
{
  const QString testName = QStringLiteral( "legend_filter_by_map" );

  QgsLayerTreeModel legendModel( mRoot );

  QgsMapSettings mapSettings;
  // extent and size to include only the red and green points
  mapSettings.setExtent( QgsRectangle( 0, 0, 10.0, 4.0 ) );
  mapSettings.setOutputSize( QSize( 400, 100 ) );
  mapSettings.setOutputDpi( 96 );
  mapSettings.setLayers( QgsProject::instance()->mapLayers().values() );

  QgsLayerTreeFilterSettings filterSettings( mapSettings );
  legendModel.setFilterSettings( &filterSettings );

  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );
}

void TestQgsLegendRenderer::testFilterByMapSameSymbol()
{
  QgsVectorLayer *vl4 = new QgsVectorLayer( QStringLiteral( "Point" ), QStringLiteral( "Point Layer" ), QStringLiteral( "memory" ) );
  {
    QgsVectorDataProvider *pr = vl4->dataProvider();
    QList<QgsField> attrs;
    attrs << QgsField( QStringLiteral( "test_attr" ), QMetaType::Type::Int );
    pr->addAttributes( attrs );

    QgsFields fields;
    fields.append( attrs.back() );

    QList<QgsFeature> features;
    QgsFeature f1( fields, 1 );
    f1.setAttribute( 0, 1 );
    const QgsGeometry f1G = QgsGeometry::fromPointXY( QgsPointXY( 1.0, 1.0 ) );
    f1.setGeometry( f1G );
    QgsFeature f2( fields, 2 );
    f2.setAttribute( 0, 2 );
    const QgsGeometry f2G = QgsGeometry::fromPointXY( QgsPointXY( 9.0, 1.0 ) );
    f2.setGeometry( f2G );
    QgsFeature f3( fields, 3 );
    f3.setAttribute( 0, 3 );
    const QgsGeometry f3G = QgsGeometry::fromPointXY( QgsPointXY( 5.0, 5.0 ) );
    f3.setGeometry( f3G );
    features << f1 << f2 << f3;
    pr->addFeatures( features );
    vl4->updateFields();
  }
  QgsProject::instance()->addMapLayer( vl4 );

  //setup categorized renderer with duplicate symbols
  QgsCategoryList cats;
  QgsMarkerSymbol *sym4_1 = new QgsMarkerSymbol();
  sym4_1->setColor( Qt::red );
  cats << QgsRendererCategory( 1, sym4_1, QStringLiteral( "Red1" ) );
  QgsMarkerSymbol *sym4_2 = new QgsMarkerSymbol();
  sym4_2->setColor( Qt::red );
  cats << QgsRendererCategory( 2, sym4_2, QStringLiteral( "Red2" ) );
  QgsMarkerSymbol *sym4_3 = new QgsMarkerSymbol();
  sym4_3->setColor( Qt::red );
  cats << QgsRendererCategory( 3, sym4_3, QStringLiteral( "Red3" ) );
  QgsCategorizedSymbolRenderer *r4 = new QgsCategorizedSymbolRenderer( QStringLiteral( "test_attr" ), cats );
  vl4->setRenderer( r4 );

  const QString testName = QStringLiteral( "legend_filter_by_map_dupe" );

  std::unique_ptr<QgsLayerTree> root( new QgsLayerTree() );
  root->addLayer( vl4 );
  QgsLayerTreeModel legendModel( root.get() );

  QgsMapSettings mapSettings;
  // extent and size to include only the red and green points
  mapSettings.setExtent( QgsRectangle( 0, 0, 10.0, 4.0 ) );
  mapSettings.setOutputSize( QSize( 400, 100 ) );
  mapSettings.setOutputDpi( 96 );
  mapSettings.setLayers( QList<QgsMapLayer *>() << vl4 );

  QgsLayerTreeFilterSettings filterSettings( mapSettings );
  legendModel.setFilterSettings( &filterSettings );

  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );

  QgsProject::instance()->removeMapLayer( vl4 );
}

bool TestQgsLegendRenderer::_testLegendColumns( int itemCount, int columnCount, const QString &testName, double symbolSpacing )
{
  QgsFillSymbol *sym = new QgsFillSymbol();
  sym->setColor( Qt::cyan );

  std::unique_ptr<QgsLayerTree> root( new QgsLayerTree() );

  QList<QgsVectorLayer *> layers;
  for ( int i = 1; i <= itemCount; ++i )
  {
    QgsVectorLayer *vl = new QgsVectorLayer( QStringLiteral( "Polygon" ), QStringLiteral( "Layer %1" ).arg( i ), QStringLiteral( "memory" ) );
    QgsProject::instance()->addMapLayer( vl );
    vl->setRenderer( new QgsSingleSymbolRenderer( sym->clone() ) );
    root->addLayer( vl );
    layers << vl;
  }
  delete sym;

  QgsLayerTreeModel legendModel( root.get() );
  QgsLegendSettings settings;
  settings.setColumnCount( columnCount );
  settings.rstyle( QgsLegendStyle::Style::Symbol ).setMargin( QgsLegendStyle::Side::Top, symbolSpacing );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  const QImage res = renderLegend( &legendModel, settings );
  const bool result = _verifyImage( res, testName );

  for ( QgsVectorLayer *l : layers )
  {
    QgsProject::instance()->removeMapLayer( l );
  }
  return result;
}

void TestQgsLegendRenderer::testColumns_data()
{
  QTest::addColumn<QString>( "testName" );
  QTest::addColumn<int>( "items" );
  QTest::addColumn<int>( "columns" );

  QTest::newRow( "2 items, 2 columns" ) << "legend_2_by_2" << 2 << 2;
  QTest::newRow( "3 items, 2 columns" ) << "legend_3_by_2" << 3 << 2;
  QTest::newRow( "4 items, 2 columns" ) << "legend_4_by_2" << 4 << 2;
  QTest::newRow( "5 items, 2 columns" ) << "legend_5_by_2" << 5 << 2;
  QTest::newRow( "3 items, 3 columns" ) << "legend_3_by_3" << 3 << 3;
  QTest::newRow( "4 items, 3 columns" ) << "legend_4_by_3" << 4 << 3;
  QTest::newRow( "5 items, 3 columns" ) << "legend_5_by_3" << 5 << 3;
  QTest::newRow( "6 items, 3 columns" ) << "legend_6_by_3" << 6 << 3;
  QTest::newRow( "7 items, 3 columns" ) << "legend_7_by_3" << 7 << 3;
  QTest::newRow( "27 items, 3 columns" ) << "legend_27_by_3" << 27 << 3;
  QTest::newRow( "27 items, 9 columns" ) << "legend_27_by_9" << 27 << 9;
}

void TestQgsLegendRenderer::testColumns()
{
  //test rendering legend with different combinations of columns and items

  QFETCH( QString, testName );
  QFETCH( int, items );
  QFETCH( int, columns );

  for ( double symbolSpacing : { 2.5, 5.0, 6.0 } )
  {
    QVERIFY( _testLegendColumns( items, columns, testName + ( symbolSpacing != 2.5 ? QStringLiteral( "_spacing_%1" ).arg( QString::number( symbolSpacing ).replace( ".", "_" ) ) : QString() ), symbolSpacing ) );
  }
}

void TestQgsLegendRenderer::testColumnBreaks()
{
  const QString testName = QStringLiteral( "legend_column_breaks" );

  QgsLayerTreeModel legendModel( mRoot );

  QgsLayerTreeLayer *layer = legendModel.rootGroup()->findLayer( mVL2 );
  layer->setCustomProperty( QStringLiteral( "legend/column-break" ), true );

  layer = legendModel.rootGroup()->findLayer( mVL3 );
  QgsMapLayerLegendUtils::setLegendNodeColumnBreak( layer, 1, true );
  legendModel.refreshLayerLegend( layer );

  layer = legendModel.rootGroup()->findLayer( mRL );
  QgsMapLayerLegendUtils::setLegendNodeColumnBreak( layer, 1, true );
  legendModel.refreshLayerLegend( layer );

  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );
}

void TestQgsLegendRenderer::testColumnBreaks2()
{
  const QString testName = QStringLiteral( "legend_column_breaks2" );

  QgsLayerTreeModel legendModel( mRoot );

  QgsLayerTreeLayer *layer = legendModel.rootGroup()->findLayer( mVL3 );
  QgsMapLayerLegendUtils::setLegendNodeColumnBreak( layer, 0, true );
  legendModel.refreshLayerLegend( layer );

  layer = legendModel.rootGroup()->findLayer( mRL );
  QgsMapLayerLegendUtils::setLegendNodeColumnBreak( layer, 0, true );
  legendModel.refreshLayerLegend( layer );

  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );
}

void TestQgsLegendRenderer::testColumnBreaks3()
{
  const QString testName = QStringLiteral( "legend_column_breaks3" );

  QgsLayerTreeModel legendModel( mRoot );

  QgsLayerTreeLayer *layer = legendModel.rootGroup()->findLayer( mVL3 );
  layer->setCustomProperty( QStringLiteral( "legend/column-break" ), true );

  layer = legendModel.rootGroup()->findLayer( mRL );
  layer->setCustomProperty( QStringLiteral( "legend/column-break" ), true );

  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );
}

void TestQgsLegendRenderer::testColumnBreaks4()
{
  const QString testName = QStringLiteral( "legend_column_breaks4" );

  QgsLayerTreeModel legendModel( mRoot );

  QgsLayerTreeLayer *layer = legendModel.rootGroup()->findLayer( mVL3 );
  QgsMapLayerLegendUtils::setLegendNodeColumnBreak( layer, 0, true );
  legendModel.refreshLayerLegend( layer );

  layer = legendModel.rootGroup()->findLayer( mRL );
  QgsMapLayerLegendUtils::setLegendNodeColumnBreak( layer, 0, true );
  legendModel.refreshLayerLegend( layer );

  QgsLegendSettings settings;
  settings.setColumnCount( 5 );
  settings.setSplitLayer( true );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );
}

void TestQgsLegendRenderer::testColumnBreaks5()
{
  const QString testName = QStringLiteral( "legend_column_breaks5" );

  QgsLayerTreeModel legendModel( mRoot );

  QgsLayerTreeLayer *layer = legendModel.rootGroup()->findLayer( mVL3 );
  QgsMapLayerLegendUtils::setLegendNodeColumnBreak( layer, 0, true );
  legendModel.refreshLayerLegend( layer );

  layer = legendModel.rootGroup()->findLayer( mRL );
  QgsMapLayerLegendUtils::setLegendNodeColumnBreak( layer, 0, true );
  legendModel.refreshLayerLegend( layer );

  QgsLegendSettings settings;
  settings.setColumnCount( 4 );
  settings.setSplitLayer( false );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );
}

void TestQgsLegendRenderer::testLayerColumnSplittingAlwaysAllow()
{
  const QString testName = QStringLiteral( "legend_layer_column_splitting_allow" );

  QgsLayerTreeModel legendModel( mRoot );

  QgsLayerTreeLayer *layer = legendModel.rootGroup()->findLayer( mVL3 );
  layer->setLegendSplitBehavior( QgsLayerTreeLayer::AllowSplittingLegendNodesOverMultipleColumns );

  QgsLegendSettings settings;
  settings.setColumnCount( 4 );
  settings.setSplitLayer( false );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );
}

void TestQgsLegendRenderer::testLayerColumnSplittingAlwaysPrevent()
{
  const QString testName = QStringLiteral( "legend_layer_column_splitting_prevent" );

  QgsLayerTreeModel legendModel( mRoot );

  QgsLayerTreeLayer *layer = legendModel.rootGroup()->findLayer( mVL3 );
  layer->setLegendSplitBehavior( QgsLayerTreeLayer::PreventSplittingLegendNodesOverMultipleColumns );

  QgsLegendSettings settings;
  settings.setColumnCount( 4 );
  settings.setSplitLayer( true );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );
}

void TestQgsLegendRenderer::testRasterStroke()
{
  const QString testName = QStringLiteral( "legend_raster_border" );

  std::unique_ptr<QgsLayerTree> root( new QgsLayerTree() );
  root->addLayer( mRL );

  QgsLayerTreeModel legendModel( root.get() );

  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  settings.setRasterStrokeWidth( 2 );
  settings.setRasterStrokeColor( Qt::green );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );
}

void TestQgsLegendRenderer::testFilterByPolygon()
{
  const QString testName = QStringLiteral( "legend_filter_by_polygon" );

  QgsLayerTreeModel legendModel( mRoot );

  QgsMapSettings mapSettings;
  // extent and size to include only the red and green points
  mapSettings.setExtent( QgsRectangle( 0, 0, 10.0, 4.0 ) );
  mapSettings.setOutputSize( QSize( 400, 100 ) );
  mapSettings.setOutputDpi( 96 );
  mapSettings.setLayers( QgsProject::instance()->mapLayers().values() );

  // select only within a map settings extent
  QgsLayerTreeFilterSettings filterSettings( mapSettings );
  legendModel.setFilterSettings( &filterSettings );

  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );

  // now with filter polygon
  const QgsGeometry geom( QgsGeometry::fromWkt( QStringLiteral( "POLYGON((0 0,2 0,2 2,0 2,0 0))" ) ) );
  filterSettings.setFilterPolygon( geom );
  legendModel.setFilterSettings( &filterSettings );

  const QString testName2 = testName + "2";
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName2 ) );
}

void TestQgsLegendRenderer::testFilterByExpression()
{
  const QString testName = QStringLiteral( "legend_filter_by_expression" );

  QgsLayerTreeModel legendModel( mRoot );

  QgsMapSettings mapSettings;
  // extent and size to include only the red and green points
  mapSettings.setExtent( QgsRectangle( 0, 0, 10.0, 4.0 ) );
  mapSettings.setOutputSize( QSize( 400, 100 ) );
  mapSettings.setOutputDpi( 96 );
  mapSettings.setLayers( QgsProject::instance()->mapLayers().values() );

  // use an expression to only include the red point
  QgsLayerTreeLayer *layer = legendModel.rootGroup()->findLayer( mVL3->id() );
  QVERIFY( layer );
  QgsLayerTreeUtils::setLegendFilterByExpression( *layer, QStringLiteral( "test_attr=1" ) );

  QgsLayerTreeFilterSettings filterSettings( mapSettings );
  filterSettings.setLayerFilterExpressionsFromLayerTree( mRoot );
  legendModel.setFilterSettings( &filterSettings );

  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );

  // test again with setLegendFilter and only expressions
  filterSettings.setFlags( Qgis::LayerTreeFilterFlag::SkipVisibilityCheck );
  legendModel.setFilterSettings( &filterSettings );

  const QString testName2 = testName + "2";
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName2 ) );
}

void TestQgsLegendRenderer::testFilterByExpressionWithContext()
{
  const QString testName = QStringLiteral( "legend_filter_by_expression_context" );

  std::unique_ptr<QgsLayerTree> root = std::make_unique<QgsLayerTree>();
  root->addLayer( mVL3 );
  QgsLayerTreeModel legendModel( root.get() );

  QgsMapSettings mapSettings;
  // extent and size to include all red and green points
  mapSettings.setExtent( QgsRectangle( 0, 0, 10.0, 4.0 ) );
  mapSettings.setOutputSize( QSize( 400, 100 ) );
  mapSettings.setOutputDpi( 96 );
  mapSettings.setLayers( QgsProject::instance()->mapLayers().values() );


  QgsExpressionContext context;
  std::unique_ptr<QgsExpressionContextScope> scope = std::make_unique<QgsExpressionContextScope>( QStringLiteral( "test_scope" ) );
  scope->setVariable( QStringLiteral( "test_var" ), QStringLiteral( "test_value" ) );
  context.appendScope( scope.release() );

  mapSettings.setExpressionContext( context );

  // Point layer
  QgsLayerTreeLayer *layer = legendModel.rootGroup()->findLayer( mVL3->id() );
  QVERIFY( layer );
  QgsLayerTreeUtils::setLegendFilterByExpression( *layer, QStringLiteral( "@test_var = 'test_value'" ) );

  QgsLayerTreeFilterSettings filterSettings( mapSettings );
  filterSettings.setLayerFilterExpressionsFromLayerTree( root.get() );

  legendModel.setFilterSettings( &filterSettings );
  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );

  // Skip visibility
  filterSettings.setFlags( Qgis::LayerTreeFilterFlag::SkipVisibilityCheck );
  legendModel.setFilterSettings( &filterSettings );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName + "2" ) );

  // False expression
  QgsLayerTreeUtils::setLegendFilterByExpression( *layer, QStringLiteral( "@test_var != 'test_value'" ) );
  filterSettings.setFlags( Qgis::LayerTreeFilterFlag::SkipVisibilityCheck );
  filterSettings.setLayerFilterExpressionsFromLayerTree( root.get() );
  legendModel.setFilterSettings( &filterSettings );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName + "3" ) );

  // False expression with visibility check
  filterSettings.setFlags( Qgis::LayerTreeFilterFlags() );
  legendModel.setFilterSettings( &filterSettings );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName + "3" ) );
}

void TestQgsLegendRenderer::testDiagramAttributeLegend()
{
  QgsVectorLayer *vl4 = new QgsVectorLayer( QStringLiteral( "Point" ), QStringLiteral( "Point Layer" ), QStringLiteral( "memory" ) );
  QgsProject::instance()->addMapLayer( vl4 );

  QgsMarkerSymbol *sym3_1 = new QgsMarkerSymbol();
  sym3_1->setColor( Qt::red );
  vl4->setRenderer( new QgsSingleSymbolRenderer( sym3_1 ) );

  QgsDiagramSettings ds;
  ds.categoryColors = QList<QColor>() << QColor( 255, 0, 0 ) << QColor( 0, 255, 0 );
  ds.categoryAttributes = QList<QString>() << QStringLiteral( "\"cat1\"" ) << QStringLiteral( "\"cat2\"" );
  ds.categoryLabels = QStringList() << QStringLiteral( "cat 1" ) << QStringLiteral( "cat 2" );

  QgsLinearlyInterpolatedDiagramRenderer *dr = new QgsLinearlyInterpolatedDiagramRenderer();
  dr->setLowerValue( 0.0 );
  dr->setLowerSize( QSizeF( 0.0, 0.0 ) );
  dr->setUpperValue( 10 );
  dr->setUpperSize( QSizeF( 40, 40 ) );
  dr->setClassificationField( QString() );
  dr->setDiagram( new QgsPieDiagram() );
  dr->setDiagramSettings( ds );
  dr->setDataDefinedSizeLegend( nullptr );
  dr->setAttributeLegend( true );
  vl4->setDiagramRenderer( dr );

  QgsDiagramLayerSettings dls = QgsDiagramLayerSettings();
  dls.setPlacement( QgsDiagramLayerSettings::OverPoint );
  dls.setShowAllDiagrams( true );
  vl4->setDiagramLayerSettings( dls );

  std::unique_ptr<QgsLayerTree> root( new QgsLayerTree() );
  root->addLayer( vl4 );
  QgsLayerTreeModel legendModel( root.get() );

  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );

  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, QStringLiteral( "legend_diagram_attributes" ) ) );

  QgsProject::instance()->removeMapLayer( vl4 );
}

void TestQgsLegendRenderer::testDiagramMeshLegend()
{
  const QString uri_1( QString( TEST_DATA_DIR ) + QStringLiteral( "/mesh/mesh_z_ws_d_vel.nc" ) ); //mesh with dataset group "Bed Elevation", "Water Level", "Depth" and "Velocity"

  QTemporaryDir tempDir;
  const QString uri( tempDir.filePath( QStringLiteral( "mesh.nc" ) ) );

  QFile::copy( uri_1, uri );
  QgsMeshLayer *layer = new QgsMeshLayer( uri, QStringLiteral( "mesh" ), QStringLiteral( "mdal" ) );
  QVERIFY( layer->isValid() );
  QCOMPARE( layer->datasetGroupCount(), 4 );

  QgsProject::instance()->addMapLayer( layer );

  int scalarIndex = 0;
  int vectorIndex = -1;

  QgsMeshRendererSettings rendererSettings = layer->rendererSettings();
  rendererSettings.setActiveScalarDatasetGroup( scalarIndex );
  rendererSettings.setActiveVectorDatasetGroup( vectorIndex );
  layer->setRendererSettings( rendererSettings );

  std::unique_ptr<QgsLayerTree> root( std::make_unique<QgsLayerTree>() );
  root->addLayer( layer );
  std::unique_ptr<QgsLayerTreeModel> legendModel( std::make_unique<QgsLayerTreeModel>( root.get() ) );

  QgsLegendSettings settings;

  setStandardTestFont( settings );
  QImage res = renderLegend( legendModel.get(), settings );

  QVERIFY( _verifyImage( res, QStringLiteral( "legend_mesh_diagram_no_vector" ), 30, QSize( 8, 12 ) ) );

  //red vector
  QgsMeshLayer *layer2 = layer->clone();
  QgsProject::instance()->removeMapLayer( layer );
  QgsProject::instance()->addMapLayer( layer2 );

  vectorIndex = 2;
  rendererSettings.setActiveVectorDatasetGroup( vectorIndex );
  QgsMeshRendererVectorSettings vectorSettings = rendererSettings.vectorSettings( vectorIndex );
  vectorSettings.setColor( Qt::red );
  rendererSettings.setVectorSettings( vectorIndex, vectorSettings );
  layer2->setRendererSettings( rendererSettings );

  root = std::make_unique<QgsLayerTree>();
  root->addLayer( layer2 );
  legendModel = std::make_unique<QgsLayerTreeModel>( root.get() );

  res = renderLegend( legendModel.get(), settings );
  QVERIFY( _verifyImage( res, QStringLiteral( "legend_mesh_diagram_red_vector" ), 30, QSize( 8, 13 ) ) );

  //color ramp vector
  QgsMeshLayer *layer3 = layer2->clone();
  QgsProject::instance()->removeMapLayer( layer2 );
  QgsProject::instance()->addMapLayer( layer3 );

  const QgsColorRampShader fcn = rendererSettings.scalarSettings( vectorIndex ).colorRampShader();
  vectorSettings.setColorRampShader( fcn );
  vectorSettings.setColoringMethod( QgsInterpolatedLineColor::ColorRamp );
  rendererSettings.setVectorSettings( vectorIndex, vectorSettings );
  layer3->setRendererSettings( rendererSettings );

  root = std::make_unique<QgsLayerTree>();
  root->addLayer( layer3 );
  legendModel = std::make_unique<QgsLayerTreeModel>( root.get() );

  res = renderLegend( legendModel.get(), settings );
  QVERIFY( _verifyImage( res, QStringLiteral( "legend_mesh_diagram_color_ramp_vector" ), 30, QSize( 8, 19 ) ) );

  QgsProject::instance()->removeMapLayer( layer3 );
}

void TestQgsLegendRenderer::testDiagramSizeLegend()
{
  QgsVectorLayer *vl4 = new QgsVectorLayer( QStringLiteral( "Point" ), QStringLiteral( "Point Layer" ), QStringLiteral( "memory" ) );
  QgsMarkerSymbol *sym3_1 = new QgsMarkerSymbol();
  sym3_1->setColor( Qt::red );
  vl4->setRenderer( new QgsSingleSymbolRenderer( sym3_1 ) );

  QgsProject::instance()->addMapLayer( vl4 );

  QgsDiagramSettings ds;
  ds.categoryColors = QList<QColor>() << QColor( 255, 0, 0 ) << QColor( 0, 255, 0 );
  ds.categoryAttributes = QList<QString>() << QStringLiteral( "\"cat1\"" ) << QStringLiteral( "\"cat2\"" );
  ds.categoryLabels = QStringList() << QStringLiteral( "cat 1" ) << QStringLiteral( "cat 2" );
  ds.scaleByArea = false;

  QgsLinearlyInterpolatedDiagramRenderer *dr = new QgsLinearlyInterpolatedDiagramRenderer();
  dr->setLowerValue( 0.0 );
  dr->setLowerSize( QSizeF( 1, 1 ) );
  dr->setUpperValue( 10 );
  dr->setUpperSize( QSizeF( 20, 20 ) );
  dr->setClassificationField( QStringLiteral( "a" ) );
  dr->setDiagram( new QgsPieDiagram() );
  dr->setDiagramSettings( ds );
  dr->setDataDefinedSizeLegend( new QgsDataDefinedSizeLegend() );
  dr->setAttributeLegend( false );
  vl4->setDiagramRenderer( dr );

  QgsDiagramLayerSettings dls = QgsDiagramLayerSettings();
  dls.setPlacement( QgsDiagramLayerSettings::OverPoint );
  dls.setShowAllDiagrams( true );
  vl4->setDiagramLayerSettings( dls );

  std::unique_ptr<QgsLayerTree> root( new QgsLayerTree() );
  root->addLayer( vl4 );
  QgsLayerTreeModel legendModel( root.get() );

  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );

  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, QStringLiteral( "legend_diagram_size" ) ) );

  QgsProject::instance()->removeMapLayer( vl4 );
}


void TestQgsLegendRenderer::testDataDefinedSizeCollapsed()
{
  const QString testName = QStringLiteral( "legend_data_defined_size_collapsed" );

  QgsVectorLayer *vlDataDefinedSize = new QgsVectorLayer( QStringLiteral( "Point" ), QStringLiteral( "Point Layer" ), QStringLiteral( "memory" ) );
  {
    QgsVectorDataProvider *pr = vlDataDefinedSize->dataProvider();
    QList<QgsField> attrs;
    attrs << QgsField( QStringLiteral( "test_attr" ), QMetaType::Type::Int );
    pr->addAttributes( attrs );

    QgsFields fields;
    fields.append( attrs.back() );

    const QgsGeometry g = QgsGeometry::fromPointXY( QgsPointXY( 1.0, 1.0 ) );

    QList<QgsFeature> features;
    QgsFeature f1( fields, 1 );
    f1.setAttribute( 0, 100 );
    f1.setGeometry( g );
    QgsFeature f2( fields, 2 );
    f2.setAttribute( 0, 200 );
    f2.setGeometry( g );
    QgsFeature f3( fields, 3 );
    f3.setAttribute( 0, 300 );
    f3.setGeometry( g );
    features << f1 << f2 << f3;
    pr->addFeatures( features );
    vlDataDefinedSize->updateFields();
  }

  QVariantMap props;
  props[QStringLiteral( "name" )] = QStringLiteral( "circle" );
  props[QStringLiteral( "color" )] = QStringLiteral( "200,200,200" );
  props[QStringLiteral( "outline_color" )] = QStringLiteral( "0,0,0" );
  QgsMarkerSymbol *symbol = QgsMarkerSymbol::createSimple( props );
  QgsProperty ddsProperty = QgsProperty::fromField( QStringLiteral( "test_attr" ) );
  ddsProperty.setTransformer( new QgsSizeScaleTransformer( QgsSizeScaleTransformer::Linear, 100, 300, 10, 30 ) ); // takes ownership
  symbol->setDataDefinedSize( ddsProperty );

  QgsDataDefinedSizeLegend *ddsLegend = new QgsDataDefinedSizeLegend();
  ddsLegend->setLegendType( QgsDataDefinedSizeLegend::LegendCollapsed );
  ddsLegend->setFont( QgsFontUtils::getStandardTestFont( QStringLiteral( "Bold" ) ) );

  QgsSingleSymbolRenderer *r = new QgsSingleSymbolRenderer( symbol ); // takes ownership
  r->setDataDefinedSizeLegend( ddsLegend );
  vlDataDefinedSize->setRenderer( r );

  QgsLayerTree *root = new QgsLayerTree();
  root->addLayer( vlDataDefinedSize );

  QgsLayerTreeModel legendModel( root );

  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );

  delete root;
}

void TestQgsLegendRenderer::testDataDefinedSizeSeparated()
{
  const QString testName = QStringLiteral( "legend_data_defined_size_separated" );

  QgsVectorLayer *vlDataDefinedSize = new QgsVectorLayer( QStringLiteral( "Point" ), QStringLiteral( "Point Layer" ), QStringLiteral( "memory" ) );
  {
    QgsVectorDataProvider *pr = vlDataDefinedSize->dataProvider();
    QList<QgsField> attrs;
    attrs << QgsField( QStringLiteral( "test_attr" ), QMetaType::Type::Int );
    pr->addAttributes( attrs );

    QgsFields fields;
    fields.append( attrs.back() );

    const QgsGeometry g = QgsGeometry::fromPointXY( QgsPointXY( 1.0, 1.0 ) );

    QList<QgsFeature> features;
    QgsFeature f1( fields, 1 );
    f1.setAttribute( 0, 100 );
    f1.setGeometry( g );
    QgsFeature f2( fields, 2 );
    f2.setAttribute( 0, 200 );
    f2.setGeometry( g );
    QgsFeature f3( fields, 3 );
    f3.setAttribute( 0, 300 );
    f3.setGeometry( g );
    features << f1 << f2 << f3;
    pr->addFeatures( features );
    vlDataDefinedSize->updateFields();
  }

  QVariantMap props;
  props[QStringLiteral( "name" )] = QStringLiteral( "circle" );
  props[QStringLiteral( "color" )] = QStringLiteral( "200,200,200" );
  props[QStringLiteral( "outline_color" )] = QStringLiteral( "0,0,0" );
  QgsMarkerSymbol *symbol = QgsMarkerSymbol::createSimple( props );
  QgsProperty ddsProperty = QgsProperty::fromField( QStringLiteral( "test_attr" ) );
  ddsProperty.setTransformer( new QgsSizeScaleTransformer( QgsSizeScaleTransformer::Linear, 100, 300, 10, 30 ) ); // takes ownership
  symbol->setDataDefinedSize( ddsProperty );

  QgsDataDefinedSizeLegend *ddsLegend = new QgsDataDefinedSizeLegend();
  ddsLegend->setLegendType( QgsDataDefinedSizeLegend::LegendSeparated );
  ddsLegend->setFont( QgsFontUtils::getStandardTestFont( QStringLiteral( "Bold" ) ) );

  QgsSingleSymbolRenderer *r = new QgsSingleSymbolRenderer( symbol ); // takes ownership
  r->setDataDefinedSizeLegend( ddsLegend );
  vlDataDefinedSize->setRenderer( r );

  QgsLayerTree *root = new QgsLayerTree();
  root->addLayer( vlDataDefinedSize );

  QgsLayerTreeModel legendModel( root );

  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );

  delete root;
}

void TestQgsLegendRenderer::testDataDefinedSizeCollapsedFilterByMap()
{
  const QString testName = QStringLiteral( "legend_data_defined_size_filter_by_map" );

  QgsVectorLayer *vlDataDefinedSize = new QgsVectorLayer( QStringLiteral( "Point" ), QStringLiteral( "Point Layer" ), QStringLiteral( "memory" ) );
  {
    QgsVectorDataProvider *pr = vlDataDefinedSize->dataProvider();
    QList<QgsField> attrs;
    attrs << QgsField( QStringLiteral( "test_attr" ), QMetaType::Type::Int );
    pr->addAttributes( attrs );

    QgsFields fields;
    fields.append( attrs.back() );

    const QgsGeometry g = QgsGeometry::fromPointXY( QgsPointXY( 1.0, 1.0 ) );

    QList<QgsFeature> features;
    QgsFeature f1( fields, 1 );
    f1.setAttribute( 0, 100 );
    f1.setGeometry( g );
    QgsFeature f2( fields, 2 );
    f2.setAttribute( 0, 200 );
    f2.setGeometry( g );
    QgsFeature f3( fields, 3 );
    f3.setAttribute( 0, 300 );
    f3.setGeometry( g );
    features << f1 << f2 << f3;
    pr->addFeatures( features );
    vlDataDefinedSize->updateFields();
  }

  QVariantMap props;
  props[QStringLiteral( "name" )] = QStringLiteral( "circle" );
  props[QStringLiteral( "color" )] = QStringLiteral( "200,200,200" );
  props[QStringLiteral( "outline_color" )] = QStringLiteral( "0,0,0" );
  QgsMarkerSymbol *symbol = QgsMarkerSymbol::createSimple( props );
  QgsProperty ddsProperty = QgsProperty::fromField( QStringLiteral( "test_attr" ) );
  ddsProperty.setTransformer( new QgsSizeScaleTransformer( QgsSizeScaleTransformer::Linear, 100, 300, 10, 30 ) ); // takes ownership
  symbol->setDataDefinedSize( ddsProperty );

  QgsDataDefinedSizeLegend *ddsLegend = new QgsDataDefinedSizeLegend();
  ddsLegend->setLegendType( QgsDataDefinedSizeLegend::LegendCollapsed );
  ddsLegend->setFont( QgsFontUtils::getStandardTestFont( QStringLiteral( "Bold" ) ) );

  QgsSingleSymbolRenderer *r = new QgsSingleSymbolRenderer( symbol ); // takes ownership
  r->setDataDefinedSizeLegend( ddsLegend );
  vlDataDefinedSize->setRenderer( r );

  QgsLayerTree *root = new QgsLayerTree();
  root->addLayer( vlDataDefinedSize );

  QgsLayerTreeModel legendModel( root );

  QgsMapSettings mapSettings;
  // extent and size to include only the red and green points
  mapSettings.setExtent( QgsRectangle( 10, 10, 20, 20 ) );
  mapSettings.setOutputSize( QSize( 400, 100 ) );
  mapSettings.setOutputDpi( 96 );
  mapSettings.setLayers( QgsProject::instance()->mapLayers().values() );

  QgsLayerTreeFilterSettings filterSettings( mapSettings );
  legendModel.setFilterSettings( &filterSettings );

  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );

  delete root;
}

void TestQgsLegendRenderer::testDataDefinedSizeSeparatedFilterByMap()
{
  const QString testName = QStringLiteral( "legend_data_defined_size_filter_by_map" );

  QgsVectorLayer *vlDataDefinedSize = new QgsVectorLayer( QStringLiteral( "Point" ), QStringLiteral( "Point Layer" ), QStringLiteral( "memory" ) );
  {
    QgsVectorDataProvider *pr = vlDataDefinedSize->dataProvider();
    QList<QgsField> attrs;
    attrs << QgsField( QStringLiteral( "test_attr" ), QMetaType::Type::Int );
    pr->addAttributes( attrs );

    QgsFields fields;
    fields.append( attrs.back() );

    const QgsGeometry g = QgsGeometry::fromPointXY( QgsPointXY( 1.0, 1.0 ) );

    QList<QgsFeature> features;
    QgsFeature f1( fields, 1 );
    f1.setAttribute( 0, 100 );
    f1.setGeometry( g );
    QgsFeature f2( fields, 2 );
    f2.setAttribute( 0, 200 );
    f2.setGeometry( g );
    QgsFeature f3( fields, 3 );
    f3.setAttribute( 0, 300 );
    f3.setGeometry( g );
    features << f1 << f2 << f3;
    pr->addFeatures( features );
    vlDataDefinedSize->updateFields();
  }

  QVariantMap props;
  props[QStringLiteral( "name" )] = QStringLiteral( "circle" );
  props[QStringLiteral( "color" )] = QStringLiteral( "200,200,200" );
  props[QStringLiteral( "outline_color" )] = QStringLiteral( "0,0,0" );
  QgsMarkerSymbol *symbol = QgsMarkerSymbol::createSimple( props );
  QgsProperty ddsProperty = QgsProperty::fromField( QStringLiteral( "test_attr" ) );
  ddsProperty.setTransformer( new QgsSizeScaleTransformer( QgsSizeScaleTransformer::Linear, 100, 300, 10, 30 ) ); // takes ownership
  symbol->setDataDefinedSize( ddsProperty );

  QgsDataDefinedSizeLegend *ddsLegend = new QgsDataDefinedSizeLegend();
  ddsLegend->setLegendType( QgsDataDefinedSizeLegend::LegendSeparated );
  ddsLegend->setFont( QgsFontUtils::getStandardTestFont( QStringLiteral( "Bold" ) ) );

  QgsSingleSymbolRenderer *r = new QgsSingleSymbolRenderer( symbol ); // takes ownership
  r->setDataDefinedSizeLegend( ddsLegend );
  vlDataDefinedSize->setRenderer( r );

  QgsLayerTree *root = new QgsLayerTree();
  root->addLayer( vlDataDefinedSize );

  QgsLayerTreeModel legendModel( root );

  QgsMapSettings mapSettings;
  // extent and size to include only the red and green points
  mapSettings.setExtent( QgsRectangle( 10, 10, 20, 20 ) );
  mapSettings.setOutputSize( QSize( 400, 100 ) );
  mapSettings.setOutputDpi( 96 );
  mapSettings.setLayers( QgsProject::instance()->mapLayers().values() );

  QgsLayerTreeFilterSettings filterSettings( mapSettings );
  legendModel.setFilterSettings( &filterSettings );

  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );

  delete root;
}

void TestQgsLegendRenderer::testTextOnSymbol()
{
  const QString testName = QStringLiteral( "legend_text_on_symbol" );

  QgsVectorLayer *vl = new QgsVectorLayer( QStringLiteral( "Polygon" ), QStringLiteral( "Polygon Layer" ), QStringLiteral( "memory" ) );

  QgsCategoryList cats;
  QgsFillSymbol *sym_1 = new QgsFillSymbol();
  sym_1->setColor( Qt::red );
  cats << QgsRendererCategory( 1, sym_1, QStringLiteral( "Red" ) );
  QgsFillSymbol *sym_2 = new QgsFillSymbol();
  sym_2->setColor( Qt::green );
  cats << QgsRendererCategory( 2, sym_2, QStringLiteral( "Green" ) );
  QgsFillSymbol *sym_3 = new QgsFillSymbol();
  sym_3->setColor( Qt::blue );
  cats << QgsRendererCategory( 3, sym_3, QStringLiteral( "Blue" ) );
  QgsCategorizedSymbolRenderer *r = new QgsCategorizedSymbolRenderer( QStringLiteral( "test_attr" ), cats );
  vl->setRenderer( r );

  QgsDefaultVectorLayerLegend *legend = new QgsDefaultVectorLayerLegend( vl );
  legend->setTextOnSymbolEnabled( true );
  QHash<QString, QString> content;
  content[cats[0].uuid()] = "Rd";
  content[cats[2].uuid()] = "Bl";
  legend->setTextOnSymbolContent( content );
  QgsTextFormat textFormat;
  textFormat.setFont( QgsFontUtils::getStandardTestFont( QStringLiteral( "Roman" ) ) );
  textFormat.setSize( 9 );
  legend->setTextOnSymbolTextFormat( textFormat );
  vl->setLegend( legend );

  QgsLayerTree *root = new QgsLayerTree();
  root->addLayer( vl );

  QgsLayerTreeModel legendModel( root );

  QgsLegendSettings settings;
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );

  delete root;
}

void TestQgsLegendRenderer::testColumnsMixedSymbolSize()
{
  QgsMarkerSymbol *sym = new QgsMarkerSymbol();
  sym->setColor( Qt::cyan );

  std::unique_ptr<QgsLayerTree> root( new QgsLayerTree() );

  QList<QgsVectorLayer *> layers;
  for ( double size : { 4, 5, 16 } )
  {
    QgsVectorLayer *vl = new QgsVectorLayer( QStringLiteral( "Polygon" ), QStringLiteral( "Layer %1" ).arg( size ), QStringLiteral( "memory" ) );
    QgsProject::instance()->addMapLayer( vl );
    sym->setSize( size );
    vl->setRenderer( new QgsSingleSymbolRenderer( sym->clone() ) );
    root->addLayer( vl );
    layers << vl;
  }
  delete sym;

  QgsLayerTreeModel legendModel( root.get() );
  QgsLegendSettings settings;
  settings.setColumnCount( 2 );
  settings.rstyle( QgsLegendStyle::Style::Symbol ).setMargin( QgsLegendStyle::Side::Top, 9 );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  const QImage res = renderLegend( &legendModel, settings );

  for ( QgsVectorLayer *l : layers )
  {
    QgsProject::instance()->removeMapLayer( l );
  }
  QVERIFY( _verifyImage( res, QStringLiteral( "columns_with_mixed_symbol_sizes" ) ) );
}

void TestQgsLegendRenderer::testBasicJson()
{
  QgsLayerTreeModel legendModel( mRoot );

  QgsLegendSettings settings;
  settings.setTitle( QStringLiteral( "Legend" ) );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  const QJsonObject json = renderJsonLegend( &legendModel, settings );

  QCOMPARE( json["title"].toString(), QString( "Legend" ) );

  const QJsonArray root = json["nodes"].toArray();

  const QJsonObject grp1 = root[0].toObject();
  QCOMPARE( grp1["title"].toString(), QString( "Line + Polygon" ) );
  QCOMPARE( grp1["type"].toString(), QString( "group" ) );

  const QJsonArray grp1_nodes = grp1["nodes"].toArray();

  const QJsonObject line_layer = grp1_nodes[0].toObject();
  QCOMPARE( line_layer["title"].toString(), QString( "Line Layer" ) );
  QCOMPARE( line_layer["type"].toString(), QString( "layer" ) );
  const QImage line_layer_icon = base64ToImage( line_layer["icon"].toString() );
  QString test_name = "line_layer_icon";
  QVERIFY( _verifyImage( line_layer_icon, test_name, 5 ) );

  const QJsonObject polygon_layer = grp1_nodes[1].toObject();
  QCOMPARE( polygon_layer["title"].toString(), QString( "Polygon Layer" ) );
  QCOMPARE( polygon_layer["type"].toString(), QString( "layer" ) );
  const QImage polygon_layer_icon = base64ToImage( polygon_layer["icon"].toString() );
  test_name = "polygon_layer_icon";
  QVERIFY( _verifyImage( polygon_layer_icon, test_name, 5 ) );

  const QJsonObject point_layer = root[1].toObject();
  QCOMPARE( point_layer["title"].toString(), QString( "Point Layer" ) );
  QCOMPARE( point_layer["type"].toString(), QString( "layer" ) );
  const QJsonArray point_layer_symbols = point_layer["symbols"].toArray();

  const QJsonObject point_layer_symbol_red = point_layer_symbols[0].toObject();
  QCOMPARE( point_layer_symbol_red["title"].toString(), QString( "Red" ) );
  const QImage point_layer_icon_red = base64ToImage( point_layer_symbol_red["icon"].toString() );
  test_name = "point_layer_icon_red";
  QVERIFY( _verifyImage( point_layer_icon_red, test_name, 5 ) );

  const QJsonObject point_layer_symbol_green = point_layer_symbols[1].toObject();
  QCOMPARE( point_layer_symbol_green["title"].toString(), QString( "Green" ) );
  const QImage point_layer_icon_green = base64ToImage( point_layer_symbol_green["icon"].toString() );
  test_name = "point_layer_icon_green";
  QVERIFY( _verifyImage( point_layer_icon_green, test_name, 5 ) );

  const QJsonObject point_layer_symbol_blue = point_layer_symbols[2].toObject();
  QCOMPARE( point_layer_symbol_blue["title"].toString(), QString( "Blue" ) );
  const QImage point_layer_icon_blue = base64ToImage( point_layer_symbol_blue["icon"].toString() );
  test_name = "point_layer_icon_blue";
  QVERIFY( _verifyImage( point_layer_icon_blue, test_name, 5 ) );

  const QJsonObject raster_layer = root[2].toObject();
  QCOMPARE( raster_layer["title"].toString(), QString( "Raster Layer" ) );
  QCOMPARE( raster_layer["type"].toString(), QString( "layer" ) );
  const QJsonArray raster_layer_symbols = raster_layer["symbols"].toArray();

  const QJsonObject raster_layer_symbol_1 = raster_layer_symbols[0].toObject();
  QCOMPARE( raster_layer_symbol_1["title"].toString(), QString( "1" ) );
  const QImage raster_layer_icon_1 = base64ToImage( raster_layer_symbol_1["icon"].toString() );
  test_name = "raster_layer_icon_1";
  QVERIFY( _verifyImage( raster_layer_icon_1, test_name, 5 ) );

  const QJsonObject raster_layer_symbol_2 = raster_layer_symbols[1].toObject();
  QCOMPARE( raster_layer_symbol_2["title"].toString(), QString( "2" ) );
  const QImage raster_layer_icon_2 = base64ToImage( raster_layer_symbol_2["icon"].toString() );
  test_name = "raster_layer_icon_2";
  QVERIFY( _verifyImage( raster_layer_icon_2, test_name, 5 ) );
}

void TestQgsLegendRenderer::testOpacityJson()
{
  const int opacity = mVL3->opacity();
  mVL3->setOpacity( 0.5 );
  QgsLayerTreeModel legendModel( mRoot );

  QgsLegendSettings settings;
  settings.setTitle( QStringLiteral( "Legend" ) );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  const QJsonObject json = renderJsonLegend( &legendModel, settings );

  const QJsonArray root = json["nodes"].toArray();

  const QJsonObject point_layer = root[1].toObject();
  const QJsonArray point_layer_symbols = point_layer["symbols"].toArray();

  const QJsonObject point_layer_symbol_red = point_layer_symbols[0].toObject();
  const QImage point_layer_icon_red = base64ToImage( point_layer_symbol_red["icon"].toString() );
  QString test_name = "point_layer_icon_red_opacity";
  QVERIFY( _verifyImage( point_layer_icon_red, test_name, 5 ) );

  const QJsonObject point_layer_symbol_green = point_layer_symbols[1].toObject();
  const QImage point_layer_icon_green = base64ToImage( point_layer_symbol_green["icon"].toString() );
  test_name = "point_layer_icon_green_opacity";
  QVERIFY( _verifyImage( point_layer_icon_green, test_name, 5 ) );

  const QJsonObject point_layer_symbol_blue = point_layer_symbols[2].toObject();
  const QImage point_layer_icon_blue = base64ToImage( point_layer_symbol_blue["icon"].toString() );
  test_name = "point_layer_icon_blue_opacity";
  QVERIFY( _verifyImage( point_layer_icon_blue, test_name, 5 ) );

  mVL3->setOpacity( opacity );
}

void TestQgsLegendRenderer::testBigMarkerJson()
{
  QgsMarkerSymbol *sym = new QgsMarkerSymbol();
  sym->setColor( Qt::red );
  sym->setSize( sym->size() * 6 );
  QgsCategorizedSymbolRenderer *catRenderer = dynamic_cast<QgsCategorizedSymbolRenderer *>( mVL3->renderer() );
  QVERIFY( catRenderer );
  catRenderer->updateCategorySymbol( 0, sym );

  QgsLayerTreeModel legendModel( mRoot );

  QgsLegendSettings settings;
  settings.setTitle( QStringLiteral( "Legend" ) );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  const QJsonObject json = renderJsonLegend( &legendModel, settings );

  const QJsonArray root = json["nodes"].toArray();

  const QJsonObject point_layer = root[1].toObject();
  const QJsonArray point_layer_symbols = point_layer["symbols"].toArray();

  const QJsonObject point_layer_symbol_red = point_layer_symbols[0].toObject();
  const QImage point_layer_icon_red = base64ToImage( point_layer_symbol_red["icon"].toString() );
  const QString test_name = "point_layer_icon_red_big";
  QVERIFY( _verifyImage( point_layer_icon_red, test_name, 50 ) );
}

void TestQgsLegendRenderer::testLabelLegend()
{
  const QString testName( "test_label_legend" );
  QgsPalLayerSettings *labelSettings = new QgsPalLayerSettings();
  QgsTextFormat format;
  format.setFont( QgsFontUtils::getStandardTestFont( QStringLiteral( "Bold" ) ) );
  format.setSize( 12 );
  format.setNamedStyle( QStringLiteral( "Bold" ) );
  format.setColor( QColor( 255, 0, 255 ) );
  labelSettings->setFormat( format );

  labelSettings->fieldName = QStringLiteral( "test_attr" );
  QgsRuleBasedLabeling::Rule *rootRule = new QgsRuleBasedLabeling::Rule( nullptr ); //root rule
  QgsRuleBasedLabeling::Rule *labelingRule = new QgsRuleBasedLabeling::Rule( labelSettings, 0, 0, QString(), QStringLiteral( "labelingRule" ) );
  rootRule->appendChild( labelingRule );
  QgsRuleBasedLabeling *labeling = new QgsRuleBasedLabeling( rootRule );
  mVL3->setLabeling( labeling );
  const bool bkLabelsEnabled = mVL3->labelsEnabled();
  mVL3->setLabelsEnabled( true );

  QgsDefaultVectorLayerLegend *vLayerLegend = dynamic_cast<QgsDefaultVectorLayerLegend *>( mVL3->legend() );
  if ( !vLayerLegend )
  {
    QFAIL( "No vector layer legend" );
  }
  const bool bkLabelLegendEnabled = vLayerLegend->showLabelLegend();
  vLayerLegend->setShowLabelLegend( true );

  QgsLayerTreeModel legendModel( mRoot );
  QgsLegendSettings settings;

  //first test if label legend nodes are present in json
  const QJsonObject json = renderJsonLegend( &legendModel, settings );
  const QJsonArray nodes = json["nodes"].toArray();
  const QJsonObject point_layer = nodes[1].toObject();
  const QJsonArray point_layer_symbols = point_layer["symbols"].toArray();
  const QJsonObject point_layer_labeling_symbol = point_layer_symbols[3].toObject();
  const QString labelTitle = point_layer_labeling_symbol["title"].toString();

  QVERIFY( labelTitle == "labelingRule" );

  //test rendered legend against reference image
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QImage res = renderLegend( &legendModel, settings );
  QVERIFY( _verifyImage( res, testName ) );

  vLayerLegend->setShowLabelLegend( bkLabelLegendEnabled );
  mVL3->setLabelsEnabled( bkLabelsEnabled );
}

void TestQgsLegendRenderer::testHeatmap()
{
  std::unique_ptr<QgsLayerTree> root( new QgsLayerTree() );

  QgsVectorLayer *vl = new QgsVectorLayer( QStringLiteral( "Points" ), QStringLiteral( "Points" ), QStringLiteral( "memory" ) );
  QgsProject::instance()->addMapLayer( vl );
  QgsHeatmapRenderer *renderer = new QgsHeatmapRenderer();
  renderer->setColorRamp( new QgsGradientColorRamp( QColor( 255, 0, 0 ), QColor( 255, 200, 100 ) ) );
  QgsColorRampLegendNodeSettings rampSettings;

  QFont font( QgsFontUtils::getStandardTestFont( QStringLiteral( "Bold" ) ) );
  QgsTextFormat f;
  f.setSize( 16 );
  f.setFont( font );
  rampSettings.setTextFormat( f );
  rampSettings.setMinimumLabel( "min" );
  rampSettings.setMaximumLabel( "max" );
  renderer->setLegendSettings( rampSettings );

  vl->setRenderer( renderer );
  vl->setLegend( new QgsDefaultVectorLayerLegend( vl ) );
  root->addLayer( vl );

  QgsLayerTreeModel legendModel( root.get() );
  QgsLegendSettings settings;
  settings.rstyle( QgsLegendStyle::Style::Symbol ).setMargin( QgsLegendStyle::Side::Top, 9 );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  const QImage res = renderLegend( &legendModel, settings );

  QgsProject::instance()->removeMapLayer( vl );
  QVERIFY( _verifyImage( res, QStringLiteral( "heatmap" ) ) );
}

void TestQgsLegendRenderer::testFilteredVector()
{
  const QString testName = QStringLiteral( "legend_filtered_vector" );

  QgsLayerTreeModel legendModel( mRoot );

  QgsLegendSettings settings;
  settings.setTitle( QStringLiteral( "Legend" ) );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QgsLegendRenderer legendRenderer( &legendModel, settings );
  legendRenderer.proxyModel()->setFilters( Qgis::LayerFilter::VectorLayer );

  const QImage res = renderLegend( legendRenderer );
  QVERIFY( _verifyImage( res, testName ) );
}

void TestQgsLegendRenderer::testFilteredRaster()
{
  const QString testName = QStringLiteral( "legend_filtered_raster" );

  QgsLayerTreeModel legendModel( mRoot );

  QgsLegendSettings settings;
  settings.setTitle( QStringLiteral( "Legend" ) );
  setStandardTestFont( settings, QStringLiteral( "Bold" ) );
  QgsLegendRenderer legendRenderer( &legendModel, settings );
  legendRenderer.proxyModel()->setFilters( Qgis::LayerFilter::RasterLayer );

  const QImage res = renderLegend( legendRenderer );
  QVERIFY( _verifyImage( res, testName ) );
}


QGSTEST_MAIN( TestQgsLegendRenderer )
#include "testqgslegendrenderer.moc"
