﻿// Copyright (c) Microsoft Open Technologies, Inc. All rights reserved. See License.txt in the project root for license information.
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;

namespace System.Linq
{
    public static partial class EnumerableEx
    {
        /// <summary>
        /// Generates an enumerable sequence by repeating a source sequence as long as the given loop condition holds.
        /// </summary>
        /// <typeparam name="TResult">Result sequence element type.</typeparam>
        /// <param name="condition">Loop condition.</param>
        /// <param name="source">Sequence to repeat while the condition evaluates true.</param>
        /// <returns>Sequence generated by repeating the given sequence while the condition evaluates to true.</returns>
        public static IEnumerable<TResult> While<TResult>(Func<bool> condition, IEnumerable<TResult> source)
        {
            if (condition == null)
                throw new ArgumentNullException("condition");
            if (source == null)
                throw new ArgumentNullException("source");

            return WhileCore(condition, source).Concat();
        }

        static IEnumerable<IEnumerable<TSource>> WhileCore<TSource>(Func<bool> condition, IEnumerable<TSource> source)
        {
            while (condition())
                yield return source;
        }

        /// <summary>
        /// Returns an enumerable sequence based on the evaluation result of the given condition.
        /// </summary>
        /// <typeparam name="TResult">Result sequence element type.</typeparam>
        /// <param name="condition">Condition to evaluate.</param>
        /// <param name="thenSource">Sequence to return in case the condition evaluates true.</param>
        /// <param name="elseSource">Sequence to return in case the condition evaluates false.</param>
        /// <returns>Either of the two input sequences based on the result of evaluating the condition.</returns>
        public static IEnumerable<TResult> If<TResult>(Func<bool> condition, IEnumerable<TResult> thenSource, IEnumerable<TResult> elseSource)
        {
            if (condition == null)
                throw new ArgumentNullException("condition");
            if (thenSource == null)
                throw new ArgumentNullException("thenSource");
            if (elseSource == null)
                throw new ArgumentNullException("elseSource");

            return EnumerableEx.Defer(() => condition() ? thenSource : elseSource);
        }

        /// <summary>
        /// Returns an enumerable sequence if the evaluation result of the given condition is true, otherwise returns an empty sequence.
        /// </summary>
        /// <typeparam name="TResult">Result sequence element type.</typeparam>
        /// <param name="condition">Condition to evaluate.</param>
        /// <param name="thenSource">Sequence to return in case the condition evaluates true.</param>
        /// <returns>The given input sequence if the condition evaluates true; otherwise, an empty sequence.</returns>
        public static IEnumerable<TResult> If<TResult>(Func<bool> condition, IEnumerable<TResult> thenSource)
        {
            if (condition == null)
                throw new ArgumentNullException("condition");
            if (thenSource == null)
                throw new ArgumentNullException("thenSource");

            return EnumerableEx.Defer(() => condition() ? thenSource : Enumerable.Empty<TResult>());
        }

        /// <summary>
        /// Generates an enumerable sequence by repeating a source sequence as long as the given loop postcondition holds.
        /// </summary>
        /// <typeparam name="TResult">Result sequence element type.</typeparam>
        /// <param name="source">Source sequence to repeat while the condition evaluates true.</param>
        /// <param name="condition">Loop condition.</param>
        /// <returns>Sequence generated by repeating the given sequence until the condition evaluates to false.</returns>
        public static IEnumerable<TResult> DoWhile<TResult>(this IEnumerable<TResult> source, Func<bool> condition)
        {
            if (source == null)
                throw new ArgumentNullException("source");
            if (condition == null)
                throw new ArgumentNullException("condition");

            return source.Concat(While(condition, source));
        }

        /// <summary>
        /// Returns a sequence from a dictionary based on the result of evaluating a selector function.
        /// </summary>
        /// <typeparam name="TValue">Type of the selector value.</typeparam>
        /// <typeparam name="TResult">Result sequence element type.</typeparam>
        /// <param name="selector">Selector function used to pick a sequence from the given sources.</param>
        /// <param name="sources">Dictionary mapping selector values onto resulting sequences.</param>
        /// <returns>The source sequence corresponding with the evaluated selector value; otherwise, an empty sequence.</returns>
        public static IEnumerable<TResult> Case<TValue, TResult>(Func<TValue> selector, IDictionary<TValue, IEnumerable<TResult>> sources)
        {
            if (selector == null)
                throw new ArgumentNullException("selector");
            if (sources == null)
                throw new ArgumentNullException("sources");

            return Case(selector, sources, Enumerable.Empty<TResult>());
        }

        /// <summary>
        /// Returns a sequence from a dictionary based on the result of evaluating a selector function, also specifying a default sequence.
        /// </summary>
        /// <typeparam name="TValue">Type of the selector value.</typeparam>
        /// <typeparam name="TResult">Result sequence element type.</typeparam>
        /// <param name="selector">Selector function used to pick a sequence from the given sources.</param>
        /// <param name="sources">Dictionary mapping selector values onto resulting sequences.</param>
        /// <param name="defaultSource">Default sequence to return in case there's no corresponding source for the computed selector value.</param>
        /// <returns>The source sequence corresponding with the evaluated selector value; otherwise, the default source.</returns>
        public static IEnumerable<TResult> Case<TValue, TResult>(Func<TValue> selector, IDictionary<TValue, IEnumerable<TResult>> sources, IEnumerable<TResult> defaultSource)
        {
            if (selector == null)
                throw new ArgumentNullException("selector");
            if (sources == null)
                throw new ArgumentNullException("sources");
            if (defaultSource == null)
                throw new ArgumentNullException("defaultSource");

            return EnumerableEx.Defer(() =>
            {
                IEnumerable<TResult> result;
                if (!sources.TryGetValue(selector(), out result))
                    result = defaultSource;
                return result;
            });
        }

        /// <summary>
        /// Generates a sequence by enumerating a source sequence, mapping its elements on result sequences, and concatenating those sequences.
        /// </summary>
        /// <typeparam name="TSource">Source sequence element type.</typeparam>
        /// <typeparam name="TResult">Result sequence element type.</typeparam>
        /// <param name="source">Source sequence.</param>
        /// <param name="resultSelector">Result selector to evaluate for each iteration over the source.</param>
        /// <returns>Sequence concatenating the inner sequences that result from evaluating the result selector on elements from the source.</returns>
        public static IEnumerable<TResult> For<TSource, TResult>(IEnumerable<TSource> source, Func<TSource, IEnumerable<TResult>> resultSelector)
        {
            if (source == null)
                throw new ArgumentNullException("source");
            if (resultSelector == null)
                throw new ArgumentNullException("resultSelector");

            return ForCore(source, resultSelector).Concat();
        }

        static IEnumerable<IEnumerable<TResult>> ForCore<TSource, TResult>(IEnumerable<TSource> source, Func<TSource, IEnumerable<TResult>> resultSelector)
        {
            return source.Select(resultSelector);
        }
    }
}
