/***************************************************************************
 * SPDX-FileCopyrightText: 2024 S. MANKOWSKI stephane@mankowski.fr
 * SPDX-FileCopyrightText: 2024 G. DE BURE support@mankowski.fr
 * SPDX-License-Identifier: GPL-3.0-or-later
 ***************************************************************************/
/** @file
 * This file is a test script.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgtestmacro.h"
#include "skgbankincludes.h"
#include "skgimportexportmanager.h"

/**
 * The main function of the unit test
 * @param argc the number of arguments
 * @param argv the list of arguments
 */
int main(int argc, char** argv)
{
    Q_UNUSED(argc)
    Q_UNUSED(argv)

    // Init test
    SKGINITTEST(true)

    {
        // Test import GNUCASH
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GNUCASH"), err)
            SKGImportExportManager impmissing(&document1, QUrl::fromLocalFile(QLatin1String("/not-existing/missingfile.uncompressed")));
            SKGTESTERROR(QLatin1String("imp1.importFile"), impmissing.importFile(), false)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgnucash/test_data.uncompressed"));
            SKGTESTERROR(QLatin1String("GNUCASH.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("Accounts Receivable")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("231.11"))
        }
    }

    {
        // Test import GNUCASH
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GNUCASH"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgnucash/all.uncompressed"));
            SKGTESTERROR(QLatin1String("GNUCASH.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("COMPTE COURANT")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-700"))
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("COMPTE EPARGNE")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("250"))
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("Actif")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-420"))
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("TITRE")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("520"))
        }
    }

    {
        // Test import GNUCASH
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GNUCASH"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgnucash/initial_balance.uncompressed"));
            SKGTESTERROR(QLatin1String("GNUCASH.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("CCP")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("100"))
        }

        int nb = 0;
        SKGTESTERROR(QLatin1String("GNUCASH:getNbObjects(unit, PRIMARY)"), document1.getNbObjects(QLatin1String("unit"), QLatin1String("t_type='1'"), nb), true)
        SKGTEST(QLatin1String("GNUCASH:getNbObjects(unit, PRIMARY)"), nb, 1)
        SKGTESTERROR(QLatin1String("GNUCASH:getNbObjects(unit, SHARE)"), document1.getNbObjects(QLatin1String("unit"), QLatin1String("t_type='S'"), nb), true)
        SKGTEST(QLatin1String("GNUCASH:getNbObjects(unit, SHARE)"), nb, 13)
    }

    {
        // Test import GNUCASH
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GNUCASH"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgnucash/transfer-CPP-vers-PEE.uncompressed"));
            SKGTESTERROR(QLatin1String("GNUCASH.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("CCP")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("750"))
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("CEL Donald")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("150"))
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("PEEs")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("190"))
        }

        {
            int nb = 0;
            SKGTESTERROR(QLatin1String("GNUCASH:getNbObjects"), document1.getNbObjects(QLatin1String("category"), QLatin1String(""), nb), true)
            SKGTEST(QLatin1String("GNUCASH:getNbObjects"), nb, 27)
        }
    }

    {
        // Test import GNUCASH
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GNUCASH"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgnucash/transfer-PEE-vers-CCP.uncompressed"));
            SKGTESTERROR(QLatin1String("GNUCASH.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("CCP")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("1140"))
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("CEL Donald")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("150"))
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("PEEs")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("100"))
        }

        {
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GNUCASH"), err)
            SKGCategoryObject cat;
            SKGTESTERROR(QLatin1String("GNUCASH.createPathCategory"), SKGCategoryObject::createPathCategory(&document1, QLatin1String("Depenses > Frais bancaires"), cat), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(cat.getCurrentAmount()), QLatin1String("-10"))
        }

        SKGTESTERROR(QLatin1String("DOC:dump"), document1.dump(DUMPCATEGORY), true)
    }

    {
        // Test import GNUCASH
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GNUCASH"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgnucash/interet-revenue-frais.gnucash"));
            SKGTESTERROR(QLatin1String("GNUCASH.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("CCP")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("750"))
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("CEL Donald")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("150"))
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("PEEs")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("640"))
        }
    }

    {
        // Test import GNUCASH
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GNUCASH"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgnucash/interet-revenue-frais-emprunt.gnucash"));
            SKGTESTERROR(QLatin1String("GNUCASH.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("CCP")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("250"))
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("Emprunts")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-99620"))
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("Maison")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("100000"))
        }
    }

    {
        // Test import GNUCASH
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GNUCASH"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgnucash/rapprochement-incorrect.uncompressed"));
            SKGTESTERROR(QLatin1String("GNUCASH.importFile"), imp1.importFile(), true)
        }

        {
            int nb = 0;
            SKGTESTERROR(QLatin1String("GNUCASH:getNbObjects"), document1.getNbObjects(QLatin1String("operation"), QLatin1String("t_status='Y'"), nb), true)
            SKGTEST(QLatin1String("GNUCASH:getNbObjects"), nb, 7)
        }
    }

    {
        // Test import GNUCASH
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GNUCASH"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgnucash/scheduled-action.uncompressed"));
            SKGTESTERROR(QLatin1String("GNUCASH.importFile"), imp1.importFile(), true)
        }

        {
            SKGObjectBase::SKGListSKGObjectBase recurrentoperations;
            SKGTESTERROR(QLatin1String("GNUCASH:SKGListSKGObjectBase"), document1.getObjects(QLatin1String("recurrentoperation"), QLatin1String(""), recurrentoperations), true)
            SKGTEST(QLatin1String("GNUCASH:nb"), recurrentoperations.count(), 1)
            if (recurrentoperations.count() == 1) {
                SKGRecurrentOperationObject recu(recurrentoperations.at(0));
                SKGTEST(QLatin1String("GNUCASH:getDate"), recu.getDate().toString(), QDate(2010, 03, 02).toString())
                SKGTEST(QLatin1String("GNUCASH:getAutoWriteDays"), recu.getAutoWriteDays(), 10)
                SKGTEST(QLatin1String("GNUCASH:getPeriodIncrement"), recu.getPeriodIncrement(), 1)
                SKGTEST(QLatin1String("GNUCASH:getPeriodUnit"), static_cast<unsigned int>(recu.getPeriodUnit()), static_cast<unsigned int>(SKGRecurrentOperationObject::MONTH))
                SKGTEST(QLatin1String("GNUCASH:getWarnDays"), recu.getWarnDays(), 0)
                SKGTESTBOOL("GNUCASH:isAutoWriteEnabled", recu.isAutoWriteEnabled(), false)
                SKGTESTBOOL("GNUCASH:isWarnEnabled", recu.isWarnEnabled(), false)
                SKGTESTBOOL("GNUCASH:hasTimeLimit", recu.hasTimeLimit(), false)
            }
        }
    }

    {
        // Test import GNUCASH - bug 228808
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GNUCASH"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgnucash/228808.gnc"));
            SKGTESTERROR(QLatin1String("GNUCASH.importFile"), imp1.importFile(), true)
        }


        {
            SKGObjectBase::SKGListSKGObjectBase recurrentoperations;
            SKGTESTERROR(QLatin1String("GNUCASH:SKGListSKGObjectBase"), document1.getObjects(QLatin1String("recurrentoperation"), QLatin1String("1=1 ORDER BY d_date"), recurrentoperations), true)
            SKGTEST(QLatin1String("GNUCASH:nb"), recurrentoperations.count(), 2)
            if (recurrentoperations.count() == 2) {
                SKGRecurrentOperationObject recu(recurrentoperations.at(0));
                SKGOperationObject op;
                SKGTESTERROR(QLatin1String("GNUCASH:getParentOperation"), recu.getParentOperation(op), true)
                SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), op.getCurrentAmount(), -50)

                recu = recurrentoperations.at(1);
                SKGTESTERROR(QLatin1String("GNUCASH:getParentOperation"), recu.getParentOperation(op), true)
                SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), op.getCurrentAmount(), -18.75)
            }
        }
    }

    {
        // Test import GNUCASH - bug 228904
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GNUCASH"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgnucash/228904.gnucash"));
            SKGTESTERROR(QLatin1String("GNUCASH.importFile"), imp1.importFile(), true)
        }


        {
            SKGObjectBase::SKGListSKGObjectBase recurrentoperations;
            SKGTESTERROR(QLatin1String("GNUCASH:SKGListSKGObjectBase"), document1.getObjects(QLatin1String("recurrentoperation"), QLatin1String("1=1 ORDER BY d_date"), recurrentoperations), true)
            SKGTEST(QLatin1String("GNUCASH:nb"), recurrentoperations.count(), 3)
        }
    }

    {
        // Test import 228901
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GNUCASH"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgnucash/228901.gnucash"));
            SKGTESTERROR(QLatin1String("GNUCASH.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("Compte emprunt")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-9476.01"))
        }

        // Test second import
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GNUCASH"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgnucash/228901.gnucash"));
            SKGTESTERROR(QLatin1String("GNUCASH.importFile"), imp1.importFile(), true)
        }
    }

    {
        // Test import 234608
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GNUCASH"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgnucash/234608.gnc"));
            SKGTESTERROR(QLatin1String("GNUCASH.importFile"), imp1.importFile(), true)
        }
    }

    {
        // Test import 234597
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GNUCASH"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgnucash/234597.gnc"));
            SKGTESTERROR(QLatin1String("GNUCASH.importFile"), imp1.importFile(), true)
        }
        {
            int nb = 0;
            SKGTESTERROR(QLatin1String("GNUCASH:getNbObjects"), document1.getNbObjects(QLatin1String("account"), QLatin1String(""), nb), true)
            SKGTEST(QLatin1String("GNUCASH:getNbObjects"), nb, 11)
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("Compte cheques")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-260"))
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("Compte cheques (2)")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-40"))
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("Argent du porte-monnaie")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("0"))
        }
    }

    {
        // Test import 243738
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GNUCASH"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgnucash/243738.gnucash"));
            SKGTESTERROR(QLatin1String("GNUCASH.importFile"), imp1.importFile(), true)
        }
    }

    {
        // Test import without book
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GNUCASH"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgnucash/without_book.gnucash"));
            SKGTESTERROR(QLatin1String("GNUCASH.importFile"), imp1.importFile(), true)
        }
    }

    {
        // Test import 302388
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GNUCASH"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgnucash/302388.gnc"));
            SKGTESTERROR(QLatin1String("GNUCASH.importFile"), imp1.importFile(), true)
        }
    }

    {
        // Test import 325174
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GNUCASH"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgnucash/325174.gnucash"));
            SKGTESTERROR(QLatin1String("GNUCASH.importFile"), imp1.importFile(), true)
        }

        int nb = 0;
        SKGTESTERROR(QLatin1String("GNUCASH:getNbObjects(account)"), document1.getNbObjects(QLatin1String("account"), QLatin1String(""), nb), true)
        SKGTEST(QLatin1String("GNUCASH:getNbObjects(account)"), nb, 11)
    }

    {
        // Test import wallet
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GNUCASH"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgnucash/wallet.gnucash"));
            SKGTESTERROR(QLatin1String("GNUCASH.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("Espece")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGBankObject bank;
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.getBank(bank), true)
            SKGTEST(QLatin1String("GNUCASH:getName"), bank.getName(), QLatin1String(""))
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("Courant")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGBankObject bank;
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.getBank(bank), true)
            SKGTEST(QLatin1String("GNUCASH:getName"), bank.getName(), QLatin1String("GNUCASH"))
        }
    }


    {
        // 407257
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GNUCASH"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgnucash/407257.gnucash"));
            SKGTESTERROR(QLatin1String("GNUCASH.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("AccountName (Nom de compte)")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
            SKGTEST(QLatin1String("GNUCASH:getNumber"), account.getNumber(), QLatin1String("123_AccountCode(CodeCompte)"))
            SKGTEST(QLatin1String("GNUCASH:getComment"), account.getComment(), QLatin1String("DescriptionAccount(Compte123)"))

            SKGUnitObject unit;
            SKGTESTERROR(QLatin1String("GNUCASH.getUnit"), account.getUnit(unit), true)
            SKGTEST(QLatin1String("GNUCASH:getName"), unit.getName(), QLatin1String("Canadian Dollar (CAD)"))
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GNUCASH.setName"), account.setName(QLatin1String("Account2Name(no transaction)")), true)
            SKGTESTERROR(QLatin1String("GNUCASH.load"), account.load(), true)
        }
    }
    // End test
    SKGENDTEST()
}
