/***************************************************************************
 * SPDX-FileCopyrightText: 2024 S. MANKOWSKI stephane@mankowski.fr
 * SPDX-FileCopyrightText: 2024 G. DE BURE support@mankowski.fr
 * SPDX-License-Identifier: GPL-3.0-or-later
 ***************************************************************************/
/** @file
 * This file is a test script.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgtestmacro.h"
#include "skgbankincludes.h"
#include "skgimportexportmanager.h"

/**
 * The main function of the unit test
 * @param argc the number of arguments
 * @param argv the list of arguments
 */
int main(int argc, char** argv)
{
    Q_UNUSED(argc)
    Q_UNUSED(argv)

    // Init test
    SKGINITTEST(true)

    QDate now = QDate::currentDate();

    {
        // Test import SKGImportExportManager::CSV skrooge
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)

        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_CSV"), err)
            SKGImportExportManager impmissing(&document1, QUrl::fromLocalFile(QLatin1String("missingfile.csv")));
            impmissing.setAutomaticApplyRules(true);
            SKGTESTERROR(QLatin1String("imp1.importFile"), impmissing.importFile(), false)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/skrooge_partial.csv"));
            SKGTEST(QLatin1String("imp1:getImportMimeTypeFilter"), imp1.getImportMimeTypeFilter().split('\n').count(), 2);
            SKGTEST(QLatin1String("imp1:getExportMimeTypeFilter"), imp1.getExportMimeTypeFilter().split('\n').count(), 2);

            SKGImportExportManager generic;
            SKGTEST(QLatin1String("SKGImportExportManager().getImportMimeTypeFilter"), generic.getImportMimeTypeFilter().split('\n').count(), 19);
            SKGTEST(QLatin1String("SKGImportExportManager().getExportMimeTypeFilter"), generic.getExportMimeTypeFilter().split('\n').count(), 11);
            imp1.setCodec(QLatin1String(""));
            SKGTESTERROR(QLatin1String("imp1.importFile"), imp1.importFile(), true)
        }
        SKGAccountObject account;
        SKGTESTERROR(QLatin1String("ACCOUNT.getObjectByName"), SKGNamedObject::getObjectByName(&document1, QLatin1String("v_account"), QLatin1String("Courant steph"), account), true)
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-935"))
        SKGTEST(QLatin1String("document1:getCategoryForPayee"), document1.getCategoryForPayee(QLatin1String("Anthony Hopkins"), false), QLatin1String("Entertain > Movie"))
        SKGTEST(QLatin1String("document1:getCategoryForPayee"), document1.getCategoryForPayee(QLatin1String("NOT FOUND")), QLatin1String(""))
    }

    {
        // Test import QIF 1
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGBankObject bank(&document1);
        SKGAccountObject account;
        SKGUnitObject unit_euro(&document1);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_INIT"), err)

            // Creation bank
            SKGTESTERROR(QLatin1String("BANK:setName"), bank.setName(QLatin1String("CREDIT COOP")), true)
            SKGTESTERROR(QLatin1String("BANK:save"), bank.save(), true)

            // Creation account
            SKGTESTERROR(QLatin1String("BANK:addAccount"), bank.addAccount(account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setName"), account.setName(QLatin1String("Courant steph")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setNumber"), account.setNumber(QLatin1String("12345P")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:save"), account.save(), true)

            // Creation unit
            SKGTESTERROR(QLatin1String("UNIT:setName"), unit_euro.setName(QLatin1String("euro")), true)
            SKGTESTERROR(QLatin1String("UNIT:save"), unit_euro.save(), true)

            // Creation unitvalue
            SKGUnitValueObject unit_euro_val1;
            SKGTESTERROR(QLatin1String("UNIT:addUnitValue"), unit_euro.addUnitValue(unit_euro_val1), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:setQuantity"), unit_euro_val1.setQuantity(1), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:setDate"), unit_euro_val1.setDate(now), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:save"), unit_euro_val1.save(), true)
        }

        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_CSV"), err)
            // Date;Libelle;Libelle complementaire;Montant;Sens;Numero de cheque
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/coopanet.csv"));
            QMap<QString, QString> parameters = imp1.getImportParameters();
            parameters[QLatin1String("automatic_search_header")] = 'N';
            parameters[QLatin1String("columns_positions")] = QLatin1String("date|comment||amount|sign|number");
            imp1.setImportParameters(parameters);
            SKGTESTERROR(QLatin1String("QIF.setDefaultAccount"), imp1.setDefaultAccount(&account), true)
            SKGTESTERROR(QLatin1String("imp1.importFile"), imp1.importFile(), true)
        }
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-680.28"))
    }

    {
        // Test import bankperfect
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGBankObject bank(&document1);
        SKGAccountObject account1;
        SKGAccountObject account2;
        SKGUnitObject unit_euro(&document1);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_INIT"), err)

            // Creation bank
            SKGTESTERROR(QLatin1String("BANK:setName"), bank.setName(QLatin1String("CREDIT COOP")), true)
            SKGTESTERROR(QLatin1String("BANK:save"), bank.save(), true)

            // Creation account
            SKGTESTERROR(QLatin1String("BANK:addAccount"), bank.addAccount(account1), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setName"), account1.setName(QLatin1String("Courant steph")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setNumber"), account1.setNumber(QLatin1String("12345P")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:save"), account1.save(), true)

            SKGTESTERROR(QLatin1String("BANK:addAccount"), bank.addAccount(account2), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setName"), account2.setName(QLatin1String("PEL")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:save"), account2.save(), true)

            // Creation unit
            SKGTESTERROR(QLatin1String("UNIT:setName"), unit_euro.setName(QLatin1String("euro")), true)
            SKGTESTERROR(QLatin1String("UNIT:save"), unit_euro.save(), true)

            // Creation unitvalue
            SKGUnitValueObject unit_euro_val1;
            SKGTESTERROR(QLatin1String("UNIT:addUnitValue"), unit_euro.addUnitValue(unit_euro_val1), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:setQuantity"), unit_euro_val1.setQuantity(1), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:setDate"), unit_euro_val1.setDate(now), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:save"), unit_euro_val1.save(), true)
        }

        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_BP_CSV"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/bankperfect.csv"));
            SKGTESTERROR(QLatin1String("QIF.setDefaultAccount"), imp1.setDefaultAccount(&account1), true)
            SKGTESTERROR(QLatin1String("imp1.importFile"), imp1.importFile(), true)
        }

        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_BP_CSV"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/bankperfect2.csv"));
            SKGTESTERROR(QLatin1String("QIF.setDefaultAccount"), imp1.setDefaultAccount(&account2), true)
            SKGTESTERROR(QLatin1String("imp1.importFile"), imp1.importFile(), true)

            int NbOperationsMerged = 0;
            SKGTESTERROR(QLatin1String("imp1.findAndGroupTransfers"), imp1.findAndGroupTransfers(NbOperationsMerged), true)
            SKGTEST(QLatin1String("imp1:NbOperationsMerged"), NbOperationsMerged, 6)
        }
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account1.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account1.getCurrentAmount()), QLatin1String("2624.071111"))

        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account2.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account2.getCurrentAmount()), QLatin1String("1500"))

        SKGImportExportManager exp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportcsv/bankperfect.csv"));
        SKGTESTERROR(QLatin1String("SKGImportExportManager::CSV.exportFile"), exp1.exportFile(), true)

        {
            SKGStringListList oTable;
            SKGTESTERROR(QLatin1String("SKGImportExportManager::CSV.exportFile"), document1.getConsolidatedView(QLatin1String("v_operation_display"), QLatin1String("d_DATEMONTH"), QLatin1String("t_CATEGORY"), QLatin1String("f_CURRENTAMOUNT"), QLatin1String("SUM"), QLatin1String(""), oTable), true)
            QStringList dump = SKGServices::tableToDump(oTable, SKGServices::DUMP_TEXT);
            int nbl = dump.count();
            for (int i = 0; i < nbl; ++i) {
                SKGTRACE << dump.at(i) << Qt::endl;
            }
        }
        {
            SKGStringListList oTable;
            SKGTESTERROR(QLatin1String("SKGImportExportManager::CSV.exportFile"), document1.getConsolidatedView(QLatin1String("v_operation_display"), QLatin1String(""), QLatin1String("t_CATEGORY"), QLatin1String("f_CURRENTAMOUNT"), QLatin1String("SUM"), QLatin1String(""), oTable), true)
            QStringList dump = SKGServices::tableToDump(oTable, SKGServices::DUMP_TEXT);
            int nbl = dump.count();
            for (int i = 0; i < nbl; ++i) {
                SKGTRACE << dump.at(i) << Qt::endl;
            }
        }
        {
            SKGStringListList oTable;
            SKGTESTERROR(QLatin1String("SKGImportExportManager::CSV.exportFile"), document1.getConsolidatedView(QLatin1String("v_operation_display"), QLatin1String("d_DATEWEEK"), QLatin1String(""), QLatin1String("f_CURRENTAMOUNT"), QLatin1String("SUM"), QLatin1String(""), oTable), true)
            SKGTESTERROR(QLatin1String("SKGImportExportManager::CSV.exportFile"), document1.getConsolidatedView(QLatin1String("v_operation_display"), QLatin1String("d_DATEQUARTER"), QLatin1String(""), QLatin1String("f_CURRENTAMOUNT"), QLatin1String("SUM"), QLatin1String(""), oTable), true)
            SKGTESTERROR(QLatin1String("SKGImportExportManager::CSV.exportFile"), document1.getConsolidatedView(QLatin1String("v_operation_display"), QLatin1String("d_DATESEMESTER"), QLatin1String(""), QLatin1String("f_CURRENTAMOUNT"), QLatin1String("SUM"), QLatin1String(""), oTable), true)
            SKGTESTERROR(QLatin1String("SKGImportExportManager::CSV.exportFile"), document1.getConsolidatedView(QLatin1String("v_operation_display"), QLatin1String("d_DATEYEAR"), QLatin1String(""), QLatin1String("f_CURRENTAMOUNT"), QLatin1String("SUM"), QLatin1String(""), oTable), true)
            SKGTESTERROR(QLatin1String("SKGImportExportManager::CSV.exportFile"), document1.getConsolidatedView(QLatin1String("v_operation_display"), QLatin1String("d_date"), QLatin1String(""), QLatin1String("f_CURRENTAMOUNT"), QLatin1String("SUM"), QLatin1String(""), oTable), true)
            SKGTESTERROR(QLatin1String("SKGImportExportManager::CSV.exportFile"), document1.getConsolidatedView(QLatin1String("v_operation_display"), QLatin1String("d_DATEMONTH"), QLatin1String(""), QLatin1String("f_CURRENTAMOUNT"), QLatin1String("SUM"), QLatin1String(""), oTable), true)
            QStringList dump = SKGServices::tableToDump(oTable, SKGServices::DUMP_TEXT);
            int nbl = dump.count();
            for (int i = 0; i < nbl; ++i) {
                SKGTRACE << dump.at(i) << Qt::endl;
            }
        }
    }

    {
        // Test import skrooge
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_SKROOGE_CSV"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportcsv/bankperfect.csv"));
            SKGTESTERROR(QLatin1String("imp1.importFile"), imp1.importFile(), true)
        }

        SKGImportExportManager exp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportcsv/bankperfect2.csv"));
        SKGTESTERROR(QLatin1String("SKGImportExportManager::CSV.exportFile"), exp1.exportFile(), true)
    }

    {
        // Test import skrooge+optimization
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_OPTIM"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/bankperfect.csv"));
            SKGTESTERROR(QLatin1String("OPTIM.importFile"), imp1.importFile(), true)
        }

        SKGTESTERROR(QLatin1String("OPTIM.undoRedoTransaction"), document1.undoRedoTransaction(SKGDocument::UNDOLASTSAVE), true)
    }

    {
        // Test import skrooge in double to check merge
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/remi_1.csv"));
            SKGTESTERROR(QLatin1String("OPTIM.importFile"), imp1.importFile(), true)
        }
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/remi_1.csv"));
            SKGTESTERROR(QLatin1String("OPTIM.importFile"), imp1.importFile(), true)
        }

        SKGAccountObject account(&document1);
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.setName(QLatin1String("remi 1")), true)
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-767.26"))
    }
    {
        // Test import 2638120
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/XXXXXXMxxxXXXXXXX.csv"));
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }

        SKGAccountObject account(&document1);
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.setName(QLatin1String("XXXXXXMxxxXXXXXXX")), true)
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("8114.26"))
    }
    {
        // Test import 206894
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/206894.csv"));
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }

        SKGAccountObject account(&document1);
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.setName(QLatin1String("206894")), true)
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-2986.39"))
    }
    {
        // Test import 397055
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/397055.csv"));
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }

        SKGAccountObject account(&document1);
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.setName(QLatin1String("397055")), true)
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("50"))
    }
    {
        // Test import with tabulation
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/mutual fund.csv"));
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }


        SKGAccountObject account(&document1);
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.setName(QLatin1String("Janus Twenty Fund")), true)
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("24.51428572"))
    }

    {
        // Test import shares with original amount
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/209705.csv"));
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/209705_2.csv"));
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1);

            int out = 0;
            SKGTESTERROR(QLatin1String("CSV.findAndGroupTransfers"), imp1.findAndGroupTransfers(out), true)
            SKGTEST(QLatin1String("CSV:nb"), out, 2)
        }

        SKGObjectBase::SKGListSKGObjectBase grouped;
        SKGTESTERROR(QLatin1String("CSV.getObjects"), document1.getObjects(QLatin1String("operation"), QLatin1String("i_group_id!=0"), grouped), true)
        SKGTEST(QLatin1String("CSV:grouped.count"), grouped.count(), 2)
    }

    {
        // Test import transactions split and grouped
        SKGAccountObject la;
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/compte.csv"));
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)

            SKGObjectBase::SKGListSKGObjectBase banks;
            SKGTESTERROR(QLatin1String("CSV.getObjects"), document1.getObjects(QLatin1String("bank"), QLatin1String(""), banks), true)
            SKGBankObject bank(banks.at(0));
            bank.setName(QLatin1String("bp"));
            bank.save();
        }

        SKGObjectBase::SKGListSKGObjectBase grouped;
        SKGTESTERROR(QLatin1String("CSV.getObjects"), document1.getObjects(QLatin1String("operation"), QLatin1String(""), grouped), true)
        SKGTEST(QLatin1String("CSV:grouped.count"), grouped.count(), 4)
        SKGTESTERROR(QLatin1String("CSV.getObjects"), document1.getObjects(QLatin1String("operation"), QLatin1String("i_group_id!=0"), grouped), true)
        SKGTEST(QLatin1String("CSV:grouped.count"), grouped.count(), 2)

        SKGObjectBase::SKGListSKGObjectBase result;
        SKGTESTERROR(QLatin1String("DOC.getObjects"), document1.getObjects(QLatin1String("account"), QLatin1String("t_name='PEL'"), result), true)
        SKGTEST(QLatin1String("DOC.getObjects.count"), result.count(), 1)
        if (result.count() != 0) {
            la = result.at(0);
        }

        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("EXPORT_CSV"), err)
            SKGImportExportManager exp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportcsv/export_all.csv"));
            SKGTESTERROR(QLatin1String("QIF.exportFile"), exp1.exportFile(), true)
        }
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("EXPORT_CSV"), err)
            SKGImportExportManager exp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportcsv/export_la.csv"));
            QMap<QString, QString> params;
            params[QLatin1String("uuid_of_selected_accounts_or_operations")] = la.getUniqueID();
            exp1.setExportParameters(params);
            SKGTESTERROR(QLatin1String("QIF.exportFile"), exp1.exportFile(), true)
        }
    }

    {
        // Test import mmex
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/mmex.csv"));
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }

        int nb = 0;
        SKGTESTERROR(QLatin1String("CATEGORY.getNbObjects"), document1.getNbObjects(QLatin1String("category"), QLatin1String("t_fullname='Alimentation > Restaurant'"), nb), true)
        SKGTEST(QLatin1String("CATEGORY:nb"), nb, 1)

        SKGAccountObject account(&document1);
        SKGTESTERROR(QLatin1String("ACCOUNT.setName"), account.setName(QLatin1String("mmex")), true)
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("1418.44"))
    }
    {
        // Test import mmex
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/mmex_no_header.csv"));
            QMap<QString, QString> parameters = imp1.getImportParameters();
            parameters[QLatin1String("automatic_search_columns")] = 'N';
            parameters[QLatin1String("columns_positions")] = QLatin1String("date|comment|sign|amount|category");
            parameters[QLatin1String("automatic_search_header")] = 'N';
            parameters[QLatin1String("header_position")] = '0';
            imp1.setImportParameters(parameters);
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }


        SKGAccountObject account(&document1);
        SKGTESTERROR(QLatin1String("ACCOUNT.setName"), account.setName(QLatin1String("mmex no header")), true)
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("1418.44"))
    }
    {
        // 263263
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/263263.csv"));
            QMap<QString, QString> parameters = imp1.getImportParameters();
            parameters[QLatin1String("mapping_debit")] = QLatin1String("kreditrente");
            parameters[QLatin1String("automatic_search_columns")] = 'N';
            parameters[QLatin1String("columns_positions")] = QLatin1String("date||number|sign|comment|amount|amount");
            parameters[QLatin1String("automatic_search_header")] = 'N';
            parameters[QLatin1String("header_position")] = '1';
            imp1.setImportParameters(parameters);
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }

        SKGAccountObject account(&document1);
        SKGTESTERROR(QLatin1String("ACCOUNT.setName"), account.setName(QLatin1String("263263")), true)
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("800.09"))
    }
    {
        // CREDIT-DEBIT
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/credit-debit.csv"));
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }

        SKGAccountObject account(&document1);
        SKGTESTERROR(QLatin1String("ACCOUNT.setName"), account.setName(QLatin1String("credit debit")), true)
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("1500"))
    }
    {
        // BACKSLASHES
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/backslashes.csv"));
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }

        SKGAccountObject account(&document1);
        SKGTESTERROR(QLatin1String("ACCOUNT.setName"), account.setName(QLatin1String("backslashes")), true)
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("1000"))
    }
    {
        // MULTILINE
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/multiline.csv"));
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }

        SKGAccountObject account(&document1);
        SKGTESTERROR(QLatin1String("ACCOUNT.setName"), account.setName(QLatin1String("multiline")), true)
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("3000"))
    }

    {
        // MULTILINE
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/non_numerical_amount.csv"));
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }

        SKGAccountObject account(&document1);
        SKGTESTERROR(QLatin1String("ACCOUNT.setName"), account.setName(QLatin1String("non numerical amount")), true)
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-119.56"))
    }

    {
        // Test import 320112
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/320112.csv"));
            QMap<QString, QString> parameters = imp1.getImportParameters();
            parameters[QLatin1String("automatic_search_columns")] = 'N';
            parameters[QLatin1String("columns_positions")] = QLatin1String("date|mode|payee|comment|amount");
            imp1.setImportParameters(parameters);
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }

        SKGAccountObject account(&document1);
        SKGTESTERROR(QLatin1String("ACCOUNT.setName"), account.setName(QLatin1String("320112")), true)
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("6.13"))
    }

    {
        // Test import date DDMMMYYYY
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/date_DDMMMYYYY.csv"));
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }

        SKGAccountObject account(&document1);
        SKGTESTERROR(QLatin1String("ACCOUNT.setName"), account.setName(QLatin1String("date DDMMMYYYY")), true)
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getAmount"), SKGServices::doubleToString(account.getAmount(QDate(2004, 12, 31))), QLatin1String("35"))
    }

    {
        // Test import separator tab
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/tabs.csv"));
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }
    }

    {
        // Test import separator comma
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/commas.csv"));
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }
    }


    {
        // BUG 406488
        //
        // "Date","Type","Number","Payee","Withdrawal (-)","Amount"
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/406488.csv"));
            QMap<QString, QString> parameters = imp1.getImportParameters();
            parameters[QLatin1String("automatic_search_columns")] = 'N';
            parameters[QLatin1String("columns_positions")] = QLatin1String("date|mode|number|payee|amount|amount");
            imp1.setImportParameters(parameters);
            SKGError err = imp1.importFile();
            SKGTESTERROR(QLatin1String("CSV.importFile"), err, false)
            SKGTEST(QLatin1String("CSV:error message"), err.getMessage(), QLatin1String("Invalid number of columns in line 2. Expected 6. Found 1."))
        }
    }

    {
        // Test import separator comma
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/349961.csv"));
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }
        int nb = 0;
        SKGTESTERROR(QLatin1String("CSV.getObjects"), document1.getNbObjects(QLatin1String("operation"), QLatin1String(""), nb), true)
        SKGTEST(QLatin1String("CSV:nb transactions"), nb, 1)
        SKGTESTERROR(QLatin1String("CSV.getObjects"), document1.getNbObjects(QLatin1String("suboperation"), QLatin1String("d_date='2015-07-07'"), nb), true)
        SKGTEST(QLatin1String("CSV:nb subtransactions 2015-07-07"), nb, 1)
        SKGTESTERROR(QLatin1String("CSV.getObjects"), document1.getNbObjects(QLatin1String("suboperation"), QLatin1String("d_date='2015-07-08'"), nb), true)
        SKGTEST(QLatin1String("CSV:nb subtransactions 2015-07-08"), nb, 1)
        SKGTESTERROR(QLatin1String("CSV.getObjects"), document1.getNbObjects(QLatin1String("suboperation"), QLatin1String("d_date='2015-07-09'"), nb), true)
        SKGTEST(QLatin1String("CSV:nb subtransactions 2015-07-09"), nb, 1)
    }

    {
        // Test import separator comma
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/362231.csv"));
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }
        int nb = 0;
        SKGTESTERROR(QLatin1String("CSV.getObjects"), document1.getNbObjects(QLatin1String("operation"), QLatin1String("d_date!='0000-00-00'"), nb), true)
        SKGTEST(QLatin1String("CSV:nb operation"), nb, 1)
        SKGTESTERROR(QLatin1String("CSV.getObjects"), document1.getNbObjects(QLatin1String("suboperation"), QLatin1String("d_date='2016-05-15'"), nb), true)
        SKGTEST(QLatin1String("CSV:nb subtransactions 2016-05-15"), nb, 1)
        SKGTESTERROR(QLatin1String("CSV.getObjects"), document1.getNbObjects(QLatin1String("suboperation"), QLatin1String("d_date='2016-05-20'"), nb), true)
        SKGTEST(QLatin1String("CSV:nb subtransactions 2016-05-20"), nb, 1)
    }

    {
        // 381562
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGBankObject bank(&document1);
        SKGAccountObject account;
        SKGUnitObject unit_euro(&document1);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_INIT"), err)

            // Creation bank
            SKGTESTERROR(QLatin1String("BANK:setName"), bank.setName(QLatin1String("CREDIT COOP")), true)
            SKGTESTERROR(QLatin1String("BANK:save"), bank.save(), true)

            // Creation account
            SKGTESTERROR(QLatin1String("BANK:addAccount"), bank.addAccount(account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setName"), account.setName(QLatin1String("Courant steph")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setNumber"), account.setNumber(QLatin1String("DE00 1234 5678 9012 3456 78")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:save"), account.save(), true)

            // Creation unit
            SKGTESTERROR(QLatin1String("UNIT:setName"), unit_euro.setName(QLatin1String("euro")), true)
            SKGTESTERROR(QLatin1String("UNIT:save"), unit_euro.save(), true)

            // Creation unitvalue
            SKGUnitValueObject unit_euro_val1;
            SKGTESTERROR(QLatin1String("UNIT:addUnitValue"), unit_euro.addUnitValue(unit_euro_val1), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:setQuantity"), unit_euro_val1.setQuantity(1), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:setDate"), unit_euro_val1.setDate(now), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:save"), unit_euro_val1.save(), true)
        }

        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_CSV"), err)
            // Date;Libelle;Libelle complementaire;Montant;Sens;Numero de cheque
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/DE00 1234 5678 9012 3456 78.csv"));
            QMap<QString, QString> parameters = imp1.getImportParameters();
            parameters[QLatin1String("automatic_search_header")] = 'N';
            parameters[QLatin1String("columns_positions")] = QLatin1String("date|comment||amount|sign|number");
            imp1.setImportParameters(parameters);
            SKGTESTERROR(QLatin1String("imp1.importFile"), imp1.importFile(), true)
        }
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-680.28"))
    }

    {
        // Test 411958
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/411958.csv"));
            QMap<QString, QString> parameters = imp1.getImportParameters();
            parameters[QLatin1String("mapping_date")] = QLatin1String("^Buchungstag");
            parameters[QLatin1String("mapping_payee")] = QLatin1String("^payee|^tiers|^.*Zahlungspflichtiger");
            parameters[QLatin1String("mapping_comment")] = QLatin1String("^comment|^libell?|^d?tail|^info|^Vorgang.*");
            parameters[QLatin1String("mapping_amount")] = QLatin1String("^value|^amount|^valeur|^montant|^credit|^debit|^Umsatz");
            parameters[QLatin1String("mapping_account")] = QLatin1String("^Konto");
            parameters[QLatin1String("mapping_sign")] = QLatin1String("^sign|^sens");
            parameters[QLatin1String("mapping_unit")] = QStringLiteral("^Währung");
            parameters[QLatin1String("mapping_debit")] = QLatin1String("^S");
            parameters[QLatin1String("automatic_search_columns")] = 'Y';
            parameters[QLatin1String("automatic_search_header")] = 'Y';
            imp1.setImportParameters(parameters);

            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }

        SKGAccountObject account(&document1);
        SKGTESTERROR(QLatin1String("ACCOUNT.setName"), account.setName(QLatin1String("411958")), true)
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("658.88"))
    }

    {
        // Test import with footer
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/footer.csv"));
            QMap<QString, QString> parameters = imp1.getImportParameters();
            parameters[QLatin1String("automatic_search_columns")] = 'N';
            parameters[QLatin1String("columns_positions")] = QLatin1String("date|mode|payee|amount|");
            imp1.setImportParameters(parameters);
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }

        SKGAccountObject account(&document1);
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.setName(QLatin1String("footer")), true)
        SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-2"))
    }

    {
        // Test import separator comma
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/commas_2.csv"));
            QMap<QString, QString> parameters = imp1.getImportParameters();
            parameters[QLatin1String("automatic_search_columns")] = 'N';
            parameters[QLatin1String("header_position")] = QLatin1String("1");
            parameters[QLatin1String("columns_positions")] = QLatin1String("|date||payee|amount||||mode");
            imp1.setImportParameters(parameters);
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }
    }

    {
        // Test import BUG 420557
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/420557_1.csv"));
            QMap<QString, QString> parameters = imp1.getImportParameters();
            parameters[QLatin1String("automatic_search_columns")] = 'N';
            parameters[QLatin1String("header_position")] = QLatin1String("1");
            parameters[QLatin1String("columns_positions")] = QLatin1String("date|account|idgroup|idtransaction|value|unit");
            imp1.setImportParameters(parameters);
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }

        int nb = 0;
        SKGTESTERROR(QLatin1String("OPERATION.getNbObjects"), document1.getNbObjects(QLatin1String("operation"), QLatin1String("i_group_id=1"), nb), true)
        SKGTEST(QLatin1String("OPERATION:nb"), nb, 2)
    }

    {
        // Test import BUG 420557
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/420557_2.csv"));
            QMap<QString, QString> parameters = imp1.getImportParameters();
            parameters[QLatin1String("automatic_search_columns")] = 'N';
            parameters[QLatin1String("header_position")] = QLatin1String("1");
            parameters[QLatin1String("columns_positions")] = QLatin1String("date|account|idgroup|idtransaction|value|unit");
            imp1.setImportParameters(parameters);
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
        }

        int nb = 0;
        SKGTESTERROR(QLatin1String("OPERATION.getNbObjects"), document1.getNbObjects(QLatin1String("operation"), QLatin1String("i_group_id=1"), nb), true)
        SKGTEST(QLatin1String("OPERATION:nb"), nb, 2)
    }

    {
        // 421302
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Create an account without number
            SKGBEGINTRANSACTION(document1, QLatin1String("CREATE_ACCOUNT"), err)
            SKGTESTERROR(QLatin1String("DOC.addOrModifyAccount"), document1.addOrModifyAccount(QLatin1String("COURANT"), QLatin1String("5823485"), QLatin1String("BANK")), true)
            SKGObjectBase account;
            SKGTESTERROR(QLatin1String("DOC.getObject"), document1.getObject(QLatin1String("v_account"), QLatin1String("t_name='COURANT'"), account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.setProperty"), account.setProperty(QLatin1String("alias"), QLatin1String("XXXX485")), true)
        }
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_CSV"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/XXXX485.csv"));
            QMap<QString, QString> parameters = imp1.getImportParameters();
            parameters[QLatin1String("automatic_search_columns")] = 'N';
            parameters[QLatin1String("header_position")] = QLatin1String("1");
            parameters[QLatin1String("columns_positions")] = QLatin1String("|date||payee|amount||||mode");
            imp1.setImportParameters(parameters);
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)
            int nb2 = 0;
            SKGTESTERROR(QLatin1String("imp1.getNbObjects"), document1.getNbObjects(QLatin1String("account"), QString(), nb2), true)
            SKGTEST(QLatin1String("ACCOUNT:nb"), SKGServices::intToString(nb2), QLatin1String("1"))
        }
    }

    {
        // 494159
        QLocale::setDefault(QLocale::German);

        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_CSV"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportcsv/494159.csv"));
            QMap<QString, QString> parameters = imp1.getImportParameters();
            parameters[QLatin1String("automatic_search_columns")] = 'N';
            parameters[QLatin1String("header_position")] = QLatin1String("1");
            parameters[QLatin1String("columns_positions")] = QLatin1String("|date||payee|||||amount");
            imp1.setImportParameters(parameters);
            SKGTESTERROR(QLatin1String("CSV.importFile"), imp1.importFile(), true)

            document1.dump(DUMPOPERATION | DUMPACCOUNT);
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.setName(QLatin1String("494159")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
            SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-12468.93"))
        }
    }

    // End test
    SKGENDTEST()
}  // NOLINT(readability/fn_size)
