/*
 * This file is part of the coreboot project.
 *
 * Copyright (C) 2009 One Laptop per Child, Association, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#if CONFIG_HAVE_ACPI_RESUME
#include <arch/acpi.h>
#endif
#include "../vx800.h"

#include <arch/io.h>
#include "pci_rawops.h"

static const struct VIA_PCI_REG_INIT_TABLE mSbStage1InitTbl[] = {
	// Combine Stage1 registers
	{0x00, 0xFF, SB_LPC_REG(0x41), 0x40, 0x40},

	// Acpi init registers in sb stage1
	{0x00, 0xFF, SB_LPC_REG(0x40), 0x04, 0x04},	// Enable 4D0/4D1 support
	{0x00, 0xFF, SB_LPC_REG(0x4E), 0x00, 0x08},	// Enable RTC port 74/75
	{0x00, 0xFF, SB_LPC_REG(0x51), 0x0D, 0x0D},	// and KBC
	{0x00, 0xFF, SB_LPC_REG(0x52), 0x0F, 0x09},	// Enable Serial IRQ
	{0x00, 0xFF, SB_LPC_REG(0x67), 0x00, 0x04},	// Set FERR voltage to 1.5v
	{0x00, 0xFF, SB_LPC_REG(0x98), 0xFF, 0x00},	// Disable GP3 Timer

	{0x00, 0xFF, SB_IDEC_REG(0xb9), 0x01, 0x01},

	{0x00, 0xFF, SB_VLINK_REG(0xE6), 0xFF, 0x39},	// Enable SMM A-Seg, MSI and Io APIC
	///// SPI-BAR.
	//// SPI_BASE_ADDRESS = 0xFED1 0000
	{0x00, 0xFF, SB_LPC_REG(0xBC), 0xFF, 0x00},
	{0x00, 0xFF, SB_LPC_REG(0xBD), 0xFF, 0xD1},
	{0x00, 0xFF, SB_LPC_REG(0xBE), 0xFF, 0xFE},
//      {0x00, 0xFF, ((0x11<<16)|(0x00<<8)|0xBC), 0xFF, 0x00},//this , for the different macro
//      {0x00, 0xFF, ((0x11<<16)|(0x00<<8)|0xBD), 0xFF, 0xD1},
//      {0x00, 0xFF, ((0x11<<16)|(0x00<<8)|0xBE), 0xFF, 0xFE},
	///// End of 2008-04-17

	{0x00, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},	// End of Table
};

static const struct VIA_PCI_REG_INIT_TABLE mNbStage2InitTable[] = {
	// D0F0: AGP Feature. For 3353, No AGP Feature

	// D0F2~D0F3 is configured by MemoryInit Peim

	// D0F4: NB PMU
	{ 0x00, 0xFF, NB_PMU_REG(0x84), 0x00, 0xDB },
	{ 0x00, 0xFF, NB_PMU_REG(0x85), 0x00, 0x05 },
	{ 0x00, 0xFF, NB_PMU_REG(0x89), 0x00, 0xF8 },
	{ 0x00, 0xFF, NB_PMU_REG(0x8B), 0x00, 0xBF },
	{ 0x00, 0xFF, NB_PMU_REG(0x8D), 0x00, 0xFC },
	{ 0x00, 0xFF, NB_PMU_REG(0x8E), 0x00, 0x19 },
	{ 0x00, 0xFF, NB_PMU_REG(0x8F), 0x03, 0x00 },
	{ 0x00, 0xFF, NB_PMU_REG(0x90), 0x00, 0xFF },
	{ 0x00, 0xFF, NB_PMU_REG(0x91), 0x00, 0xFF },
	{ 0x00, 0xFF, NB_PMU_REG(0x92), 0x00, 0xCC },
	{ 0x00, 0xFF, NB_PMU_REG(0xA0), 0x00, 0x80 },
	{ 0x00, 0xFF, NB_PMU_REG(0xA1), 0x00, 0xE0 },
	{ 0x00, 0xFF, NB_PMU_REG(0xA2), 0x00, 0xD6 },
	{ 0x00, 0xFF, NB_PMU_REG(0xA3), 0x00, 0x80 },
	{ 0x00, 0xFF, NB_PMU_REG(0xA8), 0x00, 0x20 },

	// D0F5: NB APIC, PXPTRF and MSGC
	//Note: the Rx6A, RCRBH Base Address, is not set, which is related to PCIE Root Complex.
	//Note: the Rx60, Extended CFG Address. Support and Rx61, Extended CFG Address, are set by NB Peim that is in the PEI Phase.
	//Note: the Rx42, APIC Interrupt((BT_INTR)) Control, is set by NB Peim that is in PEI phase.
	{ 0x00, 0xFF, NB_PXPTRF_REG(0x50), 0x00, 0x00 },
	{ 0x00, 0xFF, NB_PXPTRF_REG(0x54), 0x00, 0x80 },
	{ 0x00, 0xFF, NB_PXPTRF_REG(0x55), 0x00, 0x04 },
	{ 0x00, 0xFF, NB_PXPTRF_REG(0x58), 0x00, 0x00 },
	{ 0x00, 0xFF, NB_PXPTRF_REG(0x59), 0x00, 0x02 },
	{ 0x00, 0xFF, NB_PXPTRF_REG(0x5E), 0x00, 0x00 },
	{ 0x00, 0xFF, NB_PXPTRF_REG(0x5F), 0x00, 0x06 },
	{ 0x00, 0xFF, NB_PXPTRF_REG(0x80), 0x00, 0x18 },	//Set RVC1DM, RTHBHIT, RUWRDYD, RUPRRDY1, RUWPOPHD to 1.
	{ 0x00, 0xFF, NB_PXPTRF_REG(0x82), 0x00, 0x00 },	//Set RVC1RPSW, RVC1RQ1T to 1.
	{ 0x00, 0xFF, NB_PXPTRF_REG(0x83), 0x00, 0x81 },
	{ 0x00, 0xFF, NB_PXPTRF_REG(0x84), 0x00, 0x28 },
	{ 0x00, 0xFF, NB_PXPTRF_REG(0x85), 0x00, 0xC0 },
	{ 0x00, 0xFF, NB_MSGC_REG(0xA3), 0x00, 0x01 },	// RWAKEEN
	//{ 0x00, 0xFF, NB_PXPTRF_REG(0x64), 0x40, 0x00 }, //RTDNP2B32EN
	{ 0x00, 0xFF, NB_PXPTRF_REG(0xF3), 0xFC, 0x20 },
	{ 0x00, 0xFF, NB_PXPTRF_REG(0x85), 0x00, 0x00 },	//RP2P1ABORT


// fine-tune
// If no settings, C7 will hang or reboot in XP, but CN will not.
	{ 0x00, 0xFF, NB_HOST_REG(0x51), 0x84, 0x00 },
	{ 0x00, 0xFF, NB_HOST_REG(0x52), 0x0F, 0x03 },
	{ 0x00, 0xFF, NB_HOST_REG(0x54), 0x04, 0x00 },
	{ 0x00, 0xFF, NB_HOST_REG(0x55), 0x04, 0x00 },
	{ 0x00, 0xFF, NB_HOST_REG(0x59), 0x09, 0x01 },
	{ 0x00, 0xFF, NB_HOST_REG(0x5C), 0x10, 0x10 },
	{ 0x00, 0xFF, NB_HOST_REG(0x5F), 0x0E, 0x08 },
	{ 0x00, 0xFF, NB_HOST_REG(0x92), 0xFF, 0x04 },	// ACPI Base addr
	{ 0x00, 0xFF, NB_HOST_REG(0x97), 0x01, 0x01 },	// APIC MSI
	{ 0x00, 0xFF, NB_HOST_REG(0x99), 0x02, 0x00 },	// APIC MSI
	//GTL
	{ 0x00, 0xFF, NB_HOST_REG(0x73), 0xFF, 0x66 },
	{ 0x00, 0xFF, NB_HOST_REG(0xB2), 0xFF, 0x33 },
	{ 0x00, 0xFF, NB_HOST_REG(0xB3), 0xFF, 0x33 },
	{ 0x00, 0xFF, NB_HOST_REG(0xBC), 0xFF, 0x33 },
	{ 0x00, 0xFF, NB_HOST_REG(0xBD), 0xFF, 0x33 },
	{ 0x00, 0xFF, NB_HOST_REG(0xC5), 0x30, 0x20 },
	{ 0x00, 0xFF, NB_HOST_REG(0xC8), 0x10, 0x00 },


	// End of Table
	{0x00, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},	// End of Table
};

static const struct VIA_PCI_REG_INIT_TABLE mBusControllerInitTable[] = {
	// D17F0: LPC
	{ 0x00, 0xFF, SB_LPC_REG(0x40), 0x44, 0x44 },	//  Enable I/O Recovery Time,  4D0/4D1 Support
	{ 0x00, 0xFF, SB_LPC_REG(0x42), 0xF8, 0xF0 },	//  ENLBUF, GINTREN, FLUSHEN, RBRSTRD
	{ 0x00, 0xFF, SB_LPC_REG(0x43), 0x0F, 0x0B },	//  RENDTX, ENWBTO, ENRBTO
	//{ 0x00, 0xFF, SB_LPC_REG(0x46), 0x00, 0x10 },     // It is related to INTH#
	//{ 0x00, 0xFF, SB_LPC_REG(0x48), 0x00, 0x0C },      //RMRPW, RIRPW  // Reserved in 409 by Eric

	// Internal RTC, Mouse, Keyboard // set in PEI by Eric
	//{ 0x00, 0xFF, SB_LPC_REG(0x51), 0x10, 0x0D },      // Enable Internal RTC, Internal PS2 Mouse/Keyboard

	// RTC
	{ 0x00, 0xFF, SB_LPC_REG(0x58), 0x00, 0x01 },	//RTC Rx32 Map to Centrury Byte

	//{ 0x00, 0xFF, SB_LPC_REG(0x40), 0x00, 0x02 },    // RDMEGAS
	//{ 0x00, 0xFF, SB_LPC_REG(0x4E), 0x00, 0x08 },  // Enable RTC port 74/75, ENEXRTC // set in PEI by Eric

	// Serial IRQ  // set in PEI by Eric
	//{ 0x00, 0xFF, SB_LPC_REG(0x52), 0x0F, 0x09 },    // Enable Serial IRQ, Start Frame Width is 6 PCI Clock.

	// Enable 4D0h/4D1h Port
	//{ 0x00, 0xFF, SB_LPC_REG(0x40), 0x00, 0x04 },    // EISAXT // set in PEI by Eric

	// Config ROM Interface
	// Enable SPI/Set SPI Memory Base Address
	// It is initialized in PEI Phase

	// Subsystem ID/Vendor ID Back Door
	{ 0x00, 0xFF, SB_LPC_REG(0x70), 0xFF, 0x06 },
	{ 0x00, 0xFF, SB_LPC_REG(0x71), 0xFF, 0x11 },
	{ 0x00, 0xFF, SB_LPC_REG(0x72), 0xFF, 0x09 },
	{ 0x00, 0xFF, SB_LPC_REG(0x73), 0xFF, 0x34 },

	{ 0x00, 0xFF, SB_LPC_REG(0x4C), 0xC0, 0x40 },
	{ 0x00, 0xFF, SB_LPC_REG(0x5B), 0x00, 0x51 },	// Orgin value 0x53, modify for 409 by Eric
	{ 0x00, 0xFF, SB_LPC_REG(0x67), 0x03, 0x01 },


	{ 0x00, 0xFF, SB_LPC_REG(0x50), 0x7E, 0x00 },	// Setting PCI device enable
	{ 0x00, 0xFF, SB_LPC_REG(0x51), 0xD0, 0x00 },	// Setting PCI device enable
	{ 0x00, 0xFF, SB_VLINK_REG(0xD1), 0x04, 0x00 },	// Setting HDAC enable
	{0x00, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},	// End of Table
};

static const struct VIA_PCI_REG_INIT_TABLE mPCI1InitTable[] = {
	//PCI1 Programming Sequence
	//(1)Configure D17F7
	{ 0x00, 0xFF, SB_VLINK_REG(0x04), 0x00, 0x03 },
	{ 0x00, 0xFF, SB_VLINK_REG(0x0C), 0x00, 0x08 },	// Reserved in 409 by Eric
	{ 0x00, 0xFF, SB_VLINK_REG(0x4F), 0x40, 0x41 },	//RENPPB, RP2CFLSH
	{ 0x00, 0xFF, SB_VLINK_REG(0x77), 0x00, 0x48 },	//ROP2CFLSH, RFFTMR[1:0]. ROP2CFLSH work with Rx4F[0](RP2CFLSH) assertion
	// { 0x00, 0xFF, SB_VLINK_REG(0x51), 0x00, 0x80 },      //RSUB_DEC_P2P, RSUBDECOD(Window xp). If Bit7 is set, PCI lock will occured.
	// { 0x00, 0xFF, SB_VLINK_REG(0x51), 0x00, 0x81 },      //RSUB_DEC_P2P, RSUBDECOD(Window Vista)
	//(2)Configure D19F0
	{ 0x00, 0xFF, SB_P2PB_REG(0x04), 0x00, 0x07 },

	//(3)Performance Recommended Setting

	//Save Power
	{ 0x00, 0xFF, SB_VLINK_REG(0xE2), 0x1F, 0x01 },
	{ 0x00, 0xFF, SB_VLINK_REG(0xE3), 0xF1, 0x5E },
	{ 0x00, 0xFF, SB_VLINK_REG(0x74), 0x40, 0x00 },
	//Enhence Host To PCI cycle performance and PCI-To-Host Cycle performance
	{ 0x00, 0xFF, SB_VLINK_REG(0x70), 0x00, 0x82 },
	{ 0x00, 0xFF, SB_VLINK_REG(0x71), 0x30, 0xC0 },
	{ 0x00, 0xFF, SB_VLINK_REG(0x72), 0x00, 0xEE },

	//Cycle Control
	{ 0x00, 0xFF, SB_VLINK_REG(0x73), 0x00, 0x01 },
	{ 0x00, 0xFF, SB_VLINK_REG(0x74), 0x00, 0x0C },
	//Arbitration control
	{ 0x00, 0xFF, SB_VLINK_REG(0x75), 0x00, 0x0F },
	{ 0x00, 0xFF, SB_VLINK_REG(0x76), 0x00, 0xD0 },
	{0x00, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},	// End of Table
};

static const struct VIA_PCI_REG_INIT_TABLE mCCAInitTable[] = {
	{ 0x00, 0xFF, SB_VLINK_REG(0xFC), 0x02, 0x08 },	//RVWREQ, ROABKDOOR

	//CCA's Register Programming sequence
	{ 0x00, 0xFF, SB_VLINK_REG(0x50), 0x00, 0x08 },	//Config Azalia's upstream cycle high priority and other low priority
	{ 0x00, 0xFF, SB_VLINK_REG(0x51), 0x40, 0x80 },	//Disable bypass asynchronous circuit
	{ 0x00, 0xFF, SB_VLINK_REG(0x52), 0x00, 0x11 },	// Set SM Internal Device and HDAC Occupy Timer
	{ 0x00, 0xFF, SB_VLINK_REG(0x53), 0x00, 0x11 },	// Set SM Internal Device and HDAC Promote Timer
	{ 0x00, 0xFF, SB_VLINK_REG(0x54), 0xFF, 0x02 },	//Use SB internal devices's original REQ
	{ 0x00, 0xFF, SB_VLINK_REG(0x73), 0x10, 0x00 },	//RPINOWSC. Enable APIC Cycle Block P2C Write Cycle
	{ 0x00, 0xFF, SB_VLINK_REG(0x74), 0x00, 0x3C },	//RLCKXP2C, RFSBVK.
	{ 0x00, 0xFF, SB_VLINK_REG(0xE1), 0x07, 0x00 },	//RBLKAPIC, RAZC3
	{ 0x00, 0xFF, SB_VLINK_REG(0x7C), 0x04, 0x02 },	//RNMIFSB, RFSBVK
	{ 0x00, 0xFF, SB_VLINK_REG(0xE0), 0xF0, 0x90 },	//RCCA_NEWCLK, RCCA_CLKON. Use New dynamic clock scheme
	{ 0x00, 0xFF, SB_VLINK_REG(0xE7), 0xFF, 0x00 },	//Let CCA use dynamic clock.
	//The CCA is also relate to D17F0
	//{ 0x00, 0xFF, SB_LPC_REG(0x49), 0x1F, 0x00 },       //Disable CCA Test Mode
	{ 0x00, 0xFF, SB_LPC_REG(0x74), 0xFF, 0x00 },	// Let DMA cycles from internal devices directly go to NB // Reserved in 409 by Eric
	{0x00, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},	// End of Table
};

static const struct VIA_PCI_REG_INIT_TABLE IDEC_INIT[] = {
	//{0x00, 0xFF, SB_IDEC_REG(0x09), 0x00, 0x05}, //set to native mode
	{0x00, 0xFF, SB_IDEC_REG(0x04), 0x00, 0x07},
	//{0x00, 0xFF, SB_VLINK_REG(0x7C), 0x00, 0x7F},
	{0x00, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},	// End of Table
};

static const struct VIA_PCI_REG_INIT_TABLE mSbApicInitTable[] = {
	{ 0x00, 0xFF, SB_LPC_REG(0x4D), 0x04, 0x00 },
	{ 0x00, 0xFF, SB_LPC_REG(0x5B), 0x0E, 0x00 },
	{ 0x00, 0xFF, SB_LPC_REG(0x6C), 0x08, 0x00 },
	{ 0x00, 0xFF, SB_LPC_REG(0x58), 0x00, 0x40 },
	{ 0x00, 0xFF, SB_VLINK_REG(0x74), 0x00, 0x04 },
	//{ 0x00, 0xFF, SB_VLINK_REG(0x7C), 0x00, 0x7F },
	{ 0x00, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },	// End of Table
};

void AcpiInit(void)
{
	device_t_raw rawdevice = 0;
	u8 sbchiprev;
	rawdevice = PCI_RAWDEV(0, 0x11, 0);
	// Set the PMIO base io address
	pci_rawmodify_config16(rawdevice, 0x88, VX800_ACPI_IO_BASE,
			       0xff80);
	// Enable PMIO
	pci_rawmodify_config16(rawdevice, 0x80, 0x8000, 0x8000);
	// Enable Soft Resume
	outw(inw(VX800_ACPI_IO_BASE + 0x04) | 0x8000,
	     VX800_ACPI_IO_BASE + 0x04);

	// Get SB Revision
	sbchiprev = pci_rawread_config8(rawdevice, 0xf6);
	printk(BIOS_DEBUG, "SB chip revision =%x\n", sbchiprev);

	// Fill Register Table
	via_pci_inittable(sbchiprev, mSbStage1InitTbl);

	// Close all SMI/Io Traps
	outb(0x00, VX800_ACPI_IO_BASE + 0x42);
}

void Stage2NbInit(void)
{
	device_t_raw rawdevice = 0;
	u8 nbchiprev;
	u32 subid = 0;
	rawdevice = PCI_RAWDEV(0, 0, 4);
	nbchiprev = pci_rawread_config8(rawdevice, 0xf6);
	printk(BIOS_DEBUG, "NB chip revision =%x\n", nbchiprev);

	via_pci_inittable(nbchiprev, mNbStage2InitTable);

	rawdevice = PCI_RAWDEV(0, 0, 0);

	subid = PCI_DEVICE_ID_VIA_VX855_D0F0 << 16 + PCI_VENDOR_ID_VIA;
	pci_rawwrite_config32(rawdevice, 0x2C, subid);

	//vx855 NB no pcie bus
	//vx855 NB no apic
}

void IDECSupportOption(u8 sbchiprev)
{
	pci_rawmodify_config8(PCI_RAWDEV(0, 0x11, 0), 0x50, 0, 0x08);

	pci_rawmodify_config8(PCI_RAWDEV(0, 0xf, 0), 0x45, 0x00, 0x80);
	pci_rawmodify_config8(PCI_RAWDEV(0, 0xf, 0), 0x0A, 0x01, 0xFF);
	pci_rawmodify_config8(PCI_RAWDEV(0, 0xf, 0), 0x45, 0x80, 0x00);
	pci_rawmodify_config8(PCI_RAWDEV(0, 0xf, 0), 0x40, 0x02, 0x00);

	pci_rawmodify_config8(PCI_RAWDEV(0, 0xf, 0), 0x09, 0x00, 0x05);	//COMPATIBLE MODE
//      pci_rawmodify_config8(PCI_RAWDEV(0, 0xf, 0), 0x09, 0x05, 0x05);//native MODE

	via_pci_inittable(sbchiprev, IDEC_INIT);
}

void InitIDEC(u8 sbchiprev)
{
	IDECSupportOption(sbchiprev);
}

void InitUHCI(u8 Number, u8 bEnable)
{
	u8 Mask, Value;
	u16 D16;
	u8 BaseAddress;
	u8 BitShift;
	// USB Device 16
	// Function : Number
	//         0      :      0
	//         1      :      1
	//         2      :      2
	// The BitShift is got from Datasheet.

	switch (Number) {
	case 0:
		BaseAddress = 0;
		BitShift = 4;
		break;
	case 1:
		BaseAddress = 1;
		BitShift = 5;
		break;
	case 2:
	default:
		BaseAddress = 2;
		BitShift = 2;
		break;
	}

	if (bEnable) {
		Mask = 0x1 << BitShift;
		Value = 0x0;
	} else {
		Mask = 0x0;
		Value = 0x1 << BitShift;
	}
	pci_rawmodify_config8(PCI_RAWDEV(0, 0x11, 0), 0x50, Value, Mask);


	if (bEnable) {
		D16 = 0;
		pci_rawwrite_config16(PCI_RAWDEV(0, 0x10, BaseAddress),
				      0x20, D16);

		// Config some Control Register
		Mask = 0x00;
		Value = 0x12;

		pci_rawmodify_config8(PCI_RAWDEV(0, 0x10, BaseAddress),
				      0x41, Value, Mask);
		Mask = 0x00;
		Value = 0xEB;
		pci_rawmodify_config8(PCI_RAWDEV(0, 0x10, BaseAddress),
				      0x4B, Value, Mask);
	}
	return;
}

static const struct VIA_PCI_REG_INIT_TABLE mEHCIInitTable[] = {
	//EHCI
	{ 0x00, 0xFF, SB_EHCI_REG(0x43), 0x00, 0xC0 },
	{ 0x00, 0xFF, SB_EHCI_REG(0x50), 0x00, 0x80 },
	{ 0x00, 0xFF, SB_EHCI_REG(0x48), 0x20, 0x9E },
	{ 0x00, 0xFF, SB_EHCI_REG(0x49), 0x10, 0x68 },
	{ 0x00, 0xFF, SB_EHCI_REG(0x4B), 0x00, 0x69 },
	{ 0x00, 0xFF, SB_EHCI_REG(0x4D), 0x00, 0x94 },
	{ 0x00, 0xFF, SB_EHCI_REG(0x52), 0x08, 0x00 },
	{ 0x00, 0xFF, SB_EHCI_REG(0x5A), 0x00, 0x8A },
	{ 0x00, 0xFF, SB_EHCI_REG(0x5B), 0x00, 0x89 },
	{ 0x00, 0xFF, SB_EHCI_REG(0x5C), 0x00, 0x03 },
	{ 0x00, 0xFF, SB_EHCI_REG(0x5D), 0x00, 0x9A },
	{ 0x00, 0xFF, SB_EHCI_REG(0x5E), 0x00, 0x00 },
	{ 0x00, 0xFF, SB_EHCI_REG(0x6B), 0x00, 0x00 },
	{ 0x00, 0xFF, SB_EHCI_REG(0x6D), 0x00, 0x00 },
	{ 0x00, 0xFF, SB_EHCI_REG(0x6F), 0xF0, 0x00 },
	{ 0x00, 0xFF, SB_EHCI_REG(0x4E), 0x01, 0x01 },
	{ 0x00, 0xFF, SB_EHCI_REG(0x4F), 0x00, 0x11 },
	{ 0x00, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },	// End of Table
};

void InitEHCI(u8 Number, u8 bEnable)
{
	u8 Mask, Value;
	u8 EHCIRevision;

	if (bEnable) {
		Mask = 0x1 << 1;
		Value = 0x0;
	} else {
		Mask = 0x0;
		Value = 0x1 << 1;
	}
	pci_rawmodify_config8(PCI_RAWDEV(0, 0x11, 0), 0x50, Value, Mask);


	if (bEnable) {
		// Get Chipset Revision
		EHCIRevision =
		    pci_rawread_config8(PCI_RAWDEV(0, 0x10, 4), 0xF6);
		printk(BIOS_DEBUG, "EHCI Revision =%x\n", EHCIRevision);
		via_pci_inittable(EHCIRevision, mEHCIInitTable);
	}
}

#define TRUE  1
void InitUSBC(u8 sbchiprev)
{
	InitUHCI(0, TRUE);
	InitUHCI(1, TRUE);
	InitUHCI(2, TRUE);
	InitEHCI(0, TRUE);
}

void WriteSbApicIndexedReg(u8 Idx, u32 Data)
{
	u32 Data32;
	u32 ApicIdxAdr = VX800SB_APIC_BASE;
	u32 ApicDataAdr = VX800SB_APIC_BASE + VX800SB_APIC_DATA_OFFSET;
	*((u8 *) ApicIdxAdr) = Idx;
	Data32 = (*((u32 *) ApicDataAdr));	//this read is needed when write APIC ID ,dont know why.
	Data32 = Data;
	*((u32 *) ApicDataAdr) = Data32;
}

void SbApicMmioRegInit(void)
{
	u32 Offset;
	WriteSbApicIndexedReg(3, 1);
	WriteSbApicIndexedReg(0, 4);
	for (Offset = 0x10; Offset < VX800SB_APIC_ENTRY_NUMBER;
	     Offset += 2) {
		WriteSbApicIndexedReg(Offset + 1, 0);
		WriteSbApicIndexedReg(Offset, 0x10000);
	}
}

void SbApicInit(u8 sbchiprev)
{
	via_pci_inittable(sbchiprev, mSbApicInitTable);
	SbApicMmioRegInit();
}

void SbAcpiInit(void)
{
	u8 Mask, Value;
	// Enable ACPI
	Mask = 0x01;
	Value = 0x01;
	io_rawmodify_config8(VX800_ACPI_IO_BASE + 0x04, Value, Mask);
}

#define HPET_ENABLE_BIT			0x80
#define R_SB_HPET_CONTROL		0x68
#define HPET_BASE_ADDRESS		0xFED0	// 0xFED00000
#define R_SB_HPET_ADDRESS		0x69

void HpetInit(void)
{
	u8 HpetEnable = HPET_ENABLE_BIT;
	u16 HpetBase = HPET_BASE_ADDRESS;
	pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), R_SB_HPET_CONTROL,
			     HpetEnable);
	pci_rawwrite_config16(PCI_RAWDEV(0, 0x11, 0),
			      R_SB_HPET_ADDRESS + 1, HpetBase);
}

static const struct VIA_PCI_REG_INIT_TABLE mPMUInitTable[] = {
	// Power Management
	{ 0x00, 0xFF, SB_LPC_REG(0x80), 0x00, 0x20 },
	{ 0x00, 0xFF, SB_LPC_REG(0x8C), 0x02, 0x00 },
	{ 0x00, 0xFF, SB_LPC_REG(0x8D), 0x00, 0x18 },

	//Miscellaneous Configuration 1
	{ 0x00, 0xFF, SB_LPC_REG(0x94), 0xF0, 0x28 },
	{ 0x00, 0xFF, SB_LPC_REG(0x95), 0x00, 0xC1 },
	{ 0x00, 0xFF, SB_LPC_REG(0x96), 0xFF, 0x10 },
	{ 0x00, 0xFF, SB_LPC_REG(0x97), 0x00, 0xB2 },

	//Voltage Change Function Enable
	{ 0x00, 0xFF, SB_LPC_REG(0x9F), 0x00, 0x21 },
	//Internal PCIe and NM PLL Control
	{ 0x00, 0xFF, SB_LPC_REG(0xE2), 0x00, 0xEA },

	{ 0x00, 0xFF, SB_LPC_REG(0xE7), 0x00, 0x80 },
	{ 0x00, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },	// End of Table
};

void InitPMU(u8 sbchiprev)
{
	u8 Mask, Value;
	// Set PMU Registers
	via_pci_inittable(sbchiprev, mPMUInitTable);


	// Set SCI IRQ and its level trigger
	Mask = 0x0F;
	Value = 0x09;
	pci_rawmodify_config8(PCI_RAWDEV(0, 0x11, 0), 0x82, Value, Mask);

	Mask = 0x02;
	Value = 0x02;
	io_rawmodify_config8(0x4d1, Value, Mask);
}

#define R_SB_MULTI_FUNCTION_SELECT_1  0xE4
#define R_SB_CX_STATE_BREAK_EVENT_ENABLE_1  0xE6
#define PMIO_PROCESSOR_CONTROL  0x26
#define R_SB_PCI_ARBITRATION_2      0x76
#define R_SB_AUTO_SWITCH_P_STATE        0x8A

void InitCPUCStatueSupport()
{
	u8 Mask, Value;

	// Config Cx State
	// Now it is C2 & C4 Up Down Mode
	Mask = 0xFF;
	Value = 0x30;
	pci_rawmodify_config8(PCI_RAWDEV(0, 0x11, 0), R_SB_CX_STATE_BREAK_EVENT_ENABLE_1, Value, Mask);	//SB_LPC_REG

	Mask = 0xFF;
	Value = 0x1F;
	io_rawmodify_config8(VX800_ACPI_IO_BASE + PMIO_PROCESSOR_CONTROL,
			     Value, Mask);

	Mask = 0x00;
	Value = 0x80;
	pci_rawmodify_config8(PCI_RAWDEV(0, 0x11, 7), R_SB_PCI_ARBITRATION_2, Value, Mask);	//SB_VLINK_REG

	Mask = 0xFF;
	Value = 0x00;
	pci_rawmodify_config8(PCI_RAWDEV(0, 0x11, 0), R_SB_MULTI_FUNCTION_SELECT_1, Value, Mask);	//SB_VLINK_REG

	Mask = 0xFF;
	Value = 0x1F;
	pci_rawmodify_config8(PCI_RAWDEV(0, 0x11, 0), R_SB_AUTO_SWITCH_P_STATE, Value, Mask);	//SB_VLINK_REG
}

void InitSBPM(u8 sbchiprev)
{
	InitPMU(sbchiprev);
	SbAcpiInit();
	InitCPUCStatueSupport();
}

void Stage2SbInit(void)
{
	device_t_raw rawdevice = 0;
	u8 sbchiprev;

	rawdevice = PCI_RAWDEV(0, 11, 0);
	sbchiprev = pci_rawread_config8(rawdevice, 0xf6);
	printk(BIOS_DEBUG, "SB chip revision =%x\n", sbchiprev);

	//SBBasicInit
	via_pci_inittable(sbchiprev, mBusControllerInitTable);
	via_pci_inittable(sbchiprev, mPCI1InitTable);
	via_pci_inittable(sbchiprev, mCCAInitTable);


	InitIDEC(sbchiprev);

	InitUSBC(sbchiprev);

	InitSBPM(sbchiprev);

	SbApicInit(sbchiprev);

	HpetInit();

	//pci_rawmodify_config8(PCI_RAWDEV(0, 0x11, 0), 0x50, 0x76, 0);//SB_VLINK_REG

}

void init_VIA_chipset(void)
{
	printk(BIOS_DEBUG, "In: init_VIA_chipset\n");
	//1.nbstage1 is done in raminit.
	//2.sbstage1
	AcpiInit();
	//3.nbstage2
	Stage2NbInit();

	//4.sbstage2
	Stage2SbInit();

	//5.open hdac
	pci_rawmodify_config32(PCI_RAWDEV(0, 0x11, 7), 0xd1, 0, 0x04);
	printk(BIOS_DEBUG, "End: init_VIA_chipset\n");
}

/**
 * @brief Main function of the DRAM part of coreboot.
 *
 * Coreboot is divided into Pre-DRAM part and DRAM part.
 *
 *
 * Device Enumeration:
 *	In the dev_enumerate() phase,
 */

void main(void)
{
	struct lb_memory *lb_mem;
#if CONFIG_HAVE_ACPI_RESUME
	void *wake_vec;
#endif

	u16 tmp;
	tmp = inw(VX800_ACPI_IO_BASE + 0x04);
	acpi_sleep_type = ((tmp & (7 << 10)) >> 10) == 1 ? 3 : 0;

	u8 y, x;
	init_VIA_chipset();
	printk(BIOS_EMERG, "file '%s', line %d\n\n", __FILE__, __LINE__);

#if 0
	pci_rawwrite_config8(PCI_RAWDEV(0, 0, 4), 0xa3, 0x80);
	pci_rawwrite_config8(PCI_RAWDEV(0, 17, 7), 0x60, 0x20);
	pci_rawwrite_config8(PCI_RAWDEV(0, 17, 7), 0xE5,
			     pci_rawread_config8(PCI_RAWDEV(0, 3, 0),
						 0x88));
#endif

	pci_rawmodify_config8(PCI_RAWDEV(0, 0x11, 0), 0x51, 0x40, 0x40);	//close CE-ATA (Consumer Electronics-ATA) and NFC

	//pci_rawmodify_config8(PCI_RAWDEV(0, 0x11, 0), 0x50, 0x0, 0x40);//open USB Device Mode Enable
	pci_rawmodify_config8(PCI_RAWDEV(0, 0x11, 0), 0x50, 0x40, 0x40);	//close USB Device Mode

	//pci_rawmodify_config8(PCI_RAWDEV(0, 0x11, 0), 0x50, 0x04, 0x04);//close USB 1.1 UHCI Port 4-5
	//pci_rawmodify_config8(PCI_RAWDEV(0, 0x11, 0), 0x50, 0x02, 0x02);//close USB 2.0 ehci


	//pci_rawmodify_config8(PCI_RAWDEV(0, 0x11, 0), 0x50, 0x00, 0x76);//open all usb and usb mode
	//pci_rawmodify_config8(PCI_RAWDEV(0, 0x11, 0), 0x50, 0x76, 0x76);//close all usb

	printk(BIOS_INFO, "=================SB 50h=%02x \n",
		    pci_rawread_config8(PCI_RAWDEV(0, 0x11, 0), 0x50));

	/* FIXME: Is there a better way to handle this? */
	init_timer();
	printk(BIOS_EMERG, "file '%s', line %d\n\n", __FILE__, __LINE__);

	/* Find the devices we don't have hard coded knowledge about. */
	dev_enumerate();
	printk(BIOS_EMERG, "file '%s', line %d\n\n", __FILE__, __LINE__);
#if 0
	x = y = 0;
	printk(BIOS_INFO, "dump ehci3 \n");
	for (; x < 16; x++) {
		y = 0;
		for (; y < 16; y++) {
			printk(BIOS_INFO, "%02x ",
				    pci_rawread_config8(PCI_RAWDEV
							(0, 0x10, 4),
							x * 16 + y));
		}
		printk(BIOS_INFO, "\n");
	}
#endif

	post_code(0x66);
	/* Now compute and assign the bus resources. */
	dev_configure();
	printk(BIOS_EMERG, "file '%s', line %d\n\n", __FILE__, __LINE__);
#if 0
	x = y = 0;
	printk(BIOS_INFO, "dump ehci3 \n");
	for (; x < 16; x++) {
		y = 0;
		for (; y < 16; y++) {
			printk(BIOS_INFO, "%02x ",
			   pci_rawread_config8(PCI_RAWDEV (0, 0x10, 4), x * 16 + y));
		}
		printk(BIOS_INFO, "\n");
	}
#endif

	post_code(0x88);
	/* Now actually enable devices on the bus */
	dev_enable();
	printk(BIOS_EMERG, "file '%s', line %d\n\n", __FILE__, __LINE__);
	/* And of course initialize devices on the bus */
#if 0
	x = y = 0;
	printk(BIOS_INFO, "dump ehci3 \n");
	for (; x < 16; x++) {
		y = 0;
		for (; y < 16; y++) {
			printk(BIOS_INFO, "%02x ",
			    pci_rawread_config8(PCI_RAWDEV (0, 0x10, 4), x * 16 + y));
		}
		printk(BIOS_INFO, "\n");
	}
#endif

	dev_initialize();
	post_code(0x89);
	printk(BIOS_EMERG, "file '%s', line %d\n\n", __FILE__, __LINE__);

//          pci_rawwrite_config16(PCI_RAWDEV(0, 0xf, 0), 0xBA, 0x0571);

#if 0
	x = y = 0;
	printk(BIOS_INFO, "dump ehci3 \n");
	for (; x < 16; x++) {
		y = 0;
		for (; y < 16; y++) {
			printk(BIOS_INFO, "%02x ",
			    pci_rawread_config8(PCI_RAWDEV (0, 0x10, 4), x * 16 + y));
		}
		printk(BIOS_INFO, "\n");
	}
#endif

#if 0
	y = pci_rawread_config8(PCI_RAWDEV(0, 0xf, 0), 0x0d);
	y &= 0x0f;
	y |= 0x40;
	pci_rawwrite_config8(PCI_RAWDEV(0, 0xf, 0), 0x0d, y);
#endif

#if 0
	static const d0f0pcitable[5] = { 0xD0, 0, 0, 0, 0xFD };
	static const d0f2pcitable[16 * 7 + 1] = {
		0x88, 0xF8, 0xEF, 0x44, 0x7C, 0x24, 0x63, 0x01, 0x00, 0x09,
		    0x00, 0x00, 0x10, 0xA2, 0x88, 0xCE,
		0xFF, 0x0F, 0x00, 0xAA, 0x0A, 0x00, 0x00, 0x00, 0x01, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0xCC, 0x66, 0xAA, 0x55, 0x30, 0x38, 0x0C, 0x00, 0x00, 0x00,
		    0x00, 0x22, 0x00, 0xAA, 0x00, 0x00,
		0x44, 0x44, 0x55, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x0B, 0x00, 0x08, 0x00, 0x00, 0x00, 0x0A, 0x01, 0x41, 0x06,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x88, 0x56, 0x70, 0x77, 0x77, 0x07, 0x77, 0x77, 0x04,
		    0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
		0x77, 0x77, 0x33, 0x33, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
		    0x77, 0x77, 0x44, 0x44, 0x14, 0x00,
		0x75
	};

	static const d0f4pcitable[16 * 6 + 3] = {
		0x30, 0x00, 0x00,
		0xFF, 0xFF, 0xCC, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x80, 0xE0, 0xD6, 0x80, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x04, 0x00, 0x00, 0x04, 0x04,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x12, 0x12, 0x00, 0x00, 0x08, 0xF4,
		    0x01, 0x01, 0x79, 0x79, 0x0A, 0x00,
	};
	static const d0f5pcitable[16 * 10] = {
		0x13, 0x0E, 0x00, 0x00, 0xD2, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x18, 0x9A, 0x00, 0x81, 0x28, 0xC0, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x26, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	};

	static const d0f7pcitable[16 * 9] = {
		0x00, 0x2A, 0x00, 0xE0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x29, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	};
	static const d1f0pcitable[3] = {
		0x01, 0x80, 0x40
	};

	static const dcf0pcitable[96] = {
		0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x01, 0x00, 0xC2, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00,
		    0x40, 0x01, 0x03, 0x01, 0x7E, 0x01,
		0x08, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF9,
		    0x03, 0x00, 0x00, 0x00, 0x00, 0x00,
	};

	static const d10f4pcitable[48] = {
		0x00, 0x20, 0x43, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x9E, 0x68,
		    0x00, 0x09, 0x13, 0x94, 0x03, 0x10,
		0x80, 0x60, 0x11, 0xBF, 0x00, 0xFF, 0x0F, 0x00, 0x04, 0x0B,
		    0xCC, 0xCC, 0x00, 0xCC, 0x00, 0x00,
		0x20, 0x20, 0x01, 0x00, 0x05, 0x00, 0x00, 0x00, 0x01, 0x00,
		    0x00, 0x00, 0x00, 0x20, 0x00, 0xC0,
	};

	static const d11f0pcitable[16 * 12] = {
		0x44, 0x40, 0xF0, 0x0B, 0x00, 0x00, 0x00, 0x03, 0x00, 0x20,
		    0x00, 0x00, 0x04, 0x00, 0x08, 0x00,
		0xC1, 0x4D, 0x19, 0x80, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00,
		    0x00, 0x51, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x80, 0x00,
		    0xD0, 0xFE, 0x00, 0x00, 0x00, 0x00,
		0x06, 0x11, 0x09, 0x84, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x20, 0x00, 0x00, 0x00,
		0x20, 0x84, 0x49, 0x00, 0x88, 0x00, 0x00, 0x00, 0x01, 0x08,
		    0x1F, 0x00, 0x07, 0x1A, 0x00, 0x00,
		0x00, 0x6E, 0xBC, 0x88, 0x28, 0xC1, 0x10, 0x80, 0x00, 0x80,
		    0x20, 0x88, 0x00, 0x00, 0x00, 0xAD,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00,
		    0x00, 0x00, 0x10, 0xD0, 0xFE, 0x90,
		0x01, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x01, 0x04, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0xE7, 0x03, 0xA0, 0x60, 0x20, 0xC0, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
	};

	static const d11f7pcitable[192] = {
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x80, 0x43,
		0x08, 0x80, 0x11, 0x11, 0x02, 0x0F, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x80, 0x00, 0x00, 0x20, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x80, 0x00, 0x00, 0x00, 0x00,
		0x82, 0xC8, 0xEE, 0x01, 0x0C, 0x0F, 0xD0, 0x48, 0x00, 0x00,
		    0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
		0x07, 0x00, 0x21, 0x00, 0x0C, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x93, 0x08, 0x00, 0x5E, 0x00, 0x80, 0x29, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x48, 0x00, 0x00, 0x00,
	};

#define OPTION_1 1
#define NOOPTION_1 1
#ifdef OPTION_1
	static const OPTION_1_d11f0pcitable[16 * 12] = {
		0x44, 0x80, 0xf0, 0x0b, 0x00, 0x00, 0x00, 0x03, 0x00, 0x20,
		    0x00, 0x00, 0x04, 0x01, 0x08, 0x00,
		0xc0, 0x4d, 0x19, 0x80, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00,
		    0x00, 0x53, 0x00, 0xfe, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x04, 0x80, 0x00,
		    0xd0, 0xfe, 0x00, 0x00, 0xdf, 0x00,
		0x06, 0x11, 0x53, 0x83, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x20, 0x84, 0x4a, 0x00, 0xda, 0x40, 0x00, 0x00, 0x01, 0x40,
		    0x1f, 0x00, 0x07, 0x18, 0x00, 0x00,
		0x00, 0x2e, 0xbc, 0x00, 0x28, 0xc1, 0x10, 0x80, 0x00, 0x80,
		    0x08, 0x88, 0x00, 0x00, 0x00, 0xad,
		0x06, 0x11, 0x53, 0x83, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x40, 0x80, 0x40,
		    0x00, 0x00, 0x00, 0xd3, 0xfe, 0x53,
		0x01, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x01, 0x41, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0xeb, 0x03, 0xa0, 0x60, 0x20, 0x80, 0x00, 0x00,
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
		    0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
	};
#endif

	u8 i;
/* error form ---- but add the chance to resume
	for(i=0;i<5;i++) {
		pci_rawwrite_config8(PCI_RAWDEV(0, 0, 0), i, d0f0pcitable[i+0xcb]);
	}
*/

/* RO reg
	for(i=0;i<5;i++) {
		pci_rawwrite_config8(PCI_RAWDEV(0, 0, 0), i+0xcb, d0f0pcitable[i]);
	}
*/

//boot ok, resume still err in linux
#if 1
	for (i = 0; i < 9; i++) {
		pci_rawwrite_config8(PCI_RAWDEV(0, 0, 2), i + 0x50,
				     d0f2pcitable[i]);
	}
	//9 is warm reset reg,   // boot err in coreboot
	for (i = 10; i < 64; i++) {
		pci_rawwrite_config8(PCI_RAWDEV(0, 0, 2), i + 0x50,
				     d0f2pcitable[i]);
	}
	//0x90 look d0f2 appendixA1 ,   if set this to 09 or 0b, then some ddr2 will crash.
	for (i = 65; i < 113; i++) {
		pci_rawwrite_config8(PCI_RAWDEV(0, 0, 2), i + 0x50,
				     d0f2pcitable[i]);
	}
#endif
#ifdef OPTION_1
	pci_rawwrite_config8(PCI_RAWDEV(0, 0, 2), 0x66, 0x09);
	pci_rawwrite_config8(PCI_RAWDEV(0, 0, 2), 0x70, 0xdd);
	//  pci_rawwrite_config8(PCI_RAWDEV(0, 0, 2), 0x90, 0x09);
	pci_rawwrite_config8(PCI_RAWDEV(0, 0, 2), 0x92, 0x40);

#endif



#if 1


//d0f3
/*		*/
	//      pci_rawwrite_config8(PCI_RAWDEV(0, 0, 3), 0x86, 0x3b);  setting, my lspci is 0x29
	//set bit4 cause the ide not be found
//              pci_rawwrite_config8(PCI_RAWDEV(0, 0, 3), 0x86, 0x2b);
	//set bit1 cause the ide not be found

//              pci_rawwrite_config8(PCI_RAWDEV(0, 0, 3), 0x86, 0x29);
	pci_rawwrite_config8(PCI_RAWDEV(0, 0, 3), 0x95, 0x05);
	pci_rawwrite_config8(PCI_RAWDEV(0, 0, 3), 0x99, 0x12);

	pci_rawwrite_config8(PCI_RAWDEV(0, 0, 3), 0xde, 0x00);
#endif

//boot ok,  resume err in coreboot
#if 1
	for (i = 0; i < 99; i++) {
		pci_rawwrite_config8(PCI_RAWDEV(0, 0, 4), i + 0x8d,
				     d0f4pcitable[i]);
	}
#endif

#ifdef OPTION_1
	pci_rawwrite_config8(PCI_RAWDEV(0, 0, 4), 0xe9, 0x90);
	pci_rawwrite_config8(PCI_RAWDEV(0, 0, 4), 0xec, 0x0);
	pci_rawwrite_config8(PCI_RAWDEV(0, 0, 4), 0xed, 0x0);
	pci_rawwrite_config8(PCI_RAWDEV(0, 0, 4), 0xee, 0x0);
#endif


#if 1
//boot ok, resume still err in linux
	for (i = 0; i < 160; i++) {
		pci_rawwrite_config8(PCI_RAWDEV(0, 0, 5), i + 0x60,
				     d0f5pcitable[i]);
	}
	for (i = 0; i < 144; i++) {
		pci_rawwrite_config8(PCI_RAWDEV(0, 0, 7), i + 0x60,
				     d0f7pcitable[i]);
	}
	for (i = 0; i < 3; i++) {
		pci_rawwrite_config8(PCI_RAWDEV(0, 1, 0), i + 0xb0,
				     d1f0pcitable[i]);
	}
	for (i = 0; i < 96; i++) {
		pci_rawwrite_config8(PCI_RAWDEV(0, 0xc, 0), i + 0x40,
				     dcf0pcitable[i]);
	}
#endif

#ifdef OPTION_1
	pci_rawwrite_config8(PCI_RAWDEV(0, 0, 7), 0x61, 0x0);
	pci_rawwrite_config8(PCI_RAWDEV(0, 0, 7), 0x63, 0x0);
	pci_rawwrite_config8(PCI_RAWDEV(0, 0, 7), 0x76, 0xd0);
	pci_rawwrite_config8(PCI_RAWDEV(0, 0xc, 0), 0x88, 0x81);
	pci_rawwrite_config8(PCI_RAWDEV(0, 0xc, 0), 0x89, 0x01);
	pci_rawwrite_config8(PCI_RAWDEV(0, 0xc, 0), 0x8A, 0x60);
#endif

//d15f0

#if 1
	pci_rawwrite_config8(PCI_RAWDEV(0, 0x10, 0), 0x4a, 0xa2);	// no affect.
	pci_rawwrite_config8(PCI_RAWDEV(0, 0x10, 1), 0x4a, 0xa2);
	pci_rawwrite_config8(PCI_RAWDEV(0, 0x10, 2), 0x4a, 0xa2);

//boot ok, resume still err in linux,  and if disable USB, then all ok
//      for(i=0;i<48;i++){
	for (i = 0; i < 44; i++) {
		pci_rawwrite_config8(PCI_RAWDEV(0, 0x10, 4), i + 0x40,
				     d10f4pcitable[i]);
	}
#endif

//#ifdef NOOPTION_1
#if 0
	pci_rawwrite_config8(PCI_RAWDEV(0, 0x10, 4), 0x6b, 0x01);
	pci_rawwrite_config8(PCI_RAWDEV(0, 0x10, 4), 0x6d, 0x00);
	pci_rawwrite_config8(PCI_RAWDEV(0, 0x10, 4), 0x6e, 0x08);
	pci_rawwrite_config8(PCI_RAWDEV(0, 0x10, 4), 0x6f, 0x80);
#endif

#if 1
//before (11.0)is add, s3 resume has already always dead in first resume(more frequenly), and sleep ok
//      for(i=0;i<192;i++){
	for (i = 0; i < 6; i++) {
		pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), i + 0x40,
				     d11f0pcitable[i]);
	}
	//6 is uart and dvp vcp,   will have // HAVE no com1 ,and no gui show,textmode ok ,s3 sleep ok, resume fail

	//7-18 is my familar part
	for (i = 7; i < 18; i++) {	//sleep ok ,resume sleep err 2
		pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), i + 0x40,
				     d11f0pcitable[i]);
	}

	for (i = 18; i < 21; i++) {	//sleep ok ,   sleep err 1, resume
		pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), i + 0x40,
				     d11f0pcitable[i]);
	}
	//0x55 56 57 irq intA#B#C# linkA#linkB#linkC#
	for (i = 24; i < 27; i++) {	//sleep ok , resume sleep err 1  resume  1
		pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), i + 0x40,
				     d11f0pcitable[i]);
	}
	//5b port 80h
	pci_rawmodify_config8(PCI_RAWDEV(0, 0x11, 0), 0x5b, 0x0, 0x08);
	//          i++;
	//      pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), i+0x40, d11f0pcitable[i]);

	for (i = 28; i < 72; i++) {	//sleep ok , resume  sleep err 1 , resume 1ci
		pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), i + 0x40,
				     d11f0pcitable[i]);
	}
	//7273ACPI BASE

	for (i = 74; i < 112; i++) {	//boot ok, resume still err in linux
		pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), i + 0x40,
				     d11f0pcitable[i]);
	}

	//B0B4B5 dvp vcp,  if copy this ,then no uart, no gui(of unbuntu)
	// pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), 0xb0, d11f0pcitable[112]);
	i = pci_rawread_config8(PCI_RAWDEV(0, 17, 0), 0xB0);
	//multiplex with VCP
	//    i = i | 0x30;
	i = i & 0xf7;
	pci_rawwrite_config8(PCI_RAWDEV(0, 17, 0), 0xB0, i);



	for (i = 113; i < 114; i++) {	//boot ok, resume still err in linux
		pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), i + 0x40,
				     d11f0pcitable[i]);
	}

	for (i = 115; i < 116; i++) {	//boot ok, resume still err in linux
		pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), i + 0x40,
				     d11f0pcitable[i]);
	}


	for (i = 118; i < 192; i++) {	//boot ok, resume still err in linux
		pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), i + 0x40,
				     d11f0pcitable[i]);
	}
#endif
#ifdef NOOPTION_1
//      for(i=0;i<192;i++){
	for (i = 0; i < 6; i++) {
		pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), i + 0x40,
				     OPTION_1_d11f0pcitable[i]);
	}
	//6 is uart and dvp vcp,   will have // HAVE no com1 ,and no gui show,textmode ok ,s3 sleep ok, resume fail

	//7-18 is my familar part
	for (i = 7; i < 18; i++) {	//   sleep err 2
		pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), i + 0x40,
				     OPTION_1_d11f0pcitable[i]);
	}

	for (i = 18; i < 21; i++) {	//sleep ok , resume ???
		pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), i + 0x40,
				     d11f0pcitable[i]);
	}
	//0x55 56 57 irq intA#B#C# linkA#linkB#linkC#
	for (i = 24; i < 27; i++) {	//sleep ok , resume ???
		pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), i + 0x40,
				     d11f0pcitable[i]);
	}
	//5b port 80h
	i++;
	pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), i + 0x40,
			     OPTION_1_d11f0pcitable[i]);

	for (i = 28; i < 72; i++) {	//sleep ok , resume???
		pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), i + 0x40,
				     OPTION_1_d11f0pcitable[i]);
	}
	//7273ACPI BASE

	for (i = 74; i < 112; i++) {	//boot ok, resume still err in linux
		pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), i + 0x40,
				     OPTION_1_d11f0pcitable[i]);
	}

	//B0B4B5 dvp vcp,  if copy this ,then no uart, no gui(of unbuntu)
	// pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), 0xb0, OPTION_1_d11f0pcitable[112]);
	i = pci_rawread_config8(PCI_RAWDEV(0, 17, 0), 0xB0);
	//multiplex with VCP
	//    i = i | 0x30;
	i = i & 0xf7;
	pci_rawwrite_config8(PCI_RAWDEV(0, 17, 0), 0xB0, i);



	for (i = 113; i < 114; i++) {	//boot ok, resume still err in linux
		pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), i + 0x40,
				     OPTION_1_d11f0pcitable[i]);
	}

	for (i = 115; i < 116; i++) {	//boot ok, resume still err in linux
		pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), i + 0x40,
				     OPTION_1_d11f0pcitable[i]);
	}

	for (i = 118; i < 192; i++) {	//boot ok, resume still err in linux
		pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), i + 0x40,
				     OPTION_1_d11f0pcitable[i]);
	}
#endif

#if 1
	pci_rawwrite_config16(PCI_RAWDEV(0, 0xf, 0), 0xBA, PCI_DEVICE_ID_VIA_VX855_IDE);	//5324
	pci_rawwrite_config16(PCI_RAWDEV(0, 0xf, 0), 0xBE,
			      PCI_DEVICE_ID_VIA_VX855_IDE);
	pci_rawwrite_config16(PCI_RAWDEV(0, 0x11, 0), 0xA0,
			      PCI_VENDOR_ID_VIA);
	pci_rawwrite_config16(PCI_RAWDEV(0, 0x11, 0), 0xA2, PCI_DEVICE_ID_VIA_VX855_LPC);	//8353
	i = pci_rawread_config8(PCI_RAWDEV(0, 0x11, 0), 0x79);
	i &= ~0x40;
	i |= 0x40;
	pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 0), 0x79, i);
	pci_rawwrite_config16(PCI_RAWDEV(0, 0x11, 0), 0x72,
			      PCI_DEVICE_ID_VIA_VX855_LPC);

//boot ok, resume still err in linux
	for (i = 0; i < 192; i++) {
		pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 7), i + 0x40,
				     d11f7pcitable[i]);
	}
#endif
#ifdef OPTION_1
	pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 7), 0x61, 0x2a);
	pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 7), 0x63, 0xa0);
	pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 7), 0x64, 0xaa);
	pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 7), 0x84, 0x0);
	pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 7), 0x88, 0x02);
	pci_rawwrite_config8(PCI_RAWDEV(0, 0x11, 7), 0xe6, 0x3f);
#endif
	pci_rawwrite_config8(PCI_RAWDEV(0, 0x14, 0), 0x40, 0x20);
	pci_rawwrite_config8(PCI_RAWDEV(0, 0x14, 0), 0x41, 0x31);

#ifdef OPTION_1
	pci_rawwrite_config8(PCI_RAWDEV(0, 0x14, 0), 0x40, 0x00);
#endif
#endif

	u8 i911;
	//disable CHB L.L
	//set VGA memory selection
	i911 = pci_rawread_config8(PCI_RAWDEV(0, 0x1, 0), 0xb0);
	i911 &= 0xF8;
	//ByteVal |= 0x03;
	i911 |= 0x01;
	pci_rawwrite_config8(PCI_RAWDEV(0, 0x1, 0), 0xb0, i911);

#if 1
	struct device *dev;
	printk(BIOS_INFO, "========= dump all  devices...\n");
	for (dev = all_devices; dev; dev = dev->next) {
		if (dev->path.type == DEVICE_PATH_PCI) {
			printk(BIOS_DEBUG, "%s dump\n", dev_path(dev));
			x = y = 0;
			for (; x < 16; x++) {
				y = 0;
				for (; y < 16; y++) {
					printk(BIOS_INFO, "%02x ",
					    pci_read_config8(dev, x * 16 + y));
				}
				printk(BIOS_INFO, "\n");
			}
		}
		printk(BIOS_INFO, "\n");
	}
#endif
	//pci_rawmodify_config8(PCI_RAWDEV(0, 0x10, 4), 0x04, 0x17, 0x17);//
	//pci_rawmodify_config8(PCI_RAWDEV(0, 0x10, 4), 0x0c, 0x08, 0xff);///
}
