package models

import (
	"testing"

	"github.com/google/go-cmp/cmp"

	"github.com/anchore/grype/grype/version"
)

func Test_sortVersions(t *testing.T) {
	tests := []struct {
		name     string
		versions []string
		expected []string
	}{
		{
			name:     "empty slice",
			versions: []string{},
			expected: []string{},
		},
		{
			name:     "single version",
			versions: []string{"1.0.0"},
			expected: []string{"1.0.0"},
		},
		{
			name:     "already sorted versions",
			versions: []string{"1.0.0", "1.1.0", "2.0.0"},
			expected: []string{"1.0.0", "1.1.0", "2.0.0"},
		},
		{
			name:     "unsorted versions",
			versions: []string{"2.0.0", "1.0.0", "1.1.0"},
			expected: []string{"1.0.0", "1.1.0", "2.0.0"},
		},
		{
			name:     "patch versions",
			versions: []string{"1.0.2", "1.0.1", "1.0.0"},
			expected: []string{"1.0.0", "1.0.1", "1.0.2"},
		},
		{
			name:     "versions with pre-release",
			versions: []string{"1.0.0", "1.0.0-alpha", "1.0.0-beta"},
			expected: []string{"1.0.0-alpha", "1.0.0-beta", "1.0.0"},
		},
		{
			name:     "mixed pre-release and regular",
			versions: []string{"2.0.0", "1.0.0-alpha", "1.0.0", "1.0.0-beta"},
			expected: []string{"1.0.0-alpha", "1.0.0-beta", "1.0.0", "2.0.0"},
		},
		{
			name:     "versions with build metadata",
			versions: []string{"1.0.0+build.2", "1.0.0+build.1", "1.0.0"},
			expected: []string{"1.0.0+build.2", "1.0.0+build.1", "1.0.0"},
		},
		{
			name:     "complex semantic versions",
			versions: []string{"1.0.0-alpha.1", "1.0.0-alpha", "1.0.0-beta.2", "1.0.0-beta.11", "1.0.0-rc.1"},
			expected: []string{"1.0.0-alpha", "1.0.0-alpha.1", "1.0.0-beta.2", "1.0.0-beta.11", "1.0.0-rc.1"},
		},
		{
			name:     "invalid versions are appended to the end (in the order they were found in)",
			versions: []string{"invalid", "2.0.0", "also-invalid", "1.0.0"},
			expected: []string{"1.0.0", "2.0.0", "invalid", "also-invalid"},
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			result := sortVersions(tt.versions, version.SemanticFormat)

			if d := cmp.Diff(tt.expected, result); d != "" {
				t.Errorf("sortVersions() mismatch (-want +got):\n%s", d)
			}
		})
	}
}
