package bandersnatch

// Copyright 2020 ConsenSys Software Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// Code generated by consensys/gnark-crypto DO NOT EDIT


import (
	"fmt"
	"math/big"
	"math/bits"
	"math/rand"
	"runtime"
	"sync"
	"testing"

	"github.com/crate-crypto/go-ipa/bandersnatch/fr"
	"github.com/leanovate/gopter"
	"github.com/leanovate/gopter/prop"
	gnarkbandersnatch "github.com/consensys/gnark-crypto/ecc/bls12-381/bandersnatch"
)

// GenFr generates an Fr element
func GenFr() gopter.Gen {
	return func(genParams *gopter.GenParameters) *gopter.GenResult {
		var elmt fr.Element
		var b [fr.Bytes]byte
		_, err := rand.Read(b[:])
		if err != nil {
			panic(err)
		}
		elmt.SetBytes(b[:])
		genResult := gopter.NewGenResult(elmt, gopter.NoShrinker)
		return genResult
	}
}

func TestMultiExpPointAffine(t *testing.T) {

	parameters := gopter.DefaultTestParameters()
	parameters.MinSuccessfulTests = 2

	properties := gopter.NewProperties(parameters)

	genScalar := GenFr()

	var genAff = GetEdwardsCurve().Base
	var Generator PointProj
	Generator.FromAffine(&genAff)

	// size of the multiExps
	const nbSamples = 143

	// multi exp points
	var samplePoints [nbSamples]PointAffine
	var g PointProj
	g.Set(&Generator)
	for i := 1; i <= nbSamples; i++ {
		samplePoints[i-1].FromProj(&g)
		g.Add(&g, &Generator)
	}

	// final scalar to use in double and add method (without mixer factor)
	// n(n+1)(2n+1)/6  (sum of the squares from 1 to n)
	var scalar big.Int
	scalar.SetInt64(nbSamples)
	scalar.Mul(&scalar, new(big.Int).SetInt64(nbSamples+1))
	scalar.Mul(&scalar, new(big.Int).SetInt64(2*nbSamples+1))
	scalar.Div(&scalar, new(big.Int).SetInt64(6))

	// ensure a multiexp that's splitted has the same result as a non-splitted one..
	properties.Property("[G1] Multi exponentation (c=16) should be consistant with splitted multiexp", prop.ForAll(
		func(mixer fr.Element) bool {
			var samplePointsLarge [nbSamples * 13]PointAffine
			for i := 0; i < 13; i++ {
				copy(samplePointsLarge[i*nbSamples:], samplePoints[:])
			}

			var r16, splitted1, splitted2 PointProj

			// mixer ensures that all the words of a fpElement are set
			var sampleScalars [nbSamples * 13]fr.Element

			for i := 1; i <= nbSamples; i++ {
				sampleScalars[i-1].SetUint64(uint64(i)).
					Mul(&sampleScalars[i-1], &mixer).
					FromMont()
			}

			scalars16, _ := partitionScalars(sampleScalars[:], 16, false, runtime.NumCPU())
			msmC16(&r16,samplePoints[:], scalars16, true)

			MultiExp(&splitted1, samplePointsLarge[:], sampleScalars[:], MultiExpConfig{NbTasks: 128})
			MultiExp(&splitted2,samplePointsLarge[:], sampleScalars[:], MultiExpConfig{NbTasks: 51})
			return r16.Equal(&splitted1) && r16.Equal(&splitted2)
		},
		genScalar,
	))

	if testing.Short() {
		// we test only c = 5 and c = 16

		properties.Property("[G1] Multi exponentation (c=5, c=16) should be consistant with sum of square", prop.ForAll(
			func(mixer fr.Element) bool {

				var expected PointProj

				// compute expected result with double and add
				var finalScalar, mixerBigInt big.Int
				finalScalar.Mul(&scalar, mixer.ToBigIntRegular(&mixerBigInt))
				expected.ScalarMultiplication(&Generator, &finalScalar)

				// mixer ensures that all the words of a fpElement are set
				var sampleScalars [nbSamples]fr.Element

				for i := 1; i <= nbSamples; i++ {
					sampleScalars[i-1].SetUint64(uint64(i)).
						Mul(&sampleScalars[i-1], &mixer).
						FromMont()
				}

				scalars5, _ := partitionScalars(sampleScalars[:], 5, false, runtime.NumCPU())
				scalars16, _ := partitionScalars(sampleScalars[:], 16, false, runtime.NumCPU())

				var r5, r16 PointProj
				msmC5(&r5,samplePoints[:], scalars5, false)
				msmC16(&r16,samplePoints[:], scalars16, true)
				return (r5.Equal(&expected) && r16.Equal(&expected))
			},
			genScalar,
		))
	} else {

		properties.Property("[G1] Multi exponentation (c=4) should be consistant with sum of square", prop.ForAll(
			func(mixer fr.Element) bool {

				var result, expected PointProj

				// mixer ensures that all the words of a fpElement are set
				var sampleScalars [nbSamples]fr.Element

				for i := 1; i <= nbSamples; i++ {
					sampleScalars[i-1].SetUint64(uint64(i)).
						Mul(&sampleScalars[i-1], &mixer).
						FromMont()
				}

				scalars, _ := partitionScalars(sampleScalars[:], 4, false, runtime.NumCPU())
				msmC4(&result,samplePoints[:], scalars, false)

				// compute expected result with double and add
				var finalScalar, mixerBigInt big.Int
				finalScalar.Mul(&scalar, mixer.ToBigIntRegular(&mixerBigInt))
				expected.ScalarMultiplication(&Generator, &finalScalar)

				return result.Equal(&expected)
			},
			genScalar,
		))

		properties.Property("[G1] Multi exponentation (c=5) should be consistant with sum of square", prop.ForAll(
			func(mixer fr.Element) bool {

				var result, expected PointProj

				// mixer ensures that all the words of a fpElement are set
				var sampleScalars [nbSamples]fr.Element

				for i := 1; i <= nbSamples; i++ {
					sampleScalars[i-1].SetUint64(uint64(i)).
						Mul(&sampleScalars[i-1], &mixer).
						FromMont()
				}

				scalars, _ := partitionScalars(sampleScalars[:], 5, false, runtime.NumCPU())
				msmC5(&result,samplePoints[:], scalars, false)

				// compute expected result with double and add
				var finalScalar, mixerBigInt big.Int
				finalScalar.Mul(&scalar, mixer.ToBigIntRegular(&mixerBigInt))
				expected.ScalarMultiplication(&Generator, &finalScalar)

				return result.Equal(&expected)
			},
			genScalar,
		))

		properties.Property("[G1] Multi exponentation (c=6) should be consistant with sum of square", prop.ForAll(
			func(mixer fr.Element) bool {

				var result, expected PointProj

				// mixer ensures that all the words of a fpElement are set
				var sampleScalars [nbSamples]fr.Element

				for i := 1; i <= nbSamples; i++ {
					sampleScalars[i-1].SetUint64(uint64(i)).
						Mul(&sampleScalars[i-1], &mixer).
						FromMont()
				}

				scalars, _ := partitionScalars(sampleScalars[:], 6, false, runtime.NumCPU())
				msmC6(&result,samplePoints[:], scalars, false)

				// compute expected result with double and add
				var finalScalar, mixerBigInt big.Int
				finalScalar.Mul(&scalar, mixer.ToBigIntRegular(&mixerBigInt))
				expected.ScalarMultiplication(&Generator, &finalScalar)

				return result.Equal(&expected)
			},
			genScalar,
		))

		properties.Property("[G1] Multi exponentation (c=7) should be consistant with sum of square", prop.ForAll(
			func(mixer fr.Element) bool {

				var result, expected PointProj

				// mixer ensures that all the words of a fpElement are set
				var sampleScalars [nbSamples]fr.Element

				for i := 1; i <= nbSamples; i++ {
					sampleScalars[i-1].SetUint64(uint64(i)).
						Mul(&sampleScalars[i-1], &mixer).
						FromMont()
				}

				scalars, _ := partitionScalars(sampleScalars[:], 7, false, runtime.NumCPU())
				msmC7(&result,samplePoints[:], scalars, false)

				// compute expected result with double and add
				var finalScalar, mixerBigInt big.Int
				finalScalar.Mul(&scalar, mixer.ToBigIntRegular(&mixerBigInt))
				expected.ScalarMultiplication(&Generator, &finalScalar)

				return result.Equal(&expected)
			},
			genScalar,
		))

		properties.Property("[G1] Multi exponentation (c=8) should be consistant with sum of square", prop.ForAll(
			func(mixer fr.Element) bool {

				var result, expected PointProj

				// mixer ensures that all the words of a fpElement are set
				var sampleScalars [nbSamples]fr.Element

				for i := 1; i <= nbSamples; i++ {
					sampleScalars[i-1].SetUint64(uint64(i)).
						Mul(&sampleScalars[i-1], &mixer).
						FromMont()
				}

				scalars, _ := partitionScalars(sampleScalars[:], 8, false, runtime.NumCPU())
				msmC8(&result,samplePoints[:], scalars, false)

				// compute expected result with double and add
				var finalScalar, mixerBigInt big.Int
				finalScalar.Mul(&scalar, mixer.ToBigIntRegular(&mixerBigInt))
				expected.ScalarMultiplication(&Generator, &finalScalar)

				return result.Equal(&expected)
			},
			genScalar,
		))

		properties.Property("[G1] Multi exponentation (c=9) should be consistant with sum of square", prop.ForAll(
			func(mixer fr.Element) bool {

				var result, expected PointProj

				// mixer ensures that all the words of a fpElement are set
				var sampleScalars [nbSamples]fr.Element

				for i := 1; i <= nbSamples; i++ {
					sampleScalars[i-1].SetUint64(uint64(i)).
						Mul(&sampleScalars[i-1], &mixer).
						FromMont()
				}

				scalars, _ := partitionScalars(sampleScalars[:], 9, false, runtime.NumCPU())
				msmC9(&result,samplePoints[:], scalars, false)

				// compute expected result with double and add
				var finalScalar, mixerBigInt big.Int
				finalScalar.Mul(&scalar, mixer.ToBigIntRegular(&mixerBigInt))
				expected.ScalarMultiplication(&Generator, &finalScalar)

				return result.Equal(&expected)
			},
			genScalar,
		))

		properties.Property("[G1] Multi exponentation (c=10) should be consistant with sum of square", prop.ForAll(
			func(mixer fr.Element) bool {

				var result, expected PointProj

				// mixer ensures that all the words of a fpElement are set
				var sampleScalars [nbSamples]fr.Element

				for i := 1; i <= nbSamples; i++ {
					sampleScalars[i-1].SetUint64(uint64(i)).
						Mul(&sampleScalars[i-1], &mixer).
						FromMont()
				}

				scalars, _ := partitionScalars(sampleScalars[:], 10, false, runtime.NumCPU())
				msmC10(&result,samplePoints[:], scalars, false)

				// compute expected result with double and add
				var finalScalar, mixerBigInt big.Int
				finalScalar.Mul(&scalar, mixer.ToBigIntRegular(&mixerBigInt))
				expected.ScalarMultiplication(&Generator, &finalScalar)

				return result.Equal(&expected)
			},
			genScalar,
		))

		properties.Property("[G1] Multi exponentation (c=11) should be consistant with sum of square", prop.ForAll(
			func(mixer fr.Element) bool {

				var result, expected PointProj

				// mixer ensures that all the words of a fpElement are set
				var sampleScalars [nbSamples]fr.Element

				for i := 1; i <= nbSamples; i++ {
					sampleScalars[i-1].SetUint64(uint64(i)).
						Mul(&sampleScalars[i-1], &mixer).
						FromMont()
				}

				scalars, _ := partitionScalars(sampleScalars[:], 11, false, runtime.NumCPU())
				msmC11(&result,samplePoints[:], scalars, false)

				// compute expected result with double and add
				var finalScalar, mixerBigInt big.Int
				finalScalar.Mul(&scalar, mixer.ToBigIntRegular(&mixerBigInt))
				expected.ScalarMultiplication(&Generator, &finalScalar)

				return result.Equal(&expected)
			},
			genScalar,
		))

		properties.Property("[G1] Multi exponentation (c=12) should be consistant with sum of square", prop.ForAll(
			func(mixer fr.Element) bool {

				var result, expected PointProj

				// mixer ensures that all the words of a fpElement are set
				var sampleScalars [nbSamples]fr.Element

				for i := 1; i <= nbSamples; i++ {
					sampleScalars[i-1].SetUint64(uint64(i)).
						Mul(&sampleScalars[i-1], &mixer).
						FromMont()
				}

				scalars, _ := partitionScalars(sampleScalars[:], 12, false, runtime.NumCPU())
				msmC12(&result,samplePoints[:], scalars, false)

				// compute expected result with double and add
				var finalScalar, mixerBigInt big.Int
				finalScalar.Mul(&scalar, mixer.ToBigIntRegular(&mixerBigInt))
				expected.ScalarMultiplication(&Generator, &finalScalar)

				return result.Equal(&expected)
			},
			genScalar,
		))

		properties.Property("[G1] Multi exponentation (c=13) should be consistant with sum of square", prop.ForAll(
			func(mixer fr.Element) bool {

				var result, expected PointProj

				// mixer ensures that all the words of a fpElement are set
				var sampleScalars [nbSamples]fr.Element

				for i := 1; i <= nbSamples; i++ {
					sampleScalars[i-1].SetUint64(uint64(i)).
						Mul(&sampleScalars[i-1], &mixer).
						FromMont()
				}

				scalars, _ := partitionScalars(sampleScalars[:], 13, false, runtime.NumCPU())
				msmC13(&result,samplePoints[:], scalars, false)

				// compute expected result with double and add
				var finalScalar, mixerBigInt big.Int
				finalScalar.Mul(&scalar, mixer.ToBigIntRegular(&mixerBigInt))
				expected.ScalarMultiplication(&Generator, &finalScalar)

				return result.Equal(&expected)
			},
			genScalar,
		))

		properties.Property("[G1] Multi exponentation (c=14) should be consistant with sum of square", prop.ForAll(
			func(mixer fr.Element) bool {

				var result, expected PointProj

				// mixer ensures that all the words of a fpElement are set
				var sampleScalars [nbSamples]fr.Element

				for i := 1; i <= nbSamples; i++ {
					sampleScalars[i-1].SetUint64(uint64(i)).
						Mul(&sampleScalars[i-1], &mixer).
						FromMont()
				}

				scalars, _ := partitionScalars(sampleScalars[:], 14, false, runtime.NumCPU())
				msmC14(&result,samplePoints[:], scalars, false)

				// compute expected result with double and add
				var finalScalar, mixerBigInt big.Int
				finalScalar.Mul(&scalar, mixer.ToBigIntRegular(&mixerBigInt))
				expected.ScalarMultiplication(&Generator, &finalScalar)

				return result.Equal(&expected)
			},
			genScalar,
		))

		properties.Property("[G1] Multi exponentation (c=15) should be consistant with sum of square", prop.ForAll(
			func(mixer fr.Element) bool {

				var result, expected PointProj

				// mixer ensures that all the words of a fpElement are set
				var sampleScalars [nbSamples]fr.Element

				for i := 1; i <= nbSamples; i++ {
					sampleScalars[i-1].SetUint64(uint64(i)).
						Mul(&sampleScalars[i-1], &mixer).
						FromMont()
				}

				scalars, _ := partitionScalars(sampleScalars[:], 15, false, runtime.NumCPU())
				msmC15(&result,samplePoints[:], scalars, false)

				// compute expected result with double and add
				var finalScalar, mixerBigInt big.Int
				finalScalar.Mul(&scalar, mixer.ToBigIntRegular(&mixerBigInt))
				expected.ScalarMultiplication(&Generator, &finalScalar)

				return result.Equal(&expected)
			},
			genScalar,
		))

		properties.Property("[G1] Multi exponentation (c=16) should be consistant with sum of square", prop.ForAll(
			func(mixer fr.Element) bool {

				var result, expected PointProj

				// mixer ensures that all the words of a fpElement are set
				var sampleScalars [nbSamples]fr.Element

				for i := 1; i <= nbSamples; i++ {
					sampleScalars[i-1].SetUint64(uint64(i)).
						Mul(&sampleScalars[i-1], &mixer).
						FromMont()
				}

				scalars, _ := partitionScalars(sampleScalars[:], 16, false, runtime.NumCPU())
				msmC16(&result,samplePoints[:], scalars, false)

				// compute expected result with double and add
				var finalScalar, mixerBigInt big.Int
				finalScalar.Mul(&scalar, mixer.ToBigIntRegular(&mixerBigInt))
				expected.ScalarMultiplication(&Generator, &finalScalar)

				return result.Equal(&expected)
			},
			genScalar,
		))

		properties.Property("[G1] Multi exponentation (c=20) should be consistant with sum of square", prop.ForAll(
			func(mixer fr.Element) bool {

				var result, expected PointProj

				// mixer ensures that all the words of a fpElement are set
				var sampleScalars [nbSamples]fr.Element

				for i := 1; i <= nbSamples; i++ {
					sampleScalars[i-1].SetUint64(uint64(i)).
						Mul(&sampleScalars[i-1], &mixer).
						FromMont()
				}

				scalars, _ := partitionScalars(sampleScalars[:], 20, false, runtime.NumCPU())
				msmC20(&result,samplePoints[:], scalars, false)

				// compute expected result with double and add
				var finalScalar, mixerBigInt big.Int
				finalScalar.Mul(&scalar, mixer.ToBigIntRegular(&mixerBigInt))
				expected.ScalarMultiplication(&Generator, &finalScalar)

				return result.Equal(&expected)
			},
			genScalar,
		))

		properties.Property("[G1] Multi exponentation (c=21) should be consistant with sum of square", prop.ForAll(
			func(mixer fr.Element) bool {

				var result, expected PointProj

				// mixer ensures that all the words of a fpElement are set
				var sampleScalars [nbSamples]fr.Element

				for i := 1; i <= nbSamples; i++ {
					sampleScalars[i-1].SetUint64(uint64(i)).
						Mul(&sampleScalars[i-1], &mixer).
						FromMont()
				}

				scalars, _ := partitionScalars(sampleScalars[:], 21, false, runtime.NumCPU())
				msmC21(&result, samplePoints[:], scalars, false)

				// compute expected result with double and add
				var finalScalar, mixerBigInt big.Int
				finalScalar.Mul(&scalar, mixer.ToBigIntRegular(&mixerBigInt))
				expected.ScalarMultiplication(&Generator, &finalScalar)

				return result.Equal(&expected)
			},
			genScalar,
		))

		properties.Property("[G1] Multi exponentation (c=22) should be consistant with sum of square", prop.ForAll(
			func(mixer fr.Element) bool {

				var result, expected PointProj

				// mixer ensures that all the words of a fpElement are set
				var sampleScalars [nbSamples]fr.Element

				for i := 1; i <= nbSamples; i++ {
					sampleScalars[i-1].SetUint64(uint64(i)).
						Mul(&sampleScalars[i-1], &mixer).
						FromMont()
				}

				scalars, _ := partitionScalars(sampleScalars[:], 22, false, runtime.NumCPU())
				msmC22(&result,samplePoints[:], scalars, false)

				// compute expected result with double and add
				var finalScalar, mixerBigInt big.Int
				finalScalar.Mul(&scalar, mixer.ToBigIntRegular(&mixerBigInt))
				expected.ScalarMultiplication(&Generator, &finalScalar)

				return result.Equal(&expected)
			},
			genScalar,
		))

	}

	// note : this test is here as we expect to have a different multiExp than the above bucket method
	// for small number of points
	properties.Property("[G1] Multi exponentation (<50points) should be consistant with sum of square", prop.ForAll(
		func(mixer fr.Element) bool {

			var g PointProj
			g.Set(&Generator)

			var GeneratorAff PointAffine
			GeneratorAff.FromProj(&Generator)

			// mixer ensures that all the words of a fpElement are set
			samplePoints := make([]PointAffine, 30)
			sampleScalars := make([]fr.Element, 30)

			for i := 1; i <= 30; i++ {
				sampleScalars[i-1].SetUint64(uint64(i)).
					Mul(&sampleScalars[i-1], &mixer).
					FromMont()
				samplePoints[i-1].FromProj(&g)
				g.Add(&g, &Generator)
			}

			 op1MultiExp, _ := MultiExpAffine(samplePoints, sampleScalars, MultiExpConfig{})

			var finalBigScalar fr.Element
			var finalBigScalarBi big.Int
			var op1ScalarMul PointAffine
			finalBigScalar.SetString("9455").Mul(&finalBigScalar, &mixer)
			finalBigScalar.ToBigIntRegular(&finalBigScalarBi)
			op1ScalarMul.ScalarMultiplication(&GeneratorAff, &finalBigScalarBi)

			return op1ScalarMul.Equal(&op1MultiExp)
		},
		genScalar,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func BenchmarkMultiExpG1(b *testing.B) {

				var GeneratorAff = GetEdwardsCurve().Base
	// ensure every words of the scalars are filled
	var mixer fr.Element
	mixer.SetString("7716837800905789770901243404444209691916730933998574719964609384059111546487")

	const pow = (bits.UintSize / 2) - (bits.UintSize / 8) // 24 on 64 bits arch, 12 on 32 bits
	const nbSamples = 1 << pow

	var samplePoints [nbSamples]PointAffine
	var sampleScalars [nbSamples]fr.Element

	for i := 1; i <= nbSamples; i++ {
		sampleScalars[i-1].SetUint64(uint64(i)).
			Mul(&sampleScalars[i-1], &mixer).
			FromMont()
		samplePoints[i-1] = GeneratorAff
	}

	for i := 5; i <= pow; i++ {
		using := 1 << i

		b.Run(fmt.Sprintf("%d points", using), func(b *testing.B) {
			b.ResetTimer()
			for j := 0; j < b.N; j++ {
				_, _ = MultiExpAffine(samplePoints[:using], sampleScalars[:using], MultiExpConfig{})
			}
		})
	}
}

func BenchmarkMultiExpG1Reference(b *testing.B) {

					var GeneratorAff = GetEdwardsCurve().Base


	// ensure every words of the scalars are filled
	var mixer fr.Element
	mixer.SetString("7716837800905789770901243404444209691916730933998574719964609384059111546487")

	const nbSamples = 1 << 20

	var samplePoints [nbSamples]PointAffine
	var sampleScalars [nbSamples]fr.Element

	for i := 1; i <= nbSamples; i++ {
		sampleScalars[i-1].SetUint64(uint64(i)).
			Mul(&sampleScalars[i-1], &mixer).
			FromMont()
		samplePoints[i-1] = GeneratorAff
	}

	b.ResetTimer()
	for j := 0; j < b.N; j++ {
		_, _ = MultiExpAffine(samplePoints[:], sampleScalars[:], MultiExpConfig{})
	}
}

func BenchmarkManyMultiExpG1Reference(b *testing.B) {

				var GeneratorAff = GetEdwardsCurve().Base


	// ensure every words of the scalars are filled
	var mixer fr.Element
	mixer.SetString("7716837800905789770901243404444209691916730933998574719964609384059111546487")

	const nbSamples = 1 << 20

	var samplePoints [nbSamples]PointAffine
	var sampleScalars [nbSamples]fr.Element

	for i := 1; i <= nbSamples; i++ {
		sampleScalars[i-1].SetUint64(uint64(i)).
			Mul(&sampleScalars[i-1], &mixer).
			FromMont()
		samplePoints[i-1] = GeneratorAff
	}

	b.ResetTimer()
	for j := 0; j < b.N; j++ {
		var wg sync.WaitGroup
		wg.Add(3)
		go func() {
			_, _ = MultiExpAffine(samplePoints[:], sampleScalars[:], MultiExpConfig{})
			wg.Done()
		}()
		go func() {
			_,_ = MultiExpAffine(samplePoints[:], sampleScalars[:], MultiExpConfig{})
			wg.Done()
		}()
		go func() {
			_, _ = MultiExpAffine(samplePoints[:], sampleScalars[:], MultiExpConfig{})
			wg.Done()
		}()
		wg.Wait()
	}
}

func GetEdwardsCurve() gnarkbandersnatch.CurveParams {
	return CurveParams
}