// Copyright 2020-2025 Consensys Software Inc.
// Licensed under the Apache License, Version 2.0. See the LICENSE file for details.

// Code generated by consensys/gnark-crypto DO NOT EDIT

package babybear

import (
	"crypto/rand"
	"encoding/json"
	"fmt"
	"math/big"
	"math/bits"

	"testing"

	"github.com/leanovate/gopter"
	ggen "github.com/leanovate/gopter/gen"
	"github.com/leanovate/gopter/prop"

	"github.com/stretchr/testify/require"
)

// -------------------------------------------------------------------------------------------------
// benchmarks
// most benchmarks are rudimentary and should sample a large number of random inputs
// or be run multiple times to ensure it didn't measure the fastest path of the function

var benchResElement Element

func BenchmarkElementSelect(b *testing.B) {
	var x, y Element
	x.MustSetRandom()
	y.MustSetRandom()

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.Select(i%3, &x, &y)
	}
}

func BenchmarkElementSetRandom(b *testing.B) {
	var x Element
	x.MustSetRandom()

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		x.MustSetRandom()
	}
}

func BenchmarkElementSetBytes(b *testing.B) {
	var x Element
	x.MustSetRandom()
	bb := x.Bytes()
	b.ResetTimer()

	for i := 0; i < b.N; i++ {
		benchResElement.SetBytes(bb[:])
	}

}

func BenchmarkElementMulByConstants(b *testing.B) {
	b.Run("mulBy3", func(b *testing.B) {
		benchResElement.MustSetRandom()
		b.ResetTimer()
		for i := 0; i < b.N; i++ {
			MulBy3(&benchResElement)
		}
	})
	b.Run("mulBy5", func(b *testing.B) {
		benchResElement.MustSetRandom()
		b.ResetTimer()
		for i := 0; i < b.N; i++ {
			MulBy5(&benchResElement)
		}
	})
	b.Run("mulBy13", func(b *testing.B) {
		benchResElement.MustSetRandom()
		b.ResetTimer()
		for i := 0; i < b.N; i++ {
			MulBy13(&benchResElement)
		}
	})
}

func BenchmarkElementInverse(b *testing.B) {
	var x Element
	x.MustSetRandom()
	benchResElement.MustSetRandom()
	b.ResetTimer()

	for i := 0; i < b.N; i++ {
		benchResElement.Inverse(&x)
	}

}

func BenchmarkElementButterfly(b *testing.B) {
	var x Element
	x.MustSetRandom()
	benchResElement.MustSetRandom()
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		Butterfly(&x, &benchResElement)
	}
}

func BenchmarkElementExp(b *testing.B) {
	var x Element
	x.MustSetRandom()
	benchResElement.MustSetRandom()
	b1, _ := rand.Int(rand.Reader, Modulus())
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.Exp(x, b1)
	}
}

func BenchmarkElementDouble(b *testing.B) {
	benchResElement.MustSetRandom()
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.Double(&benchResElement)
	}
}

func BenchmarkElementAdd(b *testing.B) {
	var x Element
	x.MustSetRandom()
	benchResElement.MustSetRandom()
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.Add(&x, &benchResElement)
	}
}

func BenchmarkElementSub(b *testing.B) {
	var x Element
	x.MustSetRandom()
	benchResElement.MustSetRandom()
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.Sub(&x, &benchResElement)
	}
}

func BenchmarkElementNeg(b *testing.B) {
	benchResElement.MustSetRandom()
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.Neg(&benchResElement)
	}
}

func BenchmarkElementDiv(b *testing.B) {
	var x Element
	x.MustSetRandom()
	benchResElement.MustSetRandom()
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.Div(&x, &benchResElement)
	}
}

func BenchmarkElementFromMont(b *testing.B) {
	benchResElement.MustSetRandom()
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.fromMont()
	}
}

func BenchmarkElementSquare(b *testing.B) {
	benchResElement.MustSetRandom()
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.Square(&benchResElement)
	}
}

func BenchmarkElementSqrt(b *testing.B) {
	var a Element
	a.SetUint64(4)
	a.Neg(&a)
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.Sqrt(&a)
	}
}

func BenchmarkElementMul(b *testing.B) {
	x := Element{
		1172168163,
	}
	benchResElement.SetOne()
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.Mul(&benchResElement, &x)
	}
}

func BenchmarkElementCmp(b *testing.B) {
	x := Element{
		1172168163,
	}
	benchResElement = x
	benchResElement[0] = 0
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.Cmp(&x)
	}
}

func TestElementCmp(t *testing.T) {
	var x, y Element

	if x.Cmp(&y) != 0 {
		t.Fatal("x == y")
	}

	one := One()
	y.Sub(&y, &one)

	if x.Cmp(&y) != -1 {
		t.Fatal("x < y")
	}
	if y.Cmp(&x) != 1 {
		t.Fatal("x < y")
	}

	x = y
	if x.Cmp(&y) != 0 {
		t.Fatal("x == y")
	}

	x.Sub(&x, &one)
	if x.Cmp(&y) != -1 {
		t.Fatal("x < y")
	}
	if y.Cmp(&x) != 1 {
		t.Fatal("x < y")
	}
}

func TestElementNegZero(t *testing.T) {
	var a, b Element
	b.SetZero()
	for a.IsZero() {
		a.MustSetRandom()
	}
	a.Neg(&b)
	if !a.IsZero() {
		t.Fatal("neg(0) != 0")
	}
}

// -------------------------------------------------------------------------------------------------
// Gopter tests
// most of them are generated with a template

const (
	nbFuzzShort = 200
	nbFuzz      = 1000
)

// special values to be used in tests
var staticTestValues []Element

func init() {
	staticTestValues = append(staticTestValues, Element{}) // zero
	staticTestValues = append(staticTestValues, One())     // one
	staticTestValues = append(staticTestValues, rSquare)   // r²
	var e, one Element
	one.SetOne()
	e.Sub(&qElement, &one)
	staticTestValues = append(staticTestValues, e) // q - 1
	e.Double(&one)
	staticTestValues = append(staticTestValues, e) // 2

	{
		a := qElement
		a[0]--
		staticTestValues = append(staticTestValues, a)
	}
	staticTestValues = append(staticTestValues, Element{0})
	staticTestValues = append(staticTestValues, Element{1})
	staticTestValues = append(staticTestValues, Element{2})

	{
		a := qElement
		a[0]--
		staticTestValues = append(staticTestValues, a)
	}

	{
		a := qElement
		a[0] = 0
		staticTestValues = append(staticTestValues, a)
	}

}

func TestElementReduce(t *testing.T) {
	testValues := make([]Element, len(staticTestValues))
	copy(testValues, staticTestValues)

	for i := range testValues {
		s := testValues[i]
		expected := s
		reduce(&s)
		_reduceGeneric(&expected)
		if !s.Equal(&expected) {
			t.Fatal("reduce failed: asm and generic impl don't match")
		}
	}

	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := genFull()

	properties.Property("reduce should output a result smaller than modulus", prop.ForAll(
		func(a Element) bool {
			b := a
			reduce(&a)
			_reduceGeneric(&b)
			return a.smallerThanModulus() && a.Equal(&b)
		},
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))

}

func TestElementEqual(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()
	genB := gen()

	properties.Property("x.Equal(&y) iff x == y; likely false for random pairs", prop.ForAll(
		func(a testPairElement, b testPairElement) bool {
			return a.element.Equal(&b.element) == (a.element == b.element)
		},
		genA,
		genB,
	))

	properties.Property("x.Equal(&y) if x == y", prop.ForAll(
		func(a testPairElement) bool {
			b := a.element
			return a.element.Equal(&b)
		},
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestElementBytes(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("SetBytes(Bytes()) should stay constant", prop.ForAll(
		func(a testPairElement) bool {
			var b Element
			bytes := a.element.Bytes()
			b.SetBytes(bytes[:])
			return a.element.Equal(&b)
		},
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestElementInverseExp(t *testing.T) {
	// inverse must be equal to exp^-2
	exp := Modulus()
	exp.Sub(exp, new(big.Int).SetUint64(2))

	invMatchExp := func(a testPairElement) bool {
		var b Element
		b.Set(&a.element)
		a.element.Inverse(&a.element)
		b.Exp(b, exp)

		return a.element.Equal(&b)
	}

	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}
	properties := gopter.NewProperties(parameters)
	genA := gen()
	properties.Property("inv == exp^-2", prop.ForAll(invMatchExp, genA))
	properties.TestingRun(t, gopter.ConsoleReporter(false))

	parameters.MinSuccessfulTests = 1
	properties = gopter.NewProperties(parameters)
	properties.Property("inv(0) == 0", prop.ForAll(invMatchExp, ggen.OneConstOf(testPairElement{})))
	properties.TestingRun(t, gopter.ConsoleReporter(false))

}

func mulByConstant(z *Element, c uint8) {
	var y Element
	y.SetUint64(uint64(c))
	z.Mul(z, &y)
}

func TestElementMulByConstants(t *testing.T) {

	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	implemented := []uint8{0, 1, 2, 3, 5, 13}
	properties.Property("mulByConstant", prop.ForAll(
		func(a testPairElement) bool {
			for _, c := range implemented {
				var constant Element
				constant.SetUint64(uint64(c))

				b := a.element
				b.Mul(&b, &constant)

				aa := a.element
				mulByConstant(&aa, c)

				if !aa.Equal(&b) {
					return false
				}
			}

			return true
		},
		genA,
	))

	properties.Property("MulBy3(x) == Mul(x, 3)", prop.ForAll(
		func(a testPairElement) bool {
			var constant Element
			constant.SetUint64(3)

			b := a.element
			b.Mul(&b, &constant)

			MulBy3(&a.element)

			return a.element.Equal(&b)
		},
		genA,
	))

	properties.Property("MulBy5(x) == Mul(x, 5)", prop.ForAll(
		func(a testPairElement) bool {
			var constant Element
			constant.SetUint64(5)

			b := a.element
			b.Mul(&b, &constant)

			MulBy5(&a.element)

			return a.element.Equal(&b)
		},
		genA,
	))

	properties.Property("MulBy13(x) == Mul(x, 13)", prop.ForAll(
		func(a testPairElement) bool {
			var constant Element
			constant.SetUint64(13)

			b := a.element
			b.Mul(&b, &constant)

			MulBy13(&a.element)

			return a.element.Equal(&b)
		},
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))

}

func TestElementLegendre(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("legendre should output same result than big.Int.Jacobi", prop.ForAll(
		func(a testPairElement) bool {
			return a.element.Legendre() == big.Jacobi(&a.bigint, Modulus())
		},
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))

}

func TestElementBitLen(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("BitLen should output same result than big.Int.BitLen", prop.ForAll(
		func(a testPairElement) bool {
			return a.element.fromMont().BitLen() == a.bigint.BitLen()
		},
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestElementButterflies(t *testing.T) {

	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("butterfly0 == a -b; a +b", prop.ForAll(
		func(a, b testPairElement) bool {
			a0, b0 := a.element, b.element

			_butterflyGeneric(&a.element, &b.element)
			Butterfly(&a0, &b0)

			return a.element.Equal(&a0) && b.element.Equal(&b0)
		},
		genA,
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))

}

func TestElementLexicographicallyLargest(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("element.Cmp should match LexicographicallyLargest output", prop.ForAll(
		func(a testPairElement) bool {
			var negA Element
			negA.Neg(&a.element)

			cmpResult := a.element.Cmp(&negA)
			lResult := a.element.LexicographicallyLargest()

			if lResult && cmpResult == 1 {
				return true
			}
			if !lResult && cmpResult != 1 {
				return true
			}
			return false
		},
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))

}

func TestElementAdd(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()
	genB := gen()

	properties.Property("Add: having the receiver as operand should output the same result", prop.ForAll(
		func(a, b testPairElement) bool {
			var c, d Element
			d.Set(&a.element)

			c.Add(&a.element, &b.element)
			a.element.Add(&a.element, &b.element)
			b.element.Add(&d, &b.element)

			return a.element.Equal(&b.element) && a.element.Equal(&c) && b.element.Equal(&c)
		},
		genA,
		genB,
	))

	properties.Property("Add: operation result must match big.Int result", prop.ForAll(
		func(a, b testPairElement) bool {
			{
				var c Element

				c.Add(&a.element, &b.element)

				var d, e big.Int
				d.Add(&a.bigint, &b.bigint).Mod(&d, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					return false
				}
			}

			// fixed elements
			// a is random
			// r takes special values
			testValues := make([]Element, len(staticTestValues))
			copy(testValues, staticTestValues)

			for i := range testValues {
				r := testValues[i]
				var d, e, rb big.Int
				r.BigInt(&rb)

				var c Element
				c.Add(&a.element, &r)
				d.Add(&a.bigint, &rb).Mod(&d, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					return false
				}
			}
			return true
		},
		genA,
		genB,
	))

	properties.Property("Add: operation result must be smaller than modulus", prop.ForAll(
		func(a, b testPairElement) bool {
			var c Element

			c.Add(&a.element, &b.element)

			return c.smallerThanModulus()
		},
		genA,
		genB,
	))

	specialValueTest := func() {
		// test special values against special values
		testValues := make([]Element, len(staticTestValues))
		copy(testValues, staticTestValues)

		for i := range testValues {
			a := testValues[i]
			var aBig big.Int
			a.BigInt(&aBig)
			for j := range testValues {
				b := testValues[j]
				var bBig, d, e big.Int
				b.BigInt(&bBig)

				var c Element
				c.Add(&a, &b)
				d.Add(&aBig, &bBig).Mod(&d, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					t.Fatal("Add failed special test values")
				}
			}
		}
	}

	properties.TestingRun(t, gopter.ConsoleReporter(false))
	specialValueTest()

}

func TestElementSub(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()
	genB := gen()

	properties.Property("Sub: having the receiver as operand should output the same result", prop.ForAll(
		func(a, b testPairElement) bool {
			var c, d Element
			d.Set(&a.element)

			c.Sub(&a.element, &b.element)
			a.element.Sub(&a.element, &b.element)
			b.element.Sub(&d, &b.element)

			return a.element.Equal(&b.element) && a.element.Equal(&c) && b.element.Equal(&c)
		},
		genA,
		genB,
	))

	properties.Property("Sub: operation result must match big.Int result", prop.ForAll(
		func(a, b testPairElement) bool {
			{
				var c Element

				c.Sub(&a.element, &b.element)

				var d, e big.Int
				d.Sub(&a.bigint, &b.bigint).Mod(&d, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					return false
				}
			}

			// fixed elements
			// a is random
			// r takes special values
			testValues := make([]Element, len(staticTestValues))
			copy(testValues, staticTestValues)

			for i := range testValues {
				r := testValues[i]
				var d, e, rb big.Int
				r.BigInt(&rb)

				var c Element
				c.Sub(&a.element, &r)
				d.Sub(&a.bigint, &rb).Mod(&d, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					return false
				}
			}
			return true
		},
		genA,
		genB,
	))

	properties.Property("Sub: operation result must be smaller than modulus", prop.ForAll(
		func(a, b testPairElement) bool {
			var c Element

			c.Sub(&a.element, &b.element)

			return c.smallerThanModulus()
		},
		genA,
		genB,
	))

	specialValueTest := func() {
		// test special values against special values
		testValues := make([]Element, len(staticTestValues))
		copy(testValues, staticTestValues)

		for i := range testValues {
			a := testValues[i]
			var aBig big.Int
			a.BigInt(&aBig)
			for j := range testValues {
				b := testValues[j]
				var bBig, d, e big.Int
				b.BigInt(&bBig)

				var c Element
				c.Sub(&a, &b)
				d.Sub(&aBig, &bBig).Mod(&d, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					t.Fatal("Sub failed special test values")
				}
			}
		}
	}

	properties.TestingRun(t, gopter.ConsoleReporter(false))
	specialValueTest()

}

func TestElementMul(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()
	genB := gen()

	properties.Property("Mul: having the receiver as operand should output the same result", prop.ForAll(
		func(a, b testPairElement) bool {
			var c, d Element
			d.Set(&a.element)

			c.Mul(&a.element, &b.element)
			a.element.Mul(&a.element, &b.element)
			b.element.Mul(&d, &b.element)

			return a.element.Equal(&b.element) && a.element.Equal(&c) && b.element.Equal(&c)
		},
		genA,
		genB,
	))

	properties.Property("Mul: operation result must match big.Int result", prop.ForAll(
		func(a, b testPairElement) bool {
			{
				var c Element

				c.Mul(&a.element, &b.element)

				var d, e big.Int
				d.Mul(&a.bigint, &b.bigint).Mod(&d, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					return false
				}
			}

			// fixed elements
			// a is random
			// r takes special values
			testValues := make([]Element, len(staticTestValues))
			copy(testValues, staticTestValues)

			for i := range testValues {
				r := testValues[i]
				var d, e, rb big.Int
				r.BigInt(&rb)

				var c Element
				c.Mul(&a.element, &r)
				d.Mul(&a.bigint, &rb).Mod(&d, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					return false
				}
			}
			return true
		},
		genA,
		genB,
	))

	properties.Property("Mul: operation result must be smaller than modulus", prop.ForAll(
		func(a, b testPairElement) bool {
			var c Element

			c.Mul(&a.element, &b.element)

			return c.smallerThanModulus()
		},
		genA,
		genB,
	))

	specialValueTest := func() {
		// test special values against special values
		testValues := make([]Element, len(staticTestValues))
		copy(testValues, staticTestValues)

		for i := range testValues {
			a := testValues[i]
			var aBig big.Int
			a.BigInt(&aBig)
			for j := range testValues {
				b := testValues[j]
				var bBig, d, e big.Int
				b.BigInt(&bBig)

				var c Element
				c.Mul(&a, &b)
				d.Mul(&aBig, &bBig).Mod(&d, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					t.Fatal("Mul failed special test values")
				}
			}
		}
	}

	properties.TestingRun(t, gopter.ConsoleReporter(false))
	specialValueTest()

}

func TestElementDiv(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()
	genB := gen()

	properties.Property("Div: having the receiver as operand should output the same result", prop.ForAll(
		func(a, b testPairElement) bool {
			var c, d Element
			d.Set(&a.element)

			c.Div(&a.element, &b.element)
			a.element.Div(&a.element, &b.element)
			b.element.Div(&d, &b.element)

			return a.element.Equal(&b.element) && a.element.Equal(&c) && b.element.Equal(&c)
		},
		genA,
		genB,
	))

	properties.Property("Div: operation result must match big.Int result", prop.ForAll(
		func(a, b testPairElement) bool {
			{
				var c Element

				c.Div(&a.element, &b.element)

				var d, e big.Int
				d.ModInverse(&b.bigint, Modulus())
				d.Mul(&d, &a.bigint).Mod(&d, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					return false
				}
			}

			// fixed elements
			// a is random
			// r takes special values
			testValues := make([]Element, len(staticTestValues))
			copy(testValues, staticTestValues)

			for i := range testValues {
				r := testValues[i]
				var d, e, rb big.Int
				r.BigInt(&rb)

				var c Element
				c.Div(&a.element, &r)
				d.ModInverse(&rb, Modulus())
				d.Mul(&d, &a.bigint).Mod(&d, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					return false
				}
			}
			return true
		},
		genA,
		genB,
	))

	properties.Property("Div: operation result must be smaller than modulus", prop.ForAll(
		func(a, b testPairElement) bool {
			var c Element

			c.Div(&a.element, &b.element)

			return c.smallerThanModulus()
		},
		genA,
		genB,
	))

	specialValueTest := func() {
		// test special values against special values
		testValues := make([]Element, len(staticTestValues))
		copy(testValues, staticTestValues)

		for i := range testValues {
			a := testValues[i]
			var aBig big.Int
			a.BigInt(&aBig)
			for j := range testValues {
				b := testValues[j]
				var bBig, d, e big.Int
				b.BigInt(&bBig)

				var c Element
				c.Div(&a, &b)
				d.ModInverse(&bBig, Modulus())
				d.Mul(&d, &aBig).Mod(&d, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					t.Fatal("Div failed special test values")
				}
			}
		}
	}

	properties.TestingRun(t, gopter.ConsoleReporter(false))
	specialValueTest()

}

func TestElementExp(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()
	genB := gen()

	properties.Property("Exp: having the receiver as operand should output the same result", prop.ForAll(
		func(a, b testPairElement) bool {
			var c, d Element
			d.Set(&a.element)

			c.Exp(a.element, &b.bigint)
			a.element.Exp(a.element, &b.bigint)
			b.element.Exp(d, &b.bigint)

			return a.element.Equal(&b.element) && a.element.Equal(&c) && b.element.Equal(&c)
		},
		genA,
		genB,
	))

	properties.Property("Exp: operation result must match big.Int result", prop.ForAll(
		func(a, b testPairElement) bool {
			{
				var c Element

				c.Exp(a.element, &b.bigint)

				var d, e big.Int
				d.Exp(&a.bigint, &b.bigint, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					return false
				}
			}

			// fixed elements
			// a is random
			// r takes special values
			testValues := make([]Element, len(staticTestValues))
			copy(testValues, staticTestValues)

			for i := range testValues {
				r := testValues[i]
				var d, e, rb big.Int
				r.BigInt(&rb)

				var c Element
				c.Exp(a.element, &rb)
				d.Exp(&a.bigint, &rb, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					return false
				}
			}
			return true
		},
		genA,
		genB,
	))

	properties.Property("Exp: operation result must be smaller than modulus", prop.ForAll(
		func(a, b testPairElement) bool {
			var c Element

			c.Exp(a.element, &b.bigint)

			return c.smallerThanModulus()
		},
		genA,
		genB,
	))

	specialValueTest := func() {
		// test special values against special values
		testValues := make([]Element, len(staticTestValues))
		copy(testValues, staticTestValues)

		for i := range testValues {
			a := testValues[i]
			var aBig big.Int
			a.BigInt(&aBig)
			for j := range testValues {
				b := testValues[j]
				var bBig, d, e big.Int
				b.BigInt(&bBig)

				var c Element
				c.Exp(a, &bBig)
				d.Exp(&aBig, &bBig, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					t.Fatal("Exp failed special test values")
				}
			}
		}
	}

	properties.TestingRun(t, gopter.ConsoleReporter(false))
	specialValueTest()

}

func TestElementSquare(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("Square: having the receiver as operand should output the same result", prop.ForAll(
		func(a testPairElement) bool {

			var b Element

			b.Square(&a.element)
			a.element.Square(&a.element)
			return a.element.Equal(&b)
		},
		genA,
	))

	properties.Property("Square: operation result must match big.Int result", prop.ForAll(
		func(a testPairElement) bool {
			var c Element
			c.Square(&a.element)

			var d, e big.Int
			d.Mul(&a.bigint, &a.bigint).Mod(&d, Modulus())

			return c.BigInt(&e).Cmp(&d) == 0
		},
		genA,
	))

	properties.Property("Square: operation result must be smaller than modulus", prop.ForAll(
		func(a testPairElement) bool {
			var c Element
			c.Square(&a.element)
			return c.smallerThanModulus()
		},
		genA,
	))

	specialValueTest := func() {
		// test special values
		testValues := make([]Element, len(staticTestValues))
		copy(testValues, staticTestValues)

		for i := range testValues {
			a := testValues[i]
			var aBig big.Int
			a.BigInt(&aBig)
			var c Element
			c.Square(&a)

			var d, e big.Int
			d.Mul(&aBig, &aBig).Mod(&d, Modulus())

			if c.BigInt(&e).Cmp(&d) != 0 {
				t.Fatal("Square failed special test values")
			}
		}
	}

	properties.TestingRun(t, gopter.ConsoleReporter(false))
	specialValueTest()

}

func TestElementInverse(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("Inverse: having the receiver as operand should output the same result", prop.ForAll(
		func(a testPairElement) bool {

			var b Element

			b.Inverse(&a.element)
			a.element.Inverse(&a.element)
			return a.element.Equal(&b)
		},
		genA,
	))

	properties.Property("Inverse: operation result must match big.Int result", prop.ForAll(
		func(a testPairElement) bool {
			var c Element
			c.Inverse(&a.element)

			var d, e big.Int
			d.ModInverse(&a.bigint, Modulus())

			return c.BigInt(&e).Cmp(&d) == 0
		},
		genA,
	))

	properties.Property("Inverse: operation result must be smaller than modulus", prop.ForAll(
		func(a testPairElement) bool {
			var c Element
			c.Inverse(&a.element)
			return c.smallerThanModulus()
		},
		genA,
	))

	specialValueTest := func() {
		// test special values
		testValues := make([]Element, len(staticTestValues))
		copy(testValues, staticTestValues)

		for i := range testValues {
			a := testValues[i]
			var aBig big.Int
			a.BigInt(&aBig)
			var c Element
			c.Inverse(&a)

			var d, e big.Int
			d.ModInverse(&aBig, Modulus())

			if c.BigInt(&e).Cmp(&d) != 0 {
				t.Fatal("Inverse failed special test values")
			}
		}
	}

	properties.TestingRun(t, gopter.ConsoleReporter(false))
	specialValueTest()

}

func TestElementSqrt(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("Sqrt: having the receiver as operand should output the same result", prop.ForAll(
		func(a testPairElement) bool {

			b := a.element

			b.Sqrt(&a.element)
			a.element.Sqrt(&a.element)
			return a.element.Equal(&b)
		},
		genA,
	))

	properties.Property("Sqrt: operation result must match big.Int result", prop.ForAll(
		func(a testPairElement) bool {
			var c Element
			c.Sqrt(&a.element)

			var d, e big.Int
			d.ModSqrt(&a.bigint, Modulus())

			return c.BigInt(&e).Cmp(&d) == 0
		},
		genA,
	))

	properties.Property("Sqrt: operation result must be smaller than modulus", prop.ForAll(
		func(a testPairElement) bool {
			var c Element
			c.Sqrt(&a.element)
			return c.smallerThanModulus()
		},
		genA,
	))

	specialValueTest := func() {
		// test special values
		testValues := make([]Element, len(staticTestValues))
		copy(testValues, staticTestValues)

		for i := range testValues {
			a := testValues[i]
			var aBig big.Int
			a.BigInt(&aBig)
			var c Element
			c.Sqrt(&a)

			var d, e big.Int
			d.ModSqrt(&aBig, Modulus())

			if c.BigInt(&e).Cmp(&d) != 0 {
				t.Fatal("Sqrt failed special test values")
			}
		}
	}

	properties.TestingRun(t, gopter.ConsoleReporter(false))
	specialValueTest()

}

func TestElementDouble(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("Double: having the receiver as operand should output the same result", prop.ForAll(
		func(a testPairElement) bool {

			var b Element

			b.Double(&a.element)
			a.element.Double(&a.element)
			return a.element.Equal(&b)
		},
		genA,
	))

	properties.Property("Double: operation result must match big.Int result", prop.ForAll(
		func(a testPairElement) bool {
			var c Element
			c.Double(&a.element)

			var d, e big.Int
			d.Lsh(&a.bigint, 1).Mod(&d, Modulus())

			return c.BigInt(&e).Cmp(&d) == 0
		},
		genA,
	))

	properties.Property("Double: operation result must be smaller than modulus", prop.ForAll(
		func(a testPairElement) bool {
			var c Element
			c.Double(&a.element)
			return c.smallerThanModulus()
		},
		genA,
	))

	specialValueTest := func() {
		// test special values
		testValues := make([]Element, len(staticTestValues))
		copy(testValues, staticTestValues)

		for i := range testValues {
			a := testValues[i]
			var aBig big.Int
			a.BigInt(&aBig)
			var c Element
			c.Double(&a)

			var d, e big.Int
			d.Lsh(&aBig, 1).Mod(&d, Modulus())

			if c.BigInt(&e).Cmp(&d) != 0 {
				t.Fatal("Double failed special test values")
			}
		}
	}

	properties.TestingRun(t, gopter.ConsoleReporter(false))
	specialValueTest()

}

func TestElementNeg(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("Neg: having the receiver as operand should output the same result", prop.ForAll(
		func(a testPairElement) bool {

			var b Element

			b.Neg(&a.element)
			a.element.Neg(&a.element)
			return a.element.Equal(&b)
		},
		genA,
	))

	properties.Property("Neg: operation result must match big.Int result", prop.ForAll(
		func(a testPairElement) bool {
			var c Element
			c.Neg(&a.element)

			var d, e big.Int
			d.Neg(&a.bigint).Mod(&d, Modulus())

			return c.BigInt(&e).Cmp(&d) == 0
		},
		genA,
	))

	properties.Property("Neg: operation result must be smaller than modulus", prop.ForAll(
		func(a testPairElement) bool {
			var c Element
			c.Neg(&a.element)
			return c.smallerThanModulus()
		},
		genA,
	))

	specialValueTest := func() {
		// test special values
		testValues := make([]Element, len(staticTestValues))
		copy(testValues, staticTestValues)

		for i := range testValues {
			a := testValues[i]
			var aBig big.Int
			a.BigInt(&aBig)
			var c Element
			c.Neg(&a)

			var d, e big.Int
			d.Neg(&aBig).Mod(&d, Modulus())

			if c.BigInt(&e).Cmp(&d) != 0 {
				t.Fatal("Neg failed special test values")
			}
		}
	}

	properties.TestingRun(t, gopter.ConsoleReporter(false))
	specialValueTest()

}

func TestElementFixedExp(t *testing.T) {

	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	var (
		_bLegendreExponentElement *big.Int
		_bSqrtExponentElement     *big.Int
	)

	_bLegendreExponentElement, _ = new(big.Int).SetString("3c000000", 16)
	const sqrtExponentElement = "7"
	_bSqrtExponentElement, _ = new(big.Int).SetString(sqrtExponentElement, 16)

	genA := gen()

	properties.Property(fmt.Sprintf("expBySqrtExp must match Exp(%s)", sqrtExponentElement), prop.ForAll(
		func(a testPairElement) bool {
			c := a.element
			d := a.element
			c.expBySqrtExp(c)
			d.Exp(d, _bSqrtExponentElement)
			return c.Equal(&d)
		},
		genA,
	))

	properties.Property("expByLegendreExp must match Exp(3c000000)", prop.ForAll(
		func(a testPairElement) bool {
			c := a.element
			d := a.element
			c.expByLegendreExp(c)
			d.Exp(d, _bLegendreExponentElement)
			return c.Equal(&d)
		},
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestElementHalve(t *testing.T) {

	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()
	var twoInv Element
	twoInv.SetUint64(2)
	twoInv.Inverse(&twoInv)

	properties.Property("z.Halve must match z / 2", prop.ForAll(
		func(a testPairElement) bool {
			c := a.element
			d := a.element
			c.Halve()
			d.Mul(&d, &twoInv)
			return c.Equal(&d)
		},
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func combineSelectionArguments(c int64, z int8) int {
	if z%3 == 0 {
		return 0
	}
	return int(c)
}

func TestElementSelect(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := genFull()
	genB := genFull()
	genC := ggen.Int64() //the condition
	genZ := ggen.Int8()  //to make zeros artificially more likely

	properties.Property("Select: must select correctly", prop.ForAll(
		func(a, b Element, cond int64, z int8) bool {
			condC := combineSelectionArguments(cond, z)

			var c Element
			c.Select(condC, &a, &b)

			if condC == 0 {
				return c.Equal(&a)
			}
			return c.Equal(&b)
		},
		genA,
		genB,
		genC,
		genZ,
	))

	properties.Property("Select: having the receiver as operand should output the same result", prop.ForAll(
		func(a, b Element, cond int64, z int8) bool {
			condC := combineSelectionArguments(cond, z)

			var c, d Element
			d.Set(&a)
			c.Select(condC, &a, &b)
			a.Select(condC, &a, &b)
			b.Select(condC, &d, &b)
			return a.Equal(&b) && a.Equal(&c) && b.Equal(&c)
		},
		genA,
		genB,
		genC,
		genZ,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestElementSetInt64(t *testing.T) {

	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("z.SetInt64 must match z.SetString", prop.ForAll(
		func(a testPairElement, v int64) bool {
			c := a.element
			d := a.element

			c.SetInt64(v)
			d.SetString(fmt.Sprintf("%v", v))

			return c.Equal(&d)
		},
		genA, ggen.Int64(),
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestElementSetInterface(t *testing.T) {

	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()
	genInt := ggen.Int
	genInt8 := ggen.Int8
	genInt16 := ggen.Int16
	genInt32 := ggen.Int32
	genInt64 := ggen.Int64

	genUint := ggen.UInt
	genUint8 := ggen.UInt8
	genUint16 := ggen.UInt16
	genUint32 := ggen.UInt32
	genUint64 := ggen.UInt64

	properties.Property("z.SetInterface must match z.SetString with int8", prop.ForAll(
		func(a testPairElement, v int8) bool {
			c := a.element
			d := a.element

			c.SetInterface(v)
			d.SetString(fmt.Sprintf("%v", v))

			return c.Equal(&d)
		},
		genA, genInt8(),
	))

	properties.Property("z.SetInterface must match z.SetString with int16", prop.ForAll(
		func(a testPairElement, v int16) bool {
			c := a.element
			d := a.element

			c.SetInterface(v)
			d.SetString(fmt.Sprintf("%v", v))

			return c.Equal(&d)
		},
		genA, genInt16(),
	))

	properties.Property("z.SetInterface must match z.SetString with int32", prop.ForAll(
		func(a testPairElement, v int32) bool {
			c := a.element
			d := a.element

			c.SetInterface(v)
			d.SetString(fmt.Sprintf("%v", v))

			return c.Equal(&d)
		},
		genA, genInt32(),
	))

	properties.Property("z.SetInterface must match z.SetString with int64", prop.ForAll(
		func(a testPairElement, v int64) bool {
			c := a.element
			d := a.element

			c.SetInterface(v)
			d.SetString(fmt.Sprintf("%v", v))

			return c.Equal(&d)
		},
		genA, genInt64(),
	))

	properties.Property("z.SetInterface must match z.SetString with int", prop.ForAll(
		func(a testPairElement, v int) bool {
			c := a.element
			d := a.element

			c.SetInterface(v)
			d.SetString(fmt.Sprintf("%v", v))

			return c.Equal(&d)
		},
		genA, genInt(),
	))

	properties.Property("z.SetInterface must match z.SetString with uint8", prop.ForAll(
		func(a testPairElement, v uint8) bool {
			c := a.element
			d := a.element

			c.SetInterface(v)
			d.SetString(fmt.Sprintf("%v", v))

			return c.Equal(&d)
		},
		genA, genUint8(),
	))

	properties.Property("z.SetInterface must match z.SetString with uint16", prop.ForAll(
		func(a testPairElement, v uint16) bool {
			c := a.element
			d := a.element

			c.SetInterface(v)
			d.SetString(fmt.Sprintf("%v", v))

			return c.Equal(&d)
		},
		genA, genUint16(),
	))

	properties.Property("z.SetInterface must match z.SetString with uint32", prop.ForAll(
		func(a testPairElement, v uint32) bool {
			c := a.element
			d := a.element

			c.SetInterface(v)
			d.SetString(fmt.Sprintf("%v", v))

			return c.Equal(&d)
		},
		genA, genUint32(),
	))

	properties.Property("z.SetInterface must match z.SetString with uint64", prop.ForAll(
		func(a testPairElement, v uint64) bool {
			c := a.element
			d := a.element

			c.SetInterface(v)
			d.SetString(fmt.Sprintf("%v", v))

			return c.Equal(&d)
		},
		genA, genUint64(),
	))

	properties.Property("z.SetInterface must match z.SetString with uint", prop.ForAll(
		func(a testPairElement, v uint) bool {
			c := a.element
			d := a.element

			c.SetInterface(v)
			d.SetString(fmt.Sprintf("%v", v))

			return c.Equal(&d)
		},
		genA, genUint(),
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))

	{
		assert := require.New(t)
		var e Element
		r, err := e.SetInterface(nil)
		assert.Nil(r)
		assert.Error(err)

		var ptE *Element
		var ptB *big.Int

		r, err = e.SetInterface(ptE)
		assert.Nil(r)
		assert.Error(err)
		ptE = new(Element).SetOne()
		r, err = e.SetInterface(ptE)
		assert.NoError(err)
		assert.True(r.IsOne())

		r, err = e.SetInterface(ptB)
		assert.Nil(r)
		assert.Error(err)

	}
}

func TestElementNegativeExp(t *testing.T) {
	t.Parallel()

	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("x⁻ᵏ == 1/xᵏ", prop.ForAll(
		func(a, b testPairElement) bool {

			var nb, d, e big.Int
			nb.Neg(&b.bigint)

			var c Element
			c.Exp(a.element, &nb)

			d.Exp(&a.bigint, &nb, Modulus())

			return c.BigInt(&e).Cmp(&d) == 0
		},
		genA, genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestElementNewElement(t *testing.T) {
	assert := require.New(t)

	t.Parallel()

	e := NewElement(1)
	assert.True(e.IsOne())

	e = NewElement(0)
	assert.True(e.IsZero())
}

func TestElementBatchInvert(t *testing.T) {
	assert := require.New(t)

	t.Parallel()

	// ensure batchInvert([x]) == invert(x)
	for i := int64(-1); i <= 2; i++ {
		var e, eInv Element
		e.SetInt64(i)
		eInv.Inverse(&e)

		a := []Element{e}
		aInv := BatchInvert(a)

		assert.True(aInv[0].Equal(&eInv), "batchInvert != invert")

	}

	// test x * x⁻¹ == 1
	tData := [][]int64{
		{-1, 1, 2, 3},
		{0, -1, 1, 2, 3, 0},
		{0, -1, 1, 0, 2, 3, 0},
		{-1, 1, 0, 2, 3},
		{0, 0, 1},
		{1, 0, 0},
		{0, 0, 0},
	}

	for _, t := range tData {
		a := make([]Element, len(t))
		for i := 0; i < len(a); i++ {
			a[i].SetInt64(t[i])
		}

		aInv := BatchInvert(a)

		assert.True(len(aInv) == len(a))

		for i := 0; i < len(a); i++ {
			if a[i].IsZero() {
				assert.True(aInv[i].IsZero(), "0⁻¹ != 0")
			} else {
				assert.True(a[i].Mul(&a[i], &aInv[i]).IsOne(), "x * x⁻¹ != 1")
			}
		}
	}

	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("batchInvert --> x * x⁻¹ == 1", prop.ForAll(
		func(tp testPairElement, r uint8) bool {

			a := make([]Element, r)
			if r != 0 {
				a[0] = tp.element

			}
			one := One()
			for i := 1; i < len(a); i++ {
				a[i].Add(&a[i-1], &one)
			}

			aInv := BatchInvert(a)

			assert.True(len(aInv) == len(a))

			for i := 0; i < len(a); i++ {
				if a[i].IsZero() {
					if !aInv[i].IsZero() {
						return false
					}
				} else {
					if !a[i].Mul(&a[i], &aInv[i]).IsOne() {
						return false
					}
				}
			}
			return true
		},
		genA, ggen.UInt8(),
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestElementFromMont(t *testing.T) {

	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("Assembly implementation must be consistent with generic one", prop.ForAll(
		func(a testPairElement) bool {
			c := a.element
			d := a.element
			c.fromMont()
			_fromMontGeneric(&d)
			return c.Equal(&d)
		},
		genA,
	))

	properties.Property("x.fromMont().toMont() == x", prop.ForAll(
		func(a testPairElement) bool {
			c := a.element
			c.fromMont().toMont()
			return c.Equal(&a.element)
		},
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestElementJSON(t *testing.T) {
	assert := require.New(t)

	type S struct {
		A Element
		B [3]Element
		C *Element
		D *Element
	}

	// encode to JSON
	var s S
	s.A.SetString("-1")
	s.B[2].SetUint64(42)
	s.D = new(Element).SetUint64(8000)

	encoded, err := json.Marshal(&s)
	assert.NoError(err)
	// we may need to adjust "42" and "8000" values for some moduli; see Text() method for more details.
	formatValue := func(v int64) string {
		var a big.Int
		a.SetInt64(v)
		a.Mod(&a, Modulus())
		const maxUint16 = 65535
		var aNeg big.Int
		aNeg.Neg(&a).Mod(&aNeg, Modulus())
		if aNeg.Uint64() != 0 && aNeg.Uint64() <= maxUint16 {
			return "-" + aNeg.Text(10)
		}
		return a.Text(10)
	}
	expected := fmt.Sprintf("{\"A\":%s,\"B\":[0,0,%s],\"C\":null,\"D\":%s}", formatValue(-1), formatValue(42), formatValue(8000))
	assert.Equal(expected, string(encoded))

	// decode valid
	var decoded S
	err = json.Unmarshal([]byte(expected), &decoded)
	assert.NoError(err)

	assert.Equal(s, decoded, "element -> json -> element round trip failed")

	// decode hex and string values
	withHexValues := "{\"A\":\"-1\",\"B\":[0,\"0x00000\",\"0x2A\"],\"C\":null,\"D\":\"8000\"}"

	var decodedS S
	err = json.Unmarshal([]byte(withHexValues), &decodedS)
	assert.NoError(err)

	assert.Equal(s, decodedS, " json with strings  -> element  failed")

}
func TestElementMul2ExpNegN(t *testing.T) {
	t.Parallel()

	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("x * 2⁻ᵏ == Mul2ExpNegN(x, k) for 0 <= k <= 32", prop.ForAll(
		func(a testPairElement) bool {

			var b, e, two Element
			var c [33]Element
			two.SetUint64(2)
			for n := 0; n < 33; n++ {
				e.Exp(two, big.NewInt(int64(n))).Inverse(&e)
				b.Mul(&a.element, &e)
				c[n].Mul2ExpNegN(&a.element, uint32(n))
				if !c[n].Equal(&b) {
					return false
				}
			}
			return true
		},
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

type testPairElement struct {
	element Element
	bigint  big.Int
}

func gen() gopter.Gen {
	return func(genParams *gopter.GenParameters) *gopter.GenResult {
		var g testPairElement

		g.element = Element{
			uint32(genParams.NextUint64()),
		}
		if qElement[0] != ^uint32(0) {
			g.element[0] %= (qElement[0] + 1)
		}

		for !g.element.smallerThanModulus() {
			g.element = Element{
				uint32(genParams.NextUint64()),
			}
			if qElement[0] != ^uint32(0) {
				g.element[0] %= (qElement[0] + 1)
			}
		}

		g.element.BigInt(&g.bigint)
		genResult := gopter.NewGenResult(g, gopter.NoShrinker)
		return genResult
	}
}

func genRandomFq(genParams *gopter.GenParameters) Element {
	var g Element

	g = Element{
		uint32(genParams.NextUint64()),
	}

	if qElement[0] != ^uint32(0) {
		g[0] %= (qElement[0] + 1)
	}

	for !g.smallerThanModulus() {
		g = Element{
			uint32(genParams.NextUint64()),
		}
		if qElement[0] != ^uint32(0) {
			g[0] %= (qElement[0] + 1)
		}
	}

	return g
}

func genFull() gopter.Gen {
	return func(genParams *gopter.GenParameters) *gopter.GenResult {
		a := genRandomFq(genParams)

		var carry uint32
		a[0], _ = bits.Add32(a[0], qElement[0], carry)

		genResult := gopter.NewGenResult(a, gopter.NoShrinker)
		return genResult
	}
}

func genElement() gopter.Gen {
	return func(genParams *gopter.GenParameters) *gopter.GenResult {
		a := genRandomFq(genParams)
		genResult := gopter.NewGenResult(a, gopter.NoShrinker)
		return genResult
	}
}
