// Copyright 2020-2025 Consensys Software Inc.
// Licensed under the Apache License, Version 2.0. See the LICENSE file for details.

// Code generated by consensys/gnark-crypto DO NOT EDIT

package twistededwards

import (
	"crypto/rand"
	"math/big"
	"testing"

	"github.com/consensys/gnark-crypto/ecc/bn254/fr"
	"github.com/leanovate/gopter"
	"github.com/leanovate/gopter/prop"
)

// ------------------------------------------------------------
// tests

const (
	nbFuzzShort = 10
	nbFuzz      = 100
)

func TestReceiverIsOperand(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	// affine
	properties.Property("Equal affine: having the receiver as operand should output the same result", prop.ForAll(
		func() bool {
			params := GetEdwardsCurve()
			var p1 PointAffine
			p1.Set(&params.Base)

			return p1.Equal(&p1) && p1.Equal(&params.Base)
		},
	))

	properties.Property("Add affine: having the receiver as operand should output the same result", prop.ForAll(
		func() bool {

			params := GetEdwardsCurve()

			var p1, p2, p3 PointAffine
			p1.Set(&params.Base)
			p2.Set(&params.Base)
			p3.Set(&params.Base)

			res := true

			p3.Add(&p1, &p2)
			p1.Add(&p1, &p2)
			res = res && p3.Equal(&p1)

			p1.Set(&params.Base)
			p2.Add(&p1, &p2)
			res = res && p2.Equal(&p3)

			return res
		},
	))

	properties.Property("Double affine: having the receiver as operand should output the same result", prop.ForAll(
		func() bool {

			params := GetEdwardsCurve()

			var p1, p2 PointAffine
			p1.Set(&params.Base)
			p2.Set(&params.Base)

			p2.Double(&p1)
			p1.Double(&p1)

			return p2.Equal(&p1)
		},
	))

	properties.Property("Neg affine: having the receiver as operand should output the same result", prop.ForAll(
		func() bool {

			params := GetEdwardsCurve()

			var p1, p2 PointAffine
			p1.Set(&params.Base)
			p2.Set(&params.Base)

			p2.Neg(&p1)
			p1.Neg(&p1)

			return p2.Equal(&p1)
		},
	))

	properties.Property("Neg affine: having the receiver as operand should output the same result", prop.ForAll(
		func() bool {

			params := GetEdwardsCurve()

			var p1, p2 PointAffine
			p1.Set(&params.Base)
			p2.Set(&params.Base)

			var s big.Int
			s.SetUint64(10)

			p2.ScalarMultiplication(&p1, &s)
			p1.ScalarMultiplication(&p1, &s)

			return p2.Equal(&p1)
		},
	))
	properties.TestingRun(t, gopter.ConsoleReporter(false))

	// projective
	properties.Property("Equal projective: having the receiver as operand should output the same result", prop.ForAll(
		func() bool {
			params := GetEdwardsCurve()
			var p1, baseProj PointProj
			p1.FromAffine(&params.Base)
			baseProj.FromAffine(&params.Base)

			return p1.Equal(&p1) && p1.Equal(&baseProj)
		},
	))

	properties.Property("Add projective: having the receiver as operand should output the same result", prop.ForAll(
		func() bool {

			params := GetEdwardsCurve()

			var p1, p2, p3 PointProj
			p1.FromAffine(&params.Base)
			p2.FromAffine(&params.Base)
			p3.FromAffine(&params.Base)

			res := true

			p3.Add(&p1, &p2)
			p1.Add(&p1, &p2)
			res = res && p3.Equal(&p1)

			p1.FromAffine(&params.Base)
			p2.Add(&p1, &p2)
			res = res && p2.Equal(&p3)

			return res
		},
	))

	properties.Property("Double projective: having the receiver as operand should output the same result", prop.ForAll(
		func() bool {

			params := GetEdwardsCurve()

			var p1, p2 PointProj
			p1.FromAffine(&params.Base)
			p2.FromAffine(&params.Base)

			p2.Double(&p1)
			p1.Double(&p1)

			return p2.Equal(&p1)
		},
	))

	properties.Property("Neg projective: having the receiver as operand should output the same result", prop.ForAll(
		func() bool {

			params := GetEdwardsCurve()

			var p1, p2 PointProj
			p1.FromAffine(&params.Base)
			p2.FromAffine(&params.Base)

			p2.Neg(&p1)
			p1.Neg(&p1)

			return p2.Equal(&p1)
		},
	))

	// extended
	properties.Property("Equal extended: having the receiver as operand should output the same result", prop.ForAll(
		func() bool {
			params := GetEdwardsCurve()
			var p1, baseProj PointProj
			p1.FromAffine(&params.Base)
			baseProj.FromAffine(&params.Base)

			return p1.Equal(&p1) && p1.Equal(&baseProj)
		},
	))

	properties.Property("Add extended: having the receiver as operand should output the same result", prop.ForAll(
		func() bool {

			params := GetEdwardsCurve()

			var p1, p2, p3 PointProj
			p1.FromAffine(&params.Base)
			p2.FromAffine(&params.Base)
			p3.FromAffine(&params.Base)

			res := true

			p3.Add(&p1, &p2)
			p1.Add(&p1, &p2)
			res = res && p3.Equal(&p1)

			p1.FromAffine(&params.Base)
			p2.Add(&p1, &p2)
			res = res && p2.Equal(&p3)

			return res
		},
	))

	properties.Property("Double extended: having the receiver as operand should output the same result", prop.ForAll(
		func() bool {

			params := GetEdwardsCurve()

			var p1, p2 PointProj
			p1.FromAffine(&params.Base)
			p2.FromAffine(&params.Base)

			p2.Double(&p1)
			p1.Double(&p1)

			return p2.Equal(&p1)
		},
	))

	properties.Property("Neg extended: having the receiver as operand should output the same result", prop.ForAll(
		func() bool {

			params := GetEdwardsCurve()

			var p1, p2 PointProj
			p1.FromAffine(&params.Base)
			p2.FromAffine(&params.Base)

			p2.Neg(&p1)
			p1.Neg(&p1)

			return p2.Equal(&p1)
		},
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))

}

func TestField(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)
	genS := GenBigInt()

	properties.Property("MulByA(x) should match Mul(x, curve.A)", prop.ForAll(
		func(s big.Int) bool {

			params := GetEdwardsCurve()

			var z1, z2 fr.Element
			z1.SetBigInt(&s)
			z2.Mul(&z1, &params.A)
			mulByA(&z1)

			return z1.Equal(&z2)
		},
		genS,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestOps(t *testing.T) {

	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)
	genS1 := GenBigInt()
	genS2 := GenBigInt()

	// affine
	properties.Property("(affine) 0+0=2*0=0", prop.ForAll(
		func(s1 big.Int) bool {

			var p1, p2, zero PointAffine
			zero.setInfinity()

			p1.Add(&zero, &zero)
			p2.Double(&zero)

			return p1.IsOnCurve() && p1.Equal(&zero) && p1.Equal(&p2)
		},
		genS1,
	))

	properties.Property("(affine) P+0=P", prop.ForAll(
		func(s1 big.Int) bool {

			params := GetEdwardsCurve()

			var p1, p2, zero PointAffine
			p1.ScalarMultiplication(&params.Base, &s1)
			zero.setInfinity()

			p2.Add(&p1, &zero)

			return p2.IsOnCurve() && p2.Equal(&p1)
		},
		genS1,
	))

	properties.Property("(affine) P+(-P)=O", prop.ForAll(
		func(s1 big.Int) bool {

			params := GetEdwardsCurve()

			var p1, p2 PointAffine
			p1.ScalarMultiplication(&params.Base, &s1)
			p2.Neg(&p1)

			p1.Add(&p1, &p2)

			var one fr.Element
			one.SetOne()

			return p1.IsOnCurve() && p1.IsZero()
		},
		genS1,
	))

	properties.Property("(affine) P+P=2*P", prop.ForAll(
		func(s big.Int) bool {

			params := GetEdwardsCurve()

			var p1, p2, inf PointAffine
			p1.ScalarMultiplication(&params.Base, &s)
			p2.ScalarMultiplication(&params.Base, &s)

			p1.Add(&p1, &p2)
			p2.Double(&p2)

			return p1.IsOnCurve() && p1.Equal(&p2) && !p1.Equal(&inf)
		},
		genS1,
	))

	properties.Property("(affine) [a]P+[b]P = [a+b]P", prop.ForAll(
		func(s1, s2 big.Int) bool {

			params := GetEdwardsCurve()

			var p1, p2, p3, inf PointAffine
			inf.X.SetZero()
			inf.Y.SetZero()
			p1.ScalarMultiplication(&params.Base, &s1)
			p2.ScalarMultiplication(&params.Base, &s2)
			p3.Set(&params.Base)

			p2.Add(&p1, &p2)

			s1.Add(&s1, &s2)
			p3.ScalarMultiplication(&params.Base, &s1)

			return p2.IsOnCurve() && p3.Equal(&p2) && !p3.Equal(&inf)
		},
		genS1,
		genS2,
	))

	properties.Property("(affine) [a]P+[-a]P = O", prop.ForAll(
		func(s1 big.Int) bool {

			params := GetEdwardsCurve()

			var p1, p2, inf PointAffine
			inf.X.SetZero()
			inf.Y.SetOne()
			p1.ScalarMultiplication(&params.Base, &s1)
			s1.Neg(&s1)
			p2.ScalarMultiplication(&params.Base, &s1)

			p2.Add(&p1, &p2)

			return p2.IsOnCurve() && p2.Equal(&inf)
		},
		genS1,
	))

	properties.Property("(affine) [5]P=[2][2]P+P", prop.ForAll(
		func(s1 big.Int) bool {

			params := GetEdwardsCurve()

			var p1, p2 PointAffine
			p1.ScalarMultiplication(&params.Base, &s1)

			five := big.NewInt(5)
			p2.Double(&p1).Double(&p2).Add(&p2, &p1)
			p1.ScalarMultiplication(&p1, five)

			return p2.IsOnCurve() && p2.Equal(&p1)
		},
		genS1,
	))

	// projective
	properties.Property("(projective) 0+0=2*0=0", prop.ForAll(
		func(s1 big.Int) bool {

			var p1, p2, zero PointProj
			zero.setInfinity()

			p1.Add(&zero, &zero)
			p2.Double(&zero)

			return p1.Equal(&zero) && p1.Equal(&p2)
		},
		genS1,
	))

	properties.Property("(projective) P+0=P", prop.ForAll(
		func(s1 big.Int) bool {

			params := GetEdwardsCurve()

			var baseProj, p1, p2, zero PointProj
			baseProj.FromAffine(&params.Base)
			p1.ScalarMultiplication(&baseProj, &s1)
			zero.setInfinity()

			p2.Add(&p1, &zero)

			return p2.Equal(&p1)
		},
		genS1,
	))

	properties.Property("(projective) P+(-P)=O", prop.ForAll(
		func(s1 big.Int) bool {

			params := GetEdwardsCurve()

			var baseProj, p1, p2, p PointProj
			baseProj.FromAffine(&params.Base)
			p1.ScalarMultiplication(&baseProj, &s1)
			p2.Neg(&p1)

			p.Add(&p1, &p2)

			return p.IsZero()
		},
		genS1,
	))

	properties.Property("(projective) P+P=2*P", prop.ForAll(

		func(s big.Int) bool {

			params := GetEdwardsCurve()

			var baseProj, p1, p2, p PointProj
			baseProj.FromAffine(&params.Base)
			p.ScalarMultiplication(&baseProj, &s)

			p1.Add(&p, &p)
			p2.Double(&p)

			return p1.Equal(&p2)
		},
		genS1,
	))

	properties.Property("(projective) [5]P=[2][2]P+P", prop.ForAll(
		func(s1 big.Int) bool {

			params := GetEdwardsCurve()

			var baseProj, p1, p2 PointProj
			baseProj.FromAffine(&params.Base)
			p1.ScalarMultiplication(&baseProj, &s1)

			five := big.NewInt(5)
			p2.Double(&p1).Double(&p2).Add(&p2, &p1)
			p1.ScalarMultiplication(&p1, five)

			return p2.Equal(&p1)
		},
		genS1,
	))

	// extended
	properties.Property("(extended) 0+0=0", prop.ForAll(
		func(s1 big.Int) bool {

			var p1, zero PointExtended
			zero.setInfinity()

			p1.Add(&zero, &zero)

			return p1.Equal(&zero)
		},
		genS1,
	))

	properties.Property("(extended) P+0=P", prop.ForAll(
		func(s1 big.Int) bool {

			params := GetEdwardsCurve()

			var baseExtended, p1, p2, zero PointExtended
			baseExtended.FromAffine(&params.Base)
			p1.ScalarMultiplication(&baseExtended, &s1)
			zero.setInfinity()

			p2.Add(&p1, &zero)

			return p2.Equal(&p1)
		},
		genS1,
	))

	properties.Property("(extended) P+(-P)=O", prop.ForAll(
		func(s1 big.Int) bool {

			params := GetEdwardsCurve()

			var baseExtended, p1, p2, p PointExtended
			baseExtended.FromAffine(&params.Base)
			p1.ScalarMultiplication(&baseExtended, &s1)
			p2.Neg(&p1)

			p.Add(&p1, &p2)

			return p.IsZero()
		},
		genS1,
	))

	properties.Property("(extended) P+P=2*P", prop.ForAll(

		func(s big.Int) bool {

			params := GetEdwardsCurve()

			var baseExtended, p1, p2, p PointExtended
			baseExtended.FromAffine(&params.Base)
			p.ScalarMultiplication(&baseExtended, &s)

			p1.Add(&p, &p)
			p2.Double(&p)

			return p1.Equal(&p2)
		},
		genS1,
	))

	properties.Property("(extended) [5]P=[2][2]P+P", prop.ForAll(
		func(s1 big.Int) bool {

			params := GetEdwardsCurve()

			var baseExtended, p1, p2 PointExtended
			baseExtended.FromAffine(&params.Base)
			p1.ScalarMultiplication(&baseExtended, &s1)

			five := big.NewInt(5)
			p2.Double(&p1).Double(&p2).Add(&p2, &p1)
			p1.ScalarMultiplication(&p1, five)

			return p2.Equal(&p1)
		},
		genS1,
	))

	// mixed affine+extended
	properties.Property("(mixed affine+extended) P+(-P)=O", prop.ForAll(
		func(s big.Int) bool {

			params := GetEdwardsCurve()

			var baseExtended, pExtended, p PointExtended
			var pAffine PointAffine
			baseExtended.FromAffine(&params.Base)
			pExtended.ScalarMultiplication(&baseExtended, &s)
			pAffine.ScalarMultiplication(&params.Base, &s)
			pAffine.Neg(&pAffine)

			p.MixedAdd(&pExtended, &pAffine)

			return p.IsZero()
		},
		genS1,
	))

	properties.Property("(mixed affine+extended) P+P=2*P", prop.ForAll(
		func(s big.Int) bool {

			params := GetEdwardsCurve()

			var baseExtended, pExtended, p, p2 PointExtended
			var pAffine PointAffine
			baseExtended.FromAffine(&params.Base)
			pExtended.ScalarMultiplication(&baseExtended, &s)
			pAffine.ScalarMultiplication(&params.Base, &s)

			p.MixedAdd(&pExtended, &pAffine)
			p2.MixedDouble(&pExtended)

			return p.Equal(&p2)
		},
		genS1,
	))

	// mixed affine+projective
	properties.Property("(mixed affine+proj) P+(-P)=O", prop.ForAll(
		func(s big.Int) bool {

			params := GetEdwardsCurve()

			var baseProj, pProj, p PointProj
			var pAffine PointAffine
			baseProj.FromAffine(&params.Base)
			pProj.ScalarMultiplication(&baseProj, &s)
			pAffine.ScalarMultiplication(&params.Base, &s)
			pAffine.Neg(&pAffine)

			p.MixedAdd(&pProj, &pAffine)

			return p.IsZero()
		},
		genS1,
	))

	properties.Property("(mixed affine+proj) P+P=2*P", prop.ForAll(
		func(s big.Int) bool {

			params := GetEdwardsCurve()

			var baseProj, pProj, p, p2 PointProj
			var pAffine PointAffine
			baseProj.FromAffine(&params.Base)
			pProj.ScalarMultiplication(&baseProj, &s)
			pAffine.ScalarMultiplication(&params.Base, &s)

			p.MixedAdd(&pProj, &pAffine)
			p2.Double(&pProj)

			return p.Equal(&p2)
		},
		genS1,
	))

	properties.Property("scalar multiplication in Proj vs Ext should be consistent", prop.ForAll(
		func(s big.Int) bool {

			params := GetEdwardsCurve()

			var baseProj PointProj
			var baseExt PointExtended
			var p1, p2 PointAffine
			baseProj.FromAffine(&params.Base)
			baseProj.ScalarMultiplication(&baseProj, &s)
			baseExt.FromAffine(&params.Base)
			baseExt.ScalarMultiplication(&baseExt, &s)

			p1.FromProj(&baseProj)
			p2.FromExtended(&baseExt)

			return p1.Equal(&p2)
		},
		genS1,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))

}

func TestMarshal(t *testing.T) {
	t.Parallel()
	initOnce.Do(initCurveParams)

	var point, unmarshalPoint PointAffine
	point.Set(&curveParams.Base)
	for i := 0; i < 20; i++ {
		b := point.Marshal()
		unmarshalPoint.Unmarshal(b)
		if !point.Equal(&unmarshalPoint) {
			t.Fatal("error unmarshal(marshal(point))")
		}
		point.Add(&point, &curveParams.Base)
	}
}

// GenBigInt generates a big.Int
// TODO @thomas we use fr size as max bound here
func GenBigInt() gopter.Gen {
	return func(genParams *gopter.GenParameters) *gopter.GenResult {
		var s big.Int
		var b [fr.Bytes]byte
		_, err := rand.Read(b[:]) //#nosec G404 weak rng is fine here
		if err != nil {
			panic(err)
		}
		s.SetBytes(b[:])
		genResult := gopter.NewGenResult(s, gopter.NoShrinker)
		return genResult
	}
}

// ------------------------------------------------------------
// benches

func BenchmarkProjEqual(b *testing.B) {
	params := GetEdwardsCurve()

	var scalar fr.Element
	scalar.MustSetRandom()

	var baseProj PointProj
	baseProj.FromAffine(&params.Base)
	var a PointProj
	a.ScalarMultiplication(&baseProj, big.NewInt(42))

	b.Run("equal", func(b *testing.B) {
		aZScaled := a
		aZScaled.X.Mul(&aZScaled.X, &scalar)
		aZScaled.Y.Mul(&aZScaled.Y, &scalar)
		aZScaled.Z.Mul(&aZScaled.Z, &scalar)

		// Check the setup.
		if !a.Equal(&aZScaled) {
			b.Fatalf("invalid test setup")
		}

		b.ResetTimer()
		for i := 0; i < b.N; i++ {
			a.Equal(&aZScaled)
		}
	})

	b.Run("not equal", func(b *testing.B) {
		var aPlus1 PointProj
		aPlus1.Add(&a, &baseProj)

		// Check the setup.
		if a.Equal(&aPlus1) {
			b.Fatalf("invalid test setup")
		}

		b.ResetTimer()
		for i := 0; i < b.N; i++ {
			a.Equal(&aPlus1)
		}
	})
}

func BenchmarkScalarMulExtended(b *testing.B) {
	params := GetEdwardsCurve()
	var a PointExtended
	var s big.Int
	a.FromAffine(&params.Base)
	s.SetString("52435875175126190479447705081859658376581184513", 10)
	s.Add(&s, &params.Order)

	var doubleAndAdd PointExtended

	b.ResetTimer()
	for j := 0; j < b.N; j++ {
		doubleAndAdd.ScalarMultiplication(&a, &s)
	}
}

func BenchmarkScalarMulProjective(b *testing.B) {
	params := GetEdwardsCurve()
	var a PointProj
	var s big.Int
	a.FromAffine(&params.Base)
	s.SetString("52435875175126190479447705081859658376581184513", 10)
	s.Add(&s, &params.Order)

	var doubleAndAdd PointProj

	b.ResetTimer()
	for j := 0; j < b.N; j++ {
		doubleAndAdd.ScalarMultiplication(&a, &s)
	}
}

func BenchmarkNeg(b *testing.B) {
	params := GetEdwardsCurve()
	var s big.Int
	s.SetString("52435875175126190479447705081859658376581184513", 10)

	b.Run("Affine", func(b *testing.B) {
		var point PointAffine
		point.ScalarMultiplication(&params.Base, &s)

		b.ResetTimer()
		for i := 0; i < b.N; i++ {
			point.Neg(&point)
		}
	})
	b.Run("Projective", func(b *testing.B) {
		var baseProj PointProj
		baseProj.FromAffine(&params.Base)
		var point PointProj
		point.ScalarMultiplication(&baseProj, &s)

		b.ResetTimer()
		for i := 0; i < b.N; i++ {
			point.Neg(&point)
		}
	})
	b.Run("Extended", func(b *testing.B) {
		var baseProj PointExtended
		baseProj.FromAffine(&params.Base)
		var point PointExtended
		point.ScalarMultiplication(&baseProj, &s)

		b.ResetTimer()
		for i := 0; i < b.N; i++ {
			point.Neg(&point)
		}
	})
}

func BenchmarkMixedAdd(b *testing.B) {
	params := GetEdwardsCurve()
	var s big.Int
	s.SetString("52435875175126190479447705081859658376581184513", 10)
	var point PointAffine
	point.ScalarMultiplication(&params.Base, &s)

	b.Run("Projective", func(b *testing.B) {
		var accum PointProj
		accum.setInfinity()

		b.ResetTimer()
		for i := 0; i < b.N; i++ {
			accum.MixedAdd(&accum, &point)
		}
	})
	b.Run("Extended", func(b *testing.B) {
		var accum PointExtended
		accum.setInfinity()

		b.ResetTimer()
		for i := 0; i < b.N; i++ {
			accum.MixedAdd(&accum, &point)
		}
	})
}

func BenchmarkAdd(b *testing.B) {
	params := GetEdwardsCurve()
	var s big.Int
	s.SetString("52435875175126190479447705081859658376581184513", 10)

	b.Run("Affine", func(b *testing.B) {
		var point PointAffine
		point.ScalarMultiplication(&params.Base, &s)
		var accum PointAffine
		accum.setInfinity()

		b.ResetTimer()
		for i := 0; i < b.N; i++ {
			accum.Add(&accum, &point)
		}
	})
	b.Run("Projective", func(b *testing.B) {
		var pointAff PointAffine
		pointAff.ScalarMultiplication(&params.Base, &s)
		var accum, point PointProj
		point.FromAffine(&pointAff)
		accum.setInfinity()

		b.ResetTimer()
		for i := 0; i < b.N; i++ {
			accum.Add(&accum, &point)
		}
	})
	b.Run("Extended", func(b *testing.B) {
		var pointAff PointAffine
		pointAff.ScalarMultiplication(&params.Base, &s)
		var accum, point PointExtended
		point.FromAffine(&pointAff)
		accum.setInfinity()

		b.ResetTimer()
		for i := 0; i < b.N; i++ {
			accum.Add(&accum, &point)
		}
	})
}

func BenchmarkIsOnCurve(b *testing.B) {
	params := GetEdwardsCurve()
	var s big.Int
	s.SetString("52435875175126190479447705081859658376581184513", 10)

	b.Run("positive", func(b *testing.B) {
		var point PointAffine
		point.ScalarMultiplication(&params.Base, &s)

		if !point.IsOnCurve() {
			b.Fatal("point should must be on curve")
		}

		b.ResetTimer()
		for i := 0; i < b.N; i++ {
			_ = point.IsOnCurve()
		}
	})

	b.Run("negative", func(b *testing.B) {
		var point PointAffine
		point.ScalarMultiplication(&params.Base, &s)
		point.X.Add(&point.X, &point.X)

		if point.IsOnCurve() {
			b.Fatal("point should not be on curve")
		}

		b.ResetTimer()
		for i := 0; i < b.N; i++ {
			_ = point.IsOnCurve()
		}
	})
}
