import React, { useCallback, useMemo, useRef } from 'react';
import { useDragDropContext } from '../drag-drop-context/internal-context';
import { rbdInvariant } from '../drag-drop-context/rbd-invariant';
import { useDraggableData } from '../draggable/data';
import { Placeholder } from '../draggable/placeholder';
import { useDropTargetForDraggable } from '../hooks/use-drop-target-for-draggable';

/**
 * The virtual placeholder exists specifically for virtual lists,
 * to ensure that the injected placeholder is correctly positioned.
 *
 * Standard placeholders are rendered as siblings, and do not need explicit
 * positioning.
 *
 * Because virtual placeholders are injected through a portal, they need to be
 * absolutely positioned so that they cover the gap left by the dragging item.
 *
 * This placeholder is important because it acts as the drop target for the
 * dragging item.
 */
export function VirtualPlaceholder(_ref) {
  var draggableId = _ref.draggableId,
    droppableId = _ref.droppableId,
    type = _ref.type,
    direction = _ref.direction,
    isDropDisabled = _ref.isDropDisabled;
  var ref = useRef(null);
  var _useDragDropContext = useDragDropContext(),
    contextId = _useDragDropContext.contextId,
    getDragState = _useDragDropContext.getDragState;
  var dragState = getDragState();
  rbdInvariant(dragState.isDragging, 'The virtual placeholder should only be rendered during a drag');
  var getIndex = useCallback(function () {
    return dragState.sourceLocation.index;
  }, [dragState.sourceLocation.index]);
  var data = useDraggableData({
    draggableId: draggableId,
    droppableId: droppableId,
    getIndex: getIndex,
    contextId: contextId,
    type: type
  });

  /**
   * This sets up the drop target for the dragging item.
   */
  useDropTargetForDraggable({
    elementRef: ref,
    data: data,
    direction: direction,
    contextId: contextId,
    isDropDisabled: isDropDisabled,
    type: type
  });
  var style = useMemo(function () {
    return {
      position: 'absolute',
      top: dragState.draggableInitialOffsetInSourceDroppable.top,
      left: dragState.draggableInitialOffsetInSourceDroppable.left,
      margin: 0
    };
  }, [dragState.draggableInitialOffsetInSourceDroppable.left, dragState.draggableInitialOffsetInSourceDroppable.top]);

  // eslint-disable-next-line @atlaskit/ui-styling-standard/enforce-style-prop -- Ignored via go/DSP-18766
  return /*#__PURE__*/React.createElement(Placeholder, {
    ref: ref,
    style: style
  });
}