/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ERROR_DETAILS_PAGE_BASE_URL } from './error_details_base_url';
/**
 * Class that represents a runtime error.
 * Formats and outputs the error message in a consistent way.
 *
 * Example:
 * ```
 *  throw new RuntimeError(
 *    RuntimeErrorCode.INJECTOR_ALREADY_DESTROYED,
 *    ngDevMode && 'Injector has already been destroyed.');
 * ```
 *
 * Note: the `message` argument contains a descriptive error message as a string in development
 * mode (when the `ngDevMode` is defined). In production mode (after tree-shaking pass), the
 * `message` argument becomes `false`, thus we account for it in the typings and the runtime logic.
 */
export class RuntimeError extends Error {
    constructor(code, message) {
        super(formatRuntimeError(code, message));
        this.code = code;
    }
}
/**
 * Called to format a runtime error.
 * See additional info on the `message` argument type in the `RuntimeError` class description.
 */
export function formatRuntimeError(code, message) {
    // Error code might be a negative number, which is a special marker that instructs the logic to
    // generate a link to the error details page on angular.io.
    const fullCode = `NG0${Math.abs(code)}`;
    let errorMessage = `${fullCode}${message ? ': ' + message.trim() : ''}`;
    if (ngDevMode && code < 0) {
        const addPeriodSeparator = !errorMessage.match(/[.,;!?]$/);
        const separator = addPeriodSeparator ? '.' : '';
        errorMessage =
            `${errorMessage}${separator} Find more at ${ERROR_DETAILS_PAGE_BASE_URL}/${fullCode}`;
    }
    return errorMessage;
}
//# sourceMappingURL=data:application/json;base64,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