#
# Copyright 2022 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

#
# Appendix C - Intermediate verification of input and output
#

import numpy as np


NBYTES = (32e3 * np.array([ 7.5e-3, 10e-3 ]) / 8).astype(int)


### C.3.1.1  PCM Input

X_PCM_10M = np.array([

    [      0,   3212,   6392,   9512,  12539,  15446,  18205,  20788,
       23170,  25328,  27244,  28898,  30272,  31357,  32137,  32609,
       32767,  32609,  32138,  31356,  30272,  28898,  27245,  25330,
       23169,  20787,  18205,  15446,  12539,   9511,   6393,   3212,
           0,  -3212,  -6393,  -9512, -12540, -15446, -18204, -20787,
      -23170, -25329, -27245, -28898, -30273, -31356, -32137, -32610,
      -32766, -32609, -32137, -31356, -30272, -28898, -27244, -25329,
      -23171, -20787, -18204, -15446, -12539,  -9511,  -6393,  -3212,
          -1,   3212,   6393,   9512,  12540,  15446,  18204,  20788,
       23169,  25329,  27245,  28898,  30273,  31356,  32137,  32609,
       32767,  32609,  32137,  31356,  30273,  28898,  27245,  25330,
       23170,  20787,  18204,  15446,  12540,   9512,   6393,   3212,
           0,  -3212,  -6393,  -9512, -12539, -15447, -18204, -20787,
      -23170, -25330, -27244, -28898, -30272, -31356, -32137, -32609,
      -32767, -32609, -32137, -31356, -30273, -28898, -27244, -25330,
      -23169, -20787, -18204, -15446, -12540,  -9511,  -6393,  -3212,
           0,   3212,   6392,   9511,  12539,  15446,  18205,  20787,
       23169,  25329,  27245,  28898,  30273,  31356,  32137,  32609,
       32767,  32610,  32137,  31356,  30273,  28898,  27244,  25329,
       23170,  20787,  18204,  15446,  12540,   9511,   6392,   3211 ],

    [      0,  -3211,  -6393,  -9512, -12539, -15446, -18204, -20788,
      -23170, -25329, -27245, -28898, -30273, -31356, -32137, -32609,
      -32767, -32609, -32138, -31356, -30273, -28898, -27245, -25329,
      -23170, -20788, -18205, -15447, -12539,  -9512,  -6392,  -3211,
           0,   3211,   6393,   9512,  12539,  15446,  18204,  20787,
       23170,  25329,  27244,  28897,  30273,  31356,  32137,  32609,
       32767,  32609,  32137,  31356,  30273,  28897,  27244,  25330,
       23170,  20787,  18205,  15446,  12539,   9512,   6393,   3213,
           0,  -3212,  -6393,  -9512, -12540, -15447, -18205, -20787,
      -23169, -25329, -27245, -28897, -30273, -31356, -32138, -32609,
      -32767, -32609, -32138, -31356, -30273, -28898, -27244, -25330,
      -23170, -20787, -18205, -15446, -12540,  -9513,  -6392,  -3212,
           0,   3212,   6393,   9512,  12540,  15446,  18204,  20788,
       23170,  25329,  27245,  28898,  30272,  31356,  32138,  32609,
       32767,  32609,  32138,  31356,  30273,  28898,  27244,  25330,
       23170,  20787,  18204,  15446,  12540,   9512,   6392,   3211,
           0,  -3211,  -6393,  -9512, -12539, -15446, -18205, -20786,
      -23170, -25329, -27245, -28898, -30272, -31356, -32137, -32609,
      -32766, -32609, -32137, -31356, -30273, -28898, -27245, -25329,
      -23170, -20787, -18204, -15446, -12539,  -9511,  -6393,  -3212 ]

])

X_PCM_7M5 = np.array([

    [      0,   3212,   6392,   9512,  12539,  15446,  18205,  20788,
       23170,  25328,  27244,  28898,  30272,  31357,  32137,  32609,
       32767,  32609,  32138,  31356,  30272,  28898,  27245,  25330,
       23169,  20787,  18205,  15446,  12539,   9511,   6393,   3212,
           0,  -3212,  -6393,  -9512, -12540, -15446, -18204, -20787,
      -23170, -25329, -27245, -28898, -30273, -31356, -32137, -32610,
      -32766, -32609, -32137, -31356, -30272, -28898, -27244, -25329,
      -23171, -20787, -18204, -15446, -12539,  -9511,  -6393,  -3212,
          -1,   3212,   6393,   9512,  12540,  15446,  18204,  20788,
       23169,  25329,  27245,  28898,  30273,  31356,  32137,  32609,
       32767,  32609,  32137,  31356,  30273,  28898,  27245,  25330,
       23170,  20787,  18204,  15446,  12540,   9512,   6393,   3212,
           0,  -3212,  -6393,  -9512, -12539, -15447, -18204, -20787,
      -23170, -25330, -27244, -28898, -30272, -31356, -32137, -32609,
      -32767, -32609, -32137, -31356, -30273, -28898, -27244, -25330 ],

    [ -23169, -20787, -18204, -15446, -12540,  -9511,  -6393,  -3212,
           0,   3212,   6392,   9511,  12539,  15446,  18205,  20787,
       23169,  25329,  27245,  28898,  30273,  31356,  32137,  32609,
       32767,  32610,  32137,  31356,  30273,  28898,  27244,  25329,
       23170,  20787,  18204,  15446,  12540,   9511,   6392,   3211,
           0,  -3211,  -6393,  -9512, -12539, -15446, -18204, -20788,
      -23170, -25329, -27245, -28898, -30273, -31356, -32137, -32609,
      -32767, -32609, -32138, -31356, -30273, -28898, -27245, -25329,
      -23170, -20788, -18205, -15447, -12539,  -9512,  -6392,  -3211,
           0,   3211,   6393,   9512,  12539,  15446,  18204,  20787,
       23170,  25329,  27244,  28897,  30273,  31356,  32137,  32609,
       32767,  32609,  32137,  31356,  30273,  28897,  27244,  25330,
       23170,  20787,  18205,  15446,  12539,   9512,   6393,   3213,
           0,  -3212,  -6393,  -9512, -12540, -15447, -18205, -20787,
      -23169, -25329, -27245, -28897, -30273, -31356, -32138, -32609 ]

])

X_PCM = [ X_PCM_7M5, X_PCM_10M ]


### C.3.1.2  MDCT

X_10M = np.array([

    [ -5.8990646e+02,  3.2262618e+04, -2.7619007e+04,  9.5178147e+04,
      -1.1418053e+05, -2.2419557e+05, -2.2347007e+03, -1.9867627e+04,
      -1.6067159e+04,  5.8154816e+02, -1.5722676e+04,  4.0158688e+03,
      -1.0088102e+04,  1.9017417e+03, -4.1954471e+03, -9.7496049e+02,
       4.7235950e+02, -2.7499647e+03,  2.4705648e+03, -2.7529252e+03,
       2.2309610e+03, -1.4814949e+03,  8.0924574e+02,  1.0419403e+02,
      -6.3151413e+02,  1.1597939e+03, -1.3202428e+03,  1.3143498e+03,
      -1.1022736e+03,  7.2762067e+02, -3.2778511e+02, -1.1134462e+02,
       4.5185190e+02, -7.0065111e+02,  8.1286568e+02, -7.8143265e+02,
       6.4645208e+02, -4.1055413e+02,  1.5009894e+02,  1.1832095e+02,
      -3.4137778e+02,  4.8690132e+02, -5.5346349e+02,  5.1921969e+02,
      -4.1808273e+02,  2.5332159e+02, -6.4719513e+01, -1.1746306e+02,
       2.7072573e+02, -3.6623733e+02,  4.0439184e+02, -3.7097974e+02,
       2.8704947e+02, -1.6297022e+02,  2.0395888e+01,  1.1218314e+02,
      -2.2577273e+02,  2.8951685e+02, -3.1083971e+02,  2.7771234e+02,
      -2.0730346e+02,  1.0688285e+02,  6.1320766e+00, -1.0817459e+02,
       1.9320762e+02, -2.3845257e+02,  2.4826403e+02, -2.1609003e+02,
       1.5207388e+02, -6.9546445e+01, -2.4211219e+01,  1.0403883e+02,
      -1.6888652e+02,  2.0050394e+02, -2.0326028e+02,  1.7179995e+02,
      -1.1399655e+02,  4.4242025e+01,  3.4454794e+01, -1.0009362e+02,
       1.5218958e+02, -1.7342213e+02,  1.7018985e+02, -1.3794243e+02,
       8.6156013e+01, -2.4921223e+01, -4.2552602e+01,  9.5559562e+01,
      -1.3740945e+02,  1.5218075e+02, -1.4462762e+02,  1.1451272e+02,
      -6.4959967e+01,  1.0627359e+01,  4.7836856e+01, -9.2854453e+01,
       1.2654514e+02, -1.3593370e+02,  1.2459754e+02, -9.4075815e+01,
       4.9816314e+01,  2.1529924e-01, -5.2425581e+01,  9.0995703e+01,
      -1.1737069e+02,  1.2263969e+02, -1.0917602e+02,  7.9712422e+01,
      -3.5722986e+01, -9.6955535e+00,  5.6602292e+01, -8.9431609e+01,
       1.0953959e+02, -1.1249540e+02,  9.6333120e+01, -6.6446434e+01,
       2.3693799e+01,  1.7708430e+01, -5.8794346e+01,  8.8078076e+01,
      -1.0478463e+02,  1.0227969e+02, -8.4957399e+01,  5.5726976e+01,
      -1.4768315e+01, -2.3981122e+01,  6.1741642e+01, -8.6508895e+01,
       9.9868691e+01, -9.6459597e+01,  7.6331011e+01, -4.6588689e+01,
       6.4212746e+00,  3.0432190e+01, -6.6116496e+01,  8.6861568e+01,
      -9.6320778e+01,  8.9835533e+01, -6.7287784e+01,  3.7477990e+01,
       1.3842189e+00, -3.5731585e+01,  6.8605400e+01, -8.7530923e+01,
       9.3616286e+01, -8.3728496e+01,  6.0230516e+01, -2.9414119e+01,
      -7.6609110e+00,  4.2149725e+01, -7.1968501e+01,  8.7743888e+01,
      -9.0802890e+01,  7.8923198e+01, -5.4973827e+01,  2.2745574e+01,
       1.5766746e+01, -4.7673661e+01,  7.5778794e+01, -8.9162686e+01 ],

    [ -6.2843560e+03, -1.4627418e+04,  4.9801516e+03, -7.0344966e+04,
       8.5564327e+04,  2.5840606e+05, -3.5208419e+04,  5.4644134e+04,
      -1.1212441e+04,  1.8683629e+04,  1.8049757e+03,  7.0569176e+03,
       1.2829514e+03,  4.0930299e+03,  1.1947052e+03,  1.5425662e+03,
       6.0325642e+02,  6.4703789e+02,  1.5715070e+02,  1.0644751e+02,
      -7.5883978e+00, -5.2773353e+01, -6.0996565e+01, -8.3409817e+01,
      -4.3996776e+01, -3.3763658e+01, -1.2075849e+01, -2.9293481e+00,
       5.2907677e+00,  1.4501256e+01,  1.0866309e+01,  9.1905374e+00,
       3.0606323e+00,  1.6027594e+00, -1.1174900e+00, -5.4280671e+00,
      -4.5919914e+00, -4.4199737e+00, -2.0085059e+00, -4.2065884e-01,
       6.3446132e-01,  2.7718649e+00,  3.5654183e+00,  2.6588468e+00,
       1.3730777e+00,  1.3705866e+00,  6.3801955e-02, -1.1191414e+00,
      -2.1495458e+00, -2.1211746e+00, -1.1188698e+00, -1.3646427e+00,
      -1.2245570e+00,  1.5630676e-01,  1.0243080e+00,  2.0104419e+00,
       1.8985278e+00,  1.2447591e+00,  1.7309919e+00,  9.4234385e-01,
      -3.3619810e-01, -1.6030368e+00, -7.1471558e-01, -2.0710920e+00,
      -2.9423847e+00, -1.7873501e+00, -8.0633559e-01, -1.6093125e-03,
       7.9415802e-01,  8.7937487e-01,  1.7154607e+00,  1.0383457e+00,
       2.3742697e+00,  9.7376296e-01,  1.5290469e+00,  1.1078186e-01,
      -1.3382843e+00, -8.0183060e-01, -1.1937948e+00, -1.6144857e+00,
      -2.4688787e+00,  2.7158214e-01, -2.8802354e-01,  1.5704268e-01,
      -2.7983134e-01,  7.3794617e-01,  1.0364997e+00,  6.9563470e-01,
       9.9846216e-02,  7.4018952e-01, -1.6599106e-02, -6.7106650e-01,
       9.4477394e-02, -1.1325310e+00, -1.2183007e+00, -1.1326694e+00,
      -7.0637699e-01, -1.1096511e+00,  1.4381563e-01,  4.6121573e-01,
       7.9281879e-01,  1.4577665e+00,  1.9515924e+00,  1.5493961e+00,
       7.6923395e-01,  1.0666962e-01, -5.8376568e-01, -1.7768814e-01,
      -7.8840041e-01, -8.4273142e-01, -1.1227955e-01,  1.7007988e-01,
      -1.2788265e+00, -8.2037634e-01,  2.9027089e-02,  1.1299878e+00,
       1.5466537e+00,  5.7106120e-01,  1.0054291e+00,  7.7078972e-01,
      -1.2498850e-01,  1.4337381e-01, -9.1649732e-03,  1.5189923e-01,
      -1.8656702e+00, -1.3065376e+00, -1.0474639e+00, -6.1982978e-01,
      -4.0826276e-01, -2.5749212e-01,  7.5196564e-01,  5.9204803e-01,
       3.8971675e-01,  4.7356386e-01,  9.6497659e-01,  1.0654369e+00,
       1.0179577e-01, -8.7312829e-01, -1.0535862e+00, -5.5302243e-01,
      -1.4681184e+00, -9.4220508e-01, -7.2255455e-01, -5.3132915e-01,
      -1.4926868e-01,  3.6245889e-01,  1.6544183e-01,  7.2654545e-01,
      -7.6464228e-02,  1.9158155e-02,  8.5964508e-01,  5.0113123e-02,
      -3.5523428e-02, -1.0719814e+00, -1.0268355e+00, -3.9656991e-01,
      -3.7302065e-01, -7.6860159e-01, -3.9431418e-01, -8.4906570e-02 ]

])

X_7M5 = np.array([

    [ -2.7808220e+04,  3.1171944e+04, -1.0484449e+05, -1.6109747e+05,
       7.1020534e+03,  6.1529776e+04, -4.3352596e+03,  1.6872730e+03,
       8.3021669e+03, -8.0904023e+03,  2.8986710e+03,  1.6300470e+03,
      -3.9442485e+03,  3.3369609e+03, -7.5314157e+02, -1.5665608e+03,
       2.3627282e+03, -1.5968140e+03, -2.9454372e+01,  1.3170795e+03,
      -1.5221502e+03,  7.5501127e+02,  3.4796318e+02, -1.0725998e+03,
       9.9368163e+02, -2.9346750e+02, -4.7457374e+02,  8.5309969e+02,
      -6.3326698e+02,  2.2874568e+01,  5.0703180e+02, -6.5956792e+02,
       3.7615149e+02,  1.3645752e+02, -4.8970087e+02,  4.9257049e+02,
      -1.8802536e+02, -2.2788339e+02,  4.4305145e+02, -3.4924777e+02,
       5.0697164e+01,  2.7264213e+02, -3.8309643e+02,  2.2685761e+02,
       4.8724694e+01, -2.8629820e+02,  3.1493179e+02, -1.2493286e+02,
      -1.1855917e+02,  2.7825778e+02, -2.4421869e+02,  3.9752437e+01,
       1.6382158e+02, -2.5263703e+02,  1.7444462e+02,  2.6833647e+01,
      -1.8770774e+02,  2.1765469e+02, -1.1027623e+02, -7.8965164e+01,
       1.9573957e+02, -1.7599945e+02,  4.9679803e+01,  1.1484006e+02,
      -1.8898112e+02,  1.3092016e+02,  7.2222470e-01, -1.3789235e+02,
       1.7321190e+02, -8.5147249e+01, -4.3896934e+01,  1.4839226e+02,
      -1.4915199e+02,  4.1079567e+01,  7.7806126e+01, -1.4826509e+02,
       1.1954125e+02, -9.5155767e-01, -1.0279192e+02,  1.3980505e+02,
      -8.5889140e+01, -3.4465766e+01,  1.1923625e+02, -1.2267683e+02,
       5.1023135e+01,  6.4082513e+01, -1.2613323e+02,  9.9909559e+01,
      -1.6509945e+01, -8.6980952e+01,  1.2384580e+02, -7.3451876e+01,
      -1.5354773e+01,  1.0245575e+02, -1.1445423e+02,  4.5501562e+01,
       4.3867941e+01, -1.1121715e+02,  1.0028146e+02, -1.5168901e+01,
      -6.9443433e+01,  1.1170827e+02, -7.9410276e+01, -1.4798645e+01,
       8.7563510e+01, -1.0510000e+02,  5.5384253e+01,  4.1531846e+01,
      -1.0069765e+02,  9.2428209e+01, -2.7848167e+01, -6.4462037e+01,
       1.0628096e+02, -7.4978815e+01,  1.6316106e-01,  8.3599643e+01,
      -1.0561445e+02,  5.2787841e+01,  2.7101412e+01, -9.7922081e+01 ],

     [ 3.8121925e+03, -2.2031854e+04,  6.2134301e+04, -2.5584480e+05,
      -4.0289726e+04, -1.1591648e+04,  9.9946415e+03, -5.6931570e+03,
      -1.4027155e+03, -8.4816729e+01, -1.6692931e+02,  3.7467791e+01,
       1.5876357e+02,  1.0909061e+02, -1.2446699e+01, -7.8016012e+01,
      -5.9665224e+01,  1.7150538e+01,  4.1435247e+01,  1.4446691e+01,
      -3.8364171e+00, -9.0141422e+00, -1.1905776e+01, -1.1758083e+01,
       2.4695821e+00,  1.6421675e+01,  1.7958775e+01, -1.0112627e+00,
      -1.7276148e+01, -1.4579197e+01, -3.0796621e-01,  3.4903800e+00,
       8.9247274e+00,  1.0542251e+01,  5.1085251e+00, -6.3561781e+00,
      -1.5196238e+01, -8.5814612e+00,  5.9801475e+00,  8.8788786e+00,
       6.6693886e+00,  4.0520010e+00, -2.5648927e+00, -7.3376816e+00,
      -9.4020988e+00, -3.7886737e+00,  9.8324354e+00,  1.0369412e+01,
       2.7236309e+00, -1.1806094e+00, -5.4027659e+00, -7.1662503e+00,
      -5.2723002e+00,  2.2261048e+00,  1.0161892e+01,  7.6651861e+00,
      -2.2518007e+00, -5.1245529e+00, -5.3667831e+00, -4.6532034e+00,
      -9.0748070e-01,  2.8558608e+00,  8.7396007e+00,  5.8069435e+00,
      -4.9824625e+00, -6.3131200e+00, -4.2366242e+00, -1.2878986e+00,
       1.9593460e+00,  4.4854081e+00,  6.1847341e+00,  1.9350643e+00,
      -6.7952522e+00, -7.6836416e+00, -1.1341833e+00,  8.9242434e-01,
       4.4208241e+00,  4.7103946e+00,  3.3250393e+00, -5.6553445e-01,
      -6.6710206e+00, -5.9125999e+00,  8.2386239e-01,  4.8587135e+00,
       4.1530321e+00,  3.6100791e+00,  3.4896727e-01, -3.5268730e+00,
      -6.6259985e+00, -4.6374540e+00,  4.0614812e+00,  4.8600515e+00,
       3.4816753e+00,  7.5221655e-01, -1.5176717e+00, -3.5752094e+00,
      -5.3492120e+00, -1.8813288e+00,  4.5633890e+00,  5.8660550e+00,
       2.6127035e+00, -1.4095531e+00, -3.2585158e+00, -4.0492745e+00,
      -3.0513819e+00,  1.0864826e+00,  6.1166169e+00,  5.3791204e+00,
      -2.6627677e-01, -3.3401114e+00, -4.3742918e+00, -3.7773803e+00,
      -3.2386710e-01,  2.3283535e+00,  6.5557289e+00,  4.4470718e+00,
      -2.5793855e+00, -4.6626375e+00, -3.9104033e+00, -1.1065239e+00 ]

])

X = [ X_7M5, X_10M ]


### C.3.1.3  12.8 kHz resampler

X_TILDE_12K8D_10M = np.array([

    [  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00, -2.7136560e+00, -2.6569981e+00, -8.2098309e+00,
       2.5435023e+00, -1.2692479e+01,  3.2383771e+00, -4.3038498e+01,
       1.6475668e+02,  3.9002834e+03,  7.6897871e+03,  1.1168030e+04,
       1.4388464e+04,  1.7228860e+04,  1.9693442e+04,  2.1734211e+04,
       2.3341435e+04,  2.4458687e+04,  2.5091505e+04,  2.5237788e+04,
       2.4898746e+04,  2.4056493e+04,  2.2750447e+04,  2.1002836e+04,
       1.8837538e+04,  1.6271528e+04,  1.3371876e+04,  1.0177959e+04,
       6.7345474e+03,  3.0859120e+03, -6.8926189e+02, -4.5387403e+03,
      -8.4127977e+03, -1.2248756e+04, -1.5972309e+04, -1.9535642e+04,
      -2.2893718e+04, -2.5977602e+04, -2.8736936e+04, -3.1140674e+04,
      -3.3157374e+04, -3.4729421e+04, -3.5844044e+04, -3.6489767e+04,
      -3.6657691e+04, -3.6317413e+04, -3.5493490e+04, -3.4198956e+04,
      -3.2451848e+04, -3.0250170e+04, -2.7655246e+04, -2.4700058e+04,
      -2.1422013e+04, -1.7854871e+04, -1.4073383e+04, -1.0130011e+04,
      -6.0696279e+03, -1.9523115e+03,  2.1432690e+03,  6.1682081e+03,
       1.0075311e+04,  1.3793150e+04,  1.7261182e+04,  2.0439154e+04,
       2.3290746e+04,  2.5752435e+04,  2.7792879e+04,  2.9395551e+04,
       3.0539419e+04,  3.1182642e+04,  3.1334658e+04,  3.1000121e+04,
       3.0185153e+04,  2.8876238e+04,  2.7121382e+04,  2.4947537e+04,
       2.2382237e+04,  1.9447889e+04,  1.6216333e+04,  1.2730773e+04,
       9.0366635e+03,  5.1844000e+03,  1.2542494e+03, -2.7040631e+03,
      -6.6402694e+03, -1.0487968e+04, -1.4176374e+04, -1.7659977e+04,
      -2.0896015e+04, -2.3817756e+04, -2.6379477e+04, -2.8555427e+04,
      -3.0316787e+04, -3.1610147e+04, -3.2431837e+04, -3.2774372e+04,
      -3.2634728e+04, -3.1986767e+04, -3.0864057e+04, -2.9283835e+04,
      -2.7267471e+04, -2.4822384e+04, -2.2011114e+04, -1.8874753e+04,
      -1.5449273e+04                                                 ],

    [ -1.5449273e+04, -1.1776342e+04, -7.9350741e+03, -3.9753420e+03,
       5.6212790e+01,  4.0948091e+03,  8.0660721e+03,  1.1920829e+04,
       1.5610547e+04,  1.9067628e+04,  2.2236206e+04,  2.5079838e+04,
       2.7562773e+04,  2.9625190e+04,  3.1247708e+04,  3.2411870e+04,
       3.3105731e+04,  3.3290869e+04,  3.2986189e+04,  3.2200486e+04,
       3.0944512e+04,  2.9210715e+04,  2.7055339e+04,  2.4507368e+04,
       2.1598895e+04,  1.8359789e+04,  1.4864140e+04,  1.1157303e+04,
       7.2876208e+03,  3.3081055e+03, -7.0156289e+02, -4.6891569e+03,
      -8.6058356e+03, -1.2384088e+04, -1.5958973e+04, -1.9284607e+04,
      -2.2323146e+04, -2.5008095e+04, -2.7303120e+04, -2.9181339e+04,
      -3.0623681e+04, -3.1579731e+04, -3.2052556e+04, -3.2039965e+04,
      -3.1544863e+04, -3.0546815e+04, -2.9086911e+04, -2.7186800e+04,
      -2.4871192e+04, -2.2155336e+04, -1.9106913e+04, -1.5768099e+04,
      -1.2180718e+04, -8.3886907e+03, -4.4727275e+03, -4.8397983e+02,
       3.5287168e+03,  7.5007937e+03,  1.1356312e+04,  1.5052293e+04,
       1.8540697e+04,  2.1752202e+04,  2.4638955e+04,  2.7166972e+04,
       2.9306556e+04,  3.0998943e+04,  3.2231603e+04,  3.2995051e+04,
       3.3278595e+04,  3.3050597e+04,  3.2338337e+04,  3.1153755e+04,
       2.9512841e+04,  2.7417674e+04,  2.4925451e+04,  2.2071406e+04,
       1.8894430e+04,  1.5423055e+04,  1.1739349e+04,  7.8892265e+03,
       3.9209702e+03, -1.0578292e+02, -4.1130091e+03, -8.0523247e+03,
      -1.1874762e+04, -1.5509335e+04, -1.8896353e+04, -2.1995901e+04,
      -2.4770068e+04, -2.7154380e+04, -2.9121926e+04, -3.0652080e+04,
      -3.1724909e+04, -3.2299883e+04, -3.2385523e+04, -3.1985436e+04,
      -3.1105919e+04, -2.9735628e+04, -2.7920239e+04, -2.5686848e+04,
      -2.3064137e+04, -2.0073280e+04, -1.6788431e+04, -1.3249779e+04,
      -9.5054623e+03, -5.6021829e+03, -1.6246392e+03,  2.3816986e+03,
       6.3633660e+03,  1.0254738e+04,  1.3986117e+04,  1.7511038e+04,
       2.0787917e+04,  2.3746769e+04,  2.6347708e+04,  2.8559387e+04,
       3.0354770e+04,  3.1683451e+04,  3.2536757e+04,  3.2910967e+04,
       3.2802210e+04,  3.2182725e+04,  3.1087832e+04,  2.9535420e+04,
       2.7544339e+04,  2.5123821e+04,  2.2336463e+04,  1.9222835e+04,
       1.5819986e+04                                                 ]

])

X_TILDE_12K8D_7M5 = np.array([

     [ 0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00, -2.7136560e+00, -2.6569981e+00, -8.2098309e+00,
       2.5435023e+00, -1.2692479e+01,  3.2383771e+00, -4.3038498e+01,
       1.6475668e+02,  3.9002834e+03,  7.6897871e+03,  1.1168030e+04,
       1.4388464e+04,  1.7228860e+04,  1.9693442e+04,  2.1734211e+04,
       2.3341435e+04,  2.4458687e+04,  2.5091505e+04,  2.5237788e+04,
       2.4898746e+04,  2.4056493e+04,  2.2750447e+04,  2.1002836e+04,
       1.8837538e+04,  1.6271528e+04,  1.3371876e+04,  1.0177959e+04,
       6.7345474e+03,  3.0859120e+03, -6.8926189e+02, -4.5387403e+03,
      -8.4127977e+03, -1.2248756e+04, -1.5972309e+04, -1.9535642e+04,
      -2.2893718e+04, -2.5977602e+04, -2.8736936e+04, -3.1140674e+04,
      -3.3157374e+04, -3.4729421e+04, -3.5844044e+04, -3.6489767e+04,
      -3.6657691e+04, -3.6317413e+04, -3.5493490e+04, -3.4198956e+04,
      -3.2451848e+04, -3.0250170e+04, -2.7655246e+04, -2.4700058e+04,
      -2.1422013e+04                                                 ],

    [ -2.1422013e+04, -1.7854871e+04, -1.4073383e+04, -1.0130011e+04,
      -6.0696279e+03, -1.9523115e+03,  2.1432690e+03,  6.1682081e+03,
       1.0075311e+04,  1.3793150e+04,  1.7261182e+04,  2.0439154e+04,
       2.3290746e+04,  2.5752435e+04,  2.7792879e+04,  2.9395551e+04,
       3.0539419e+04,  3.1182642e+04,  3.1334658e+04,  3.1000121e+04,
       3.0185153e+04,  2.8876238e+04,  2.7121382e+04,  2.4947537e+04,
       2.2382237e+04,  1.9447889e+04,  1.6216333e+04,  1.2730773e+04,
       9.0366635e+03,  5.1844000e+03,  1.2542494e+03, -2.7040631e+03,
      -6.6402694e+03, -1.0487968e+04, -1.4176374e+04, -1.7659977e+04,
      -2.0896015e+04, -2.3817756e+04, -2.6379477e+04, -2.8555427e+04,
      -3.0316787e+04, -3.1610147e+04, -3.2431837e+04, -3.2774372e+04,
      -3.2634728e+04, -3.1986767e+04, -3.0864057e+04, -2.9283835e+04,
      -2.7267471e+04, -2.4822384e+04, -2.2011114e+04, -1.8874753e+04,
      -1.5449273e+04, -1.1776342e+04, -7.9350741e+03, -3.9753420e+03,
       5.6212790e+01,  4.0948091e+03,  8.0660721e+03,  1.1920829e+04,
       1.5610547e+04,  1.9067628e+04,  2.2236206e+04,  2.5079838e+04,
       2.7562773e+04,  2.9625190e+04,  3.1247708e+04,  3.2411870e+04,
       3.3105731e+04,  3.3290869e+04,  3.2986189e+04,  3.2200486e+04,
       3.0944512e+04,  2.9210715e+04,  2.7055339e+04,  2.4507368e+04,
       2.1598895e+04,  1.8359789e+04,  1.4864140e+04,  1.1157303e+04,
       7.2876208e+03,  3.3081055e+03, -7.0156289e+02, -4.6891569e+03,
      -8.6058356e+03, -1.2384088e+04, -1.5958973e+04, -1.9284607e+04,
      -2.2323146e+04, -2.5008095e+04, -2.7303120e+04, -2.9181339e+04,
      -3.0623681e+04, -3.1579731e+04, -3.2052556e+04, -3.2039965e+04,
      -3.1544863e+04                                                 ]

])

X_TILDE_12K8D = [ X_TILDE_12K8D_7M5, X_TILDE_12K8D_10M ]


### C.3.1.4  Pitch analysis

T_CURR_10M = np.array([ 25, 26 ])
T_CURR_7M5 = np.array([ 22, 25 ])
T_CURR = [ T_CURR_7M5, T_CURR_10M ]

NORMCORR_10M = np.array([ 0.677220, 0.992748 ])
NORMCORR_7M5 = np.array([ 0.473429, 0.952099 ])
NORMCORR = [ NORMCORR_7M5, NORMCORR_10M ]

T1_10M = np.array([ 25, 26 ])
T1_7M5 = np.array([ 22, 25 ])
T1 = [ T1_7M5, T1_10M ]

T2_10M = np.array([ 21, 26 ])
T2_7M5 = np.array([ 21, 25 ])
T2 = [ T2_7M5, T2_10M ]

NORMCORR1_10M = np.array([ 0.677220, 0.992748 ])
NORMCORR1_7M5 = np.array([ 0.473429, 0.952099 ])
NORMCORR1 = [ NORMCORR1_7M5, NORMCORR1_10M ]

NORMCORR2_10M = np.array([ 0.276293, 0.992748 ])
NORMCORR2_7M5 = np.array([ 0.000000, 0.952099 ])
NORMCORR2 = [ NORMCORR2_7M5, NORMCORR2_10M ]


### C.3.1.5  LTPF encoder

PITCH_PRESENT_10M = np.array([ 1, 1 ])
PITCH_PRESENT_7M5 = np.array([ 0, 1 ])
PITCH_PRESENT = [ PITCH_PRESENT_7M5, PITCH_PRESENT_10M ]

PITCH_INDEX_10M = np.array([ 76, 76 ])
PITCH_INDEX_7M5 = np.array([  0, 72 ])
PITCH_INDEX = [ PITCH_INDEX_7M5, PITCH_INDEX_10M ]

LTPF_ACTIVE_10M = np.array([ 0, 0 ])
LTPF_ACTIVE_7M5 = np.array([ 0, 0 ])
LTPF_ACTIVE = [ LTPF_ACTIVE_7M5, LTPF_ACTIVE_10M ]

NC_LTPF_10M = np.array([ 0.690317, 0.998707 ])
NC_LTPF_7M5 = np.array([ 0.000000, 0.963121 ])
NC_LTPF = [ NC_LTPF_7M5, NC_LTPF_10M ]


### C.3.1.6  Per-band energy

E_B_10M = np.array([

    [  3.4798963e+05,  1.0408765e+09,  7.6280953e+08,  9.0588797e+09,
       1.3037195e+10,  5.0263652e+10,  4.9938874e+06,  3.9472260e+08,
       2.5815358e+08,  3.3819826e+05,  2.4720253e+08,  1.6127203e+07,
       1.0176981e+08,  3.6166214e+06,  1.7601776e+07,  9.5054796e+05,
       2.2312350e+05,  7.5623058e+06,  6.1036902e+06,  7.5785971e+06,
       4.9771870e+06,  2.1948272e+06,  6.5487867e+05,  1.0856397e+04,
       3.9881009e+05,  1.3451218e+06,  1.7430411e+06,  1.7275154e+06,
       8.7221944e+05,  5.9920352e+04,  3.4754106e+05,  6.3569381e+05,
       2.9322749e+05,  1.8264770e+04,  1.7680584e+05,  2.8795546e+05,
       1.1948250e+05,  8.9930924e+03,  1.0371110e+05,  1.5057936e+05,
       3.6457561e+04,  4.9126130e+04,  7.2240065e+04,  7.7210962e+03,
       5.1941280e+04,  2.4886024e+04,  1.3310972e+04,  3.7010599e+04,
       6.5396090e+03,  2.5307401e+04,  4.7465699e+03,  1.9017662e+04,
       3.8107631e+03,  1.2269623e+04,  7.9690160e+03,  4.5694858e+03,
       9.2694695e+03,  4.6138651e+03,  4.2316029e+03,  5.8000267e+03,
       5.7863671e+03,  4.4037279e+03,  3.9216878e+03,  3.7117332e+03 ],

    [  3.9493130e+07,  2.1396136e+08,  2.4801910e+07,  4.9484143e+09,
       7.3212541e+09,  6.6773691e+10,  1.2396327e+09,  2.9859813e+09,
       1.2571883e+08,  3.4907798e+08,  3.2579374e+06,  4.9800086e+07,
       1.6459643e+06,  1.6752894e+07,  1.4273206e+06,  2.3795106e+06,
       3.6391831e+05,  4.1865804e+05,  2.4696343e+04,  1.1331072e+04,
       5.7583781e+01,  2.7850268e+03,  3.7205810e+03,  6.9571975e+03,
       1.9357163e+03,  1.1399846e+03,  1.4582614e+02,  8.5810802e+00,
       1.1913932e+02,  1.0127132e+02,  5.9681539e+00,  1.5356348e+01,
       2.0311276e+01,  2.1055250e+00,  4.0428882e+00,  9.8908370e+00,
       1.8819250e+00,  6.2827408e-01,  4.5599643e+00,  1.5570596e+00,
       8.5772617e-01,  3.0652366e+00,  1.3324581e+00,  2.4566558e+00,
       4.1674750e+00,  4.6799657e-01,  3.2193746e+00,  1.0994905e+00,
       1.6164118e+00,  1.5691847e+00,  5.4527735e-01,  2.5211389e-01,
       1.0146872e+00,  5.1845169e-01,  1.7875047e+00,  3.4334672e-01,
       7.2301137e-01,  8.6777500e-01,  1.0547766e+00,  2.8083700e-01,
       6.9555959e-01,  6.1528702e-01,  1.8631657e-01,  4.6476980e-01 ]

])

E_B_7M5 = np.array([

    [  7.7329708e+08,  9.7169006e+08,  1.0992368e+10,  2.5952396e+10,
       5.0439162e+07,  3.7859133e+09,  1.8794475e+07,  2.8468901e+06,
       6.8925976e+07,  6.5454609e+07,  8.4022938e+06,  2.6570531e+06,
       1.5557097e+07,  1.1135308e+07,  5.6722222e+05,  2.4541127e+06,
       5.5824847e+06,  2.5498151e+06,  8.6756001e+02,  1.7346984e+06,
       2.3169412e+06,  5.7004201e+05,  1.2107837e+05,  1.1504704e+06,
       9.8740319e+05,  8.6123175e+04,  2.2522023e+05,  7.2777908e+05,
       4.0102706e+05,  5.2324588e+02,  2.5708125e+05,  4.3502984e+05,
       1.4148995e+05,  1.8620654e+04,  2.4121631e+05,  4.3642188e+04,
       1.5913430e+05,  3.8451966e+04,  9.9113627e+04,  4.2170377e+04,
       5.7395127e+04,  4.5741834e+04,  3.0611513e+04,  4.5331490e+04,
       1.5575486e+04,  4.1303881e+04,  9.1981721e+03,  3.4644893e+04,
       1.7123395e+04,  1.2051636e+04,  1.3059786e+04,  1.5318037e+04,
       1.4108814e+04,  1.0037512e+04,  7.5940392e+03,  7.2531779e+03,
       8.4324439e+03,  7.8664771e+03,  7.3659535e+03,  6.8968988e+03,
       6.3095928e+03,  5.8688253e+03,  4.3696732e+03,  6.2506180e+03 ],

    [  1.4532812e+07,  4.8540259e+08,  3.8606713e+09,  6.5456561e+10,
       1.6232620e+09,  1.3436631e+08,  9.9892859e+07,  3.2412037e+07,
       1.9676109e+06,  7.1938775e+03,  2.7865394e+04,  1.4038353e+03,
       2.5205872e+04,  1.1900762e+04,  1.5492031e+02,  6.0864981e+03,
       3.5599390e+03,  2.9414095e+02,  1.7168797e+03,  2.0870689e+02,
       1.4718096e+01,  8.1254760e+01,  1.4174751e+02,  1.3825251e+02,
       6.0988360e+00,  2.6967143e+02,  3.2251759e+02,  1.0226523e+00,
       2.9846530e+02,  2.1255299e+02,  9.4843183e-02,  1.2182752e+01,
       7.9650760e+01,  1.1113906e+02,  3.3249014e+01,  1.5228356e+02,
       5.7298325e+01,  3.0449728e+01,  3.0210123e+01,  5.1376755e+01,
       1.0210074e+02,  4.4060019e+00,  4.0272511e+01,  1.6376346e+01,
       8.1009561e+01,  1.5665824e+01,  2.5227331e+01,  4.4897310e+00,
       4.4975382e+01,  1.9821051e+01,  2.0736286e+01,  3.6319425e+01,
       7.2088263e+00,  1.1187844e+01,  2.6713368e+01,  1.7962481e+01,
       1.9492612e+01,  1.3200905e+01,  1.1809729e+01,  1.6012045e+01,
       9.3764812e+00,  1.9393796e+01,  1.6381346e+01,  1.2937103e+01 ]

])

E_B = [ E_B_7M5, E_B_10M ]


### C.3.1.7  Bandwidth detector

P_BW_10M = np.array([ 1, 1 ])
P_BW_7M5 = np.array([ 1, 1 ])
P_BW = [ P_BW_7M5, P_BW_10M ]


### C.3.1.8  SNS gains

SCF_10M = np.array([

    [  3.5009846e+00,  4.4942639e+00,  2.3071956e+00,  1.2022551e+00,
       6.2362294e-01,  9.4722039e-03, -3.2828840e-02, -3.3840570e-01,
      -7.2946152e-01, -1.0018093e+00, -1.2127892e+00, -1.4678244e+00,
      -1.6640459e+00, -1.8392946e+00, -1.9307510e+00, -1.9205837e+00 ],

    [  3.7432369e+00,  5.6927098e+00,  3.2662471e+00,  1.4474935e+00,
      -4.4505556e-01, -1.2458756e+00, -1.2458756e+00, -1.2458756e+00,
      -1.2458756e+00, -1.2458756e+00, -1.2458756e+00, -1.2458756e+00,
      -1.2458756e+00, -1.2458756e+00, -1.2458756e+00, -1.2458756e+00 ],

    [ -1.5649514e+00, -1.1656014e+00, -1.5624815e+00, -1.1411195e+00,
      -7.4976482e-01, -2.4654068e-01,  1.3534391e-01, -1.1293867e-01,
      -1.6932960e-01,  5.7623565e-02,  6.7159547e-01,  9.3859612e-01,
       1.0824257e+00,  1.2057632e+00,  1.4053510e+00,  1.2160286e+00 ]

])

SCF_7M5 = np.array([

    [  4.4048340e+00,  3.5389298e+00,  1.5267043e+00,  7.9358598e-01,
       3.0615231e-01, -3.2867352e-02, -2.7847443e-01, -4.6371063e-01,
      -6.8557046e-01, -8.7944953e-01, -1.0976367e+00, -1.2286102e+00,
      -1.3879732e+00, -1.5027094e+00, -1.5071962e+00, -1.5060084e+00 ],

    [  5.4182466e+00,  4.3372862e+00,  3.1487482e-01, -7.7464674e-01,
      -7.7464674e-01, -7.7464674e-01, -7.7464674e-01, -7.7464674e-01,
      -7.7464674e-01, -7.7464674e-01, -7.7464674e-01, -7.7464674e-01,
      -7.7464674e-01, -7.7464674e-01, -7.7464674e-01, -7.7464674e-01 ]

])

SCF = [ SCF_7M5, SCF_10M ]


### C.3.1.9  SNS quantization: stage 2

T2ROT_10M = np.array([

    [ -3.8060310e-01,  2.8077898e-01, -6.7415911e-01, -4.4050504e-01,
      -5.0680535e-02,  1.2946234e+00, -3.1905543e-02, -8.3983883e-01,
      -3.0573474e-02, -1.7180800e-01, -3.1795511e-01, -5.6801435e-01,
      -3.4459445e-01, -5.3266246e-02, -6.2922325e-02, -2.0856957e-01 ],

    [ -6.9862836e-01,  7.4316023e-01,  1.6797292e-01,  1.7318569e+00,
       2.0540381e-01, -3.3073095e-01, -5.2816094e-01, -1.0380535e+00,
      -6.4017558e-01, -3.8876809e-01, -3.3239735e-01, -4.8437565e-01,
      -4.2394514e-01, -3.6228481e-01, -2.3217161e-01, -2.2684893e-01 ],

    [ -1.0349648e+00, -8.7888573e-01,  1.9107834e-01, -9.0835649e-01,
       7.6273219e-01,  5.4107875e-01,  7.5161773e-01, -7.8846551e-02,
       1.4299991e-01, -1.5545871e-01, -6.6321266e-02, -7.2264622e-02,
      -1.3018946e-01, -2.4094909e-01, -1.6785267e-01,  7.1244633e-02 ]

])

T2ROT_7M5 = np.array([

    [  5.5508969e-01,  3.8471081e-01,  4.0948426e-02, -7.5029612e-01,
      -9.7881975e-01,  4.9702346e-01, -8.2168015e-02, -4.3546804e-01,
       1.8914981e-01, -1.9792621e-01, -2.4794744e-01, -2.9000112e-01,
      -2.2928306e-01, -8.1890752e-02, -1.0731157e-01, -2.2349961e-01 ],

    [ -1.9137509e-01, -1.3751444e-01,  4.4707625e-01,  4.0374158e-01,
       1.6628366e+00,  3.1663673e-01,  5.0786462e-01,  5.0214496e-01,
      -2.8554914e-01, -6.2625497e-01, -4.7743904e-01, -5.5803079e-01,
      -4.8903072e-01, -4.6108945e-01, -2.3248007e-01, -1.2652277e-01 ],

])

T2ROT = [ T2ROT_7M5, T2ROT_10M ]

SNS_Y0_10M = np.array([
    [ -1, 1,-2,-1, 0, 3, 0,-2, 0, 0, 0,-1, 0, 0, 0, 0 ],
    [ -1, 1, 0, 3, 0, 0,-1,-2,-1,-1, 0,-1, 0, 0, 0, 0 ],
    [ -2,-2, 0,-2, 2, 1, 1, 0, 0, 0, 0, 0, 0,-1, 0, 0 ]
])

SNS_Y1_10M = np.array([
    [ -1, 1,-2,-1, 0, 3, 0,-2, 0, 0 ],
    [ -1, 1, 0, 3, 0, 0,-1,-2,-1,-1 ],
    [ -2,-2, 0,-2, 2, 1, 1, 0, 0, 0 ]
])

SNS_Y2_10M = np.array([
    [ -1, 0,-1,-1, 0, 2, 0,-1, 0, 0, 0,-1,-1, 0, 0, 0 ],
    [ -1, 1, 0, 2, 0, 0,-1,-1,-1, 0, 0,-1, 0, 0, 0, 0 ],
    [ -2,-1, 0,-2, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0 ]
])

SNS_Y3_10M = np.array([
    [  0, 0,-1,-1, 0, 2, 0,-1, 0, 0, 0,-1, 0, 0, 0, 0 ],
    [ -1, 1, 0, 2, 0, 0, 0,-1,-1, 0, 0, 0, 0, 0, 0, 0 ],
    [ -1,-1, 0,-1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0 ]
])

SNS_Y0_7M5 = np.array([
    [  1, 1, 0,-2,-3, 1, 0,-1, 0,-1, 0,-1, 0, 0, 0, 0 ],
    [  0, 0, 1, 1, 3, 1, 1, 1,-1,-1, 0,-1, 0, 0, 0, 0 ]
])

SNS_Y1_7M5 = np.array([
    [  1, 1, 0,-2,-3, 1, 0,-1, 0,-1 ],
    [  0, 0, 1, 1, 3, 1, 1, 1,-1,-1 ]
])

SNS_Y2_7M5 = np.array([
    [  1, 1, 0,-1,-2, 1, 0,-1, 0, 0, 0,-1, 0, 0, 0, 0 ],
    [  0, 0, 0, 0, 2, 0, 1, 1, 0,-1,-1,-1,-1, 0, 0, 0 ]
])

SNS_Y3_7M5 = np.array([
    [  1, 0, 0,-1,-2, 1, 0,-1, 0, 0, 0, 0, 0, 0, 0, 0 ],
    [  0, 0, 0, 0, 2, 0, 1, 1, 0,-1, 0,-1, 0, 0, 0, 0 ]
])

SNS_Y0 = [ SNS_Y0_7M5, SNS_Y0_10M ]
SNS_Y1 = [ SNS_Y1_7M5, SNS_Y1_10M ]
SNS_Y2 = [ SNS_Y2_7M5, SNS_Y2_10M ]
SNS_Y3 = [ SNS_Y3_7M5, SNS_Y3_10M ]

SNS_XQ0_10M = np.array([

    [ -2.1821789e-01,  2.1821789e-01, -4.3643578e-01, -2.1821789e-01,
      -0.0000000e+00,  6.5465367e-01, -0.0000000e+00, -4.3643578e-01,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -2.1821789e-01,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00 ],

    [ -2.2941573e-01,  2.2941573e-01,  0.0000000e+00,  6.8824720e-01,
       0.0000000e+00, -0.0000000e+00, -2.2941573e-01, -4.5883147e-01,
      -2.2941573e-01, -2.2941573e-01, -0.0000000e+00, -2.2941573e-01,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00 ],

    [ -4.5883147e-01, -4.5883147e-01,  0.0000000e+00, -4.5883147e-01,
       4.5883147e-01,  2.2941573e-01,  2.2941573e-01, -0.0000000e+00,
       0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -2.2941573e-01, -0.0000000e+00,  0.0000000e+00 ]

])

SNS_XQ1_10M = np.array([

    [ -2.2360680e-01,  2.2360680e-01, -4.4721360e-01, -2.2360680e-01,
      -0.0000000e+00,  6.7082039e-01, -0.0000000e+00, -4.4721360e-01,
      -0.0000000e+00, -0.0000000e+00 ],

    [ -2.3570226e-01,  2.3570226e-01,  0.0000000e+00,  7.0710678e-01,
       0.0000000e+00, -0.0000000e+00, -2.3570226e-01, -4.7140452e-01,
      -2.3570226e-01, -2.3570226e-01 ],

    [ -4.7140452e-01, -4.7140452e-01, 0.0000000e+00, -4.7140452e-01,
       4.7140452e-01,  2.3570226e-01, 2.3570226e-01, -0.0000000e+00,
       0.0000000e+00, -0.0000000e+00 ],

])

SNS_XQ2_10M = np.array([

    [ -3.1622777e-01,  0.0000000e+00, -3.1622777e-01, -3.1622777e-01,
      -0.0000000e+00,  6.3245553e-01, -0.0000000e+00, -3.1622777e-01,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -3.1622777e-01,
      -3.1622777e-01, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00 ],

    [ -3.1622777e-01,  3.1622777e-01,  0.0000000e+00,  6.3245553e-01,
       0.0000000e+00, -0.0000000e+00, -3.1622777e-01, -3.1622777e-01,
      -3.1622777e-01, -0.0000000e+00, -0.0000000e+00, -3.1622777e-01,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00 ],

    [ -5.7735027e-01, -2.8867513e-01,  0.0000000e+00, -5.7735027e-01,
       2.8867513e-01,  2.8867513e-01,  2.8867513e-01, -0.0000000e+00,
       0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,  0.0000000e+00 ]

])

SNS_XQ3_10M = np.array([

    [ -0.0000000e+00,  0.0000000e+00, -3.5355339e-01, -3.5355339e-01,
      -0.0000000e+00,  7.0710678e-01, -0.0000000e+00, -3.5355339e-01,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -3.5355339e-01,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00 ],

    [ -3.5355339e-01,  3.5355339e-01,  0.0000000e+00,  7.0710678e-01,
       0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -3.5355339e-01,
      -3.5355339e-01, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00 ],

    [ -4.0824829e-01, -4.0824829e-01,  0.0000000e+00, -4.0824829e-01,
       4.0824829e-01,  4.0824829e-01,  4.0824829e-01, -0.0000000e+00,
       0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,  0.0000000e+00 ]

])

SNS_XQ0_7M5 = np.array([

    [  2.2941573e-01,  2.2941573e-01,  0.0000000e+00, -4.5883147e-01,
      -6.8824720e-01,  2.2941573e-01, -0.0000000e+00, -2.2941573e-01,
       0.0000000e+00, -2.2941573e-01, -0.0000000e+00, -2.2941573e-01,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00 ],

    [ -0.0000000e+00, -0.0000000e+00,  2.4253563e-01,  2.4253563e-01,
       7.2760688e-01,  2.4253563e-01,  2.4253563e-01,  2.4253563e-01,
      -2.4253563e-01, -2.4253563e-01, -0.0000000e+00, -2.4253563e-01,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00 ]

])

SNS_XQ1_7M5 = np.array([

    [  2.3570226e-01,  2.3570226e-01,  0.0000000e+00, -4.7140452e-01,
      -7.0710678e-01,  2.3570226e-01, -0.0000000e+00, -2.3570226e-01,
       0.0000000e+00, -2.3570226e-01 ],

    [ -0.0000000e+00, -0.0000000e+00,  2.5000000e-01,  2.5000000e-01,
       7.5000000e-01,  2.5000000e-01,  2.5000000e-01,  2.5000000e-01,
      -2.5000000e-01, -2.5000000e-01 ]

])

SNS_XQ2_7M5 = np.array([

    [  3.1622777e-01,  3.1622777e-01,  0.0000000e+00, -3.1622777e-01,
      -6.3245553e-01,  3.1622777e-01, -0.0000000e+00, -3.1622777e-01,
       0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -3.1622777e-01,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00 ],

    [ -0.0000000e+00, -0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       6.3245553e-01,  0.0000000e+00,  3.1622777e-01,  3.1622777e-01,
      -0.0000000e+00, -3.1622777e-01, -3.1622777e-01, -3.1622777e-01,
      -3.1622777e-01, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00 ]

])

SNS_XQ3_7M5 = np.array([

    [  3.5355339e-01,  0.0000000e+00,  0.0000000e+00, -3.5355339e-01,
      -7.0710678e-01,  3.5355339e-01, -0.0000000e+00, -3.5355339e-01,
       0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00 ],

    [ -0.0000000e+00, -0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       7.0710678e-01,  0.0000000e+00,  3.5355339e-01,  3.5355339e-01,
      -0.0000000e+00, -3.5355339e-01, -0.0000000e+00, -3.5355339e-01,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00 ]

])

SNS_XQ0 = [ SNS_XQ0_7M5, SNS_XQ0_10M ]
SNS_XQ1 = [ SNS_XQ1_7M5, SNS_XQ1_10M ]
SNS_XQ2 = [ SNS_XQ2_7M5, SNS_XQ2_10M ]
SNS_XQ3 = [ SNS_XQ3_7M5, SNS_XQ3_10M ]


### C.3.1.10/12  SNS quantization gains

IND_LF_10M = np.array([ 25, 25,  4 ])
IND_LF_7M5 = np.array([ 17, 17 ])
IND_LF = [ IND_LF_7M5, IND_LF_10M ]

IND_HF_10M = np.array([  8,  9, 27 ])
IND_HF_7M5 = np.array([  8, 25 ])
IND_HF = [ IND_HF_7M5, IND_HF_10M ]

SUBMODE_MSB_10M = np.array([ 0, 0, 1 ])
SUBMODE_MSB_7M5 = np.array([ 0, 0 ])
SUBMODE_MSB = [ SUBMODE_MSB_7M5, SUBMODE_MSB_10M ]

SUBMODE_LSB_10M = np.array([ 0, 0, 1 ])
SUBMODE_LSB_7M5 = np.array([ 1, 0 ])
SUBMODE_LSB = [ SUBMODE_LSB_7M5, SUBMODE_LSB_10M ]

G_IND_10M = np.array([ 0, 0, 3 ])
G_IND_7M5 = np.array([ 0, 0 ])
G_IND = [ G_IND_7M5, G_IND_10M ]

LS_IND_A_10M = np.array([ 1, 1, 1 ])
LS_IND_A_7M5 = np.array([ 0, 0 ])
LS_IND_A = [ LS_IND_A_7M5, LS_IND_A_10M ]

LS_IND_B_10M = np.array([ True, True, None ])
LS_IND_B_7M5 = np.array([ None, 1 ])
LS_IND_B = [ LS_IND_B_7M5, LS_IND_B_10M ]

IDX_A_10M = np.array([  865837, 1023911, 61886 ])
IDX_A_7M5 = np.array([ 1025681, 2213651 ])
IDX_A = [ IDX_A_7M5, IDX_A_10M ]

IDX_B_10M = np.array([ 1, 1, None ])
IDX_B_7M5 = np.array([ None, 1 ])
IDX_B = [ IDX_B_7M5, IDX_B_10M ]

SCF_Q_10M = np.array([

    [  3.6627046e+00,  4.1100042e+00,  2.4746060e+00,  1.1870402e+00,
       5.2877727e-01,  7.5258069e-02,  2.3608016e-01, -2.5249380e-01,
      -6.8201580e-01, -1.1328318e+00, -1.0800359e+00, -1.6239492e+00,
      -1.7143487e+00, -1.9073626e+00, -2.2321087e+00, -2.0267285e+00 ],

    [  3.8168009e+00,  5.1162010e+00,  3.4807329e+00,  1.1997530e+00,
      -2.2081921e-01, -1.0181629e+00, -1.0657728e+00, -1.1230690e+00,
      -1.2811443e+00, -1.3048218e+00, -1.2081961e+00, -1.2804827e+00,
      -8.0867787e-01, -1.0022261e+00, -1.3971454e+00, -1.1057621e+00 ],

    [ -1.3038278e+00, -1.3183731e+00, -1.4433352e+00, -1.3025832e+00,
      -7.2407159e-01, -7.0739510e-02,  3.0891592e-01,  1.1110561e-01,
      -2.3826205e-01,  2.2790355e-01,  7.5958402e-01,  1.0128069e+00,
       1.0886346e+00,  1.3319042e+00,  1.4367016e+00,  1.0146522e+00 ]

])


SCF_Q_7M5 = np.array([

    [  4.3811806e+00,  3.2082316e+00,  1.7782399e+00,  6.4980749e-01,
       2.0221356e-01,  7.0332445e-02, -2.8668104e-01, -8.5700304e-01,
      -6.1964726e-01, -8.0165679e-01, -1.0516986e+00, -1.3477232e+00,
      -1.4003932e+00, -1.5566567e+00, -1.6059594e+00, -1.5454836e+00 ],

    [  5.8316083e+00,  3.9835316e+00,  8.6468396e-01, -9.5443316e-01,
      -8.1173658e-01, -6.2522498e-01, -6.4450858e-01, -7.4038105e-01,
      -1.0412349e+00, -6.5113903e-01, -5.8671926e-01, -9.6639127e-01,
      -5.7817825e-01, -5.4629017e-01, -9.2263684e-01, -8.4544942e-01 ],

])

SCF_Q = [ SCF_Q_7M5, SCF_Q_10M ]


### C.3.1.11  SNS interpolation

G_SNS_10M = np.array([

    [  7.8961620e-02,  7.8961620e-02,  7.5959959e-02,  7.0294618e-02,
       6.5051816e-02,  6.0200039e-02,  6.6727283e-02,  8.8588922e-02,
       1.1761302e-01,  1.5614618e-01,  2.0114945e-01,  2.5143043e-01,
       3.1428006e-01,  3.9284010e-01,  4.6498069e-01,  5.2116367e-01,
       5.8413517e-01,  6.5471542e-01,  7.2092077e-01,  7.7986290e-01,
       8.4362411e-01,  9.1259840e-01,  9.3603811e-01,  9.1031240e-01,
       8.8529374e-01,  8.6096267e-01,  8.8576235e-01,  9.6402006e-01,
       1.0491919e+00,  1.1418887e+00,  1.2364330e+00,  1.3319727e+00,
       1.4348949e+00,  1.5457699e+00,  1.6682873e+00,  1.8038405e+00,
       1.9504078e+00,  2.1088841e+00,  2.1828793e+00,  2.1629996e+00,
       2.1433009e+00,  2.1237816e+00,  2.2161032e+00,  2.4351378e+00,
       2.6758212e+00,  2.9402932e+00,  3.1064119e+00,  3.1554571e+00,
       3.2052766e+00,  3.2558828e+00,  3.3368236e+00,  3.4503169e+00,
       3.5676705e+00,  3.6890154e+00,  3.8582746e+00,  4.0816214e+00,
       4.3178972e+00,  4.5678504e+00,  4.6153375e+00,  4.4539677e+00,
       4.2982400e+00,  4.1479572e+00,  4.0029288e+00,  3.8629712e+00 ],

    [  7.0962424e-02,  7.0962424e-02,  6.3406480e-02,  5.0622552e-02,
       4.0416103e-02,  3.2267464e-02,  3.3220843e-02,  4.4105412e-02,
       5.8556233e-02,  7.7741760e-02,  1.0915042e-01,  1.6206389e-01,
       2.4062852e-01,  3.5727938e-01,  4.9237129e-01,  6.2979823e-01,
       8.0558272e-01,  1.0304308e+00,  1.2487522e+00,  1.4337794e+00,
       1.6462221e+00,  1.8901424e+00,  2.0337102e+00,  2.0505581e+00,
       2.0675455e+00,  2.0846737e+00,  2.1037086e+00,  2.1246996e+00,
       2.1459001e+00,  2.1673122e+00,  2.2081350e+00,  2.2694571e+00,
       2.3324822e+00,  2.3972575e+00,  2.4353075e+00,  2.4453201e+00,
       2.4553739e+00,  2.4654690e+00,  2.4499352e+00,  2.4092551e+00,
       2.3692505e+00,  2.3299102e+00,  2.3250020e+00,  2.3543088e+00,
       2.3839851e+00,  2.4140354e+00,  2.3319019e+00,  2.1488367e+00,
       1.9801431e+00,  1.8246926e+00,  1.7812270e+00,  1.8419813e+00,
       1.9048079e+00,  1.9697773e+00,  2.0728144e+00,  2.2196325e+00,
       2.3768496e+00,  2.5452025e+00,  2.5681376e+00,  2.4416843e+00,
       2.3214575e+00,  2.2071505e+00,  2.0984720e+00,  1.9951447e+00 ]

])

G_SNS_7M5 = np.array([

    [  4.7988064e-02,  4.7988064e-02,  5.3121439e-02,  6.5094311e-02,
       7.9765711e-02,  9.7743851e-02,  1.2247145e-01,  1.5691066e-01,
       2.0103424e-01,  2.5756547e-01,  3.2148295e-01,  3.9091375e-01,
       4.7533955e-01,  5.7799882e-01,  6.6256861e-01,  7.1600434e-01,
       7.7374963e-01,  8.3615204e-01,  8.7920505e-01,  8.9952910e-01,
       9.2032297e-01,  9.4159751e-01,  9.8233993e-01,  1.0450324e+00,
       1.1117258e+00,  1.1826756e+00,  1.2816224e+00,  1.4147546e+00,
       1.5617162e+00,  1.7239439e+00,  1.7744028e+00,  1.7029011e+00,
       1.6342806e+00,  1.5684252e+00,  1.5609220e+00,  1.6109378e+00,
       1.6625563e+00,  1.7158288e+00,  1.7812772e+00,  1.8601545e+00,
       1.9425246e+00,  2.0285422e+00,  2.1268255e+00,  2.2387723e+00,
       2.3566116e+00,  2.4806533e+00,  2.5567427e+00,  2.5801849e+00,
       2.6038421e+00,  2.6277162e+00,  2.6757181e+00,  2.7491622e+00,
       2.8246222e+00,  2.9021534e+00,  2.9543066e+00,  2.9796549e+00,
       3.0052206e+00,  3.0310057e+00,  3.0280729e+00,  2.9965054e+00,
       2.9652670e+00,  2.9343543e+00,  2.9037639e+00,  2.8734924e+00 ],

    [  1.7559453e-02,  1.7559453e-02,  2.0608757e-02,  2.8387910e-02,
       3.9103448e-02,  5.3863762e-02,  8.2831558e-02,  1.4220423e-01,
       2.4413451e-01,  4.1912721e-01,  6.4291757e-01,  8.8116553e-01,
       1.2077018e+00,  1.6552436e+00,  1.9140070e+00,  1.8672589e+00,
       1.8216525e+00,  1.7771601e+00,  1.7271850e+00,  1.6722548e+00,
       1.6190716e+00,  1.5675798e+00,  1.5450306e+00,  1.5502021e+00,
       1.5553909e+00,  1.5605971e+00,  1.5762459e+00,  1.6026515e+00,
       1.6294995e+00,  1.6567972e+00,  1.7147376e+00,  1.8065050e+00,
       1.9031836e+00,  2.0050361e+00,  1.9895925e+00,  1.8595441e+00,
       1.7379963e+00,  1.6243933e+00,  1.5616667e+00,  1.5443306e+00,
       1.5271869e+00,  1.5102336e+00,  1.5520535e+00,  1.6576004e+00,
       1.7703250e+00,  1.8907155e+00,  1.8893168e+00,  1.7663992e+00,
       1.6514784e+00,  1.5440344e+00,  1.4888436e+00,  1.4806393e+00,
       1.4724802e+00,  1.4643660e+00,  1.5087288e+00,  1.6104012e+00,
       1.7189253e+00,  1.8347627e+00,  1.8829418e+00,  1.8579241e+00,
       1.8332388e+00,  1.8088814e+00,  1.7848477e+00,  1.7611333e+00 ],

])

G_SNS = [ G_SNS_7M5, G_SNS_10M ]


### C.3.1.13  Spectral shaping

X_S_10M = np.array([

    [ -4.6579970e+01,  2.5475086e+03, -2.0979386e+03,  6.6905115e+03,
      -7.4276511e+03, -1.3496582e+04, -1.4911551e+02, -1.7600517e+03,
      -1.8897070e+03,  9.0806525e+01, -3.1626076e+03,  1.0097116e+03,
      -3.1704894e+03,  7.4708040e+02, -1.9508019e+03, -5.0811399e+02,
       2.7592179e+02, -1.8004443e+03,  1.7810815e+03, -2.1469042e+03,
       1.8820925e+03, -1.3520099e+03,  7.5748485e+02,  9.4849121e+01,
      -5.5907550e+02,  9.9853924e+02, -1.1694214e+03,  1.2670596e+03,
      -1.1564965e+03,  7.6341371e+02, -3.7429412e+02, -1.2714316e+02,
       5.5868459e+02, -8.6630814e+02,  1.0827149e+03, -1.0408470e+03,
       9.2759078e+02, -5.8910202e+02,  2.3201842e+02,  1.8289697e+02,
      -5.6951623e+02,  8.1229132e+02, -9.9835988e+02,  9.3658953e+02,
      -8.1543182e+02,  4.9408042e+02, -1.3648595e+02, -2.4771597e+02,
       5.9096160e+02, -7.9945187e+02,  8.7469937e+02, -8.0242901e+02,
       6.1523337e+02, -3.4929423e+02,  4.3714524e+01,  2.3825249e+02,
      -4.7949197e+02,  6.1487056e+02, -6.8885288e+02,  6.1543922e+02,
      -4.5940587e+02,  2.6027447e+02,  1.4932451e+01, -2.6342003e+02,
       5.1698905e+02, -6.3805644e+02,  6.6431016e+02, -6.3536804e+02,
       4.4714179e+02, -2.0448694e+02, -7.5210020e+01,  3.2318745e+02,
      -5.2463109e+02,  6.3268159e+02, -6.4137909e+02,  5.4210738e+02,
      -3.6539049e+02,  1.4180793e+02,  1.1043715e+02, -3.2082774e+02,
       4.9551143e+02, -5.6464214e+02,  5.5411819e+02, -4.4912438e+02,
       2.8748742e+02, -8.3157725e+01, -1.4199053e+02,  3.1886540e+02,
      -4.7410616e+02,  5.2507181e+02, -4.9901112e+02,  3.9510519e+02,
      -2.3175576e+02,  3.7914915e+01,  1.7066614e+02, -3.3127409e+02,
       4.6682698e+02, -5.0146152e+02,  4.5964223e+02, -3.4704713e+02,
       1.8377315e+02,  8.3068359e-01, -2.0227229e+02,  3.5108641e+02,
      -4.5284836e+02,  4.7317762e+02, -4.4561519e+02,  3.2535592e+02,
      -1.4580770e+02, -3.9573578e+01,  2.3102912e+02, -3.8615649e+02,
       4.7298066e+02, -4.8574358e+02,  4.1595651e+02, -2.8690887e+02,
       1.0822973e+02,  8.0889458e+01, -2.6856378e+02,  4.0232748e+02,
      -4.7864051e+02,  4.7205528e+02, -3.9210707e+02,  2.5719880e+02,
      -6.8160760e+01, -1.1068097e+02,  2.8495852e+02, -3.8530783e+02,
       4.4481193e+02, -4.2962793e+02,  3.3997586e+02, -2.0750452e+02,
       2.8600150e+01,  1.3080486e+02, -2.8418457e+02,  3.7335187e+02,
      -4.1400983e+02,  3.8613469e+02, -2.8921905e+02,  1.5545710e+02,
       5.7416806e+00, -1.4821308e+02,  2.8457226e+02, -3.6307452e+02,
       3.8831635e+02, -3.4730222e+02,  2.4109847e+02, -1.1774263e+02,
      -3.0666081e+01,  1.6872235e+02, -2.8808479e+02,  3.5123254e+02,
      -3.6347751e+02,  3.0487804e+02, -2.1236231e+02,  8.7865496e+01,
       6.0906484e+01, -1.8416198e+02,  2.9273130e+02, -3.4443289e+02 ],

    [ -4.4595314e+02, -1.0379970e+03,  3.1577388e+02, -3.5610417e+03,
       3.4581766e+03,  8.3381081e+03, -1.1696533e+03,  2.4101021e+03,
      -6.5655829e+02,  1.4524982e+03,  1.9701386e+02,  1.1436715e+03,
       3.0871470e+02,  1.4623552e+03,  5.8823857e+02,  9.7150548e+02,
       4.8597295e+02,  6.6672780e+02,  1.9624228e+02,  1.5262224e+02,
      -1.2492188e+01, -9.9749152e+01, -1.2404934e+02, -1.7103668e+02,
      -9.0965339e+01, -7.0386211e+01, -2.5404068e+01, -6.2239848e+00,
       1.1353459e+01,  3.1118246e+01,  2.3550684e+01,  1.9918764e+01,
       6.7582893e+00,  3.5391093e+00, -2.5360957e+00, -1.2318765e+01,
      -1.0710738e+01, -1.0309510e+01, -4.8149059e+00, -1.0084275e+00,
       1.5451084e+00,  6.7503435e+00,  8.7185892e+00,  6.5017316e+00,
       3.3714192e+00,  3.3653026e+00,  1.5730174e-01, -2.7592084e+00,
      -5.2662479e+00, -5.1967404e+00, -2.6956428e+00, -3.2877724e+00,
      -2.9012824e+00,  3.7032988e-01,  2.4268422e+00,  4.6841491e+00,
       4.4233993e+00,  2.9001769e+00,  4.0245597e+00,  2.1909513e+00,
      -7.8166125e-01, -3.7740437e+00, -1.6826612e+00, -4.8759902e+00,
      -7.0146012e+00, -4.2610160e+00, -1.9222920e+00, -3.8849374e-03,
       1.9171256e+00,  2.1228421e+00,  4.0002861e+00,  2.4213202e+00,
       5.5365640e+00,  2.0924576e+00,  3.2856722e+00,  2.3805212e-01,
      -2.6499944e+00, -1.5877393e+00, -2.3638844e+00, -3.1969127e+00,
      -4.5049447e+00,  4.9555393e-01, -5.2555443e-01,  2.8655462e-01,
      -4.9844313e-01,  1.3144496e+00,  1.8462413e+00,  1.2390833e+00,
       1.8391487e-01,  1.3634153e+00, -3.0575242e-02, -1.2360920e+00,
       1.7996128e-01, -2.1572539e+00, -2.3206289e+00, -2.1575175e+00,
      -1.3914054e+00, -2.1857655e+00,  2.8328476e-01,  9.0849227e-01,
       1.5616765e+00,  3.0216795e+00,  4.0452889e+00,  3.2116107e+00,
       1.5944792e+00,  2.2110634e-01, -1.2957452e+00, -3.9440236e-01,
      -1.7499591e+00, -1.8705540e+00, -2.4921933e-01,  4.0425431e-01,
      -3.0395784e+00, -1.9499112e+00,  6.8993026e-02,  2.6858112e+00,
       3.9365468e+00,  1.4534664e+00,  2.5590207e+00,  1.9618160e+00,
      -3.1812104e-01,  3.6820368e-01, -2.3536912e-02,  3.9009812e-01,
      -4.7912977e+00, -3.3553682e+00, -2.6900314e+00, -1.5134286e+00,
      -9.9684877e-01, -6.2871446e-01,  1.8360627e+00,  1.4455944e+00,
       9.5156525e-01,  1.0993584e+00,  2.2401521e+00,  2.4733664e+00,
       2.3631456e-01, -2.0269302e+00, -2.4458555e+00, -1.2206038e+00,
      -3.2403583e+00, -2.0795885e+00, -1.5947867e+00, -1.1727234e+00,
      -3.2945845e-01,  8.0000134e-01,  3.4717506e-01,  1.5246353e+00,
      -1.6045804e-01,  4.0202852e-02,  1.8039411e+00,  1.0516099e-01,
      -7.4544919e-02, -2.1387580e+00, -2.0486855e+00, -7.9121437e-01,
      -7.4423018e-01, -1.5334714e+00, -7.8671386e-01, -1.6940090e-01 ]

])

X_S_7M5 = np.array([

    [ -1.3344626e+03,  1.4958812e+03, -5.5694903e+03, -1.0486529e+04,
       5.6650034e+02,  6.0141573e+03, -5.3094551e+02,  2.6475111e+02,
       1.6690199e+03, -2.0838083e+03,  9.3187330e+02,  6.3720777e+02,
      -1.8748573e+03,  1.9287595e+03, -4.9900796e+02, -1.1216643e+03,
       1.8281601e+03, -1.3351793e+03, -2.5896432e+01,  1.1847513e+03,
      -1.4008698e+03,  7.1091673e+02,  3.4181812e+02, -1.1209015e+03,
       1.1047015e+03, -3.4707684e+02, -6.0822434e+02,  1.2069267e+03,
      -9.8898328e+02,  3.9434472e+01,  8.9967867e+02, -1.1231789e+03,
       6.1473708e+02,  2.1402341e+02, -7.6438486e+02,  7.6886411e+02,
      -3.0289717e+02, -3.6710597e+02,  7.3659799e+02, -5.8064409e+02,
       8.6987653e+01,  4.6780721e+02, -6.8240093e+02,  4.0409628e+02,
       9.0635459e+01, -5.3255888e+02,  6.1176277e+02, -2.4268516e+02,
      -2.4050228e+02,  5.6445766e+02, -5.1941054e+02,  8.4546495e+01,
       3.6675922e+02, -5.6559679e+02,  4.1109821e+02,  6.3236483e+01,
      -4.6563784e+02,  5.3992584e+02, -2.8194795e+02, -2.0189361e+02,
       5.0504429e+02, -4.5411112e+02,  1.2935836e+02,  2.9902539e+02,
      -4.9207698e+02,  3.4402101e+02,  1.8978015e+00, -3.6234196e+02,
       4.6346622e+02, -2.2783004e+02, -1.1745582e+02,  4.0795439e+02,
      -4.1004303e+02,  1.1293439e+02,  2.1977291e+02, -4.1879288e+02,
       3.3765885e+02, -2.7615664e+00, -2.9831792e+02,  4.0573572e+02,
      -2.5374285e+02, -1.0182244e+02,  3.5226045e+02, -3.6553462e+02,
       1.5203133e+02,  1.9094377e+02, -3.7905818e+02,  3.0025026e+02,
      -4.9616026e+01, -2.6139695e+02,  3.7537731e+02, -2.2263305e+02,
      -4.6540403e+01,  3.1054395e+02, -3.4657574e+02,  1.3778205e+02,
       1.3283532e+02, -3.3677365e+02,  3.0049394e+02, -4.5453695e+01,
      -2.0808762e+02,  3.3473443e+02, -2.3547267e+02, -4.3881934e+01,
       2.5964919e+02, -3.1164958e+02,  1.6251702e+02,  1.2186915e+02,
      -2.9548260e+02,  2.7121711e+02, -8.0864501e+01, -1.8718254e+02,
       3.0861481e+02, -2.1772078e+02,  4.7378118e-01,  2.4022293e+02,
      -3.0348233e+02,  1.5168546e+02,  7.7875700e+01, -2.8137835e+02 ],

    [  6.6940015e+01, -3.8686730e+02,  1.2805107e+03, -7.2628992e+03,
      -1.5754672e+03, -6.2436979e+02,  8.2787173e+02, -8.0959099e+02,
      -3.4245127e+02, -3.5548999e+01, -1.0732178e+02,  3.3015325e+01,
       1.9173905e+02,  1.8057154e+02, -2.3823068e+01, -1.4567609e+02,
      -1.0868931e+02,  3.0479251e+01,  7.1566338e+01,  2.4158550e+01,
      -6.2114342e+00, -1.4130388e+01, -1.8394789e+01, -1.8227404e+01,
       3.8411657e+00,  2.5627619e+01,  2.8307446e+01, -1.6207017e+00,
      -2.8151474e+01, -2.4154773e+01, -5.2808122e-01,  6.3053889e+00,
       1.6985395e+01,  2.1137594e+01,  1.0163883e+01, -1.2646204e+01,
      -2.8258074e+01, -1.5957606e+01,  1.0393474e+01,  1.5431458e+01,
       1.0833710e+01,  6.5820435e+00, -4.0055074e+00, -1.1459013e+01,
      -1.4519949e+01, -5.8509646e+00,  1.5015967e+01,  1.5836030e+01,
       4.1133189e+00, -1.7829961e+00, -8.3853815e+00, -1.1122404e+01,
      -8.7393669e+00,  3.6899921e+00,  1.7989851e+01,  1.3569871e+01,
      -4.2575144e+00, -9.6890713e+00, -1.0139553e+01, -8.7913755e+00,
      -1.6029731e+00,  5.0445901e+00,  1.4433262e+01,  9.5900419e+00,
      -8.2284294e+00, -9.7476743e+00, -6.5414935e+00, -1.9885597e+00,
       2.9171599e+00,  6.6780713e+00,  9.2081019e+00,  2.8651322e+00,
      -1.0061317e+01, -1.1376702e+01, -1.6700624e+00,  1.3140771e+00,
       6.5095758e+00,  6.8977417e+00,  4.8690746e+00, -8.2814942e-01,
      -1.0064761e+01, -8.9205098e+00,  1.2429849e+00,  7.8244782e+00,
       6.6880480e+00,  5.8136759e+00,  5.9984867e-01, -6.0624312e+00,
      -1.1389596e+01, -7.9714369e+00,  7.4518543e+00,  8.9170413e+00,
       6.3880481e+00,  1.3801389e+00, -2.8576875e+00, -6.7319113e+00,
      -1.0072255e+01, -3.5424326e+00,  8.4784304e+00,  1.0898685e+01,
       4.8542048e+00, -2.6188427e+00, -5.9736376e+00, -7.4232870e+00,
      -5.5939115e+00,  1.9917820e+00,  1.1064235e+01,  9.7301910e+00,
      -4.8166311e-01, -6.0418655e+00, -7.8074447e+00, -6.7420485e+00,
      -5.7805346e-01,  4.1557564e+00,  1.1700978e+01,  7.8318863e+00,
      -4.5426418e+00, -8.2115262e+00, -6.8867416e+00, -1.9487360e+00 ]

])

X_S = [ X_S_7M5, X_S_10M ]


### C.3.1.14  TNS coder

X_F_10M = np.array([

    [ -4.6579970e+01,  2.5475086e+03, -2.0979386e+03,  6.6905115e+03,
      -7.4276511e+03, -1.3496582e+04, -1.4911551e+02, -1.7600517e+03,
      -1.8897070e+03,  9.0806525e+01, -3.1626076e+03,  1.0097116e+03,
      -3.1704894e+03, -1.4629727e+03,  1.6177531e+02, -2.8070849e+02,
       1.6836280e+02, -4.8794939e+02,  4.8735463e+01, -3.2812648e+02,
       2.0567762e+02, -2.7612607e+02,  1.7089464e+02, -1.4903447e+02,
       1.7634897e+02, -1.7636863e+02, -1.4114282e+01,  1.5326904e+02,
      -2.0051680e+02,  7.5885344e+01, -1.2030117e+02,  8.6222257e+01,
       3.4247381e+01, -3.1564007e+01,  1.3038568e+02, -1.1547149e+02,
       1.5722828e+02, -1.1575179e+02,  8.3918034e+01, -4.2997589e+01,
      -4.9151429e+01,  5.7178185e+01, -1.4462317e+02,  1.2240116e+02,
      -1.4658879e+02,  1.0329710e+02, -4.8119513e+01,  1.5046477e+01,
       4.7974194e+01, -6.8870459e+01,  7.5850716e+01, -1.0491255e+02,
       1.0512982e+02, -9.6951573e+01,  6.8788954e+01, -3.9997908e+01,
      -8.7506165e+00,  3.5520535e+01, -9.4932383e+01,  8.5065288e+01,
      -6.7462450e+01,  9.2204890e+01, -2.2005176e+01, -2.2102766e+01,
       7.7845918e+01, -7.8223738e+01,  7.3211489e+01, -1.4466156e+02,
       7.8730279e+01, -2.1569349e+01,  2.0934594e+01,  1.0330210e+01,
      -5.6199662e+01,  7.4347857e+01, -8.7431560e+01,  8.2441742e+01,
      -7.4246878e+01,  5.8052214e+01, -1.2858317e+01, -8.5491490e+00,
       4.9073587e+01, -6.1606297e+01,  7.0756502e+01, -7.3226896e+01,
       6.7149964e+01, -4.3873945e+01, -3.2798184e-01,  1.2618393e+01,
      -6.0263717e+01,  6.9581326e+01, -6.2258677e+01,  7.0807773e+01,
      -5.6473108e+01,  2.9499750e+01,  6.4843952e+00, -2.1637431e+01,
       6.5900289e+01, -6.9209851e+01,  5.7574118e+01, -6.3239986e+01,
       5.0201888e+01, -2.5161732e+01, -1.8407647e+01,  3.3057952e+01,
      -5.1394596e+01,  6.6274453e+01, -8.9093497e+01,  6.1636186e+01,
      -1.8970991e+01,  1.8110690e+01,  1.4664315e+01, -5.7379639e+01,
       5.9659304e+01, -6.7425126e+01,  6.6740221e+01, -5.6441736e+01,
       3.3370764e+01, -8.8727172e+00, -2.8768291e+01,  4.7072609e+01,
      -6.4504039e+01,  7.0679425e+01, -6.4413968e+01,  5.7895311e+01,
      -2.2822099e+01,  8.2462832e+00,  2.1908355e+01, -3.0461871e+01,
       5.6741507e+01, -7.3393219e+01,  5.3252893e+01, -4.6826469e+01,
       2.0122779e+01, -5.9699302e+00, -2.3230042e+01,  3.8784526e+01,
      -5.3705628e+01,  6.2675409e+01, -4.5504385e+01,  3.5709340e+01,
      -1.9387569e+01,  5.2553786e+00,  3.0873151e+01, -4.6681999e+01,
       5.1345152e+01, -5.3802726e+01,  3.5105259e+01, -3.7404776e+01,
       2.0074043e+01,  1.0637688e+01, -3.4769882e+01,  4.1709646e+01,
      -5.1828676e+01,  4.2299294e+01, -4.3335588e+01,  3.5793638e+01,
       1.1265725e+00, -1.0395933e+01,  3.3544261e+01, -4.9299157e+01 ],

    [ -4.4595314e+02, -1.0379970e+03,  3.1577388e+02, -3.5610417e+03,
       3.4581766e+03,  8.3381081e+03, -1.1696533e+03,  2.4101021e+03,
      -6.5655829e+02,  1.4524982e+03,  1.9701386e+02,  1.1436715e+03,
       3.0871470e+02,  1.3134384e+03, -2.1221169e+02,  2.4285486e+02,
      -1.3136018e+02,  2.0492262e+02, -5.7778142e+01,  2.0474887e+02,
       7.0763142e+01,  9.9953062e+01,  8.3698665e+01,  6.2817480e+01,
       7.7348745e+01,  4.8618716e+01,  1.9547983e+01, -7.2171464e+00,
      -1.9159653e+01, -1.4038329e+01, -1.8628426e+01, -1.6163696e+01,
      -1.4520857e+01, -4.8554891e+00, -5.7820395e-01, -3.7918076e+00,
       2.3294407e+00,  2.9495405e+00,  4.7407475e+00,  4.4409321e+00,
       1.5250457e+00,  2.7760466e+00,  1.9722269e+00, -2.1015209e+00,
      -3.2710967e+00, -2.8401413e-03, -1.3044069e+00, -1.6457901e+00,
      -1.6669278e+00, -2.3124530e-01,  2.3073051e+00,  8.9693576e-02,
      -9.1429477e-01,  1.6390903e+00,  1.5921541e+00,  2.0781925e+00,
       5.6774613e-01, -1.4873030e+00,  9.0090862e-01, -1.7392411e-01,
      -2.0619629e+00, -2.6804981e+00,  1.5808344e+00, -1.9739969e+00,
      -3.2934549e+00,  8.3984861e-01,  1.6902572e+00,  1.1574818e+00,
       1.5309696e+00, -4.5293583e-01,  6.7334155e-01, -1.0828159e+00,
       2.8647595e+00, -1.0355011e+00,  1.2478700e+00, -1.0934505e+00,
      -2.6607804e+00,  6.5988107e-01,  5.0203258e-01, -9.3129652e-01,
      -1.6993602e+00,  3.3832184e+00, -1.1600967e-01, -2.6664726e-01,
      -1.2900776e+00,  4.8121635e-01,  5.7458524e-01, -1.3748852e-02,
      -1.0422755e+00,  9.6404835e-01, -6.5515813e-01, -1.2033320e+00,
       1.2628304e+00, -1.5881764e+00, -1.2024205e+00, -1.5188743e-01,
       2.1530142e-01, -1.1518121e+00,  1.5159517e+00,  7.6772612e-01,
       3.4785413e-01,  1.4223757e+00,  1.6791664e+00,  1.6342604e-03,
      -9.7881678e-01, -1.0907888e+00, -1.1879306e+00,  1.1798907e+00,
      -9.0898948e-02, -1.9025707e-01,  1.4674577e+00,  9.8643104e-01,
      -3.5040854e+00, -8.8416089e-01,  1.4586672e+00,  2.8440488e+00,
       2.5053395e+00, -1.5126041e+00, -3.8048891e-02,  5.8273982e-02,
      -1.6839084e+00,  8.3919618e-01,  8.4288521e-01,  8.2299662e-01,
      -4.3286655e+00, -8.2387031e-01,  3.8407497e-01,  8.0932325e-01,
       4.2209188e-01, -1.6088979e-01,  1.1786569e+00, -1.5473109e-01,
      -9.7397503e-01, -1.5447399e-01,  1.2455806e+00,  1.1521613e+00,
      -1.1394046e+00, -2.4879158e+00, -1.1733733e+00,  1.0462318e+00,
      -1.2964334e+00,  2.4531392e-01,  1.9106251e-01, -4.3741350e-01,
       5.7814855e-02,  7.3157372e-01, -7.7503310e-01,  5.5390581e-01,
      -1.3940056e+00, -5.7191178e-01,  1.8789088e+00, -5.6670606e-01,
      -4.8435902e-01, -1.8372728e+00, -9.7604008e-01,  1.0244293e+00,
       5.7039949e-01, -1.0034018e+00, -8.3178864e-02,  7.5770740e-02 ]

])

X_F_7M5 = np.array([

    [ -1.3344626e+03,  1.4958812e+03, -5.5694903e+03, -1.0486529e+04,
       5.6650034e+02,  6.0141573e+03, -5.3094551e+02,  2.6475111e+02,
       1.6690199e+03, -2.0838083e+03, -5.7181477e+02,  3.0979708e+02,
      -2.1393724e+02,  1.2521746e+02,  3.2684730e+02, -1.7994574e+02,
      -2.0020361e+02,  8.6111917e+00, -2.4832990e+02,  1.7213677e+02,
       1.4131656e+00, -8.5287997e+01,  1.7851510e+02, -2.3922723e+02,
       9.9313025e+01,  8.9492487e+01, -2.1678225e+02,  2.8445423e+02,
      -1.0586008e+02, -1.7011189e+02,  2.0924470e+02, -4.3658588e+01,
      -5.3305585e+01,  9.5065494e+01, -2.4959448e+01, -1.9882483e+01,
       6.3950343e+01, -1.8254423e+02,  1.4871147e+02, -2.9924201e+01,
      -3.8358507e+01,  1.3662271e+02, -9.1506608e+01, -3.6536786e+01,
       6.1379812e+01, -1.2146687e+02,  7.5261109e+01,  6.3261448e+01,
      -7.4977224e+01,  9.6981001e+01, -5.2501206e+01, -8.8542526e+01,
       8.7236325e+01, -7.1242762e+01,  3.4918529e+01,  1.0382986e+02,
      -9.4414359e+01,  4.3111460e+01, -6.8550987e+00, -1.1139641e+02,
       6.8058914e+01, -6.1872729e+00, -2.3074735e+01,  1.0246797e+02,
      -5.7063670e+01, -1.0528237e+01,  3.4668643e+01, -9.7928146e+01,
       5.4841513e+01,  3.8537271e+01, -4.1668071e+01,  9.3225595e+01,
      -3.6482817e+01, -5.6719701e+01,  4.9322737e+01, -7.0376756e+01,
       2.4449679e+01,  7.1939467e+01, -5.4394295e+01,  5.0564463e+01,
      -1.2276169e+01, -8.5637045e+01,  5.2333982e+01, -2.5308579e+01,
      -5.5784300e+00,  8.7058263e+01, -5.1715921e+01, -7.2178424e-01,
       1.4946171e+01, -8.5511725e+01,  4.3784053e+01,  1.9685956e+01,
      -2.5023328e+01,  8.0537198e+01, -3.0868111e+01, -3.4548956e+01,
       3.4272902e+01, -7.1390131e+01,  1.8631772e+01,  5.2295899e+01,
      -3.9080597e+01,  5.2223213e+01, -9.1791474e+00, -6.4321817e+01,
       3.7530035e+01, -3.6279190e+01,  1.3252369e+00,  7.2448665e+01,
      -3.9490622e+01,  1.4609831e+01,  9.3290814e+00, -7.2230799e+01,
       3.7265287e+01,  3.1414748e+00, -1.8859243e+01,  7.1138748e+01,
      -3.1848909e+01, -2.1851571e+01,  2.6573367e+01, -6.9966757e+01 ],

    [  6.6940015e+01, -3.8686730e+02,  1.2805107e+03, -7.2628992e+03,
      -1.5754672e+03, -6.2436979e+02,  8.2787173e+02, -8.0959099e+02,
      -3.4245127e+02, -3.5548999e+01, -7.4832843e+01,  9.1419785e+01,
       5.1342258e+01,  5.3971888e+01, -2.7664071e+01,  4.6162382e+01,
      -1.0556418e+01,  6.7202580e+01, -1.7808740e+01, -6.8463927e+00,
      -1.7273295e+01,  1.1657273e+00, -3.0473621e+00,  7.6704533e+00,
       3.2019440e+00, -4.9821490e+00,  9.7767521e+00, -5.6595863e+00,
      -2.2488613e+00, -2.3116498e+00,  3.6060424e+00, -1.0509700e+01,
       1.0880662e+01,  1.3223251e+00,  3.8299992e+00, -3.2105583e+00,
      -1.3455890e+00, -1.3800967e+00,  3.9392807e+00, -5.9971359e+00,
       3.0005141e+00,  3.1693816e+00, -2.4580934e+00,  7.4517245e-01,
      -2.5522465e+00, -1.7665735e+00,  6.2439566e+00, -2.5566772e+00,
       1.0487489e+00,  4.9010207e+00, -2.4551119e+00, -2.3895763e+00,
      -1.5405507e+00,  1.9495530e+00,  3.6880545e+00,  5.9918442e-01,
      -2.4705668e+00,  4.8115098e+00, -4.1881444e+00, -3.6102608e+00,
      -5.5087057e-01, -2.6206962e+00,  3.9639713e+00, -7.7640910e-01,
      -3.9510665e+00,  4.4026144e+00, -3.2255765e+00, -1.5538782e+00,
      -3.1853054e-01,  1.6928250e+00,  8.0901945e-01,  9.8517710e-01,
      -3.9968911e+00,  5.2957448e-01,  7.5228614e-01, -5.5755012e+00,
       4.1839400e+00, -8.6149818e-01,  2.4716293e+00, -3.9843760e-01,
      -1.6905671e+00, -8.5829267e-01,  1.9447429e+00, -3.4134235e-01,
       7.0851852e-02,  5.1275041e+00, -1.0122600e+00,  1.4475050e-01,
      -3.3223224e+00, -1.0374444e+00,  3.7779787e+00, -4.5459512e+00,
       4.0726493e+00, -2.5433866e-01,  2.0525115e+00, -3.0465923e+00,
      -1.7020097e+00, -1.0852672e+00,  2.3054607e+00, -5.4698297e-01,
       1.0938816e+00,  9.5295631e-01,  1.8110373e-01, -2.4274422e+00,
      -9.0034207e-01,  3.4598368e-01,  3.0697391e+00, -3.3366935e-03,
      -3.4637690e-02,  2.0742294e+00, -2.0643727e+00, -2.3004709e+00,
       3.4207323e-01, -1.8133178e+00,  5.0366054e+00, -8.8976033e-01,
      -7.4204267e-01,  1.6508374e+00, -1.2431019e+00, -1.1825352e+00 ]

])

X_F = [ X_F_7M5, X_F_10M ]

RC_ORDER_10M = np.array([ [ 6, 0 ], [ 6, 0 ] ])
RC_ORDER_7M5 = np.array([ [ 8, 0 ], [ 8, 0 ] ])
RC_ORDER = [ RC_ORDER_7M5, RC_ORDER_10M ]

RC_I_1_10M = np.array([ [ 13,  9,  4,  9,  8,  9,  8,  8 ],
                        [  4,  7,  9,  9,  9,  9,  8,  8 ] ])

RC_I_1_7M5 = np.array([ [ 12, 13,  6,  9,  7,  9,  7,  9 ],
                        [  5, 13,  8, 10,  9,  9,  7,  9 ] ])

RC_I_1 = [ RC_I_1_7M5, RC_I_1_10M ]

RC_I_2_10M = np.array([ [  8,  8,  8,  8,  8,  8,  8,  8 ],
                        [  8,  8,  8,  8,  8,  8,  8,  8 ] ])

RC_I_2_7M5 = np.array([ [  8,  8,  8,  8,  8,  8,  8,  8 ],
                        [  8,  8,  8,  8,  8,  8,  8,  8 ] ])

RC_I_2 = [ RC_I_2_7M5, RC_I_2_10M ]

RC_Q_1_10M = np.array([

    [  7.9801723e-01,  1.8374952e-01, -6.7369564e-01,  1.8374952e-01,
       0.0000000e+00,  1.8374952e-01,  0.0000000e+00,  0.0000000e+00 ],

    [ -6.7369564e-01, -1.8374952e-01,  1.8374952e-01,  1.8374952e-01,
       1.8374952e-01,  1.8374952e-01,  0.0000000e+00,  0.0000000e+00 ]

])

RC_Q_1_7M5 = np.array([

    [  6.7369564e-01,  7.9801723e-01, -3.6124167e-01,  1.8374952e-01,
      -1.8374952e-01,  1.8374952e-01, -1.8374952e-01,  1.8374952e-01 ],

    [ -5.2643216e-01,  7.9801723e-01,  0.0000000e+00,  3.6124167e-01,
       1.8374952e-01,  1.8374952e-01, -1.8374952e-01,  1.8374952e-01 ]

])

RC_Q_1 = [ RC_Q_1_7M5, RC_Q_1_10M ]

RC_Q_2_10M = np.array([

    [  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00 ],

    [  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00 ]

])

RC_Q_2_7M5 = np.array([

    [  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00 ],

    [  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00 ]

])

RC_Q_2 = [ RC_Q_2_7M5, RC_Q_2_10M ]

NUM_TNS_FILTERS_10M = [ 1, 1 ]
NUM_TNS_FILTERS_7M5 = [ 1, 1 ]
NUM_TNS_FILTERS = [ NUM_TNS_FILTERS_7M5, NUM_TNS_FILTERS_10M ]

TNS_LEV_A_10M = np.array([

    [  1.0000000e+00,  7.2228594e-01, -5.9139666e-01, -5.8613895e-01,
       1.1856746e-01,  2.9269254e-02,  8.0280074e-02,  3.8446982e-02,
       4.3781506e-02 ],

    [  1.0000000e+00, -5.6089086e-01, -2.3332924e-01,  1.3258672e-01,
      -7.2233128e-02,  1.1753190e-01,  1.9802609e-01, -6.1275417e-02,
       2.2455104e-02 ]

])

TNS_LEV_A_7M5 = np.array([

    [  1.0000000e+00,  5.2890099e-01,  5.4205760e-01, -4.4859180e-01,
       3.5954391e-01, -1.9179061e-01,  1.8237136e-01, -1.1968822e-01,
       1.6649226e-01 ],

    [  1.0000000e+00, -9.0587790e-01,  9.8849245e-01, -1.4294859e-01,
       2.5680120e-01, -5.8559598e-03,  3.0914531e-01, -2.2365546e-01,
       1.0115038e-01 ]

])

TNS_LEV_A = [ TNS_LEV_A_7M5, TNS_LEV_A_10M ]

TNS_LEV_E_10M = [ 0.412635, 1.396833 ]
TNS_LEV_E_7M5 = [ 0.397854, 0.665554 ]
TNS_LEV_E = [ TNS_LEV_E_7M5, TNS_LEV_E_10M ]

TNS_LEV_RC_10M = np.array([

    [  8.3775274e-01,  1.7149527e-01, -7.0757376e-01,  1.9028627e-01,
      -1.4202392e-02,  1.0144450e-01,  6.8373224e-03,  4.3781506e-02 ],

    [ -6.7692801e-01, -9.9895702e-02,  2.2652538e-01,  1.0193040e-01,
       2.0702019e-01,  1.7652182e-01, -4.8705113e-02,  2.2455104e-02 ]

])

TNS_LEV_RC_7M5 = np.array([

    [  6.0637394e-01,  8.3208082e-01, -4.0823140e-01,  2.1979501e-01,
      -1.5450397e-01,  2.2566273e-01, -2.1366897e-01,  1.6649226e-01 ],

    [ -5.4911119e-01,  7.7810682e-01, -2.2286927e-03,  3.4649135e-01,
       2.2484419e-01,  9.3957220e-02, -1.3339034e-01,  1.0115038e-01 ]

])

TNS_LEV_RC = [ TNS_LEV_RC_7M5, TNS_LEV_RC_10M ]

NBITS_TNS_10M = np.array([ 24, 18 ])
NBITS_TNS_7M5 = np.array([ 24, 24 ])
NBITS_TNS = [ NBITS_TNS_7M5, NBITS_TNS_10M ]


### C.3.1.15 Global gain estimation

GG_OFF_10M = np.array([ -131, -131 ])
GG_OFF_7M5 = np.array([ -127, -127 ])
GG_OFF = [ GG_OFF_7M5, GG_OFF_10M ]

GG_IND_10M = np.array([  191,  166 ])
GG_IND_7M5 = np.array([  189,  162 ])
GG_IND = [ GG_IND_7M5, GG_IND_10M ]

GG_MIN_10M = np.array([  121,  115 ])
GG_MIN_7M5 = np.array([  114,  109 ])
GG_MIN = [ GG_MIN_7M5, GG_MIN_10M ]

GG_10M = np.array([ 138.949549, 17.782794 ])
GG_7M5 = np.array([ 163.789371, 17.782794 ])
GG = [ GG_7M5, GG_10M ]

NBITS_OFFSET_10M = np.array([ 0.000000, -1.200000 ])
NBITS_OFFSET_7M5 = np.array([ 0.000000, -1.600000 ])
NBITS_OFFSET = [ NBITS_OFFSET_7M5, NBITS_OFFSET_10M ]


### C.3.1.16 Quantization

X_Q_10M = np.array([

    [    0,   18,  -15,   48,  -53,  -97,   -1,  -13,  -13,    1,
       -23,    7,  -23,  -10,    1,   -2,    1,   -3,    0,   -2,
         1,   -2,    1,   -1,    1,   -1,    0,    1,   -1,    0,
        -1,    0,    0,    0,    1,   -1,    1,   -1,    0,    0,
         0,    0,   -1,    1,   -1,    1,    0,    0,    0,    0,
         0,   -1,    1,   -1,    0,    0,    0,    0,   -1,    0,
         0,    1,    0,    0,    0,    0,    0,   -1,    0,    0,
         0,    0,    0,    0,   -1,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,   -1,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0 ],

    [  -25,  -58,   18, -200,  194,  469,  -66,  135,  -37,   82,
        11,   64,   17,   74,  -12,   14,   -7,   11,   -3,   11,
         4,    5,    5,    3,    4,    3,    1,    0,   -1,   -1,
        -1,   -1,   -1,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0 ]

])

X_Q_7M5 = np.array([

    [   -8,    9,  -34,  -64,    3,   37,   -3,    1,   10,  -13,
        -3,    2,   -1,    1,    2,   -1,   -1,    0,   -1,    1,
         0,    0,    1,   -1,    0,    0,   -1,    2,   -1,   -1,
         1,    0,    0,    0,    0,    0,    0,   -1,    1,    0,
         0,    1,    0,    0,    0,   -1,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    1,    0,    0,    0,   -1,
         0,    0,    0,    1,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0 ],

    [    4,  -22,   72, -408,  -88,  -35,   46,  -45,  -19,   -2,
        -4,    5,    3,    3,   -1,    2,    0,    4,   -1,    0,
        -1,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0 ]

])

X_Q = [ X_Q_7M5, X_Q_10M ]

LASTNZ_10M = np.array([ 108,  34 ])
LASTNZ_7M5 = np.array([  64,  22 ])
LASTNZ = [ LASTNZ_7M5, LASTNZ_10M ]

NBITS_EST_10M = np.array([ 231, 250 ])
NBITS_EST_7M5 = np.array([ 164, 140 ])
NBITS_EST = [ NBITS_EST_7M5, NBITS_EST_10M ]

LSB_MODE_10M = np.array([ 0, 0 ])
LSB_MODE_7M5 = np.array([ 0, 0 ])
LSB_MODE = [ LSB_MODE_7M5, LSB_MODE_10M ]

NBITS_SPEC_10M = np.array([ 225, 231 ])
NBITS_SPEC_7M5 = np.array([ 156, 146 ])
NBITS_SPEC = [ NBITS_SPEC_7M5, NBITS_SPEC_10M ]


### C.3.1.17 Global Gain adjustement

GG_IND_ADJ_10M = np.array([ 192, 168 ])
GG_IND_ADJ_7M5 = np.array([ 190, 162 ])
GG_IND_ADJ = [ GG_IND_ADJ_7M5, GG_IND_ADJ_10M ]

GG_ADJ_10M = np.array([ 1.508591e+02, 2.096180e+01 ])
GG_ADJ_7M5 = np.array([ 1.778279e+02, 1.778279e+01 ])
GG_ADJ = [ GG_ADJ_7M5, GG_ADJ_10M ]


### C.3.1.18 Requantization

X_Q_REQ_10M = np.array([

    [    0,   17,  -14,   44,  -49,  -89,   -1,  -12,  -12,    0,
       -21,    7,  -21,  -10,    1,   -2,    1,   -3,    0,   -2,
         1,   -2,    1,   -1,    1,   -1,    0,    1,   -1,    0,
        -1,    0,    0,    0,    1,   -1,    1,   -1,    0,    0,
         0,    0,   -1,    1,   -1,    1,    0,    0,    0,    0,
         0,   -1,    1,   -1,    0,    0,    0,    0,   -1,    0,
         0,    0,    0,    0,    0,    0,    0,   -1,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0 ],

    [  -21,  -49,   15, -170,  165,  398,  -56,  115,  -31,   69,
         9,   54,   15,   63,  -10,   11,   -6,   10,   -3,   10,
         3,    5,    4,    3,    4,    2,    1,    0,   -1,   -1,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0 ]

])

X_Q_REQ_7M5 = np.array([

    [   -7,    8,  -31,  -59,    3,   34,   -3,    1,    9,  -12,
        -3,    2,   -1,    1,    2,   -1,   -1,    0,   -1,    1,
         0,    0,    1,   -1,    0,    0,   -1,    1,    0,   -1,
         1,    0,    0,    0,    0,    0,    0,   -1,    1,    0,
         0,    1,    0,    0,    0,   -1,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,   -1,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
         0,    0,    0,    0,    0,    0,    0,    0,    0,    0 ],

])

X_Q_REQ = [ X_Q_REQ_7M5, X_Q_REQ_10M ]

LASTNZ_REQ_10M = np.array([ 68, 30 ])
LASTNZ_REQ_7M5 = np.array([ 60, None ])
LASTNZ_REQ = [ LASTNZ_REQ_7M5, LASTNZ_REQ_10M ]

NBITS_EST_REQ_10M = np.array([ 206, 237 ])
NBITS_EST_REQ_7M5 = np.array([ 147, None ])
NBITS_EST_REQ = [ NBITS_EST_REQ_7M5, NBITS_EST_REQ_10M ]

NBITS_TRUNC_REQ_10M = np.array([ 206, 229 ])
NBITS_TRUNC_REQ_7M5 = np.array([ 147, None ])
NBITS_TRUNC_REQ = [ NBITS_TRUNC_REQ_7M5, NBITS_TRUNC_REQ_10M ]

LSB_MODE_REQ_10M = np.array([ 0, 0 ])
LSB_MODE_REQ_7M5 = np.array([ 0, None ])
LSB_MODE_REQ = [ LSB_MODE_REQ_7M5, LSB_MODE_REQ_10M ]


### C.3.1.19 Residual Coding

RES_BITS_10M = np.array([
    [ 0, 1 ,1 ,0, 0, 1, 1, 0, 1, 0, 0, 1, 1, 1, 1, 0,
      0, 1, 1, 1, 1, 1, 0 ],
    [ 0, 0, 1, 1, 0, 0 ]
], dtype=object)

RES_BITS_7M5 = np.array([
    [ 0, 1, 0, 1, 1, 0, 1, 1, 1, 1, 0, 0, 0 ],
    [ 0, 1, 1, 0, 0, 0, 1, 0, 0, 1 ]
], dtype=object)

RES_BITS = [ RES_BITS_7M5, RES_BITS_10M ]


### C.3.1.20 Noise factor

F_NF_10M = np.array([ 3, 7 ])
F_NF_7M5 = np.array([ 4, 6 ])
F_NF = [ F_NF_7M5, F_NF_10M ]


### C.3.1.21 Side information encoding

BYTES_SIDE_10M = [

    bytearray([
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x19, 0x30, 0xc3, 0x8d, 0xf1, 0x88, 0xcf, 0x80, 0x43 ]),

    bytearray([
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x39, 0x30, 0xc5, 0xf7, 0x6b, 0x89, 0xcf, 0x50, 0x1d ])

]

BYTES_SIDE_7M5 = [

    bytearray([
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x04, 0x07, 0xd3, 0x48, 0x84, 0x45, 0xbe, 0x3b ]),

    bytearray([
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x18, 0x90, 0x6c, 0x0f, 0x6b, 0x8c, 0xc7, 0xa2, 0x15 ])

]

BYTES_SIDE = [ BYTES_SIDE_7M5, BYTES_SIDE_10M ]


### C.3.1.22 Arithmetic encoding

BYTES_AC_10M = [

    bytearray([
        0xa7, 0x1c, 0x2a, 0x35, 0xaf, 0x69, 0x0e, 0xf9, 0xbe, 0xf1,
        0xcb, 0x2e, 0x87, 0x5f, 0x05, 0x13, 0xd7, 0x8e, 0x3c, 0xf2,
        0xcd, 0xd5, 0xad, 0x56, 0xb6, 0x5e, 0xe1, 0x8f, 0xc1, 0xbe,
        0x80, 0x99, 0x30, 0xc3, 0x8d, 0xf1, 0x88, 0xcf, 0x80, 0x43 ]),

    bytearray([
        0x7e, 0x55, 0xff, 0x85, 0x35, 0x86, 0xf7, 0x01, 0xf8, 0x8c,
        0xaa, 0x1a, 0x1e, 0x42, 0x43, 0xd6, 0x26, 0x02, 0x37, 0x5c,
        0x78, 0xff, 0x1a, 0x55, 0xdd, 0xd2, 0x81, 0x2e, 0x68, 0xdd,
        0xc4, 0xf9, 0x30, 0xc5, 0xf7, 0x6b, 0x89, 0xcf, 0x50, 0x1d ])

]

BYTES_AC_7M5 = [

    bytearray([
        0xfe, 0x86, 0xdb, 0xa2, 0x90, 0x78, 0x54, 0xb1, 0x5d, 0x1b,
        0x1f, 0x3b, 0x24, 0x62, 0xaf, 0xb5, 0x95, 0x9c, 0xb0, 0xa0,
        0x6f, 0xbe, 0xac, 0x07, 0xd3, 0x48, 0x84, 0x45, 0xbe, 0x3b ]),

    bytearray([
        0xda, 0x52, 0x63, 0x57, 0xf8, 0x66, 0x37, 0xcf, 0x85, 0xe1,
        0xd4, 0x32, 0x46, 0xc2, 0x36, 0x6d, 0xed, 0xa9, 0x52, 0x58,
        0x17, 0x18, 0x90, 0x6c, 0x0f, 0x6b, 0x8c, 0xc7, 0xa2, 0x15 ])

]

BYTES_AC = [ BYTES_AC_7M5, BYTES_AC_10M ]


### C.3.1.23 Attack detector intermediate data

NBYTES_ATT = (88e3 * np.array([ 7.5e-3, 10e-3 ]) / 8).astype(int)

X_PCM_ATT_10M = np.array([

    [      0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,  27852,  29491,  27852,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0 ],

    [      0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0 ]

])

X_PCM_ATT_7M5 = np.array([

    [      0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,  27852,  29491,  27852,      0,
           0,      0,      0,      0,      0,      0,      0,      0 ],

    [      0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0,
           0,      0,      0,      0,      0,      0,      0,      0 ],

])

X_PCM_ATT = [ X_PCM_ATT_7M5, X_PCM_ATT_10M ]

F_ATT_7M5 = np.array([ 1, 1 ])
F_ATT_10M = np.array([ 1, 1 ])
F_ATT = [ F_ATT_7M5, F_ATT_10M ]


### C.4.1.1-4  Spectrum Decoding

X_HAT_Q_10M = np.array([

    [    0.0000,   16.8125,  -13.8125,   44.3125,  -49.3125,  -89.3125,
        -0.8125,  -11.8125,  -12.3125,    0.0000,  -20.8125,    6.8125,
       -21.3125,   -9.8125,    1.3125,   -1.8125,    1.3125,   -3.3125,
         0.0000,   -2.3125,    1.3125,   -1.8125,    1.3125,   -0.8125,
         1.0000,   -1.0000,    0.0000,    1.0000,   -1.0000,    0.0000,
        -1.0000,    0.0000,    0.0000,    0.0000,    1.0000,   -1.0000,
         1.0000,   -1.0000,    0.0000,    0.0000,    0.0000,    0.0000,
        -1.0000,    1.0000,   -1.0000,    1.0000,    0.0000,    0.0000,
         0.0000,    0.0000,    0.0000,   -1.0000,    1.0000,   -1.0000,
         0.0000,    0.0000,    0.0000,    0.0000,   -1.0000,    0.0000,
         0.0000,    0.0000,   -0.3125,   -0.3125,    0.0000,    0.0000,
         0.0000,   -1.0000,    0.0000,    0.0000,    0.0000,    0.3125,
        -0.3125,   -0.3125,   -0.3125,   -0.3125,    0.3125,   -0.3125,
        -0.3125,    0.3125,    0.3125,    0.3125,    0.3125,    0.3125,
        -0.3125,    0.3125,    0.3125,    0.3125,    0.3125,    0.3125,
         0.3125,    0.3125,    0.3125,   -0.3125,   -0.3125,   -0.3125,
        -0.3125,   -0.3125,   -0.3125,   -0.3125,   -0.3125,   -0.3125,
         0.3125,   -0.3125,   -0.3125,    0.3125,   -0.3125,   -0.3125,
        -0.3125,   -0.3125,   -0.3125,   -0.3125,    0.3125,    0.3125,
        -0.3125,    0.3125,    0.3125,    0.3125,   -0.3125,    0.3125,
         0.3125,   -0.3125,    0.3125,   -0.3125,    0.3125,    0.3125,
        -0.3125,    0.3125,    0.3125,    0.3125,   -0.3125,   -0.3125,
         0.3125,   -0.3125,    0.3125,    0.3125,   -0.3125,   -0.3125,
        -0.3125,   -0.3125,    0.3125,    0.3125,    0.3125,   -0.3125,
        -0.3125,    0.3125,    0.3125,    0.3125,   -0.3125,   -0.3125,
         0.3125,    0.3125,   -0.3125,   -0.3125,   -0.3125,   -0.3125,
        -0.3125,    0.3125,   -0.3125,   -0.3125 ],

    [  -21.3125,  -49.3125,   15.3125, -170.0000,  165.0000,  398.0000,
       -56.0000,  115.0000,  -31.0000,   69.0000,    9.0000,   54.0000,
        15.0000,   63.0000,  -10.0000,   11.0000,   -6.0000,   10.0000,
        -3.0000,   10.0000,    3.0000,    5.0000,    4.0000,    3.0000,
         4.0000,    2.0000,    1.0000,    0.0000,   -1.0000,   -1.0000,
         0.0000,    0.0000,    0.0000,    0.0625,    0.0625,   -0.0625,
        -0.0625,   -0.0625,    0.0625,    0.0625,   -0.0625,    0.0625,
        -0.0625,    0.0625,   -0.0625,   -0.0625,    0.0625,   -0.0625,
        -0.0625,   -0.0625,    0.0625,    0.0625,    0.0625,   -0.0625,
        -0.0625,   -0.0625,   -0.0625,    0.0625,   -0.0625,    0.0625,
        -0.0625,    0.0625,    0.0625,    0.0625,    0.0625,   -0.0625,
         0.0625,    0.0625,    0.0625,   -0.0625,   -0.0625,   -0.0625,
        -0.0625,    0.0625,   -0.0625,    0.0625,    0.0625,   -0.0625,
         0.0625,    0.0625,    0.0625,   -0.0625,    0.0625,   -0.0625,
        -0.0625,    0.0625,   -0.0625,   -0.0625,   -0.0625,   -0.0625,
        -0.0625,    0.0625,   -0.0625,    0.0625,   -0.0625,   -0.0625,
        -0.0625,   -0.0625,    0.0625,    0.0625,   -0.0625,    0.0625,
        -0.0625,   -0.0625,    0.0625,   -0.0625,   -0.0625,   -0.0625,
         0.0625,    0.0625,   -0.0625,   -0.0625,    0.0625,   -0.0625,
        -0.0625,    0.0625,    0.0625,    0.0625,    0.0625,    0.0625,
        -0.0625,   -0.0625,   -0.0625,    0.0625,   -0.0625,    0.0625,
         0.0625,    0.0625,   -0.0625,    0.0625,   -0.0625,   -0.0625,
        -0.0625,    0.0625,   -0.0625,   -0.0625,   -0.0625,    0.0625,
        -0.0625,   -0.0625,   -0.0625,    0.0625,    0.0625,    0.0625,
        -0.0625,   -0.0625,    0.0625,   -0.0625,    0.0625,    0.0625,
         0.0625,    0.0625,    0.0625,   -0.0625,    0.0625,    0.0625,
         0.0625,   -0.0625,    0.0625,    0.0625 ]

])

X_HAT_Q_7M5 = np.array([

    [   -7.3125,    8.3125,  -31.3125,  -58.8125,    3.3125,   33.8125,
        -2.8125,    1.3125,    9.3125,  -11.8125,   -3.0000,    2.0000,
        -1.0000,    1.0000,    2.0000,   -1.0000,   -1.0000,    0.0000,
        -1.0000,    1.0000,    0.0000,    0.0000,    1.0000,   -1.0000,
         0.0000,    0.0000,   -1.0000,    1.0000,    0.0000,   -1.0000,
         1.0000,    0.0000,    0.0000,    0.2500,   -0.2500,    0.0000,
         0.0000,   -1.0000,    1.0000,    0.0000,    0.0000,    1.0000,
         0.0000,    0.0000,    0.0000,   -1.0000,    0.0000,    0.0000,
        -0.2500,    0.2500,    0.2500,    0.2500,   -0.2500,    0.2500,
         0.2500,    0.2500,   -0.2500,    0.0000,    0.0000,   -1.0000,
         0.0000,    0.0000,    0.2500,    0.2500,   -0.2500,   -0.2500,
         0.2500,    0.2500,    0.2500,   -0.2500,   -0.2500,    0.2500,
        -0.2500,    0.2500,   -0.2500,    0.2500,   -0.2500,   -0.2500,
        -0.2500,   -0.2500,   -0.2500,   -0.2500,    0.2500,   -0.2500,
         0.2500,    0.2500,   -0.2500,   -0.2500,    0.2500,   -0.2500,
         0.2500,    0.2500,   -0.2500,   -0.2500,    0.2500,   -0.2500,
         0.2500,   -0.2500,    0.2500,    0.2500,    0.2500,    0.2500,
         0.2500,   -0.2500,   -0.2500,   -0.2500,    0.2500,    0.2500,
         0.2500,    0.2500,    0.2500,   -0.2500,   -0.2500,   -0.2500,
         0.2500,    0.2500,    0.2500,   -0.2500,    0.2500,    0.2500 ],

    [    3.8125,  -21.8125,   72.3125, -408.3125,  -88.3125,  -35.3125,
        46.3125,  -45.0000,  -19.0000,   -2.0000,   -4.0000,    5.0000,
         3.0000,    3.0000,   -1.0000,    2.0000,    0.0000,    4.0000,
        -1.0000,    0.0000,   -1.0000,    0.0000,    0.0000,   -0.1250,
        -0.1250,    0.1250,   -0.1250,   -0.1250,    0.1250,   -0.1250,
        -0.1250,   -0.1250,   -0.1250,   -0.1250,    0.1250,    0.1250,
        -0.1250,   -0.1250,    0.1250,   -0.1250,    0.1250,   -0.1250,
        -0.1250,    0.1250,    0.1250,   -0.1250,   -0.1250,    0.1250,
        -0.1250,   -0.1250,   -0.1250,    0.1250,   -0.1250,    0.1250,
        -0.1250,   -0.1250,    0.1250,   -0.1250,    0.1250,   -0.1250,
        -0.1250,    0.1250,   -0.1250,   -0.1250,   -0.1250,   -0.1250,
        -0.1250,   -0.1250,   -0.1250,    0.1250,   -0.1250,    0.1250,
         0.1250,    0.1250,   -0.1250,   -0.1250,   -0.1250,   -0.1250,
         0.1250,    0.1250,    0.1250,    0.1250,    0.1250,    0.1250,
        -0.1250,   -0.1250,   -0.1250,    0.1250,    0.1250,   -0.1250,
        -0.1250,   -0.1250,   -0.1250,   -0.1250,   -0.1250,    0.1250,
         0.1250,   -0.1250,    0.1250,    0.1250,    0.1250,   -0.1250,
        -0.1250,   -0.1250,   -0.1250,    0.1250,   -0.1250,    0.1250,
        -0.1250,   -0.1250,   -0.1250,   -0.1250,    0.1250,    0.1250,
        -0.1250,    0.1250,    0.1250,    0.1250,   -0.1250,   -0.1250 ]

])

X_HAT_Q = [ X_HAT_Q_7M5, X_HAT_Q_10M ]

### C.4.1.6  TNS

X_HAT_TNS_10M = np.array([

    [  0.0000000e+00,  2.5363181e+03, -2.0837409e+03,  6.6849426e+03,
      -7.4392379e+03, -1.3473601e+04, -1.2257300e+02, -1.7820228e+03,
      -1.8574523e+03,  0.0000000e+00, -3.1397544e+03,  1.0277274e+03,
      -3.2151839e+03,  7.6090367e+02, -1.9466503e+03, -5.2445445e+02,
       3.3140414e+02, -1.8519794e+03,  1.7920998e+03, -2.1685321e+03,
       1.8597265e+03, -1.3374728e+03,  7.4531009e+02,  1.3315563e+02,
      -6.0572955e+02,  1.0738661e+03, -1.2042433e+03,  1.2934874e+03,
      -1.0948115e+03,  6.2875792e+02, -2.6176578e+02, -3.3286816e+02,
       6.3874781e+02, -9.2546646e+02,  1.0536075e+03, -1.0082162e+03,
       8.4609116e+02, -5.3643862e+02,  8.6016577e+01,  3.1548104e+02,
      -6.4537299e+02,  7.8414243e+02, -9.2031551e+02,  8.4727748e+02,
      -7.1936859e+02,  4.6589807e+02, -6.4881797e+01, -2.6457448e+02,
       5.6215195e+02, -6.6813432e+02,  6.6226333e+02, -6.5474860e+02,
       5.2409793e+02, -4.0142502e+02,  6.3505532e+01,  1.7235901e+02,
      -4.1924611e+02,  5.0550392e+02, -6.5734435e+02,  5.0345309e+02,
      -3.6608788e+02,  8.0016434e+01,  9.3237184e+01, -3.5017721e+02,
       4.2795992e+02, -4.6886670e+02,  3.7402812e+02, -3.7888490e+02,
       1.5291801e+02, -1.5802320e+01, -1.9719846e+02,  3.4187888e+02,
      -4.2748515e+02,  3.5181826e+02, -3.0466662e+02,  7.6284275e+01,
       8.7663093e+01, -2.9659233e+02,  3.1030268e+02, -2.9594060e+02,
       2.6439080e+02, -9.3629481e+01,  5.7399177e+01,  1.6375927e+02,
      -2.3208956e+02,  3.5517121e+02, -2.5708880e+02,  2.5690950e+02,
      -5.2915469e+01,  2.3237996e+01,  1.8246977e+02, -1.5537990e+02,
       2.8029628e+02, -2.5001493e+02,  1.6287623e+02, -1.2710806e+02,
      -6.9041960e+01,  5.1949460e+01, -2.3006446e+02,  1.3249613e+02,
      -2.3061667e+02,  6.5507752e+01, -6.6442896e+00, -1.4373816e+02,
       1.3523410e+02, -1.2379774e+02,  5.2635973e+01, -6.1214557e+01,
      -4.6571211e+01,  6.8132465e+00, -1.2664460e+02,  3.6959535e+01,
      -3.5846511e+01,  2.8934234e+01, -4.3948154e+01,  8.1935863e+01,
       7.4660900e+00,  5.1207805e+01, -2.1748068e+01,  8.5958555e+01,
       4.9435547e+00, -4.1272410e+01,  1.2552229e+02, -1.6625839e+02,
       1.9307762e+02, -1.0613596e+02,  1.4547859e+01,  1.0890433e+02,
      -1.0539493e+02,  1.9916467e+02, -1.9460994e+02,  1.3162563e+02,
      -2.6679242e+01, -1.0641065e+02,  1.9984677e+02, -1.8499813e+02,
       1.3746427e+02, -1.2384807e+02, -4.8113073e+00, -1.1154394e+01,
      -4.8901484e+01,  9.9602073e+01, -6.2727442e+01,  4.0646873e+01,
      -3.8839607e+01,  5.5618701e+01,  1.5609019e+01,  2.6413984e+01,
      -1.3661998e+01, -2.1935131e+01,  7.0604596e+01, -3.6013874e+01,
      -5.2534512e+00, -1.9095831e+01, -5.9462211e+01, -2.0474437e+01,
      -8.2457334e+01,  6.6458037e+01, -1.3905842e+02,  4.4992015e+01 ],

    [ -4.4674836e+02, -1.0336788e+03,  3.2097756e+02, -3.5635060e+03,
       3.4586970e+03,  8.3427964e+03, -1.1738608e+03,  2.4106070e+03,
      -6.4981580e+02,  1.4463642e+03,  1.8865620e+02,  1.1319372e+03,
       3.1442700e+02,  1.4722656e+03,  5.9737149e+02,  9.6658866e+02,
       4.9169825e+02,  6.7177464e+02,  1.9327293e+02,  1.5499589e+02,
      -2.1600722e+01, -9.8228391e+01, -1.2744379e+02, -1.7269979e+02,
      -8.5793205e+01, -7.4701778e+01, -2.2833266e+01,  8.7668786e-01,
       1.4493209e+01,  2.7797597e+01,  4.0794920e+01,  4.3722928e+01,
       3.6950947e+01,  2.9794074e+01,  2.0218141e+01,  7.2233914e+00,
      -4.3603335e+00, -1.3552659e+01, -1.6604838e+01, -1.8237845e+01,
      -1.9241801e+01, -1.3923796e+01, -1.1242668e+01, -3.6760056e+00,
      -1.1829705e+00,  2.5681718e+00,  7.2841081e+00,  6.6584846e+00,
       6.5203815e+00,  4.4803237e+00,  5.1830531e+00,  3.8023813e+00,
       2.5801707e+00, -8.9194181e-01, -2.7289508e+00, -4.3183797e+00,
      -5.5686696e+00, -3.5658387e+00, -5.0013288e+00, -1.5969716e+00,
      -2.5434186e+00,  1.0495670e+00,  2.5178803e+00,  4.0287891e+00,
       5.1309495e+00,  2.8569829e+00,  4.5137803e+00,  3.7678816e+00,
       3.5274736e+00,  2.1055313e-01, -1.4896329e+00, -3.0388583e+00,
      -4.4849681e+00, -2.7741285e+00, -4.6088456e+00, -1.5252078e+00,
      -1.4816216e-01, -7.4985193e-01,  2.1073955e+00,  3.0534784e+00,
       4.4307001e+00,  2.0418555e+00,  3.6512371e+00,  8.8259920e-01,
      -5.5756792e-01,  2.4731249e-01, -2.4585823e+00, -3.1307473e+00,
      -4.3178686e+00, -4.4328370e+00, -4.5859669e+00, -1.9461803e+00,
      -2.7535202e+00,  5.2435412e-01, -5.7680820e-01, -1.1178729e-01,
      -4.7067976e-01, -9.7145273e-01,  1.1713253e+00,  1.5417023e+00,
      -6.2164141e-02,  1.8196547e+00, -3.3253562e-01, -8.7085257e-01,
       4.0545492e-01, -1.6971448e+00, -2.1315433e+00, -3.1522747e+00,
      -7.0840626e-01,  1.9820684e-01, -1.2521487e+00, -1.2659519e+00,
       9.9900232e-01, -5.5880815e-01, -1.1085204e+00,  6.9059170e-01,
       1.6309021e+00,  2.4853740e+00,  2.8911996e+00,  3.6269550e+00,
       1.3909661e+00,  9.7770819e-02, -1.4853328e+00, -1.9863830e-01,
      -2.7510985e+00, -8.4492066e-01, -1.7063127e-01,  1.1338053e+00,
      -4.4636302e-01,  1.7476214e+00, -3.9440033e-02, -6.3944530e-01,
      -1.7198507e+00,  7.4867347e-02, -1.8649133e+00, -2.4597157e+00,
      -2.9678215e+00, -5.7798031e-01, -2.1073509e+00, -2.2439980e+00,
      -2.4007588e+00,  2.2681540e-01,  1.3649713e+00,  2.4256778e+00,
       9.1907814e-01,  4.7699713e-01,  2.1704836e+00, -3.5175645e-01,
       1.3125148e+00,  1.2589840e+00,  2.0656111e+00,  2.3990940e+00,
       2.6834440e+00,  6.0589675e-01,  1.9858819e+00,  1.9373749e+00,
       2.1901227e+00, -3.8495770e-01,  1.1720109e+00,  1.3860966e+00 ]

])

X_HAT_TNS_7M5 = np.array([

    [ -1.3003668e+03,  1.4781948e+03, -5.5682374e+03, -1.0458506e+04,
       5.8905505e+02,  6.0128073e+03, -5.0014108e+02,  2.3339917e+02,
       1.6560227e+03, -2.1005926e+03,  9.8231590e+02,  6.5472037e+02,
      -1.8816616e+03,  1.9995050e+03, -5.2483056e+02, -1.1296152e+03,
       1.8917056e+03, -1.4074840e+03,  7.3771850e+01,  1.1607281e+03,
      -1.4654795e+03,  9.0328961e+02,  1.8790564e+02, -1.0278688e+03,
       1.0629204e+03, -5.3634594e+02, -3.2451913e+02,  9.0207993e+02,
      -8.0267777e+02,  1.2277267e+02,  5.5484886e+02, -6.8584739e+02,
       4.1901310e+02,  7.2779950e+01, -4.3460765e+02,  4.5133152e+02,
      -2.1578787e+02, -2.4628376e+02,  5.3756691e+02, -3.8146878e+02,
       3.4097661e+01,  4.5169871e+02, -5.0575362e+02,  2.7278971e+02,
       9.4493294e+01, -5.2727129e+02,  4.9704432e+02, -2.0290693e+02,
      -2.1943755e+02,  4.6316564e+02, -3.5988557e+02,  1.5354987e+02,
       8.8971222e+01, -2.1177185e+02,  2.9338708e+02, -1.6015893e+02,
      -3.0076244e+01,  1.4282354e+02, -1.8203323e+02, -3.8237583e+01,
       8.0989223e+01, -6.7911301e+01,  4.5876843e+01,  5.8160967e+01,
      -1.2140603e+02,  3.0523591e+01,  1.0177828e+02, -7.1769925e+01,
       6.9697281e+01, -4.4342614e+01, -7.7392269e+01,  1.5965917e+02,
      -1.5467801e+02,  8.0965471e+01,  5.9745822e+00, -7.8890418e+01,
       9.3802490e+01, -1.3176806e+02,  3.1336676e+01,  5.8640239e-01,
      -1.0941547e+02,  8.4322101e+01, -1.0231229e+01, -7.6312076e+01,
       1.2996536e+02, -2.7610121e+01, -8.9190623e+01,  7.8060953e+01,
       6.1715955e+00, -9.5954760e+01,  1.3405798e+02, -1.2502304e+01,
      -1.3552601e+02,  1.1952233e+02, -6.4806966e+00, -1.2142849e+02,
       1.8115437e+02, -1.4257871e+02,  3.1225251e+01,  1.6033227e+02,
      -1.6511810e+02,  1.5240738e+02,  1.2638046e+01, -1.6469393e+02,
       1.4205406e+02, -1.1641621e+02,  4.7933541e+01,  9.6516025e+01,
      -1.1306054e+02,  1.3942714e+02, -2.5241905e+01, -9.3530513e+01,
       8.8433097e+01, -1.0959238e+02,  8.7282861e+01,  3.9977748e+01,
      -7.0411977e+01,  4.1780370e+01,  1.4511943e+01,  3.1125577e+01 ],

    [  6.7796903e+01, -3.8788720e+02,  1.2859183e+03, -7.2609371e+03,
      -1.5704430e+03, -6.2795492e+02,  8.2356565e+02, -8.0022573e+02,
      -3.3787309e+02, -3.5565588e+01, -1.0363528e+02,  3.3897199e+01,
       1.9043187e+02,  1.7876914e+02, -1.5669449e+01, -1.4678018e+02,
      -1.0987901e+02,  3.8138527e+01,  7.5635735e+01,  2.8493842e+01,
      -9.6336587e+00, -2.2372156e+01, -1.9441869e+01, -2.6295536e+01,
      -6.7279947e+00,  3.2810187e+01,  3.3795132e+01,  2.0377831e+00,
      -2.2847912e+01, -2.2110999e+01, -7.4749335e+00,  8.1844245e-01,
       6.9094406e+00,  1.1427591e+01,  1.0664575e+01,  2.5808397e+00,
      -1.1296056e+01, -1.5517110e+01, -3.3427801e+00,  6.2108447e+00,
       1.0440121e+01,  4.7195530e+00, -4.6830353e+00, -4.3321706e+00,
      -4.2308894e-01, -6.6286888e-01, -3.0541198e+00,  1.0803641e+00,
       3.4706447e+00, -1.0277278e+00, -5.8300676e+00, -1.2004259e+00,
       2.9511818e+00,  4.4258015e+00, -1.2798751e-01, -5.6192448e+00,
      -1.1434112e+00,  2.9285732e-01,  2.3558315e+00,  8.5236355e-01,
      -3.6033829e+00,  3.6155080e-01, -1.9369648e-02, -3.0335453e+00,
      -3.9032015e+00, -3.0556779e+00,  3.9409059e-01,  3.5833234e-01,
      -1.1380321e+00,  3.0590734e+00,  2.1256269e+00,  1.1948865e+00,
       1.8418728e+00,  1.9339640e+00, -1.8934727e+00, -7.8211470e+00,
      -7.2315006e+00, -1.6085474e+00,  7.0887341e+00,  1.1289772e+01,
       7.3233713e+00,  5.9236776e-01, -4.6290981e+00, -4.8012930e+00,
      -5.3731448e+00, -4.9951949e+00, -1.0331793e+00,  6.8575568e+00,
       1.1172332e+01,  2.1481470e+00, -9.4111289e+00, -1.2048567e+01,
      -6.2321457e+00,  1.2834480e+00,  4.2595916e+00,  8.2235361e+00,
       9.4292179e+00, -8.2653408e-01, -7.7277654e+00, -5.0381286e+00,
       2.5809009e+00,  2.9671834e+00, -3.1758077e+00, -4.8014638e+00,
      -2.4972384e+00,  3.7424503e+00,  3.1651108e+00,  2.1258770e+00,
      -3.9826224e-01, -5.5310115e+00, -6.7195904e+00, -4.6113744e+00,
       4.7359889e+00,  1.1737180e+01,  4.3792994e+00, -2.4150647e+00,
      -3.2444202e+00, -3.3837188e-01, -2.0142283e+00, -6.2230109e+00 ]

])

X_HAT_TNS = [ X_HAT_TNS_7M5, X_HAT_TNS_10M ]

X_HAT_F_48K_10M = np.array([

       9.4517170e+02,  6.1123091e+02,  2.6864344e+03, -9.2131878e+02,
       3.9506388e+03, -1.6369062e+03,  2.4082498e+04, -3.0114303e+02,
      -7.0038117e+03, -5.0717257e+03, -4.3561383e+03, -4.8809024e+03,
      -2.4479052e+03, -6.2613898e+01,  2.0513506e+03, -1.4311748e+02,
       1.4311748e+02, -9.0641071e+02,  9.5411654e+01, -1.2880573e+03,
       5.2476410e+02, -1.1926457e+03,  4.7705827e+01, -9.5411654e+01,
       3.8164662e+02,  9.5411654e+01,  2.3852913e+02,  9.5411654e+01,
       9.5411654e+01, -9.5411654e+01, -1.9082331e+02, -1.9082331e+02,
       0.0000000e+00,  0.0000000e+00, -4.7705827e+02,  5.7246992e+02,
       2.8623496e+02,  3.3394079e+02, -4.7705827e+01, -9.5411654e+01,
      -9.5411654e+01, -9.5411654e+01, -9.5411654e+01,  0.0000000e+00,
       0.0000000e+00,  2.3852913e+02,  4.7705827e+01,  8.1099906e+02,
      -5.2476410e+02,  0.0000000e+00, -9.5411654e+01,  9.5411654e+01,
      -1.9082331e+02, -4.7705827e+01, -9.5411654e+01,  1.4311748e+02,
       4.7705827e+01,  9.5411654e+01,  0.0000000e+00,  0.0000000e+00,
      -2.3852913e+02, -9.5411654e+01, -2.8623496e+02, -3.3394079e+02,
      -1.4311748e+02, -4.7705827e+01,  4.7705827e+01,  4.7705827e+01,
       1.9082331e+02,  2.3852913e+02,  9.5411654e+01,  0.0000000e+00,
      -9.5411654e+01, -1.9082331e+02, -1.9082331e+02, -9.5411654e+01,
      -1.4311748e+02,  4.7705827e+01,  4.7705827e+01,  2.8623496e+02,
       1.4311748e+02,  2.3852913e+02,  1.9082331e+02,  1.4311748e+02,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00, -4.7705827e+01,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00, -4.7705827e+01,
       4.7705827e+01, -2.3852913e+02,  0.0000000e+00, -9.5411654e+01,
      -2.8623496e+02, -2.3852913e+02, -4.7705827e+01, -4.7705827e+01,
       0.0000000e+00,  0.0000000e+00, -4.7705827e+01,  4.7705827e+01,
      -4.7705827e+01, -9.5411654e+01, -4.7705827e+01, -4.7705827e+01,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  4.7705827e+01,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00, -4.7705827e+01, -4.7705827e+01,  0.0000000e+00,
       0.0000000e+00,  4.7705827e+01,  0.0000000e+00, -4.7705827e+01,
       0.0000000e+00, -4.7705827e+01,  1.4311748e+02, -9.5411654e+01,
      -4.7705827e+01,  4.7705827e+01, -4.7705827e+01, -4.7705827e+01,
       4.7705827e+01,  4.7705827e+01,  0.0000000e+00,  9.5411654e+01,
      -1.9082331e+02,  9.5411654e+01,  1.9082331e+02, -9.5411654e+01,
      -9.5411654e+01,  1.9082331e+02, -1.9082331e+02,  9.5411654e+01,
       4.7705827e+01,  0.0000000e+00, -4.7705827e+01,  0.0000000e+00,
       0.0000000e+00,  4.7705827e+01,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  5.9632284e+00,  5.9632284e+00,  5.9632284e+00,
       5.9632284e+00,  5.9632284e+00, -5.9632284e+00, -5.9632284e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  4.7705827e+01,
      -4.7705827e+01, -4.7705827e+01,  0.0000000e+00,  0.0000000e+00,
      -4.7705827e+01,  0.0000000e+00,  4.7705827e+01,  0.0000000e+00,
       0.0000000e+00, -4.7705827e+01,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  4.7705827e+01,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  4.7705827e+01,  0.0000000e+00, -4.7705827e+01,
      -4.7705827e+01, -4.7705827e+01, -4.7705827e+01,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  4.7705827e+01,  4.7705827e+01,
       4.7705827e+01,  0.0000000e+00,  0.0000000e+00, -4.7705827e+01,
      -9.5411654e+01, -9.5411654e+01, -9.5411654e+01,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  9.5411654e+01,
       4.7705827e+01,  4.7705827e+01,  4.7705827e+01,  4.7705827e+01,
       0.0000000e+00,  0.0000000e+00, -4.7705827e+01, -4.7705827e+01,
      -4.7705827e+01,  0.0000000e+00, -4.7705827e+01, -4.7705827e+01,
      -4.7705827e+01,  0.0000000e+00, -4.7705827e+01,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  4.7705827e+01,  4.7705827e+01,  0.0000000e+00,
       0.0000000e+00,  4.7705827e+01, -4.7705827e+01,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00, -4.7705827e+01,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00, -5.9632284e+00,
       5.9632284e+00, -5.9632284e+00,  5.9632284e+00,  5.9632284e+00,
       5.9632284e+00, -5.9632284e+00,  5.9632284e+00, -5.9632284e+00,
      -5.9632284e+00, -5.9632284e+00,  5.9632284e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  4.7705827e+01,  0.0000000e+00,
       4.7705827e+01,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
      -5.9632284e+00, -5.9632284e+00, -5.9632284e+00,  5.9632284e+00,
      -5.9632284e+00, -5.9632284e+00, -5.9632284e+00,  5.9632284e+00,
       5.9632284e+00,  5.9632284e+00, -5.9632284e+00, -5.9632284e+00,
       5.9632284e+00, -5.9632284e+00,  5.9632284e+00,  5.9632284e+00,
       5.9632284e+00,  5.9632284e+00,  5.9632284e+00, -5.9632284e+00,
       5.9632284e+00,  5.9632284e+00,  5.9632284e+00, -5.9632284e+00,
       5.9632284e+00,  5.9632284e+00, -5.9632284e+00, -5.9632284e+00,
       5.9632284e+00, -5.9632284e+00,  5.9632284e+00,  5.9632284e+00,
       5.9632284e+00,  5.9632284e+00,  5.9632284e+00, -5.9632284e+00,
       5.9632284e+00, -5.9632284e+00,  5.9632284e+00,  5.9632284e+00,
       5.9632284e+00,  5.9632284e+00,  5.9632284e+00, -5.9632284e+00,
      -5.9632284e+00,  5.9632284e+00, -5.9632284e+00,  5.9632284e+00,
       5.9632284e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       4.7705827e+01, -4.7705827e+01,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00, -5.9632284e+00,  5.9632284e+00, -5.9632284e+00,
      -5.9632284e+00, -5.9632284e+00, -5.9632284e+00, -5.9632284e+00,
      -5.9632284e+00, -5.9632284e+00, -5.9632284e+00,  5.9632284e+00,
       5.9632284e+00,  5.9632284e+00, -5.9632284e+00,  5.9632284e+00,
      -5.9632284e+00, -5.9632284e+00,  5.9632284e+00,  5.9632284e+00,
      -5.9632284e+00, -5.9632284e+00,  5.9632284e+00, -5.9632284e+00,
       5.9632284e+00,  5.9632284e+00,  5.9632284e+00, -5.9632284e+00,
      -5.9632284e+00, -5.9632284e+00, -5.9632284e+00,  5.9632284e+00,
       5.9632284e+00, -5.9632284e+00,  5.9632284e+00, -5.9632284e+00,
       5.9632284e+00, -5.9632284e+00, -5.9632284e+00,  5.9632284e+00,
       5.9632284e+00, -5.9632284e+00, -5.9632284e+00, -5.9632284e+00,
       5.9632284e+00, -5.9632284e+00, -5.9632284e+00,  5.9632284e+00,
       5.9632284e+00,  5.9632284e+00,  5.9632284e+00, -5.9632284e+00,
       5.9632284e+00, -5.9632284e+00,  5.9632284e+00, -5.9632284e+00,
      -5.9632284e+00, -5.9632284e+00, -5.9632284e+00, -5.9632284e+00,
      -5.9632284e+00, -5.9632284e+00, -5.9632284e+00,  5.9632284e+00,
       5.9632284e+00, -5.9632284e+00, -5.9632284e+00,  5.9632284e+00,
       5.9632284e+00, -5.9632284e+00,  5.9632284e+00,  5.9632284e+00,
       5.9632284e+00,  5.9632284e+00, -5.9632284e+00, -5.9632284e+00,
       5.9632284e+00,  5.9632284e+00, -5.9632284e+00,  5.9632284e+00,
       5.9632284e+00,  5.9632284e+00,  5.9632284e+00, -5.9632284e+00

])

RC_ORDER_48K_10M = np.array([ 4, 5 ])

RC_I_1_48K_10M = np.array([ 3, 10, 10, 9, 8, 8, 8, 8 ])
RC_I_2_48K_10M = np.array([ 4,  9, 11, 9, 9, 8, 8, 8 ])

X_HAT_TNS_48K_10M = np.array([

       9.4517170e+02,  6.1123091e+02,  2.6864344e+03, -9.2131878e+02,
       3.9506388e+03, -1.6369062e+03,  2.4082498e+04, -3.0114303e+02,
      -7.0038117e+03, -5.0717257e+03, -4.3561383e+03, -4.8809024e+03,
      -2.4479052e+03, -2.2398317e+03, -3.1132766e+01,  2.0090579e+02,
       1.1862042e+03,  5.7338427e+02,  6.1759350e+02, -9.7469088e+02,
      -6.4374897e+02, -2.0211697e+03, -1.7062857e+03, -1.3890132e+03,
      -4.2328832e+02,  3.5577372e+02,  1.1106915e+03,  1.4302110e+03,
       1.4201992e+03,  9.4898618e+02,  2.3607533e+02, -4.7224392e+02,
      -8.4841824e+02, -9.9022181e+02, -1.3448133e+03, -4.1592231e+02,
       2.0637545e+02,  9.3371650e+02,  1.1146864e+03,  9.6858809e+02,
       5.9597157e+02,  9.1928376e+01, -3.7626331e+02, -6.1986247e+02,
      -6.9177579e+02, -3.4667628e+02, -1.0196915e+02,  9.4981560e+02,
       5.0772033e+02,  5.6925480e+02,  2.7206846e+02,  8.9632931e+01,
      -3.0001737e+02, -4.6633999e+02, -5.8788470e+02, -3.5775041e+02,
      -1.5049111e+02,  1.4316978e+02,  2.9621193e+02,  3.6240991e+02,
       9.5814668e+01, -7.8107250e+01, -4.7386669e+02, -8.4266712e+02,
      -9.1319744e+02, -7.8870946e+02, -4.4400310e+02, -5.1962032e+01,
       4.4241817e+02,  8.5744581e+02,  9.6559775e+02,  8.1789237e+02,
       4.2722231e+02, -1.1744557e+02, -6.0875733e+02, -8.7077276e+02,
      -9.9676704e+02, -7.3639748e+02, -3.7055666e+02,  2.7449836e+02,
       6.9342929e+02,  1.0695021e+03,  1.1847878e+03,  1.0572091e+03,
       6.5809175e+02,  2.0789887e+02, -2.0474320e+02, -5.3854975e+02,
      -6.4606559e+02, -5.9469535e+02, -4.1519106e+02, -2.2005318e+02,
       6.5763023e+01, -1.8180468e+00,  1.1794585e+02,  3.7652871e+01,
      -2.6013985e+02, -4.9007184e+02, -5.2184675e+02, -4.8856445e+02,
      -3.1503130e+02, -1.1131138e+02,  2.8240046e+01,  2.1696333e+02,
       2.2485577e+02,  1.2779759e+02,  2.8796892e+01, -9.8980705e+01,
      -1.5201103e+02, -1.6768213e+02, -1.4166855e+02, -3.8081735e+01,
       1.9957705e+01,  7.3452728e+01,  9.9166879e+01,  9.4204641e+01,
       7.0145854e+01, -1.3744063e+01, -9.3049197e+01, -1.1359718e+02,
      -1.1480749e+02, -3.8801493e+01, -5.6439787e-01, -7.4563180e+00,
       2.1645294e+01, -2.1494684e+01,  1.2628610e+02,  1.3469076e+01,
      -3.1053646e+01,  1.0922488e+00, -7.3585334e+01, -1.0982479e+02,
      -4.7186648e+01,  1.5142158e+01,  4.5632808e+01,  1.6549594e+02,
      -3.6083949e+01,  5.8172372e+01,  2.0212810e+02,  6.2799792e+01,
      -3.6264662e+01,  1.1267918e+02, -1.4073978e+02, -3.0415441e+01,
       1.2090131e+00,  5.3715012e+00, -1.1377286e+01, -4.5565675e+00,
      -5.6917226e+00,  4.3600122e+01,  4.1505158e+01,  4.0418175e+01,
       3.0433405e+01,  1.8807364e+01,  8.6851800e+00,  1.3063317e+00,
      -1.6374357e+00, -5.1287186e-01, -8.3181817e+00, -1.3316624e+01,
      -1.1754958e+01, -9.3082910e+00, -4.7125436e+00,  4.7799716e+01,
      -1.4779469e+00, -4.4671371e+01, -4.7792557e+01, -5.2664965e+01,
      -8.7747532e+01, -6.2908560e+01,  7.0728829e+00,  2.9933713e+01,
       5.4684947e+01,  1.2283279e+01,  6.0871825e+00, -9.7832692e+00,
      -2.0805101e+01,  2.5453410e+01,  2.2568456e+01,  2.6671505e+01,
       2.3653407e+01,  6.0861792e+01,  4.5907030e+01, -1.3920574e+01,
      -7.4036162e+01, -1.3377313e+02, -1.7526999e+02, -1.4452509e+02,
      -9.6576994e+01, -3.4117396e+01,  7.2828977e+01,  1.5570493e+02,
       2.1295914e+02,  1.8790695e+02,  1.3270346e+02,  9.1184381e+00,
      -1.5641394e+02, -2.9335242e+02, -3.8817395e+02, -3.2871789e+02,
      -2.0655338e+02, -4.3550786e+01,  1.0968391e+02,  3.0824064e+02,
       3.3663279e+02,  3.0642586e+02,  1.8519446e+02,  5.7390329e+01,
      -1.1630607e+02, -2.0276872e+02, -2.7699576e+02, -2.4543885e+02,
      -1.7669630e+02, -1.9362678e+01,  4.7199101e+01,  1.0516333e+02,
       8.6666990e+01,  8.8657037e+01, -1.5558419e+01, -3.6914314e+01,
      -8.0595009e+01, -7.0566026e+01, -5.9248668e+01, -1.4894165e+01,
       1.6364229e+01,  9.4426516e+01,  1.2736629e+02,  9.8422987e+01,
       5.1256905e+01,  4.3261239e+01, -7.3161150e+01, -8.1108665e+01,
      -9.8148881e+01, -5.9938566e+01, -2.8972996e+01, -2.3262373e+01,
       2.3557895e+01,  4.1063112e+01,  4.9456507e+01,  2.9572229e+01,
       2.1373324e+01, -1.2116045e+01, -1.7920284e+01, -2.5397799e+01,
      -1.5011495e+01, -1.6604903e+01,  5.7220095e+00,  4.2124559e+00,
       9.1295553e+00,  2.0972476e+00,  1.0219314e+01,  1.7408319e+00,
       9.3716408e-01, -4.4755970e+00,  4.3615172e+01,  2.1619284e+01,
       7.0197660e+01,  2.9541340e+01,  2.1495384e+01, -1.5543427e+01,
      -2.9701692e+01, -4.8601263e+01, -4.1417636e+01, -2.0218456e+01,
      -4.1919983e+00,  1.0811220e+01,  1.8282903e+01,  2.9153799e+01,
       2.6637698e+01,  2.0822319e+01, -2.0049692e+00, -1.7515145e+01,
      -1.8945085e+01, -2.6096971e+01, -1.0443081e+01,  2.8259408e+00,
       1.9054250e+01,  2.6408940e+01,  2.9532072e+01,  1.1442436e+01,
       7.5316255e+00, -3.0305702e+00, -5.1747435e+00, -1.8407036e+01,
      -7.2025275e+00, -1.3142410e+00, -8.9104503e-01, -1.9580507e+00,
       9.0242278e+00,  5.0649445e-01,  9.0989185e+00,  8.4374358e+00,
       1.2180369e+01,  9.8043365e+00,  1.0390873e+01, -4.5184465e+00,
      -3.0104815e-01, -1.3617490e+01, -2.8571680e+00,  4.7226441e-02,
       1.0968668e+01,  1.4263590e+01,  1.9054635e+01,  4.5694855e+00,
      -5.1714614e+00, -5.6951038e+00, -1.5868058e+01, -6.3038212e+00,
       6.0252467e-01,  5.9026641e+00,  7.8745777e+00,  8.8744978e+00,
       5.3462027e+01, -1.8654382e+01, -4.3353244e+00, -2.6929767e+01,
      -1.5311914e+01, -2.7217677e+01,  3.5149041e-01, -3.3211226e+00,
       7.7038597e+00,  2.4167585e+00,  3.1307056e+00, -6.1700798e+00,
      -9.6229135e+00, -1.5399820e+01, -1.5266153e+01, -2.7143843e+00,
       8.4318705e+00,  1.8387062e+01,  1.1142869e+01,  1.6194808e+01,
      -9.1085598e-01, -9.8865640e+00, -9.6606173e+00, -4.4966142e+00,
      -1.0006828e+01, -8.3653492e+00,  3.2044245e+00, -2.9444546e-01,
       1.0908085e+01,  1.3460563e+01,  1.7296403e+01,  2.3331346e+00,
      -6.4643323e+00, -1.8141900e+01, -2.2182809e+01, -1.1540102e+01,
       1.1288138e+00,  2.4890144e+00,  1.6559739e+01,  8.5186198e+00,
       1.5415016e+01, -1.5380758e+00, -8.2713929e+00, -8.0722274e+00,
      -2.5013191e+00, -8.9116220e+00, -7.8813678e+00, -9.3233771e+00,
       3.7714591e+00, -1.8991746e+00, -8.2806999e-01,  5.9195021e+00,
       1.1264459e+01,  1.3340521e+01,  1.4254734e+01,  7.3923870e-01,
       2.7992830e+00, -1.2191542e+01, -4.1559084e+00, -1.4491802e+01,
      -1.1202358e+01, -1.3530333e+01, -8.4296049e+00, -7.8650114e+00,
      -4.0242598e+00, -3.9190447e+00, -3.2052868e+00,  6.8414475e+00,
       1.2298364e+01,  4.5966329e+00, -2.2198027e+00,  1.4780361e+00,
       2.6041179e+00, -5.9986780e+00,  1.8848742e+00,  5.1985761e+00,
       1.0933162e+01,  1.2576045e+01,  2.6295117e+00, -6.3792422e+00,
      -3.2379866e+00, -1.0344676e+00, -7.5143096e+00,  2.5669904e+00,
       7.6423118e+00,  1.4123706e+01,  1.5442459e+01,  4.3051286e+00

])

### C.4.1.7  Spectral shaping

X_HAT_SNS_10M = np.array([

    [  0.0000000e+00,  3.2120898e+04, -2.7432096e+04,  9.5098925e+04,
      -1.1435865e+05, -2.2381382e+05, -1.8369247e+03, -2.0115639e+04,
      -1.5792914e+04,  0.0000000e+00, -1.5609063e+04,  4.0875220e+03,
      -1.0230315e+04,  1.9369297e+03, -4.1865185e+03, -1.0063143e+03,
       5.6734153e+02, -2.8286784e+03,  2.4858485e+03, -2.7806581e+03,
       2.2044492e+03, -1.4655656e+03,  7.9623905e+02,  1.4627465e+02,
      -6.8421307e+02,  1.2472853e+03, -1.3595558e+03,  1.3417640e+03,
      -1.0434807e+03,  5.9927829e+02, -2.2923931e+02, -2.9150665e+02,
       5.1660528e+02, -7.4849706e+02,  7.9101280e+02, -7.5693454e+02,
       5.8965376e+02, -3.7385221e+02,  5.5646432e+01,  2.0409315e+02,
      -3.8684762e+02,  4.7002840e+02, -5.1019782e+02,  4.6970753e+02,
      -3.6882983e+02,  2.3887213e+02, -3.0765937e+01, -1.2545710e+02,
       2.5752773e+02, -3.0607937e+02,  3.0617821e+02, -3.0270399e+02,
       2.4452840e+02, -1.8729289e+02,  2.9629779e+01,  8.1156654e+01,
      -1.9740547e+02,  2.3802067e+02, -2.9662172e+02,  2.2717944e+02,
      -1.6519442e+02,  3.2859099e+01,  3.8288258e+01, -1.4380181e+02,
       1.5993592e+02, -1.7522348e+02,  1.3978068e+02, -1.2885956e+02,
       5.2007743e+01, -5.3744028e+00, -6.3481105e+01,  1.1005588e+02,
      -1.3761380e+02,  1.1149518e+02, -9.6552293e+01,  2.4175349e+01,
       2.7349618e+01, -9.2532521e+01,  9.6809951e+01, -9.2329191e+01,
       8.1204030e+01, -2.8757018e+01,  1.7629374e+01,  5.0296426e+01,
      -6.9554039e+01,  1.0643991e+02, -7.7045965e+01,  7.6992232e+01,
      -1.5336408e+01,  6.7350324e+00,  5.2884931e+01, -4.5033515e+01,
       7.8565630e+01, -7.0077921e+01,  4.5653385e+01, -3.5627746e+01,
      -1.8715552e+01,  1.4082202e+01, -6.2364732e+01,  3.5916392e+01,
      -6.2514422e+01,  1.6978510e+01, -1.7220883e+00, -3.7254517e+01,
       3.5050409e+01, -3.2086295e+01,  1.2895849e+01, -1.4997608e+01,
      -1.1409978e+01,  1.6692500e+00, -3.1028013e+01,  8.5596145e+00,
      -8.3018444e+00,  6.7010013e+00, -1.0178138e+01,  1.8975872e+01,
       1.6344865e+00,  1.1210482e+01, -4.7611165e+00,  1.8818163e+01,
       1.0822497e+00, -8.9424468e+00,  2.7196773e+01, -3.6023020e+01,
       4.1833912e+01, -2.2996359e+01,  3.1520683e+00,  2.4451081e+01,
      -2.3663155e+01,  4.4716235e+01, -4.3693613e+01,  2.9552445e+01,
      -5.9899945e+00, -2.4756795e+01,  4.6495022e+01, -4.3040437e+01,
       3.1981524e+01, -2.8813669e+01, -1.1193668e+00, -2.6891294e+00,
      -1.1789293e+01,  2.4012319e+01, -1.5122490e+01,  9.7992507e+00,
      -9.3635505e+00,  1.3408697e+01,  3.8993996e+00,  6.5986643e+00,
      -3.4130004e+00, -5.4797703e+00,  1.7638234e+01, -8.9968811e+00,
      -1.3124018e+00, -4.9433015e+00, -1.5392870e+01, -5.3001786e+00,
      -2.1345573e+01,  1.7203866e+01, -3.5997789e+01,  1.1646997e+01 ],

    [ -6.2955623e+03, -1.4566565e+04,  5.0622202e+03, -7.0393645e+04,
       8.5577202e+04,  2.5855135e+05, -3.5335070e+04,  5.4655582e+04,
      -1.1097295e+04,  1.8604727e+04,  1.7284056e+03,  6.9845121e+03,
       1.3066905e+03,  4.1207686e+03,  1.2132541e+03,  1.5347593e+03,
       6.1036346e+02,  6.5193569e+02,  1.5477285e+02,  1.0810302e+02,
      -1.3121390e+01, -5.1968778e+01, -6.2665661e+01, -8.4220873e+01,
      -4.1495195e+01, -3.5833798e+01, -1.0853816e+01,  4.1261732e-01,
       6.7539065e+00,  1.2953817e+01,  1.8822817e+01,  2.0173803e+01,
       1.6734007e+01,  1.3492868e+01,  8.9088009e+00,  3.1828720e+00,
      -1.8693963e+00, -5.8104021e+00, -6.9265975e+00, -7.6077959e+00,
      -7.9011791e+00, -5.7174692e+00, -4.5976263e+00, -1.5032819e+00,
      -4.8178832e-01,  1.0459392e+00,  2.9544513e+00,  2.7006970e+00,
       2.6614506e+00,  1.8287519e+00,  2.1513094e+00,  1.5782394e+00,
       1.0890240e+00, -3.7646581e-01, -1.1518203e+00, -1.8534533e+00,
      -2.3900791e+00, -1.5304619e+00, -2.1511073e+00, -6.8686891e-01,
      -1.0939425e+00,  4.4580685e-01,  1.0694775e+00,  1.7112407e+00,
       2.1522574e+00,  1.1984063e+00,  1.8933761e+00,  1.5608228e+00,
       1.4612352e+00,  8.7220397e-02, -6.3880600e-01, -1.3031673e+00,
      -1.9233091e+00, -1.2909908e+00, -2.1448096e+00, -7.0978302e-01,
      -7.4823965e-02, -3.7868573e-01,  1.0642643e+00,  1.5420494e+00,
       2.4281898e+00,  1.1190134e+00,  2.0010149e+00,  4.8369747e-01,
      -3.1302463e-01,  1.3884389e-01, -1.3802746e+00, -1.7576352e+00,
      -2.3441435e+00, -2.4065591e+00, -2.4896924e+00, -1.0565690e+00,
      -1.4455632e+00,  2.7527927e-01, -3.0281700e-01, -5.8686912e-02,
      -2.3895075e-01, -4.9317896e-01,  5.9464861e-01,  7.8267846e-01,
      -3.1558969e-02,  8.7786664e-01, -1.6042710e-01, -4.2013050e-01,
       1.9560599e-01, -8.1876351e-01, -9.6031363e-01, -1.4201787e+00,
      -3.1915476e-01,  8.9297143e-02, -5.6412436e-01, -5.3261760e-01,
       4.2030523e-01, -2.3510454e-01, -4.6638224e-01,  2.9054917e-01,
       6.4077498e-01,  9.7649360e-01,  1.1359409e+00,  1.4250163e+00,
       5.4650505e-01,  3.8070710e-02, -5.7836964e-01, -7.7347222e-02,
      -1.0712427e+00, -3.2900133e-01, -6.6441638e-02,  4.6435375e-01,
      -1.8280947e-01,  7.1574422e-01, -1.6152798e-02, -2.6188697e-01,
      -7.0437063e-01,  3.2250148e-02, -8.0333729e-01, -1.0595566e+00,
      -1.2784303e+00, -2.4897303e-01, -9.0777061e-01, -1.0166946e+00,
      -1.0877187e+00,  1.0276390e-01,  6.1843142e-01,  1.0990088e+00,
       4.1640936e-01,  2.1611445e-01,  1.0343162e+00, -1.6762504e-01,
       6.2546212e-01,  5.9995270e-01,  9.8434057e-01,  1.1432576e+00,
       1.2787609e+00,  3.0368561e-01,  9.9535729e-01,  9.7104477e-01,
       1.0977262e+00, -1.9294725e-01,  5.8743153e-01,  6.9473485e-01 ]
])

X_HAT_SNS_7M5 = np.array([

    [ -2.7097714e+04,  3.0803384e+04, -1.0482091e+05, -1.6066697e+05,
       7.3848155e+03,  6.1515964e+04, -4.0837362e+03,  1.4874654e+03,
       8.2375155e+03, -8.1555674e+03,  3.0555770e+03,  1.6748461e+03,
      -3.9585632e+03,  3.4593583e+03, -7.9211504e+02, -1.5776652e+03,
       2.4448549e+03, -1.6832871e+03,  8.3907446e+01,  1.2903730e+03,
      -1.5923535e+03,  9.5931605e+02,  1.9128372e+02, -9.8357603e+02,
       9.5609940e+02, -4.5350218e+02, -2.5320962e+02,  6.3762292e+02,
      -5.1397160e+02,  7.1216163e+01,  3.1269611e+02, -4.0275234e+02,
       2.5638994e+02,  4.6403201e+01, -2.7843009e+02,  2.8914418e+02,
      -1.3395170e+02, -1.5288222e+02,  3.2333756e+02, -2.2944713e+02,
       1.9872415e+01,  2.6325395e+02, -2.8392752e+02,  1.5314277e+02,
       5.0798626e+01, -2.8345564e+02,  2.5587542e+02, -1.0445526e+02,
      -1.0817500e+02,  2.2832437e+02, -1.6921255e+02,  7.2196740e+01,
       3.9741076e+01, -9.4592848e+01,  1.2449531e+02, -6.7961530e+01,
      -1.2124324e+01,  5.7574971e+01, -7.1197322e+01, -1.4955585e+01,
       3.1388922e+01, -2.6320323e+01,  1.7618904e+01,  2.2336595e+01,
      -4.6625728e+01,  1.1616015e+01,  3.8732599e+01, -2.7312663e+01,
       2.6048066e+01, -1.6572229e+01, -2.8923924e+01,  5.8075573e+01,
      -5.6263689e+01,  2.9450962e+01,  2.1151792e+00, -2.7929547e+01,
       3.3208863e+01, -4.5403546e+01,  1.0797732e+01,  2.0205768e-01,
      -3.7035922e+01,  2.8542095e+01, -3.4631575e+00, -2.5611045e+01,
       4.3617589e+01, -9.2662146e+00, -2.9678561e+01,  2.5975116e+01,
       2.0536248e+00, -3.1929357e+01,  4.4228879e+01, -4.1248038e+00,
      -4.4713214e+01,  3.9433226e+01, -2.1402050e+00, -4.0100915e+01,
       5.9824970e+01, -4.7085628e+01,  1.0420556e+01,  5.3506419e+01,
      -5.5103555e+01,  5.0861706e+01,  4.2620261e+00, -5.5541010e+01,
       4.7905991e+01, -3.9259942e+01,  1.6335294e+01,  3.2891742e+01,
      -3.8529956e+01,  4.7515440e+01, -8.6928230e+00, -3.2210096e+01,
       3.0454645e+01, -3.7741492e+01,  3.0058525e+01,  1.3912599e+01,
      -2.4503972e+01,  1.4539927e+01,  5.0502807e+00,  1.0831968e+01 ],

    [  3.8609917e+03, -2.2089936e+04,  6.2396694e+04, -2.5577568e+05,
      -4.0161241e+04, -1.1658208e+04,  9.9426556e+03, -5.6272992e+03,
      -1.3839628e+03, -8.4856310e+01, -1.6119528e+02,  3.8468595e+01,
       1.5768120e+02,  1.0800171e+02, -8.1867250e+00, -7.8607304e+01,
      -6.0318312e+01,  2.1460378e+01,  4.3791334e+01,  1.7039175e+01,
      -5.9501127e+00, -1.4271781e+01, -1.2583485e+01, -1.6962650e+01,
      -4.3255972e+00,  2.1024124e+01,  2.1440266e+01,  1.2715073e+00,
      -1.4021429e+01, -1.3345628e+01, -4.3592289e+00,  4.5305297e-01,
       3.6304646e+00,  5.6994442e+00,  5.3601809e+00,  1.2971700e+00,
      -6.0746373e+00, -8.3445774e+00, -1.9233529e+00,  3.5735662e+00,
       6.4270894e+00,  2.9054250e+00, -2.9987419e+00, -2.7740687e+00,
      -2.7396267e-01, -4.2922732e-01, -1.9998336e+00,  7.0742100e-01,
       2.2980847e+00, -6.8050918e-01, -3.7563574e+00, -7.7344365e-01,
       1.7803940e+00,  2.6700051e+00, -7.2296053e-02, -3.1741317e+00,
      -6.0475055e-01,  1.5489233e-01,  1.2469224e+00,  4.5114909e-01,
      -2.0399596e+00,  2.0468239e-01, -1.1728671e-02, -1.8368664e+00,
      -2.3634589e+00, -1.9790219e+00,  2.5523434e-01,  2.3207537e-01,
      -7.6437315e-01,  2.0546640e+00,  1.4277033e+00,  8.0700716e-01,
       1.2439713e+00,  1.3061682e+00, -1.2859071e+00, -5.3115466e+00,
      -4.9111022e+00, -1.0984600e+00,  4.8408213e+00,  7.7096653e+00,
       4.8540011e+00,  3.9262706e-01, -3.0682108e+00, -2.9814266e+00,
      -3.3365255e+00, -3.1018325e+00, -6.0106119e-01,  3.9894443e+00,
       6.4996032e+00,  1.2497035e+00, -5.1293438e+00, -6.5668257e+00,
      -3.3967039e+00,  6.9951716e-01,  2.2622003e+00,  4.3673873e+00,
       5.0077054e+00, -4.3895891e-01, -4.1593548e+00, -2.7116978e+00,
       1.3891315e+00,  1.5970423e+00, -1.7323481e+00, -2.6191154e+00,
      -1.3622003e+00,  2.0414418e+00,  1.7497613e+00,  1.1752440e+00,
      -2.2017045e-01, -3.0576971e+00, -3.7647976e+00, -2.5836234e+00,
       2.6534415e+00,  6.5760121e+00,  2.4535983e+00, -1.3713128e+00,
      -1.8422343e+00, -1.9213303e-01, -1.1437114e+00, -3.5335263e+00 ],
])

X_HAT_SNS = [ X_HAT_SNS_7M5, X_HAT_SNS_10M ]

### C.4.1.8  MDCT

T_HAT_MDCT_10M = np.array([

    [  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
       6.1525095e-01,  1.7117620e+00, -1.6200436e-01, -2.1107548e+00,
       9.1636911e-01,  8.7583398e+00,  1.4291828e+01,  1.4615904e+01,
       1.9312730e+01,  2.2178311e+01,  2.1178760e+01,  1.9939021e+01,
       3.2745193e+01,  3.1385007e+01,  4.5642888e+01,  3.8185668e+01,
       4.3452271e+01,  3.0174130e+01,  2.7510416e+01,  3.9048290e+00,
      -4.1911157e+00, -3.4032526e+01, -4.3089457e+01, -7.1817465e+01,
      -7.3319439e+01, -1.0231340e+02, -8.2645833e+01, -1.0207070e+02,
      -7.4253115e+01, -9.5269932e+01, -1.2210097e+02, -1.3216707e+02,
      -1.2665681e+02, -1.6033791e+02, -1.3067613e+02, -1.8796611e+01,
       6.2097263e+01,  7.2290617e+00, -1.2550979e+02, -8.9649115e+01,
       7.6135408e+02,  2.7072170e+03,  6.1558256e+03,  9.6522574e+03,
       1.2566221e+04,  1.5421574e+04,  1.8329565e+04,  2.1102710e+04,
       2.3323039e+04,  2.5451924e+04,  2.7365468e+04,  2.8891223e+04,
       3.0341813e+04,  3.1380038e+04,  3.2134454e+04,  3.2606958e+04,
       3.2683895e+04,  3.2518324e+04,  3.2036626e+04,  3.1278795e+04,
       3.0136453e+04,  2.8694957e+04,  2.7009676e+04,  2.5075921e+04,
       2.2940269e+04,  2.0564299e+04,  1.7956441e+04,  1.5202722e+04,
       1.2352834e+04,  9.3639445e+03,  6.2714619e+03,  3.1112267e+03,
      -1.0075267e+02, -3.3115389e+03, -6.4954073e+03, -9.6110111e+03,
      -1.2594144e+04, -1.5477538e+04, -1.8214406e+04, -2.0797170e+04,
      -2.3195615e+04, -2.5355124e+04, -2.7302557e+04, -2.8919494e+04,
      -3.0273227e+04, -3.1357707e+04, -3.2152224e+04, -3.2625545e+04,
      -3.2802109e+04, -3.2640407e+04, -3.2172861e+04, -3.1393517e+04,
      -3.0316009e+04, -2.8922986e+04, -2.7290775e+04, -2.5351571e+04,
      -2.3183937e+04, -2.0795771e+04, -1.8212609e+04, -1.5453610e+04,
      -1.2543978e+04, -9.4923186e+03, -6.3663932e+03, -3.1740539e+03,
       4.2407582e+01,  3.2479359e+03,  6.4369448e+03,  9.5564107e+03,
       1.2589389e+04,  1.5494437e+04,  1.8238168e+04,  2.0814647e+04,
       2.3183440e+04,  2.5340307e+04,  2.7236643e+04,  2.8879098e+04,
       3.0219987e+04,  3.1304367e+04,  3.2073455e+04,  3.2541098e+04,
       3.2660415e+04,  3.2490788e+04,  3.2004309e+04,  3.1215306e+04,
       3.0131477e+04,  2.8751611e+04,  2.7114762e+04,  2.5198099e+04,
       2.3044166e+04,  2.0655799e+04,  1.8068458e+04,  1.5312555e+04,
       1.2400688e+04,  9.3834439e+03,  6.2670869e+03,  3.1043555e+03,
      -9.0364008e+01, -3.2949280e+03, -6.4558716e+03, -9.5736656e+03,
      -1.2593429e+04, -1.5487321e+04, -1.8231003e+04, -2.0808229e+04,
      -2.3191386e+04, -2.5327915e+04, -2.7226934e+04, -2.8860268e+04,
      -3.0235350e+04, -3.1313467e+04, -3.2092397e+04, -3.2585580e+04,
      -3.2777259e+04, -3.2606534e+04, -3.2146234e+04, -3.1383325e+04,
      -3.0295384e+04, -2.8926934e+04, -2.7297985e+04, -2.5343669e+04,
      -2.3152305e+04, -2.0670493e+04, -1.7934304e+04, -1.4904360e+04,
      -1.1690361e+04, -8.2034060e+03, -4.4966301e+03, -5.9716414e+02,
       3.3525737e+03,  7.3668269e+03,  1.1354417e+04,  1.5112467e+04,
       1.8623854e+04,  2.1896001e+04,  2.4687380e+04,  2.6938007e+04,
       2.8626837e+04,  2.9696113e+04,  3.0136274e+04,  2.9938040e+04,
       2.9086081e+04,  2.7633076e+04,  2.5690739e+04,  2.3353541e+04,
       2.0706676e+04,  1.7862611e+04,  1.4893850e+04,  1.1890794e+04,
       9.0400724e+03,  6.3496693e+03,  3.8416289e+03,  1.6255098e+03,
      -2.6829587e+02, -1.8648283e+03, -3.1265144e+03, -4.0739027e+03,
      -4.7217057e+03, -5.1308259e+03, -5.2994797e+03, -5.2717085e+03,
      -5.0702632e+03, -4.7404796e+03, -4.2926332e+03, -3.7668674e+03,
      -3.1863326e+03, -2.5717341e+03, -1.9487052e+03, -1.3380710e+03,
      -7.5480786e+02, -2.1731911e+02,  2.6258546e+02,  6.7531993e+02,
       1.0134680e+03,  1.2732878e+03,  1.4518662e+03,  1.5497281e+03,
       1.5671087e+03,  1.5098872e+03,  1.3843562e+03,  1.1969411e+03,
       9.5570061e+02,  6.6987624e+02,  3.5217663e+02,  9.8694010e+00,
      -3.4394935e+02, -6.9967997e+02, -1.0489605e+03, -1.3797643e+03,
      -1.6859141e+03, -1.9573045e+03, -2.1892388e+03, -2.3761776e+03,
      -2.5135134e+03, -2.6015039e+03, -2.6377063e+03, -2.6272971e+03,
      -2.5708325e+03, -2.4734547e+03, -2.3405452e+03, -2.1774990e+03,
      -1.9928710e+03, -1.7900465e+03, -1.5787667e+03, -1.3647693e+03,
      -1.1565972e+03, -9.5713834e+02, -7.7246019e+02, -6.0536929e+02,
      -4.5916520e+02, -3.3470350e+02, -2.3251725e+02, -1.5151486e+02,
      -9.0224117e+01, -4.6447782e+01, -1.7377196e+01, -1.6206249e-01,
       8.2694380e+00,  1.0620902e+01,  9.2650945e+00,  5.7978679e+00 ],

    [ -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -5.8307275e+01, -1.5961611e+02, -3.2822433e+02, -5.8684193e+02,
      -9.5063064e+02, -1.4274461e+03, -2.0154696e+03, -2.7009630e+03,
      -3.4569464e+03, -4.2430564e+03, -5.0068652e+03, -5.6873576e+03,
      -6.2169269e+03, -6.5268185e+03, -6.5539697e+03, -6.2419962e+03,
      -5.5512117e+03, -4.4569448e+03, -2.9581517e+03, -1.0742654e+03,
       1.1530263e+03,  3.6594640e+03,  6.3646353e+03,  9.1749853e+03,
       1.1985116e+04,  1.4691422e+04,  1.7187008e+04,  1.9378907e+04,
       2.1179777e+04,  2.2522740e+04,  2.3359818e+04,  2.3657684e+04,
       2.3405693e+04,  2.2611255e+04,  2.1297773e+04,  1.9499385e+04,
       1.7264572e+04,  1.4646235e+04,  1.1703370e+04,  8.4960382e+03,
       5.0883571e+03,  1.5400010e+03, -2.0852434e+03, -5.7317898e+03,
      -9.3443119e+03, -1.2867162e+04, -1.6250847e+04, -1.9448892e+04,
      -2.2417993e+04, -2.5119550e+04, -2.7519523e+04, -2.9589531e+04,
      -3.1306279e+04, -3.2653256e+04, -3.3615026e+04, -3.4186391e+04,
      -3.4364519e+04, -3.4152077e+04, -3.3554064e+04, -3.2586709e+04,
      -3.1263292e+04, -2.9603089e+04, -2.7630063e+04, -2.5371579e+04,
      -2.2853607e+04, -2.0109903e+04, -1.7174483e+04, -1.4079475e+04,
      -1.0861936e+04, -7.5574424e+03, -4.2020683e+03, -8.2997756e+02,
       2.5249878e+03,  5.8295399e+03,  9.0523976e+03,  1.2164914e+04,
       1.5141205e+04,  1.7954597e+04,  2.0583186e+04,  2.3005366e+04,
       2.5205897e+04,  2.7165327e+04,  2.8871803e+04,  3.0312108e+04,
       3.1479927e+04,  3.2363955e+04,  3.2960351e+04,  3.3265900e+04,
       3.3277052e+04,  3.2993786e+04,  3.2419707e+04,  3.1554084e+04,
       3.0408074e+04,  2.8986068e+04,  2.7298583e+04,  2.5360460e+04,
       2.3186540e+04,  2.0794689e+04,  1.8206602e+04,  1.5444741e+04,
       1.2535402e+04,  9.5045150e+03,  6.3825434e+03,  3.1961404e+03,
      -1.9030604e+01, -3.2350097e+03, -6.4178539e+03, -9.5376996e+03,
      -1.2564818e+04, -1.5468879e+04, -1.8222757e+04, -2.0799617e+04,
      -2.3177492e+04, -2.5329860e+04, -2.7239548e+04, -2.8887388e+04,
      -3.0257022e+04, -3.1336972e+04, -3.2115659e+04, -3.2585521e+04,
      -3.2741691e+04, -3.2583544e+04, -3.2112334e+04, -3.1332786e+04,
      -3.0253039e+04, -2.8881031e+04, -2.7231757e+04, -2.5321143e+04,
      -2.3168095e+04, -2.0790131e+04, -1.8212051e+04, -1.5458479e+04,
      -1.2554090e+04, -9.5278709e+03, -6.4088050e+03, -3.2282766e+03,
      -1.5127187e+01,  3.1974595e+03,  6.3806758e+03,  9.5013370e+03,
       1.2533377e+04,  1.5443257e+04,  1.8203988e+04,  2.0793445e+04,
       2.3180292e+04,  2.5344634e+04,  2.7265053e+04,  2.8919963e+04,
       3.0294320e+04,  3.1376795e+04,  3.2154178e+04,  3.2622959e+04,
       3.2776568e+04,  3.2617286e+04,  3.2145842e+04,  3.1360660e+04,
       3.0278348e+04,  2.8901877e+04,  2.7241369e+04,  2.5301056e+04,
       2.3077170e+04,  2.0602593e+04,  1.7860920e+04,  1.4845935e+04,
       1.1575770e+04,  8.0714330e+03,  4.3590833e+03,  4.8619425e+02,
      -3.4812252e+03, -7.4811451e+03, -1.1420224e+04, -1.5218969e+04,
      -1.8767216e+04, -2.1982348e+04, -2.4766842e+04, -2.7038512e+04,
      -2.8729372e+04, -2.9797530e+04, -3.0215913e+04, -2.9983529e+04,
      -2.9130275e+04, -2.7706100e+04, -2.5778477e+04, -2.3437188e+04,
      -2.0783932e+04, -1.7920105e+04, -1.4952584e+04, -1.1982327e+04,
      -9.1037807e+03, -6.3864700e+03, -3.9012432e+03, -1.6878870e+03,
       2.1843895e+02,  1.8077853e+03,  3.0762163e+03,  4.0339692e+03,
       4.7031968e+03,  5.1098317e+03,  5.2820716e+03,  5.2546005e+03,
       5.0617380e+03,  4.7306477e+03,  4.2889148e+03,  3.7647491e+03,
       3.1840313e+03,  2.5717028e+03,  1.9493478e+03,  1.3380429e+03,
       7.5567371e+02,  2.1773747e+02, -2.6311651e+02, -6.7663706e+02,
      -1.0155643e+03, -1.2750705e+03, -1.4528246e+03, -1.5489867e+03,
      -1.5659952e+03, -1.5076498e+03, -1.3804175e+03, -1.1912334e+03,
      -9.4848033e+02, -6.6207374e+02, -3.4175876e+02,  1.6521653e+00,
       3.5767927e+02,  7.1550189e+02,  1.0651058e+03,  1.3968326e+03,
       1.7019804e+03,  1.9728594e+03,  2.2034762e+03,  2.3889564e+03,
       2.5257870e+03,  2.6127289e+03,  2.6495795e+03,  2.6378966e+03,
       2.5805079e+03,  2.4818035e+03,  2.3472272e+03,  2.1829178e+03,
       1.9955916e+03,  1.7924019e+03,  1.5804111e+03,  1.3664419e+03,
       1.1569292e+03,  9.5724044e+02,  7.7214172e+02,  6.0521399e+02,
       4.5883364e+02,  3.3442067e+02,  2.3213371e+02,  1.5121914e+02,
       9.0047461e+01,  4.6310025e+01,  1.7308037e+01,  7.2726310e-02,
      -8.3269806e+00, -1.0647845e+01, -9.2769990e+00, -5.7939041e+00 ],

])

T_HAT_MDCT_7M5 = np.array([

    [ -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
       9.0382948e-01,  2.8563300e+00,  2.0863167e+00,  3.2605273e+00,
       1.8231017e+00, -2.6473031e+00, -7.7420704e+00, -1.6971743e+01,
      -4.4169569e+00,  4.7473387e+00,  7.9882732e+00,  2.1090757e+00,
       6.9477046e+00,  7.6294361e+00,  4.5069158e+00,  1.1288109e+00,
       5.5301798e-01, -1.2320805e+00,  1.2696965e+01,  1.7998129e+01,
       1.9997378e+01,  2.3310802e+01,  3.4116671e+01,  3.1619222e+01,
       2.3643252e+01,  2.2595989e+01,  2.4150879e+01,  1.7561939e+01,
       2.4167995e+01,  2.1868269e+01,  1.2021561e+01,  1.0810360e+01,
      -1.1321816e+01, -1.3811836e+01, -2.7571991e+01, -3.3459505e+01,
      -2.6720233e+01, -4.0425004e+01, -4.1666697e+01, -4.8106995e+01,
      -7.1121739e+01, -8.5018856e+01, -6.4519501e+01, -6.1651047e+01,
      -6.2001672e+01, -4.9054098e+01,  5.3605147e+00, -2.7222279e+00,
      -6.3200946e+00, -2.8873822e+01, -5.6314175e+01, -5.9551902e+01,
      -2.1183627e+01, -9.5007617e+01, -6.7674879e+01,  7.6546124e+01,
       3.6355638e+02,  2.0908440e+02,  9.2290767e+01, -8.4453487e+01,
      -2.0810832e+02, -1.9235273e+02, -4.0634578e+02, -2.2011977e+02,
       6.2920459e+02,  3.1481663e+03,  6.2343351e+03,  9.4022080e+03,
       1.2520451e+04,  1.5313131e+04,  1.8128985e+04,  2.0762454e+04,
       2.3084787e+04,  2.5275848e+04,  2.7095495e+04,  2.8665301e+04,
       3.0094623e+04,  3.1202047e+04,  3.2006678e+04,  3.2461623e+04,
       3.2568832e+04,  3.2408327e+04,  3.1961953e+04,  3.1146555e+04,
       3.0073949e+04,  2.8725124e+04,  2.7099832e+04,  2.5196695e+04,
       2.3022972e+04,  2.0643354e+04,  1.8079103e+04,  1.5352852e+04,
       1.2476728e+04,  9.4135962e+03,  6.2948219e+03,  3.1010477e+03,
      -9.0897787e+01, -3.3383673e+03, -6.5093586e+03, -9.6214110e+03,
      -1.2638625e+04, -1.5564780e+04, -1.8289238e+04, -2.0877731e+04,
      -2.3274493e+04, -2.5456613e+04, -2.7372293e+04, -2.9018289e+04,
      -3.0399516e+04, -3.1474248e+04, -3.2213279e+04, -3.2686770e+04,
      -3.2836146e+04, -3.2655386e+04, -3.2161995e+04, -3.1363572e+04,
      -3.0296725e+04, -2.8893704e+04, -2.7226660e+04, -2.5303018e+04,
      -2.3185989e+04, -2.0860640e+04, -1.8388180e+04, -1.5768019e+04,
      -1.3017648e+04, -1.0256225e+04, -7.4288937e+03, -4.6404303e+03,
      -1.8479563e+03,  8.9695589e+02,  3.6050165e+03,  6.1774761e+03,
       8.5802409e+03,  1.0843314e+04,  1.2957357e+04,  1.4915752e+04,
       1.6688121e+04,  1.8234848e+04,  1.9622853e+04,  2.0843143e+04,
       2.1874370e+04,  2.2730604e+04,  2.3433492e+04,  2.3972978e+04,
       2.4361621e+04,  2.4632443e+04,  2.4789213e+04,  2.4834368e+04,
       2.4817473e+04,  2.4691904e+04,  2.4506825e+04,  2.4260930e+04,
       2.3952544e+04,  2.3583514e+04,  2.3158707e+04,  2.2682736e+04,
       2.2114814e+04,  2.1501602e+04,  2.0804765e+04,  2.0028682e+04,
       1.9176325e+04,  1.8229068e+04,  1.7186098e+04,  1.6064305e+04,
       1.4857345e+04,  1.3566101e+04,  1.2216378e+04,  1.0824025e+04,
       9.3566686e+03,  7.8525218e+03,  6.3408636e+03,  4.8352443e+03,
       3.3500951e+03,  1.8786022e+03,  4.4842764e+02, -8.8483949e+02,
      -2.1241927e+03, -3.2448273e+03, -4.2656467e+03, -5.1440714e+03,
      -5.9062863e+03, -6.5125637e+03, -6.9672797e+03, -7.2820785e+03,
      -7.4483481e+03, -7.4911171e+03, -7.4042834e+03, -7.2043747e+03,
      -6.8934126e+03, -6.5065747e+03, -6.0545237e+03, -5.5534407e+03,
      -5.0174587e+03, -4.4641711e+03, -3.9151968e+03, -3.3732391e+03,
      -2.8528068e+03, -2.3664418e+03, -1.9195477e+03, -1.5167365e+03,
      -1.1638365e+03, -8.6204669e+02, -6.1212532e+02, -4.0852043e+02,
      -2.5108888e+02, -1.3435049e+02, -5.3189240e+01, -1.0851528e+00,
       2.6728439e+01,  3.7348684e+01,  3.5932093e+01,  2.7568411e+01 ],

    [ -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00, -0.0000000e+00, -0.0000000e+00, -0.0000000e+00,
      -0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
       7.4140637e+01,  1.4259396e+02,  2.4292388e+02,  3.8197729e+02,
       5.6596483e+02,  8.0010169e+02,  1.0881546e+03,  1.4325879e+03,
       1.8337245e+03,  2.2902305e+03,  2.7978462e+03,  3.3513844e+03,
       3.9420820e+03,  4.5597592e+03,  5.1914056e+03,  5.8224872e+03,
       6.4359174e+03,  7.0133507e+03,  7.5344894e+03,  7.9783768e+03,
       8.3235628e+03,  8.5499202e+03,  8.6355016e+03,  8.5623082e+03,
       8.3125769e+03,  7.8732452e+03,  7.2322659e+03,  6.3830254e+03,
       5.3220070e+03,  4.0508083e+03,  2.5743934e+03,  9.0382380e+02,
      -9.4658515e+02, -2.9577774e+03, -5.1073476e+03, -7.3680800e+03,
      -9.7116793e+03, -1.2105745e+04, -1.4518157e+04, -1.6913624e+04,
      -1.9259368e+04, -2.1519698e+04, -2.3664419e+04, -2.5661182e+04,
      -2.7484674e+04, -2.9109128e+04, -3.0513566e+04, -3.1680767e+04,
      -3.2599500e+04, -3.3259046e+04, -3.3656364e+04, -3.3789359e+04,
      -3.3661417e+04, -3.3275865e+04, -3.2647991e+04, -3.1778553e+04,
      -3.0687860e+04, -2.9386993e+04, -2.7893292e+04, -2.6217793e+04,
      -2.4380305e+04, -2.2396921e+04, -2.0285631e+04, -1.8056304e+04,
      -1.5731396e+04, -1.3310749e+04, -1.0817930e+04, -8.2632546e+03,
      -5.6584950e+03, -3.0169039e+03, -3.5049466e+02,  2.3300294e+03,
       5.0072558e+03,  7.6692538e+03,  1.0296341e+04,  1.2873294e+04,
       1.5379430e+04,  1.7799709e+04,  2.0110518e+04,  2.2292425e+04,
       2.4321243e+04,  2.6179470e+04,  2.7838772e+04,  2.9288536e+04,
       3.0502052e+04,  3.1467674e+04,  3.2161346e+04,  3.2580900e+04,
       3.2712443e+04,  3.2548152e+04,  3.2079638e+04,  3.1315042e+04,
       3.0239307e+04,  2.8881574e+04,  2.7232557e+04,  2.5325302e+04,
       2.3165560e+04,  2.0776212e+04,  1.8202819e+04,  1.5452512e+04,
       1.2552870e+04,  9.5323059e+03,  6.4141989e+03,  3.2353808e+03,
       3.2008805e+01, -3.1778835e+03, -6.3568255e+03, -9.4682388e+03,
      -1.2490842e+04, -1.5403926e+04, -1.8154186e+04, -2.0745346e+04,
      -2.3121161e+04, -2.5285375e+04, -2.7200767e+04, -2.8855100e+04,
      -3.0236188e+04, -3.1303814e+04, -3.2106304e+04, -3.2610143e+04,
      -3.2801911e+04, -3.2722047e+04, -3.2333816e+04, -3.1622380e+04,
      -3.0637858e+04, -2.9319050e+04, -2.7722782e+04, -2.5854268e+04,
      -2.3707550e+04, -2.1309710e+04, -1.8683478e+04, -1.5843106e+04,
      -1.2810610e+04, -9.6245078e+03, -6.3128385e+03, -2.9018457e+03,
       5.6021860e+02,  4.0355321e+03,  7.4888401e+03,  1.0869459e+04,
       1.4146120e+04,  1.7266962e+04,  2.0195488e+04,  2.2894666e+04,
       2.5327458e+04,  2.7459548e+04,  2.9268259e+04,  3.0729712e+04,
       3.1824168e+04,  3.2542974e+04,  3.2879994e+04,  3.2838377e+04,
       3.2420961e+04,  3.1641217e+04,  3.0522280e+04,  2.9086732e+04,
       2.7364573e+04,  2.5386626e+04,  2.3192561e+04,  2.0822736e+04,
       1.8313768e+04,  1.5710203e+04,  1.3055162e+04,  1.0388760e+04,
       7.7479346e+03,  5.1773492e+03,  2.7035918e+03,  3.6336147e+02,
      -1.8203312e+03, -3.8257571e+03, -5.6281400e+03, -7.2191945e+03,
      -8.5918440e+03, -9.7361057e+03, -1.0653660e+04, -1.1351662e+04,
      -1.1834991e+04, -1.2108888e+04, -1.2194030e+04, -1.2106898e+04,
      -1.1844914e+04, -1.1451706e+04, -1.0928890e+04, -1.0302174e+04,
      -9.5993172e+03, -8.8336978e+03, -8.0218966e+03, -7.1899794e+03,
      -6.3420744e+03, -5.5028868e+03, -4.6880751e+03, -3.9103856e+03,
      -3.1844358e+03, -2.5158381e+03, -1.9161507e+03, -1.3860285e+03,
      -9.3082869e+02, -5.4957243e+02, -2.3962727e+02,  2.0859282e+00,
       1.8035744e+02,  3.0232747e+02,  3.7488265e+02,  4.0574857e+02,
       4.0326247e+02,  3.7569870e+02,  3.3102137e+02,  2.7655429e+02,
       2.1828293e+02,  1.6157727e+02,  1.1024218e+02,  6.6816369e+01 ]

])

T_HAT_MDCT = [ T_HAT_MDCT_7M5, T_HAT_MDCT_10M ]

### C.4.1.9  LTPF

X_HAT_LTPF_10M = np.array([

    [  6.1525095e-01,  1.7117620e+00, -1.6200436e-01, -2.1107548e+00,
       9.1636911e-01,  8.7583398e+00,  1.4291828e+01,  1.4615904e+01,
       1.9312730e+01,  2.2178311e+01,  2.1178760e+01,  1.9939021e+01,
       3.2745193e+01,  3.1385007e+01,  4.5642888e+01,  3.8185668e+01,
       4.3452271e+01,  3.0174130e+01,  2.7510416e+01,  3.9048290e+00,
      -4.1911157e+00, -3.4032526e+01, -4.3089457e+01, -7.1817465e+01,
      -7.3319439e+01, -1.0231340e+02, -8.2645833e+01, -1.0207070e+02,
      -7.4253115e+01, -9.5269932e+01, -1.2210097e+02, -1.3216707e+02,
      -1.2665681e+02, -1.6033791e+02, -1.3067613e+02, -1.8796611e+01,
       6.2097263e+01,  7.2290617e+00, -1.2550979e+02, -8.9649115e+01,
       7.6135408e+02,  2.7072170e+03,  6.1558256e+03,  9.6522574e+03,
       1.2566221e+04,  1.5421574e+04,  1.8329565e+04,  2.1102710e+04,
       2.3323039e+04,  2.5451924e+04,  2.7365468e+04,  2.8891223e+04,
       3.0341813e+04,  3.1380038e+04,  3.2134454e+04,  3.2606958e+04,
       3.2683895e+04,  3.2518324e+04,  3.2036626e+04,  3.1278795e+04,
       3.0136453e+04,  2.8694957e+04,  2.7009676e+04,  2.5075921e+04,
       2.2940269e+04,  2.0564299e+04,  1.7956441e+04,  1.5202722e+04,
       1.2352834e+04,  9.3639445e+03,  6.2714619e+03,  3.1112267e+03,
      -1.0075267e+02, -3.3115389e+03, -6.4954073e+03, -9.6110111e+03,
      -1.2594144e+04, -1.5477538e+04, -1.8214406e+04, -2.0797170e+04,
      -2.3195615e+04, -2.5355124e+04, -2.7302557e+04, -2.8919494e+04,
      -3.0273227e+04, -3.1357707e+04, -3.2152224e+04, -3.2625545e+04,
      -3.2802109e+04, -3.2640407e+04, -3.2172861e+04, -3.1393517e+04,
      -3.0316009e+04, -2.8922986e+04, -2.7290775e+04, -2.5351571e+04,
      -2.3183937e+04, -2.0795771e+04, -1.8212609e+04, -1.5453610e+04,
      -1.2543978e+04, -9.4923186e+03, -6.3663932e+03, -3.1740539e+03,
       4.2407582e+01,  3.2479359e+03,  6.4369448e+03,  9.5564107e+03,
       1.2589389e+04,  1.5494437e+04,  1.8238168e+04,  2.0814647e+04,
       2.3183440e+04,  2.5340307e+04,  2.7236643e+04,  2.8879098e+04,
       3.0219987e+04,  3.1304367e+04,  3.2073455e+04,  3.2541098e+04,
       3.2660415e+04,  3.2490788e+04,  3.2004309e+04,  3.1215306e+04,
       3.0131477e+04,  2.8751611e+04,  2.7114762e+04,  2.5198099e+04,
       2.3044166e+04,  2.0655799e+04,  1.8068458e+04,  1.5312555e+04,
       1.2400688e+04,  9.3834439e+03,  6.2670869e+03,  3.1043555e+03,
      -9.0364008e+01, -3.2949280e+03, -6.4558716e+03, -9.5736656e+03,
      -1.2593429e+04, -1.5487321e+04, -1.8231003e+04, -2.0808229e+04,
      -2.3191386e+04, -2.5327915e+04, -2.7226934e+04, -2.8860268e+04,
      -3.0235350e+04, -3.1313467e+04, -3.2092397e+04, -3.2585580e+04,
      -3.2777259e+04, -3.2606534e+04, -3.2146234e+04, -3.1383325e+04,
      -3.0295384e+04, -2.8926934e+04, -2.7297985e+04, -2.5343669e+04 ],

    [ -2.3210612e+04, -2.0830109e+04, -1.8262528e+04, -1.5491202e+04,
      -1.2640992e+04, -9.6308521e+03, -6.5120997e+03, -3.2981272e+03,
      -1.0437273e+02,  3.1237705e+03,  6.3475523e+03,  9.4251090e+03,
       1.2406927e+04,  1.5369182e+04,  1.8133410e+04,  2.0696011e+04,
       2.3075625e+04,  2.5239168e+04,  2.7178122e+04,  2.8863775e+04,
       3.0239107e+04,  3.1292540e+04,  3.2055374e+04,  3.2528526e+04,
       3.2691792e+04,  3.2554032e+04,  3.2080858e+04,  3.1269701e+04,
       3.0219849e+04,  2.8872409e+04,  2.7201446e+04,  2.5283194e+04,
       2.3137397e+04,  2.0746427e+04,  1.8171258e+04,  1.5425482e+04,
       1.2542866e+04,  9.5154094e+03,  6.4038902e+03,  3.2243297e+03,
       1.8093877e+01, -3.2004786e+03, -6.3778766e+03, -9.4986572e+03,
      -1.2530644e+04, -1.5438896e+04, -1.8199553e+04, -2.0786963e+04,
      -2.3172801e+04, -2.5336869e+04, -2.7256938e+04, -2.8914212e+04,
      -3.0292811e+04, -3.1379968e+04, -3.2163160e+04, -3.2636663e+04,
      -3.2797410e+04, -3.2642190e+04, -3.2169708e+04, -3.1389768e+04,
      -3.0307591e+04, -2.8933213e+04, -2.7277887e+04, -2.5361709e+04,
      -2.3197556e+04, -2.0809583e+04, -1.8223444e+04, -1.5459240e+04,
      -1.2547850e+04, -9.5147469e+03, -6.3913071e+03, -3.2061552e+03,
       1.1474435e+01,  3.2280361e+03,  6.4146913e+03,  9.5376172e+03,
       1.2570372e+04,  1.5481142e+04,  1.8242641e+04,  2.0827867e+04,
       2.3213026e+04,  2.5375280e+04,  2.7293037e+04,  2.8947339e+04,
       3.0323330e+04,  3.1406817e+04,  3.2187891e+04,  3.2660531e+04,
       3.2817886e+04,  3.2659083e+04,  3.2187190e+04,  3.1402569e+04,
       3.0317849e+04,  2.8939620e+04,  2.7281206e+04,  2.5360297e+04,
       2.3194809e+04,  2.0805310e+04,  1.8215867e+04,  1.5450539e+04,
       1.2535402e+04,  9.5045150e+03,  6.3825434e+03,  3.1961404e+03,
      -1.9030604e+01, -3.2350097e+03, -6.4178539e+03, -9.5376996e+03,
      -1.2564818e+04, -1.5468879e+04, -1.8222757e+04, -2.0799617e+04,
      -2.3177492e+04, -2.5329860e+04, -2.7239548e+04, -2.8887388e+04,
      -3.0257022e+04, -3.1336972e+04, -3.2115659e+04, -3.2585521e+04,
      -3.2741691e+04, -3.2583544e+04, -3.2112334e+04, -3.1332786e+04,
      -3.0253039e+04, -2.8881031e+04, -2.7231757e+04, -2.5321143e+04,
      -2.3168095e+04, -2.0790131e+04, -1.8212051e+04, -1.5458479e+04,
      -1.2554090e+04, -9.5278709e+03, -6.4088050e+03, -3.2282766e+03,
      -1.5127187e+01,  3.1974595e+03,  6.3806758e+03,  9.5013370e+03,
       1.2533377e+04,  1.5443257e+04,  1.8203988e+04,  2.0793445e+04,
       2.3180292e+04,  2.5344634e+04,  2.7265053e+04,  2.8919963e+04,
       3.0294320e+04,  3.1376795e+04,  3.2154178e+04,  3.2622959e+04,
       3.2776568e+04,  3.2617286e+04,  3.2145842e+04,  3.1360660e+04,
       3.0278348e+04,  2.8901877e+04,  2.7241369e+04,  2.5301056e+04 ],

])

X_HAT_LTPF_7M5 = np.array([

    [  9.0382948e-01,  2.8563300e+00,  2.0863167e+00,  3.2605273e+00,
       1.8231017e+00, -2.6473031e+00, -7.7420704e+00, -1.6971743e+01,
      -4.4169569e+00,  4.7473387e+00,  7.9882732e+00,  2.1090757e+00,
       6.9477046e+00,  7.6294361e+00,  4.5069158e+00,  1.1288109e+00,
       5.5301798e-01, -1.2320805e+00,  1.2696965e+01,  1.7998129e+01,
       1.9997378e+01,  2.3310802e+01,  3.4116671e+01,  3.1619222e+01,
       2.3643252e+01,  2.2595989e+01,  2.4150879e+01,  1.7561939e+01,
       2.4167995e+01,  2.1868269e+01,  1.2021561e+01,  1.0810360e+01,
      -1.1321816e+01, -1.3811836e+01, -2.7571991e+01, -3.3459505e+01,
      -2.6720233e+01, -4.0425004e+01, -4.1666697e+01, -4.8106995e+01,
      -7.1121739e+01, -8.5018856e+01, -6.4519501e+01, -6.1651047e+01,
      -6.2001672e+01, -4.9054098e+01,  5.3605147e+00, -2.7222279e+00,
      -6.3200946e+00, -2.8873822e+01, -5.6314175e+01, -5.9551902e+01,
      -2.1183627e+01, -9.5007617e+01, -6.7674879e+01,  7.6546124e+01,
       3.6355638e+02,  2.0908440e+02,  9.2290767e+01, -8.4453487e+01,
      -2.0810832e+02, -1.9235273e+02, -4.0634578e+02, -2.2011977e+02,
       6.2920459e+02,  3.1481663e+03,  6.2343351e+03,  9.4022080e+03,
       1.2520451e+04,  1.5313131e+04,  1.8128985e+04,  2.0762454e+04,
       2.3084787e+04,  2.5275848e+04,  2.7095495e+04,  2.8665301e+04,
       3.0094623e+04,  3.1202047e+04,  3.2006678e+04,  3.2461623e+04,
       3.2568832e+04,  3.2408327e+04,  3.1961953e+04,  3.1146555e+04,
       3.0073949e+04,  2.8725124e+04,  2.7099832e+04,  2.5196695e+04,
       2.3022972e+04,  2.0643354e+04,  1.8079103e+04,  1.5352852e+04,
       1.2476728e+04,  9.4135962e+03,  6.2948219e+03,  3.1010477e+03,
      -9.0897787e+01, -3.3383673e+03, -6.5093586e+03, -9.6214110e+03,
      -1.2638625e+04, -1.5564780e+04, -1.8289238e+04, -2.0877731e+04,
      -2.3274493e+04, -2.5456613e+04, -2.7372293e+04, -2.9018289e+04,
      -3.0399516e+04, -3.1474248e+04, -3.2213279e+04, -3.2686770e+04,
      -3.2836146e+04, -3.2655386e+04, -3.2161995e+04, -3.1363572e+04,
      -3.0296725e+04, -2.8893704e+04, -2.7226660e+04, -2.5303018e+04 ],

    [ -2.3111848e+04, -2.0718046e+04, -1.8145256e+04, -1.5386042e+04,
      -1.2451683e+04, -9.4561229e+03, -6.3407390e+03, -3.2078423e+03,
      -1.4231827e+01,  3.1871864e+03,  6.4028626e+03,  9.5288605e+03,
       1.2522323e+04,  1.5403074e+04,  1.8148763e+04,  2.0738239e+04,
       2.3124038e+04,  2.5248199e+04,  2.7157343e+04,  2.8821520e+04,
       3.0197933e+04,  3.1280524e+04,  3.2068994e+04,  3.2535286e+04,
       3.2674198e+04,  3.2505688e+04,  3.2021479e+04,  3.1217393e+04,
       3.0139480e+04,  2.8742712e+04,  2.7081219e+04,  2.5164753e+04,
       2.3005958e+04,  2.0625737e+04,  1.8051360e+04,  1.5314656e+04,
       1.2403135e+04,  9.3958576e+03,  6.2866076e+03,  3.1150574e+03,
      -8.3043055e+01, -3.2906309e+03, -6.4783209e+03, -9.5968770e+03,
      -1.2627329e+04, -1.5543027e+04, -1.8297188e+04, -2.0856742e+04,
      -2.3242831e+04, -2.5406524e+04, -2.7315500e+04, -2.8954115e+04,
      -3.0311322e+04, -3.1397263e+04, -3.2199564e+04, -3.2663393e+04,
      -3.2812053e+04, -3.2631820e+04, -3.2158938e+04, -3.1361864e+04,
      -3.0286591e+04, -2.8909485e+04, -2.7252911e+04, -2.5338382e+04,
      -2.3179744e+04, -2.0801866e+04, -1.8222214e+04, -1.5467629e+04,
      -1.2551908e+04, -9.5234786e+03, -6.4050183e+03, -3.2234113e+03,
      -1.0202956e+01,  3.2050827e+03,  6.3811438e+03,  9.5000550e+03,
       1.2526624e+04,  1.5433267e+04,  1.8190970e+04,  2.0775689e+04,
       2.3157406e+04,  2.5317424e+04,  2.7226647e+04,  2.8880016e+04,
       3.0250964e+04,  3.1333324e+04,  3.2108157e+04,  3.2579815e+04,
       3.2739171e+04,  3.2585501e+04,  3.2115570e+04,  3.1342610e+04,
       3.0239307e+04,  2.8881574e+04,  2.7232557e+04,  2.5325302e+04,
       2.3165560e+04,  2.0776212e+04,  1.8202819e+04,  1.5452512e+04,
       1.2552870e+04,  9.5323059e+03,  6.4141989e+03,  3.2353808e+03,
       3.2008805e+01, -3.1778835e+03, -6.3568255e+03, -9.4682388e+03,
      -1.2490842e+04, -1.5403926e+04, -1.8154186e+04, -2.0745346e+04,
      -2.3121161e+04, -2.5285375e+04, -2.7200767e+04, -2.8855100e+04,
      -3.0236188e+04, -3.1303814e+04, -3.2106304e+04, -3.2610143e+04 ]

])

X_HAT_LTPF = [ X_HAT_LTPF_7M5, X_HAT_LTPF_10M ]

LTPF_C2_NBITS = 320

LTPF_C2_ACTIVE = np.array([ False, True ])
LTPF_C2_PITCH_INDEX = np.array([ 60, 56 ])

LTPF_C2_C_N = np.array([
    [  0.           ,  0.           ,  0.            ],
    [  2.0480302e-01,  1.4271871e-01, -6.4036434e-03 ],
])

LTPF_C2_C_D = np.array([
    [  0.           ,  0.           ,  0.           ,  0.            ],
    [  1.5868459e-02,  1.8368837e-01,  1.8368837e-01,  1.5868459e-02 ],
])

LTPF_C2_X = np.array([

    [  2.1004800e+03,  2.0660438e+03,  2.0070810e+03,  1.9335150e+03,
       1.8437505e+03,  1.7474564e+03,  1.6433626e+03,  1.5341100e+03,
       1.4257583e+03,  1.3211557e+03,  1.2062000e+03,  1.0819606e+03,
       9.5194047e+02,  8.1893657e+02,  6.7789896e+02,  5.3732954e+02,
       4.0506079e+02,  2.7179124e+02,  1.3998655e+02,  1.9709326e+00,
      -1.3683087e+02, -2.7783589e+02, -4.1717230e+02, -5.5239764e+02,
      -6.7544678e+02, -7.8374115e+02, -8.8533384e+02, -9.8169275e+02,
      -1.0740844e+03, -1.1619601e+03, -1.2468187e+03, -1.3276334e+03,
      -1.3976625e+03, -1.4581676e+03, -1.5080701e+03, -1.5565703e+03,
      -1.6009886e+03, -1.6442955e+03, -1.6843665e+03, -1.7151734e+03,
      -1.7459758e+03, -1.7579559e+03, -1.7383847e+03, -1.6558090e+03,
      -1.4937678e+03, -1.2402230e+03, -9.0680789e+02, -5.2126546e+02,
      -1.0907639e+02,  2.9396419e+02,  6.6844988e+02,  9.9743497e+02,
       1.2728461e+03,  1.5005211e+03,  1.6749621e+03,  1.8070876e+03,
       1.9030819e+03,  1.9644916e+03,  1.9851832e+03,  1.9771133e+03,
       1.9382403e+03,  1.8819654e+03,  1.8027215e+03,  1.7116504e+03,
       1.6036716e+03,  1.4931560e+03,  1.3786289e+03,  1.2583587e+03,
       1.1369652e+03,  1.0121811e+03,  8.8292297e+02,  7.5232872e+02,
       6.2279997e+02,  4.8668604e+02,  3.4937590e+02,  2.1277380e+02,
       7.7335654e+01, -5.2960798e+01, -1.8097039e+02, -3.0070120e+02,
      -4.1472118e+02, -5.2643886e+02, -6.3396297e+02, -7.3691168e+02,
      -8.3480847e+02, -9.2600159e+02, -1.0147685e+03, -1.0954793e+03,
      -1.1736112e+03, -1.2509220e+03, -1.3222525e+03, -1.3883874e+03,
      -1.4496560e+03, -1.5064726e+03, -1.5576614e+03, -1.6028926e+03,
      -1.6434987e+03, -1.6803495e+03, -1.7069786e+03, -1.7018749e+03,
      -1.6517403e+03, -1.5416168e+03, -1.3607723e+03, -1.1195564e+03,
      -8.2569063e+02, -4.9652560e+02, -1.5319529e+02,  1.8519693e+02,
       5.0070752e+02,  7.8614834e+02,  1.0314193e+03,  1.2402508e+03,
       1.4166554e+03,  1.5555207e+03,  1.6600606e+03,  1.7324994e+03,
       1.7753655e+03,  1.7931195e+03,  1.7839794e+03,  1.7543157e+03,
       1.7049335e+03,  1.6342871e+03,  1.5549646e+03,  1.4629383e+03,
       1.3581758e+03,  1.2457595e+03,  1.1299878e+03,  1.0119864e+03,
       8.9090477e+02,  7.6719677e+02,  6.4285660e+02,  5.1408928e+02,
       3.8508372e+02,  2.5534679e+02,  1.2497756e+02, -3.9966107e+00,
      -1.2599628e+02, -2.4595825e+02, -3.6034285e+02, -4.7095724e+02,
      -5.7730810e+02, -6.8592514e+02, -7.8960041e+02, -8.9182726e+02,
      -9.8741569e+02, -1.0733871e+03, -1.1555560e+03, -1.2285264e+03,
      -1.2950383e+03, -1.3587508e+03, -1.4190449e+03, -1.4731166e+03,
      -1.5244161e+03, -1.5700435e+03, -1.6143711e+03, -1.6442522e+03,
      -1.6504480e+03, -1.6200592e+03, -1.5385467e+03, -1.4018632e+03 ],

    [ -1.2097642e+03, -9.7471997e+02, -7.0651192e+02, -4.1925510e+02,
      -1.2278127e+02,  1.6308666e+02,  4.3389868e+02,  6.8416589e+02,
       9.0871878e+02,  1.1028177e+03,  1.2670447e+03,  1.4015901e+03,
       1.5108776e+03,  1.5894427e+03,  1.6428125e+03,  1.6684610e+03,
       1.6687068e+03,  1.6467684e+03,  1.6069401e+03,  1.5504379e+03,
       1.4802355e+03,  1.3985477e+03,  1.3037872e+03,  1.1957600e+03,
       1.0853732e+03,  9.7233331e+02,  8.4944729e+02,  7.2549848e+02,
       5.9937450e+02,  4.8205790e+02,  3.5945231e+02,  2.3181221e+02,
       1.0289577e+02, -2.2787734e+01, -1.4227469e+02, -2.5954609e+02,
      -3.7614640e+02, -4.8391910e+02, -5.8341913e+02, -6.8123214e+02,
      -7.7747871e+02, -8.6932601e+02, -9.5415487e+02, -1.0385138e+03,
      -1.1184902e+03, -1.1867290e+03, -1.2499404e+03, -1.3128700e+03,
      -1.3730047e+03, -1.4324194e+03, -1.4841659e+03, -1.5297608e+03,
      -1.5601321e+03, -1.5603639e+03, -1.5252096e+03, -1.4567924e+03,
      -1.3535284e+03, -1.2050129e+03, -1.0232675e+03, -8.1556031e+02,
      -5.8338838e+02, -3.4263538e+02, -9.1284066e+01,  1.4792883e+02,
       3.7724977e+02,  5.9063752e+02,  7.8210956e+02,  9.5337152e+02,
       1.1069618e+03,  1.2346417e+03,  1.3304957e+03,  1.4081378e+03,
       1.4660754e+03,  1.5064930e+03,  1.5231194e+03,  1.5173564e+03,
       1.4891356e+03,  1.4460229e+03,  1.3890302e+03,  1.3171848e+03,
       1.2329448e+03,  1.1400293e+03,  1.0376506e+03,  9.3324803e+02,
       8.2426407e+02,  7.1443513e+02,  6.0252527e+02,  4.9088960e+02,
       3.7646014e+02,  2.6184052e+02,  1.4340035e+02,  3.0422051e+01,
      -8.5724331e+01, -1.9809318e+02, -3.0918985e+02, -4.1384220e+02,
      -5.0992503e+02, -6.0275760e+02, -6.9829113e+02, -7.9525188e+02,
      -8.9453019e+02, -9.7802721e+02, -1.0526954e+03, -1.1251736e+03,
      -1.1955435e+03, -1.2637584e+03, -1.3263901e+03, -1.3794649e+03,
      -1.4229086e+03, -1.4542303e+03, -1.4659972e+03, -1.4469638e+03,
      -1.3956114e+03, -1.3075606e+03, -1.1938217e+03, -1.0487411e+03,
      -8.7590742e+02, -6.8650630e+02, -4.8348960e+02, -2.7915529e+02,
      -7.3951415e+01,  1.2639053e+02,  3.2037647e+02,  5.0816397e+02,
       6.7805869e+02,  8.2207151e+02,  9.5456310e+02,  1.0705573e+03,
       1.1629335e+03,  1.2429314e+03,  1.3063791e+03,  1.3382875e+03,
       1.3364949e+03,  1.3197603e+03,  1.3011283e+03,  1.2904738e+03,
       1.2841453e+03,  1.2735627e+03,  1.2483109e+03,  1.1982190e+03,
       1.1236072e+03,  1.0353182e+03,  9.4811986e+02,  8.4791470e+02,
       7.4977805e+02,  6.0946439e+02,  4.1686370e+02,  1.5206161e+02,
      -7.9210822e+01, -2.2784261e+02, -3.2684361e+02, -3.9056312e+02,
      -4.6716161e+02, -5.4206330e+02, -6.2230545e+02, -7.0009951e+02,
      -7.7171489e+02, -8.3103588e+02, -8.8858361e+02, -9.5353054e+02 ]

])

LTPF_C2_PREV = np.array([

    [  3.6075890e+02,  2.5558573e+02,  1.5676072e+02,  6.2869832e+01,
      -2.7464215e+01, -1.0807999e+02, -1.8538536e+02, -2.5213341e+02,
      -3.1737297e+02, -3.7675903e+02, -4.3291912e+02, -4.8396843e+02,
      -5.3257774e+02, -5.7812590e+02, -6.2166743e+02, -6.6533460e+02,
      -7.0685062e+02, -7.4579194e+02, -7.7729679e+02, -8.0489134e+02,
      -8.3252315e+02, -8.5569655e+02, -8.7998412e+02, -9.0742496e+02,
      -9.3370927e+02, -9.5471476e+02, -9.7276449e+02, -9.9514771e+02,
      -1.0136386e+03, -1.0163600e+03, -9.9100449e+02, -9.3570452e+02,
      -8.4068970e+02, -7.0878785e+02, -5.4236585e+02, -3.4883259e+02,
      -1.3544369e+02,  7.6143761e+01,  2.8313432e+02,  4.7371481e+02,
       6.3854542e+02,  7.8591376e+02,  9.1647781e+02,  1.0326049e+03,
       1.1226394e+03,  1.1855486e+03,  1.2110267e+03,  1.2072759e+03,
       1.1810221e+03,  1.1433488e+03,  1.0943565e+03,  1.0414966e+03,
       9.8463390e+02,  9.1048007e+02,  8.1752062e+02,  7.2333732e+02,
       6.4850080e+02,  5.9874218e+02,  5.6539980e+02,  5.5634736e+02,
       5.3800637e+02,  4.9467989e+02,  4.1631880e+02,  2.8517505e+02,
       1.1305724e+02, -8.8498535e+01, -2.9818901e+02, -4.9635689e+02,
      -6.7273562e+02, -8.1632731e+02, -9.2538304e+02, -1.0146932e+03,
      -1.0962177e+03, -1.1660919e+03, -1.2306080e+03, -1.2853503e+03,
      -1.3346346e+03, -1.3883886e+03, -1.4496624e+03, -1.5155564e+03,
      -1.5793268e+03, -1.6396210e+03, -1.6937295e+03, -1.7314870e+03,
      -1.7610564e+03, -1.7775059e+03, -1.7806815e+03, -1.7684558e+03,
      -1.6949960e+03, -1.4776266e+03, -1.0700242e+03, -5.1121101e+02,
       1.0329967e+02,  6.7165782e+02,  1.1381176e+03,  1.4974226e+03,
       1.7689436e+03,  1.9620604e+03,  2.0875907e+03,  2.1420343e+03,
       2.1555856e+03,  2.1497779e+03,  2.1345923e+03,  2.1141460e+03,
       2.0831857e+03,  2.0328927e+03,  1.9628672e+03,  1.8762276e+03,
       1.7788061e+03,  1.6631901e+03,  1.5303905e+03,  1.3855681e+03,
       1.2298992e+03,  1.0770934e+03,  9.1998634e+02,  7.6869655e+02,
       6.1555385e+02,  4.7278828e+02,  3.3511742e+02,  2.0334643e+02,
       8.1069660e+01, -4.1916116e+01, -1.7411361e+02, -3.0622256e+02,
      -4.3179154e+02, -5.5570746e+02, -6.8069811e+02, -7.9809071e+02,
      -9.0053455e+02, -9.9388321e+02, -1.0834733e+03, -1.1654917e+03,
      -1.2466250e+03, -1.3184789e+03, -1.3814537e+03, -1.4386054e+03,
      -1.5024822e+03, -1.5617271e+03, -1.6148868e+03, -1.6639722e+03,
      -1.7108728e+03, -1.7519590e+03, -1.7805217e+03, -1.8036393e+03,
      -1.8080682e+03, -1.7690236e+03, -1.6458189e+03, -1.4003994e+03,
      -1.0280788e+03, -5.6864419e+02, -7.5505097e+01,  3.9987715e+02,
       8.1589313e+02,  1.1654034e+03,  1.4482132e+03,  1.6762780e+03,
       1.8458377e+03,  1.9698779e+03,  2.0528472e+03,  2.0941040e+03 ],

    [  2.1004800e+03,  2.0660438e+03,  2.0070810e+03,  1.9335150e+03,
       1.8437505e+03,  1.7474564e+03,  1.6433626e+03,  1.5341100e+03,
       1.4257583e+03,  1.3211557e+03,  1.2062000e+03,  1.0819606e+03,
       9.5194047e+02,  8.1893657e+02,  6.7789896e+02,  5.3732954e+02,
       4.0506079e+02,  2.7179124e+02,  1.3998655e+02,  1.9709326e+00,
      -1.3683087e+02, -2.7783589e+02, -4.1717230e+02, -5.5239764e+02,
      -6.7544678e+02, -7.8374115e+02, -8.8533384e+02, -9.8169275e+02,
      -1.0740844e+03, -1.1619601e+03, -1.2468187e+03, -1.3276334e+03,
      -1.3976625e+03, -1.4581676e+03, -1.5080701e+03, -1.5565703e+03,
      -1.6009886e+03, -1.6442955e+03, -1.6843665e+03, -1.7151734e+03,
      -1.7459758e+03, -1.7579559e+03, -1.7383847e+03, -1.6558090e+03,
      -1.4937678e+03, -1.2402230e+03, -9.0680789e+02, -5.2126546e+02,
      -1.0907639e+02,  2.9396419e+02,  6.6844988e+02,  9.9743497e+02,
       1.2728461e+03,  1.5005211e+03,  1.6749621e+03,  1.8070876e+03,
       1.9030819e+03,  1.9644916e+03,  1.9851832e+03,  1.9771133e+03,
       1.9382403e+03,  1.8819654e+03,  1.8027215e+03,  1.7116504e+03,
       1.6036716e+03,  1.4931560e+03,  1.3786289e+03,  1.2583587e+03,
       1.1369652e+03,  1.0121811e+03,  8.8292297e+02,  7.5232872e+02,
       6.2279997e+02,  4.8668604e+02,  3.4937590e+02,  2.1277380e+02,
       7.7335654e+01, -5.2960798e+01, -1.8097039e+02, -3.0070120e+02,
      -4.1472118e+02, -5.2643886e+02, -6.3396297e+02, -7.3691168e+02,
      -8.3480847e+02, -9.2600159e+02, -1.0147685e+03, -1.0954793e+03,
      -1.1736112e+03, -1.2509220e+03, -1.3222525e+03, -1.3883874e+03,
      -1.4496560e+03, -1.5064726e+03, -1.5576614e+03, -1.6028926e+03,
      -1.6434987e+03, -1.6803495e+03, -1.7069786e+03, -1.7018749e+03,
      -1.6517403e+03, -1.5416168e+03, -1.3607723e+03, -1.1195564e+03,
      -8.2569063e+02, -4.9652560e+02, -1.5319529e+02,  1.8519693e+02,
       5.0070752e+02,  7.8614834e+02,  1.0314193e+03,  1.2402508e+03,
       1.4166554e+03,  1.5555207e+03,  1.6600606e+03,  1.7324994e+03,
       1.7753655e+03,  1.7931195e+03,  1.7839794e+03,  1.7543157e+03,
       1.7049335e+03,  1.6342871e+03,  1.5549646e+03,  1.4629383e+03,
       1.3581758e+03,  1.2457595e+03,  1.1299878e+03,  1.0119864e+03,
       8.9090477e+02,  7.6719677e+02,  6.4285660e+02,  5.1408928e+02,
       3.8508372e+02,  2.5534679e+02,  1.2497756e+02, -3.9966107e+00,
      -1.2599628e+02, -2.4595825e+02, -3.6034285e+02, -4.7095724e+02,
      -5.7730810e+02, -6.8592514e+02, -7.8960041e+02, -8.9182726e+02,
      -9.8741569e+02, -1.0733871e+03, -1.1555560e+03, -1.2285264e+03,
      -1.2950383e+03, -1.3587508e+03, -1.4190449e+03, -1.4731166e+03,
      -1.5244161e+03, -1.5700435e+03, -1.6143711e+03, -1.6442522e+03,
      -1.6504480e+03, -1.6200592e+03, -1.5385467e+03, -1.4018632e+03 ],

])

LTPF_C2_TRANS = np.array([

      -1.2097642e+03, -9.7530715e+02, -7.0586256e+02, -4.1543418e+02,
      -1.1414107e+02,  1.7777097e+02,  4.5531279e+02,  7.1225199e+02,
       9.4297106e+02,  1.1428504e+03,  1.3122721e+03,  1.4510401e+03,
       1.5632810e+03,  1.6437088e+03,  1.6979967e+03,  1.7236213e+03,
       1.7231450e+03,  1.6998644e+03,  1.6575399e+03,  1.5978279e+03,
       1.5240838e+03,  1.4378667e+03,  1.3380626e+03,  1.2256564e+03,
       1.1110323e+03,  9.9312189e+02,  8.6572627e+02,  7.3745656e+02,
       6.0633674e+02,  4.8191123e+02,  3.5135472e+02,  2.1648189e+02,
       8.0831679e+01, -5.1111363e+01, -1.7717212e+02, -3.0113312e+02,
      -4.2348388e+02, -5.3759466e+02, -6.4602675e+02, -7.5403092e+02,
      -8.6017221e+02, -9.5944484e+02, -1.0506850e+03, -1.1398086e+03,
      -1.2228263e+03, -1.2941518e+03, -1.3612258e+03, -1.4273749e+03,
      -1.4894501e+03, -1.5495534e+03, -1.6019734e+03, -1.6490374e+03,
      -1.6814663e+03, -1.6844562e+03, -1.6513997e+03, -1.5793876e+03,
      -1.4632011e+03, -1.2939942e+03, -1.0852074e+03, -8.4480887e+02,
      -5.7732264e+02, -3.0008156e+02, -1.5103259e+01,  2.5424761e+02,
       5.1054364e+02,  7.4618547e+02,  9.5531914e+02,  1.1392454e+03,
       1.2997276e+03,  1.4305069e+03,  1.5279454e+03,  1.6037570e+03,
       1.6553093e+03,  1.6847892e+03,  1.6879843e+03,  1.6682815e+03,
       1.6264868e+03,  1.5696403e+03,  1.4983779e+03,  1.4119806e+03,
       1.3127144e+03,  1.2052332e+03,  1.0900241e+03,  9.7210566e+02,
       8.4877207e+02,  7.2467907e+02,  5.9961756e+02,  4.7505541e+02,
       3.4639404e+02,  2.1700781e+02,  8.5155762e+01, -4.0026523e+01,
      -1.6698145e+02, -2.8953979e+02, -4.0971310e+02, -5.2233024e+02,
      -6.2776248e+02, -7.3117198e+02, -8.3540014e+02, -9.3755661e+02,
      -1.0392995e+03, -1.1263481e+03, -1.2052728e+03, -1.2805748e+03,
      -1.3532553e+03, -1.4234920e+03, -1.4883360e+03, -1.5444401e+03,
      -1.5915550e+03, -1.6260829e+03, -1.6379761e+03, -1.6152110e+03,
      -1.5560714e+03, -1.4554812e+03, -1.3203947e+03, -1.1456045e+03,
      -9.3870522e+02, -7.1082654e+02, -4.6691560e+02, -2.2039906e+02,
       2.5206392e+01,  2.6135641e+02,  4.8629103e+02,  6.9790232e+02,
       8.8592993e+02,  1.0460616e+03,  1.1899193e+03,  1.3098413e+03,
       1.4023098e+03,  1.4788921e+03,  1.5346583e+03,  1.5579963e+03,
       1.5491955e+03,  1.5242006e+03,  1.4921986e+03,  1.4607118e+03,
       1.4256432e+03,  1.3811223e+03,  1.3213357e+03,  1.2406869e+03,
       1.1418577e+03,  1.0338657e+03,  9.2734345e+02,  8.0983956e+02,
       6.9568058e+02,  5.4691592e+02,  3.6168815e+02,  1.2569299e+02,
      -7.2937241e+01, -2.1016935e+02, -3.1888920e+02, -4.0476988e+02,
      -5.0359952e+02, -5.9623817e+02, -6.9164993e+02, -7.8397385e+02,
      -8.7149927e+02, -9.4943765e+02, -1.0247441e+03, -1.1018532e+03,

])

LTPF_C3_NBITS = 320

LTPF_C3_ACTIVE = np.array([ True, False ])
LTPF_C3_PITCH_INDEX = np.array([ 56, 0 ])

LTPF_C3_C_N = np.array([
    [  2.0480302e-01,  1.4271871e-01, -6.4036434e-03 ],
    [  0.           ,  0.           ,  0.            ],
])

LTPF_C3_C_D = np.array([
    [  1.5868459e-02,  1.8368837e-01,  1.8368837e-01,  1.5868459e-02 ],
    [  0.           ,  0.           ,  0.           ,  0.            ],
])

LTPF_C3_X = np.array([

    [ -3.4872147e+02, -3.2869651e+02, -3.0368708e+02, -2.7206929e+02,
      -2.3135299e+02, -1.8746149e+02, -1.4271272e+02, -9.8455678e+01,
      -4.8137684e+01,  2.0589588e-01,  4.5670854e+01,  8.5887062e+01,
       1.2485726e+02,  1.6126194e+02,  1.9191268e+02,  2.2387152e+02,
       2.4650629e+02,  2.6665533e+02,  2.8210563e+02,  2.9385531e+02,
       3.0042345e+02,  2.9898320e+02,  2.9792092e+02,  2.9201085e+02,
       2.8341711e+02,  2.7006923e+02,  2.5925000e+02,  2.4396424e+02,
       2.2871379e+02,  2.1604489e+02,  2.0086440e+02,  1.8158468e+02,
       1.6604772e+02,  1.4881816e+02,  1.3450578e+02,  1.1691585e+02,
       1.0011405e+02,  8.1066153e+01,  5.7608036e+01,  3.3232973e+01,
       8.7252186e+00, -1.9268859e+01, -5.0019465e+01, -7.8718920e+01,
      -1.1264618e+02, -1.4558774e+02, -1.7907091e+02, -2.1353821e+02,
      -2.4860083e+02, -2.7968268e+02, -3.0845126e+02, -3.3166214e+02,
      -3.4854139e+02, -3.6210272e+02, -3.6311551e+02, -3.6538831e+02,
      -3.5698782e+02, -3.4320560e+02, -3.2228293e+02, -2.9431070e+02,
      -2.6126653e+02, -2.2161989e+02, -1.7770590e+02, -1.3258934e+02,
      -8.8768668e+01, -4.1250510e+01,  4.8812815e+00,  4.8960277e+01,
       9.1265856e+01,  1.2722198e+02,  1.5854814e+02,  1.8841450e+02,
       2.1263776e+02,  2.3530944e+02,  2.5135346e+02,  2.6817644e+02,
       2.7730158e+02,  2.7972049e+02,  2.7893036e+02,  2.7638406e+02,
       2.6804826e+02,  2.6051672e+02,  2.4861891e+02,  2.3713159e+02,
       2.2260424e+02,  2.0911457e+02,  1.9338336e+02,  1.8071811e+02,
       1.6387353e+02,  1.5139501e+02,  1.3594276e+02,  1.2442513e+02,
       1.1022492e+02,  9.7036956e+01,  7.4818560e+01,  5.7999575e+01,
       3.1360863e+01,  8.7303630e+00, -1.6051537e+01, -4.5294499e+01,
      -7.3824309e+01, -1.0455773e+02, -1.3439723e+02, -1.6704153e+02,
      -1.9839956e+02, -2.3387826e+02, -2.6412575e+02, -2.9170423e+02,
      -3.1531207e+02, -3.3274995e+02, -3.4693113e+02, -3.5369132e+02,
      -3.5301724e+02, -3.4943527e+02, -3.3608402e+02, -3.1755446e+02,
      -2.9205556e+02, -2.6071672e+02, -2.2432542e+02, -1.8738248e+02,
      -1.4392604e+02, -1.0106545e+02, -5.5577429e+01, -9.8667562e+00,
       3.7353443e+01,  7.5303068e+01,  1.1105079e+02,  1.4264083e+02,
       1.7184174e+02,  1.9822361e+02,  2.2081903e+02,  2.4084291e+02,
       2.5643437e+02,  2.6758684e+02,  2.7445589e+02,  2.7569331e+02,
       2.7159104e+02,  2.6648908e+02,  2.5914969e+02,  2.4863883e+02,
       2.3470167e+02,  2.2613959e+02,  2.1126267e+02,  2.0090440e+02,
       1.8404815e+02,  1.6736836e+02,  1.5327263e+02,  1.3857671e+02,
       1.2564592e+02,  1.1183479e+02,  9.8674586e+01,  7.9462699e+01,
       6.1566145e+01,  3.4757509e+01,  9.1597341e+00, -1.6506764e+01,
      -4.4310986e+01, -7.7716465e+01, -1.1113407e+02, -1.4150980e+02 ],

    [ -1.6775203e+02, -2.0410277e+02, -2.4176001e+02, -2.7386652e+02,
      -2.9629852e+02, -3.0908748e+02, -3.0195831e+02, -2.8578850e+02,
      -2.6686979e+02, -2.6140403e+02, -2.6601879e+02, -2.7595293e+02,
      -2.8053255e+02, -2.9088609e+02, -2.8340082e+02, -2.8587804e+02,
      -2.5164112e+02, -2.1308294e+02, -1.5229837e+02, -8.7968057e+01,
      -3.9965246e+01,  1.9682469e+01,  8.3439201e+01,  1.3071313e+02,
       2.0278676e+02,  2.2530998e+02,  2.4917273e+02,  2.2837462e+02,
       2.0851337e+02,  1.8379204e+02,  1.0597931e+02,  5.0251896e+01,
       5.3816342e+01,  1.9418724e+02,  3.6073746e+02,  4.9001894e+02,
       5.6206313e+02,  5.4551465e+02,  5.0418915e+02,  3.9875911e+02,
       2.9036149e+02,  2.1569487e+02,  1.2216776e+02,  5.6342202e+01,
       1.9881173e+01, -1.0527142e+01, -2.8641229e+01, -6.5396380e+01,
      -2.2087227e+01, -1.9320385e+00,  9.9837180e+00, -1.2901176e+01,
      -4.8912099e+01, -1.0229260e+02, -1.3236180e+02, -1.1357973e+02,
      -1.6394336e+02, -2.0430337e+02, -2.3625105e+02, -2.7025726e+02,
      -2.1302135e+02, -2.2632729e+02, -2.0952571e+02, -2.1427507e+02,
      -2.5180218e+02, -2.9681097e+02, -3.2929367e+02, -3.2993365e+02,
      -3.3630273e+02, -3.0716107e+02, -2.2776881e+02, -2.2007043e+02,
      -1.7036950e+02, -1.6356561e+02, -1.0711820e+02,  3.8723772e+01,
       9.2665406e+01,  1.2719129e+02,  2.3419328e+02,  2.4827798e+02,
       2.2061102e+02,  1.4474493e+02,  1.0064081e+02,  9.5343023e+01,
       5.5501524e+01,  3.6218435e+00, -6.9277352e+01, -1.2309115e+01,
      -2.9683737e+01, -1.1059060e+01,  1.2650129e+02,  2.8907139e+02,
       5.9646998e+02,  7.2219690e+02,  6.4274592e+02,  2.5572095e+02,
       8.9009716e+00, -1.6887796e+02, -5.4165018e+02, -8.8289542e+02,
      -9.0816906e+02,  2.8400805e+02,  1.3941683e+03,  1.4288425e+03,
       1.0428507e+03,  6.0910516e+02,  4.9505488e+02,  5.1165385e+02,
      -4.9324168e+01, -5.0538446e+02, -6.1278936e+02, -6.0237215e+02,
      -1.0190721e+03, -1.3790932e+03, -1.3541381e+03,  7.7933615e+01,
       1.3253623e+03,  1.3970218e+03,  1.1115023e+03,  7.7044387e+02,
       5.1675398e+02,  2.7357534e+02, -1.1882263e+02, -3.6893899e+02,
      -5.9907798e+02, -6.8538645e+02, -1.0290340e+03, -1.2093927e+03,
      -1.3183905e+03, -6.4097534e+02,  7.8577340e+02,  1.3858651e+03,
       1.2575136e+03,  1.0978140e+03,  7.7239776e+02,  3.5047643e+02,
       1.3677282e+02, -2.9932975e+01, -3.3626785e+02, -6.8938845e+02,
      -8.0039496e+02, -8.8267000e+02, -8.9787173e+02, -8.6951038e+02,
       4.6805939e+02,  1.7552178e+03,  1.5546574e+03,  1.1148179e+03,
       7.4606055e+02,  5.7001670e+02,  3.6944970e+02,  9.7838518e+01,
      -1.1941272e+02, -4.1377871e+02, -7.6382784e+02, -8.4434380e+02,
      -7.9151095e+02, -8.6493987e+02, -2.8343467e+02,  1.2284377e+03 ],

])

LTPF_C3_PREV = np.array([

    [  6.4066155e+01,  9.8702625e+01,  1.3788395e+02,  1.7357908e+02,
       2.0965843e+02,  2.3764314e+02,  2.6644923e+02,  2.9043075e+02,
       3.1194822e+02,  3.2482869e+02,  3.3339160e+02,  3.4071606e+02,
       3.4077768e+02,  3.3516446e+02,  3.2705459e+02,  3.1766371e+02,
       3.0450863e+02,  2.8884502e+02,  2.7626398e+02,  2.5687540e+02,
       2.3783751e+02,  2.1914457e+02,  1.9792827e+02,  1.7491941e+02,
       1.5135385e+02,  1.2210867e+02,  9.3873907e+01,  6.1133113e+01,
       2.5655472e+01, -9.0068272e+00, -4.6864087e+01, -8.5961652e+01,
      -1.2549724e+02, -1.6396142e+02, -2.0212479e+02, -2.3817055e+02,
      -2.7665292e+02, -3.0851152e+02, -3.4115267e+02, -3.6978674e+02,
      -3.9095963e+02, -4.0758998e+02, -4.1787548e+02, -4.2041787e+02,
      -4.1483578e+02, -4.0572811e+02, -3.8202738e+02, -3.5555284e+02,
      -3.2171184e+02, -2.8408934e+02, -2.3936543e+02, -1.8996836e+02,
      -1.4140617e+02, -9.0899975e+01, -4.2131433e+01,  5.1447212e+00,
       5.1259588e+01,  9.2032590e+01,  1.3108193e+02,  1.6957969e+02,
       2.0599062e+02,  2.3735997e+02,  2.6672423e+02,  2.9065678e+02,
       3.1149055e+02,  3.2543573e+02,  3.3604446e+02,  3.4053351e+02,
       3.4068048e+02,  3.3756977e+02,  3.3026320e+02,  3.2068706e+02,
       3.0912932e+02,  2.9328335e+02,  2.7625928e+02,  2.6245063e+02,
       2.4670232e+02,  2.2666582e+02,  2.0533576e+02,  1.8873190e+02,
       1.6715637e+02,  1.4420637e+02,  1.1629663e+02,  8.8561616e+01,
       5.8962789e+01,  2.5166894e+01, -1.1130553e+01, -4.5737395e+01,
      -8.2162714e+01, -1.2346999e+02, -1.6150200e+02, -2.0028107e+02,
      -2.3669696e+02, -2.7393564e+02, -3.0977094e+02, -3.4398480e+02,
      -3.6896668e+02, -3.9241987e+02, -4.0955117e+02, -4.2052014e+02,
      -4.2287410e+02, -4.1573504e+02, -4.0403021e+02, -3.8178373e+02,
      -3.5516185e+02, -3.2075205e+02, -2.8058887e+02, -2.3403076e+02,
      -1.8587624e+02, -1.3565255e+02, -8.4981034e+01, -3.4327765e+01,
       1.6122604e+01,  6.2803153e+01,  1.0703665e+02,  1.4333420e+02,
       1.8149169e+02,  2.1834576e+02,  2.4733100e+02,  2.7545243e+02,
       2.9756989e+02,  3.1498586e+02,  3.2836159e+02,  3.3505102e+02,
       3.3656005e+02,  3.3923920e+02,  3.3129967e+02,  3.2145289e+02,
       3.1074707e+02,  2.9901767e+02,  2.8311559e+02,  2.6682498e+02,
       2.4940742e+02,  2.3261497e+02,  2.1196627e+02,  1.9333406e+02,
       1.7509963e+02,  1.5279050e+02,  1.3109055e+02,  1.0449469e+02,
       7.8409595e+01,  4.9363873e+01,  1.7554459e+01, -1.5539732e+01,
      -4.8805768e+01, -8.5198692e+01, -1.2189071e+02, -1.5886754e+02,
      -1.9634205e+02, -2.3335312e+02, -2.7073502e+02, -3.0557489e+02,
      -3.3663951e+02, -3.6449227e+02, -3.8546603e+02, -4.0164063e+02,
      -4.1170176e+02, -4.1157567e+02, -4.0619410e+02, -3.9613725e+02 ],

    [ -3.7147766e+02, -3.4548841e+02, -3.1333857e+02, -2.7463625e+02,
      -2.2842732e+02, -1.8027124e+02, -1.3148070e+02, -8.3137241e+01,
      -3.0054878e+01,  1.9883532e+01,  6.6714609e+01,  1.0807815e+02,
       1.4822202e+02,  1.8586131e+02,  2.1800420e+02,  2.5060047e+02,
       2.7401990e+02,  2.9513896e+02,  3.1106912e+02,  3.2286594e+02,
       3.2918272e+02,  3.2833211e+02,  3.2739842e+02,  3.2088124e+02,
       3.1147526e+02,  2.9799117e+02,  2.8657122e+02,  2.7043218e+02,
       2.5431668e+02,  2.3996550e+02,  2.2300791e+02,  2.0270053e+02,
       1.8568534e+02,  1.6689279e+02,  1.5012389e+02,  1.2962033e+02,
       1.0942755e+02,  8.6477433e+01,  5.9720239e+01,  3.1778371e+01,
       2.9431398e+00, -2.9247908e+01, -6.3765298e+01, -9.6917096e+01,
      -1.3470119e+02, -1.7108334e+02, -2.0808704e+02, -2.4574024e+02,
      -2.8326377e+02, -3.1633765e+02, -3.4671102e+02, -3.7097994e+02,
      -3.8869922e+02, -4.0237282e+02, -4.0334704e+02, -4.0408504e+02,
      -3.9385794e+02, -3.7723701e+02, -3.5253919e+02, -3.2150862e+02,
      -2.8492979e+02, -2.4106826e+02, -1.9287257e+02, -1.4370111e+02,
      -9.5563935e+01, -4.3545337e+01,  7.0713976e+00,  5.5117442e+01,
       1.0039302e+02,  1.3954207e+02,  1.7504613e+02,  2.0852764e+02,
       2.3658875e+02,  2.6248596e+02,  2.8111481e+02,  2.9971212e+02,
       3.1019868e+02,  3.1449414e+02,  3.1475549e+02,  3.1252800e+02,
       3.0472635e+02,  2.9675698e+02,  2.8382811e+02,  2.7136329e+02,
       2.5584249e+02,  2.4078158e+02,  2.2398280e+02,  2.0974116e+02,
       1.9071035e+02,  1.7568738e+02,  1.5790704e+02,  1.4374608e+02,
       1.2653380e+02,  1.0989392e+02,  8.5342354e+01,  6.5160001e+01,
       3.5365747e+01,  9.6994716e+00, -1.9148759e+01, -5.2247954e+01,
      -8.4460390e+01, -1.1917021e+02, -1.5343168e+02, -1.9000568e+02,
      -2.2535999e+02, -2.6422733e+02, -2.9746477e+02, -3.2794048e+02,
      -3.5381425e+02, -3.7289983e+02, -3.8800620e+02, -3.9445315e+02,
      -3.9341234e+02, -3.8871213e+02, -3.7329028e+02, -3.5219370e+02,
      -3.2339521e+02, -2.8848818e+02, -2.4782613e+02, -2.0517168e+02,
      -1.5630958e+02, -1.0872773e+02, -5.8413582e+01, -7.9029246e+00,
       4.3081069e+01,  8.5378547e+01,  1.2552492e+02,  1.6074903e+02,
       1.9342830e+02,  2.2272087e+02,  2.4784891e+02,  2.6963531e+02,
       2.8673587e+02,  2.9921346e+02,  3.0619842e+02,  3.0724158e+02,
       3.0343005e+02,  2.9800183e+02,  2.8972999e+02,  2.7822622e+02,
       2.6355926e+02,  2.5308771e+02,  2.3630848e+02,  2.2381546e+02,
       2.0555172e+02,  1.8801065e+02,  1.7231093e+02,  1.5597740e+02,
       1.4132322e+02,  1.2578292e+02,  1.1037371e+02,  8.8726457e+01,
       6.8198855e+01,  3.9334854e+01,  1.1725578e+01, -1.6214439e+01,
      -4.7122569e+01, -8.2888237e+01, -1.1823402e+02, -1.5159407e+02 ],

])

LTPF_C3_TRANS = np.array([

      -1.8246409e+02, -2.2173994e+02, -2.6085374e+02, -2.9483717e+02,
      -3.2011835e+02, -3.3711584e+02, -3.3663215e+02, -3.2868626e+02,
      -3.1677331e+02, -3.1352548e+02, -3.1534653e+02, -3.1872408e+02,
      -3.1513281e+02, -3.1569554e+02, -2.9910316e+02, -2.9198409e+02,
      -2.5148901e+02, -2.0982533e+02, -1.4838034e+02, -8.5301198e+01,
      -3.6694674e+01,  2.3482066e+01,  8.5828448e+01,  1.3216759e+02,
       2.0172964e+02,  2.2393590e+02,  2.4953564e+02,  2.3256572e+02,
       2.1734484e+02,  1.9615122e+02,  1.2352168e+02,  7.1940230e+01,
       7.4762310e+01,  2.0724785e+02,  3.6354237e+02,  4.8584290e+02,
       5.5511323e+02,  5.3946449e+02,  5.0046673e+02,  3.9744659e+02,
       2.9036149e+02,  2.1569487e+02,  1.2216776e+02,  5.6342202e+01,
       1.9881173e+01, -1.0527142e+01, -2.8641229e+01, -6.5396380e+01,
      -2.2087227e+01, -1.9320385e+00,  9.9837180e+00, -1.2901176e+01,
      -4.8912099e+01, -1.0229260e+02, -1.3236180e+02, -1.1357973e+02,
      -1.6394336e+02, -2.0430337e+02, -2.3625105e+02, -2.7025726e+02,
      -2.1302135e+02, -2.2632729e+02, -2.0952571e+02, -2.1427507e+02,
      -2.5180218e+02, -2.9681097e+02, -3.2929367e+02, -3.2993365e+02,
      -3.3630273e+02, -3.0716107e+02, -2.2776881e+02, -2.2007043e+02,
      -1.7036950e+02, -1.6356561e+02, -1.0711820e+02,  3.8723772e+01,
       9.2665406e+01,  1.2719129e+02,  2.3419328e+02,  2.4827798e+02,
       2.2061102e+02,  1.4474493e+02,  1.0064081e+02,  9.5343023e+01,
       5.5501524e+01,  3.6218435e+00, -6.9277352e+01, -1.2309115e+01,
      -2.9683737e+01, -1.1059060e+01,  1.2650129e+02,  2.8907139e+02,
       5.9646998e+02,  7.2219690e+02,  6.4274592e+02,  2.5572095e+02,
       8.9009716e+00, -1.6887796e+02, -5.4165018e+02, -8.8289542e+02,
      -9.0816906e+02,  2.8400805e+02,  1.3941683e+03,  1.4288425e+03,
       1.0428507e+03,  6.0910516e+02,  4.9505488e+02,  5.1165385e+02,
      -4.9324168e+01, -5.0538446e+02, -6.1278936e+02, -6.0237215e+02,
      -1.0190721e+03, -1.3790932e+03, -1.3541381e+03,  7.7933615e+01,
       1.3253623e+03,  1.3970218e+03,  1.1115023e+03,  7.7044387e+02,
       5.1675398e+02,  2.7357534e+02, -1.1882263e+02, -3.6893899e+02,
      -5.9907798e+02, -6.8538645e+02, -1.0290340e+03, -1.2093927e+03,
      -1.3183905e+03, -6.4097534e+02,  7.8577340e+02,  1.3858651e+03,
       1.2575136e+03,  1.0978140e+03,  7.7239776e+02,  3.5047643e+02,
       1.3677282e+02, -2.9932975e+01, -3.3626785e+02, -6.8938845e+02,
      -8.0039496e+02, -8.8267000e+02, -8.9787173e+02, -8.6951038e+02,
       4.6805939e+02,  1.7552178e+03,  1.5546574e+03,  1.1148179e+03,
       7.4606055e+02,  5.7001670e+02,  3.6944970e+02,  9.7838518e+01,
      -1.1941272e+02, -4.1377871e+02, -7.6382784e+02, -8.4434380e+02,
      -7.9151095e+02, -8.6493987e+02, -2.8343467e+02,  1.2284377e+03,

])

LTPF_C4_NBITS = 320

LTPF_C4_ACTIVE = np.array([ True, True ])
LTPF_C4_PITCH_INDEX = np.array([ 56, 56 ])

LTPF_C4_C_N = np.array([
    [  2.0480302e-01,  1.4271871e-01, -6.4036434e-03 ],
    [  2.0480302e-01,  1.4271871e-01, -6.4036434e-03 ],
])

LTPF_C4_C_D = np.array([
    [  1.5868459e-02,  1.8368837e-01,  1.8368837e-01,  1.5868459e-02 ],
    [  1.5868459e-02,  1.8368837e-01,  1.8368837e-01,  1.5868459e-02 ],
])

LTPF_C4_X = np.array([

    [ -1.3212378e+02,  4.5530263e+00,  1.4615982e+02,  2.9071151e+02,
       4.2471908e+02,  5.4825306e+02,  6.6032559e+02,  7.5740827e+02,
       8.3891172e+02,  9.0995714e+02,  9.6747077e+02,  1.0121710e+03,
       1.0274871e+03,  1.0037721e+03,  9.3793010e+02,  8.2480912e+02,
       6.6687812e+02,  5.0610704e+02,  3.7535508e+02,  2.8417912e+02,
       2.3507950e+02,  2.2431573e+02,  2.3503665e+02,  2.5100781e+02,
       2.5591188e+02,  2.4573096e+02,  2.3732458e+02,  2.2379033e+02,
       2.0393082e+02,  1.7940329e+02,  1.5349002e+02,  1.2421970e+02,
       8.2607174e+01,  3.3890085e+01, -1.4919514e+01, -6.7050562e+01,
      -1.2022717e+02, -1.7802094e+02, -2.3045847e+02, -2.7788664e+02,
      -3.3683640e+02, -4.1806874e+02, -5.2071432e+02, -6.3745995e+02,
      -7.6831195e+02, -8.9933303e+02, -1.0058396e+03, -1.0704297e+03,
      -1.0883680e+03, -1.0715794e+03, -1.0235340e+03, -9.4781945e+02,
      -8.5584778e+02, -7.4716794e+02, -6.2920874e+02, -5.0223965e+02,
      -3.6316475e+02, -2.2074077e+02, -7.5442666e+01,  6.7658245e+01,
       2.0420074e+02,  3.3396592e+02,  4.4656898e+02,  5.4596637e+02,
       6.3808859e+02,  7.2787698e+02,  8.1472882e+02,  8.9279612e+02,
       9.5178350e+02,  9.7288475e+02,  9.4735790e+02,  8.8244167e+02,
       7.9508887e+02,  7.0282080e+02,  6.0598321e+02,  5.1640859e+02,
       4.3937674e+02,  3.7245496e+02,  3.2160255e+02,  2.8003421e+02,
       2.5119199e+02,  2.3495839e+02,  2.2200688e+02,  2.0973737e+02,
       1.9073208e+02,  1.6793312e+02,  1.4320783e+02,  1.1276855e+02,
       8.0839691e+01,  4.8429863e+01,  1.1662609e+01, -3.3424450e+01,
      -7.9726479e+01, -1.2393767e+02, -1.7110449e+02, -2.1662821e+02,
      -2.6310613e+02, -3.1774127e+02, -3.9885548e+02, -5.1583270e+02,
      -6.4243905e+02, -7.6184139e+02, -8.6566023e+02, -9.4822201e+02,
      -1.0112142e+03, -1.0437627e+03, -1.0457937e+03, -1.0207768e+03,
      -9.6819885e+02, -8.9378058e+02, -7.9861844e+02, -6.8734848e+02,
      -5.6329990e+02, -4.2840844e+02, -2.8589443e+02, -1.4101586e+02,
      -5.6273423e-01,  1.3949815e+02,  2.7417238e+02,  4.0010557e+02,
       5.2193759e+02,  6.3439172e+02,  7.3810702e+02,  8.2658657e+02,
       8.9885978e+02,  9.5846967e+02,  9.8349123e+02,  9.6474177e+02,
       9.0987618e+02,  8.3178192e+02,  7.4721512e+02,  6.6076640e+02,
       5.7958128e+02,  5.0626404e+02,  4.3829495e+02,  3.7955641e+02,
       3.3400402e+02,  3.0012060e+02,  2.6635949e+02,  2.3180026e+02,
       1.9980913e+02,  1.7106449e+02,  1.4708346e+02,  1.1651799e+02,
       8.3457665e+01,  5.6291779e+01,  3.0698836e+01,  5.4712667e+00,
      -2.9709210e+01, -7.0496428e+01, -1.0899206e+02, -1.5337912e+02,
      -1.9940203e+02, -2.4464990e+02, -2.9496636e+02, -3.6218204e+02,
      -4.5477003e+02, -5.6091604e+02, -6.6582484e+02, -7.5807547e+02 ],

    [ -8.3474175e+02, -8.9039341e+02, -9.2355843e+02, -9.3729800e+02,
      -9.2464041e+02, -8.9038579e+02, -8.3678014e+02, -7.6008712e+02,
      -6.6935538e+02, -5.6422713e+02, -4.4941540e+02, -3.2822721e+02,
      -1.9802065e+02, -6.6766936e+01,  6.3833225e+01,  1.9485046e+02,
       3.2069998e+02,  4.3761734e+02,  5.4460736e+02,  6.4521909e+02,
       7.4169295e+02,  8.2759672e+02,  8.9638366e+02,  9.3673025e+02,
       9.4669839e+02,  9.2302657e+02,  8.8064813e+02,  8.2833056e+02,
       7.6070809e+02,  6.9391450e+02,  6.2171679e+02,  5.4789771e+02,
       4.8160318e+02,  4.2010754e+02,  3.6391872e+02,  3.1081191e+02,
       2.6834464e+02,  2.2822565e+02,  1.8578653e+02,  1.5071546e+02,
       1.1735576e+02,  8.3222831e+01,  4.9624826e+01,  1.0744867e+01,
      -2.5764012e+01, -6.4812135e+01, -1.0358083e+02, -1.4353059e+02,
      -1.8297458e+02, -2.2359073e+02, -2.6775258e+02, -3.1656152e+02,
      -3.7871430e+02, -4.4751491e+02, -5.2485941e+02, -6.0299909e+02,
      -6.8515536e+02, -7.6401821e+02, -8.2770748e+02, -8.7387481e+02,
      -8.9915932e+02, -9.0643654e+02, -8.8914063e+02, -8.5144521e+02,
      -7.9103639e+02, -7.1441804e+02, -6.2692990e+02, -5.2729939e+02,
      -4.2275301e+02, -3.0657474e+02, -1.8688744e+02, -6.1633756e+01,
       5.8686296e+01,  1.8250335e+02,  3.0377201e+02,  4.1978552e+02,
       5.3054960e+02,  6.2835118e+02,  7.2035468e+02,  7.9934358e+02,
       8.5642737e+02,  8.9765907e+02,  9.2069014e+02,  9.1957966e+02,
       8.9145864e+02,  8.3848660e+02,  7.7429387e+02,  7.0401776e+02,
       6.3210907e+02,  5.6076246e+02,  4.9172735e+02,  4.2473574e+02,
       3.6317680e+02,  3.0715534e+02,  2.5275264e+02,  2.0664538e+02,
       1.6042746e+02,  1.1820978e+02,  7.3335304e+01,  3.4381568e+01,
      -5.8542117e+00, -4.4553111e+01, -8.3479478e+01, -1.2048090e+02,
      -1.6116063e+02, -2.0192831e+02, -2.4373220e+02, -2.8256043e+02,
      -3.2919802e+02, -3.7322793e+02, -4.2010268e+02, -4.7218497e+02,
      -5.3461522e+02, -6.1386549e+02, -6.9152470e+02, -7.6738453e+02,
      -8.2527123e+02, -8.6158674e+02, -8.7895938e+02, -8.7597504e+02,
      -8.5108741e+02, -8.0480220e+02, -7.4250154e+02, -6.5876002e+02,
      -5.6356029e+02, -4.5524527e+02, -3.4038894e+02, -2.2360514e+02,
      -1.0268552e+02,  2.5676655e+01,  1.5166378e+02,  2.7540954e+02,
       3.9465179e+02,  5.0881759e+02,  6.1514014e+02,  7.1241132e+02,
       7.9504576e+02,  8.7166197e+02,  9.3226594e+02,  9.7438372e+02,
       9.8472937e+02,  9.6485548e+02,  9.1860685e+02,  8.5256605e+02,
       7.8008367e+02,  7.0149395e+02,  6.2054247e+02,  5.4108956e+02,
       4.6123090e+02,  3.8599064e+02,  3.1521582e+02,  2.4589274e+02,
       1.8211975e+02,  1.3163470e+02,  8.4985461e+01,  3.6873046e+01,
      -6.0437124e+00, -4.8770494e+01, -8.4128492e+01, -1.2196451e+02 ],

])

LTPF_C4_PREV = np.array([

    [  8.5827694e+02,  8.3064489e+02,  8.0499625e+02,  7.8056066e+02,
       7.6744901e+02,  7.5611316e+02,  7.1123694e+02,  6.4234457e+02,
       5.7975881e+02,  5.3276775e+02,  5.0354201e+02,  4.6493297e+02,
       4.2651511e+02,  3.8147439e+02,  3.2876260e+02,  2.7004720e+02,
       2.0149312e+02,  1.2434150e+02,  4.6683685e+01, -2.4488830e+01,
      -9.4758661e+01, -1.6910076e+02, -2.4454068e+02, -3.2272819e+02,
      -4.1153117e+02, -5.2406430e+02, -6.6312995e+02, -7.9501146e+02,
      -9.1517707e+02, -1.0195879e+03, -1.1003671e+03, -1.1506760e+03,
      -1.1718109e+03, -1.1569278e+03, -1.1251298e+03, -1.0925962e+03,
      -1.0572755e+03, -1.0074938e+03, -9.3394117e+02, -8.3773785e+02,
      -7.2203590e+02, -5.9088086e+02, -4.5612949e+02, -3.2040519e+02,
      -1.7104088e+02, -2.9232508e+01,  1.0254422e+02,  2.3645451e+02,
       3.6931210e+02,  4.8954097e+02,  5.9869449e+02,  7.0325516e+02,
       7.9700038e+02,  8.6886012e+02,  9.1083211e+02,  9.2243956e+02,
       9.0940850e+02,  8.7320143e+02,  8.2061884e+02,  7.7544913e+02,
       7.3767280e+02,  7.0507470e+02,  6.8093384e+02,  6.6014067e+02,
       6.3137283e+02,  6.0264808e+02,  5.6924398e+02,  5.4487108e+02,
       5.1295321e+02,  4.8260718e+02,  4.4645113e+02,  4.0584256e+02,
       3.5619699e+02,  2.8935939e+02,  2.1760106e+02,  1.4449684e+02,
       7.4653166e+01, -1.4735621e+00, -8.3781827e+01, -1.5582846e+02,
      -2.3783590e+02, -3.2355910e+02, -4.0747488e+02, -5.0371066e+02,
      -6.0048961e+02, -7.0726549e+02, -8.2713764e+02, -9.4294256e+02,
      -1.0401191e+03, -1.1242695e+03, -1.1843296e+03, -1.1977911e+03,
      -1.1759957e+03, -1.1382443e+03, -1.0869538e+03, -1.0150254e+03,
      -9.2445459e+02, -8.2418678e+02, -7.1231843e+02, -5.8566805e+02,
      -4.4749524e+02, -3.2163393e+02, -1.9483077e+02, -6.6146712e+01,
       7.2606186e+01,  2.1709089e+02,  3.6141116e+02,  4.9774327e+02,
       6.2102585e+02,  7.3301494e+02,  8.2884523e+02,  9.1772806e+02,
       9.8287592e+02,  1.0258870e+03,  1.0350466e+03,  1.0124387e+03,
       9.5526828e+02,  8.4870459e+02,  6.9698370e+02,  5.5384197e+02,
       4.7329566e+02,  4.4187783e+02,  4.2164793e+02,  4.0723567e+02,
       4.0287348e+02,  3.9294225e+02,  3.8536364e+02,  3.6737694e+02,
       3.4038387e+02,  3.2024898e+02,  2.9202545e+02,  2.5623304e+02,
       2.0059931e+02,  1.4494932e+02,  8.7316562e+01,  2.0208470e+01,
      -5.1106062e+01, -1.0972390e+02, -1.7160643e+02, -2.3635682e+02,
      -3.0057986e+02, -3.6841959e+02, -4.5812030e+02, -5.5409463e+02,
      -6.5667832e+02, -7.6531178e+02, -9.0843842e+02, -1.0761546e+03,
      -1.2069858e+03, -1.2737779e+03, -1.2929641e+03, -1.2624172e+03,
      -1.2008239e+03, -1.1122508e+03, -9.9807662e+02, -8.7056340e+02,
      -7.3753241e+02, -6.0499387e+02, -4.6029956e+02, -3.2128673e+02 ],

    [ -1.8675929e+02, -4.2406693e+01,  1.0644289e+02,  2.5996083e+02,
       4.0591306e+02,  5.4343259e+02,  6.6906247e+02,  7.7899810e+02,
       8.7298704e+02,  9.5506440e+02,  1.0209952e+03,  1.0688758e+03,
       1.0833988e+03,  1.0572928e+03,  9.8779757e+02,  8.6895392e+02,
       7.0635769e+02,  5.4680902e+02,  4.2601966e+02,  3.4987333e+02,
       3.1179292e+02,  3.0331866e+02,  3.0951274e+02,  3.1789860e+02,
       3.1563910e+02,  3.0013119e+02,  2.8624618e+02,  2.6714149e+02,
       2.4150830e+02,  2.0837106e+02,  1.7059521e+02,  1.2855863e+02,
       7.5094806e+01,  1.5176971e+01, -4.3358321e+01, -1.0315083e+02,
      -1.6398964e+02, -2.2910287e+02, -2.9037721e+02, -3.5326931e+02,
      -4.3186627e+02, -5.2797253e+02, -6.4097239e+02, -7.7021355e+02,
      -9.1916034e+02, -1.0635912e+03, -1.1696907e+03, -1.2240756e+03,
      -1.2277996e+03, -1.1942767e+03, -1.1287017e+03, -1.0350001e+03,
      -9.2445100e+02, -7.9889645e+02, -6.6691580e+02, -5.2684803e+02,
      -3.7726996e+02, -2.2821747e+02, -7.6317887e+01,  7.6079467e+01,
       2.2531300e+02,  3.6948450e+02,  4.9788068e+02,  6.1407961e+02,
       7.2081446e+02,  8.2045742e+02,  9.1242924e+02,  9.9212633e+02,
       1.0508862e+03,  1.0718419e+03,  1.0464386e+03,  9.7937941e+02,
       8.8142794e+02,  7.6477603e+02,  6.3740319e+02,  5.2412010e+02,
       4.3571759e+02,  3.6988854e+02,  3.2884698e+02,  3.0173096e+02,
       2.8691049e+02,  2.7882926e+02,  2.6731147e+02,  2.5356111e+02,
       2.3340364e+02,  2.0894382e+02,  1.8071643e+02,  1.4579425e+02,
       1.0855560e+02,  6.8052759e+01,  2.0813765e+01, -3.3555159e+01,
      -8.7826734e+01, -1.4078458e+02, -1.9733530e+02, -2.5226649e+02,
      -3.0811893e+02, -3.7371017e+02, -4.6559419e+02, -5.8909591e+02,
      -7.2198295e+02, -8.5479624e+02, -9.7874703e+02, -1.0795937e+03,
      -1.1503967e+03, -1.1796433e+03, -1.1713652e+03, -1.1317595e+03,
      -1.0619895e+03, -9.6960460e+02, -8.5726543e+02, -7.3060563e+02,
      -5.9289544e+02, -4.4495754e+02, -2.9059997e+02, -1.3479531e+02,
       1.7757467e+01,  1.7012020e+02,  3.1647370e+02,  4.5257127e+02,
       5.8125188e+02,  6.9866277e+02,  8.0708027e+02,  9.0151273e+02,
       9.8106784e+02,  1.0459462e+03,  1.0733030e+03,  1.0542555e+03,
       9.9525660e+02,  9.0830183e+02,  8.0934326e+02,  7.0399220e+02,
       6.0377076e+02,  5.1638905e+02,  4.4150465e+02,  3.8256291e+02,
       3.4048809e+02,  3.1118639e+02,  2.8406392e+02,  2.5713040e+02,
       2.3131764e+02,  2.0600503e+02,  1.8197113e+02,  1.5035197e+02,
       1.1569230e+02,  8.4266914e+01,  5.1996878e+01,  1.7843797e+01,
      -2.6863170e+01, -7.5983644e+01, -1.2245636e+02, -1.7436828e+02,
      -2.2708180e+02, -2.7906654e+02, -3.3755256e+02, -4.1588336e+02,
      -5.2295173e+02, -6.4523634e+02, -7.6683613e+02, -8.7677439e+02 ],

])

LTPF_C4_TRANS = np.array([

      -9.6976885e+02, -1.0376330e+03, -1.0764010e+03, -1.0872965e+03,
      -1.0660439e+03, -1.0190108e+03, -9.4899438e+02, -8.5471701e+02,
      -7.4561708e+02, -6.2178553e+02, -4.8798080e+02, -3.4713910e+02,
      -1.9841833e+02, -5.0343781e+01,  9.6414723e+01,  2.4225179e+02,
       3.8080735e+02,  5.0947979e+02,  6.2776176e+02,  7.3826963e+02,
       8.4169803e+02,  9.3159353e+02,  1.0031138e+03,  1.0438329e+03,
       1.0480614e+03,  1.0127914e+03,  9.5385188e+02,  8.8138762e+02,
       7.9431025e+02,  7.0979394e+02,  6.2416163e+02,  5.4300509e+02,
       4.7372434e+02,  4.1360311e+02,  3.6279724e+02,  3.1673275e+02,
       2.7937838e+02,  2.4267574e+02,  2.0418215e+02,  1.7211796e+02,
       1.3914947e+02,  1.0348883e+02,  6.8276707e+01,  2.9143910e+01,
      -7.9633499e+00, -4.9829988e+01, -9.3834050e+01, -1.3941156e+02,
      -1.8501204e+02, -2.3271716e+02, -2.8327412e+02, -3.3841588e+02,
      -4.0881202e+02, -4.9188847e+02, -5.8927352e+02, -6.8905431e+02,
      -7.8967502e+02, -8.8144454e+02, -9.5326854e+02, -1.0026317e+03,
      -1.0264190e+03, -1.0268593e+03, -9.9874748e+02, -9.4795333e+02,
      -8.7252905e+02, -7.7954835e+02, -6.7411996e+02, -5.5559866e+02,
      -4.3134824e+02, -2.9558412e+02, -1.5723885e+02, -1.5143571e+01,
       1.2171974e+02,  2.6041913e+02,  3.9322892e+02,  5.1828299e+02,
       6.3629063e+02,  7.4159801e+02,  8.3990996e+02,  9.2215495e+02,
       9.7897443e+02,  1.0129522e+03,  1.0197564e+03,  9.9746243e+02,
       9.4932784e+02,  8.7962855e+02,  8.0193420e+02,  7.2094920e+02,
       6.4030858e+02,  5.6344020e+02,  4.9240203e+02,  4.2631986e+02,
       3.6714651e+02,  3.1424334e+02,  2.6363836e+02,  2.1946532e+02,
       1.7487874e+02,  1.3448581e+02,  9.0857113e+01,  5.1852661e+01,
       1.0318345e+01, -3.0189483e+01, -7.1704052e+01, -1.1290529e+02,
      -1.5806523e+02, -2.0311837e+02, -2.4946754e+02, -2.9428673e+02,
      -3.4738868e+02, -4.0117413e+02, -4.6307505e+02, -5.3393190e+02,
      -6.1557536e+02, -7.0983150e+02, -7.9873811e+02, -8.8091592e+02,
      -9.4071232e+02, -9.7639817e+02, -9.9018818e+02, -9.8005670e+02,
      -9.4511932e+02, -8.8673987e+02, -8.1023192e+02, -7.1271293e+02,
      -6.0397885e+02, -4.8247591e+02, -3.5415373e+02, -2.2240894e+02,
      -8.6345911e+01,  5.4855371e+01,  1.9247458e+02,  3.2774740e+02,
       4.5715456e+02,  5.8013934e+02,  6.9372155e+02,  7.9720836e+02,
       8.8545637e+02,  9.6285451e+02,  1.0187327e+03,  1.0521050e+03,
       1.0516389e+03,  1.0206935e+03,  9.6358799e+02,  8.8842240e+02,
       8.0838051e+02,  7.2366647e+02,  6.3870123e+02,  5.5707357e+02,
       4.7704505e+02,  4.0312802e+02,  3.3463611e+02,  2.6847649e+02,
       2.0824825e+02,  1.5901226e+02,  1.1171069e+02,  6.3026014e+01,
       1.8987746e+01, -2.5300684e+01, -6.3942070e+01, -1.0563596e+02,

])

LTPF_C5_NBITS = 320

LTPF_C5_ACTIVE = np.array([ True, True ])
LTPF_C5_PITCH_INDEX = np.array([ 56, 52 ])

LTPF_C5_C_N = np.array([
    [  2.0480302e-01,  1.4271871e-01, -6.4036434e-03 ],
    [  2.0480302e-01,  1.4271871e-01, -6.4036434e-03 ],
])

LTPF_C5_C_D = np.array([
    [  1.5868459e-02,  1.8368837e-01,  1.8368837e-01,  1.5868459e-02 ],
    [  4.2799674e-02,  2.2003000e-01,  1.3427625e-01,  2.6795433e-03 ],
])

LTPF_C5_X = np.array([

    [ -1.6060766e+02, -1.9568387e+02, -2.3100280e+02, -2.6630342e+02,
      -3.0336764e+02, -3.3854889e+02, -3.7043669e+02, -4.1163573e+02,
      -4.5567039e+02, -5.1558927e+02, -5.7814485e+02, -6.4080974e+02,
      -6.9798700e+02, -7.4456362e+02, -7.7654511e+02, -7.9319453e+02,
      -7.9402247e+02, -7.7652860e+02, -7.4572416e+02, -6.9989705e+02,
      -6.3841065e+02, -5.6041492e+02, -4.6954673e+02, -3.7010557e+02,
      -2.6541886e+02, -1.5525665e+02, -4.1036675e+01,  7.7898304e+01,
       1.9450287e+02,  3.0691788e+02,  4.1560715e+02,  5.1731671e+02,
       6.1122047e+02,  6.9593032e+02,  7.7071271e+02,  8.3246711e+02,
       8.7294256e+02,  8.9535004e+02,  8.9058119e+02,  8.6785095e+02,
       8.2896133e+02,  7.8301121e+02,  7.2447207e+02,  6.6383224e+02,
       5.9764845e+02,  5.3040514e+02,  4.6518541e+02,  3.9659186e+02,
       3.2746766e+02,  2.6753027e+02,  2.0739840e+02,  1.4677357e+02,
       9.5287069e+01,  4.7197825e+01, -1.1700140e+00, -4.9161682e+01,
      -9.6726950e+01, -1.4038011e+02, -1.8662421e+02, -2.2897297e+02,
      -2.6864070e+02, -3.1060300e+02, -3.4908047e+02, -3.8903634e+02,
      -4.3354889e+02, -4.7982553e+02, -5.3003184e+02, -5.8409985e+02,
      -6.3676286e+02, -6.8405252e+02, -7.2580553e+02, -7.5743871e+02,
      -7.7706979e+02, -7.8082023e+02, -7.7270174e+02, -7.4653280e+02,
      -7.1067906e+02, -6.5596456e+02, -5.8829707e+02, -5.0656026e+02,
      -4.1202861e+02, -3.1175500e+02, -2.0302114e+02, -9.4710595e+01,
       2.0375247e+01,  1.3270062e+02,  2.4243959e+02,  3.5297406e+02,
       4.5553582e+02,  5.5590475e+02,  6.4792544e+02,  7.2690018e+02,
       7.9536436e+02,  8.5008231e+02,  8.8455109e+02,  9.0442411e+02,
       8.9923508e+02,  8.7803657e+02,  8.3884819e+02,  7.8786430e+02,
       7.2427475e+02,  6.5372622e+02,  5.8223046e+02,  5.0499317e+02,
       4.2757252e+02,  3.4979799e+02,  2.7384122e+02,  2.0202195e+02,
       1.3245127e+02,  6.9157310e+01,  1.0836031e+01, -5.0190037e+01,
      -1.0459364e+02, -1.5805356e+02, -2.0733214e+02, -2.4766726e+02,
      -2.9312137e+02, -3.3816241e+02, -3.7627128e+02, -4.1399424e+02,
      -4.4836893e+02, -4.8327539e+02, -5.1189667e+02, -5.4560547e+02,
      -5.8121896e+02, -6.1491595e+02, -6.4841741e+02, -6.7347617e+02,
      -6.9479054e+02, -7.0418229e+02, -7.0192552e+02, -6.8320863e+02,
      -6.5899253e+02, -6.2202504e+02, -5.7186005e+02, -5.1046639e+02,
      -4.4158751e+02, -3.6145704e+02, -2.7098565e+02, -1.7538139e+02,
      -7.6006538e+01,  2.3722024e+01,  1.2888233e+02,  2.3123425e+02,
       3.3082160e+02,  4.2819177e+02,  5.1739280e+02,  6.0102219e+02,
       6.7529473e+02,  7.3951297e+02,  8.0008252e+02,  8.4203459e+02,
       8.6655983e+02,  8.6963556e+02,  8.5619100e+02,  8.2481263e+02,
       7.8075652e+02,  7.2216895e+02,  6.6035959e+02,  5.8959592e+02 ],

    [  5.1338681e+02,  4.3458985e+02,  3.6021867e+02,  2.8040781e+02,
       2.0424899e+02,  1.3450352e+02,  6.6513343e+01,  5.6705127e+00,
      -5.2746410e+01, -1.1038385e+02, -1.6030998e+02, -2.0583244e+02,
      -2.4887172e+02, -2.9195508e+02, -3.2720109e+02, -3.6021736e+02,
      -3.9187844e+02, -4.2450350e+02, -4.5121148e+02, -4.8051913e+02,
      -5.0679429e+02, -5.3473222e+02, -5.6274008e+02, -5.8904566e+02,
      -6.1354818e+02, -6.2864496e+02, -6.3771702e+02, -6.3780508e+02,
      -6.2690341e+02, -6.0523722e+02, -5.7468865e+02, -5.3607922e+02,
      -4.8568411e+02, -4.2700735e+02, -3.5998795e+02, -2.8215398e+02,
      -2.0120100e+02, -1.1187382e+02, -2.5873841e+01,  6.7900858e+01,
       1.6266005e+02,  2.5147024e+02,  3.4066102e+02,  4.2711439e+02,
       5.0521368e+02,  5.8027967e+02,  6.4926123e+02,  7.0606425e+02,
       7.5197499e+02,  7.7936955e+02,  7.9241574e+02,  7.9273830e+02,
       7.7226162e+02,  7.4318941e+02,  6.9912979e+02,  6.4987144e+02,
       5.8935287e+02,  5.2867427e+02,  4.5818050e+02,  3.9091836e+02,
       3.1175882e+02,  2.4316163e+02,  1.6901936e+02,  1.0131712e+02,
       4.0884705e+01, -1.8284360e+01, -7.2463773e+01, -1.2615956e+02,
      -1.7449474e+02, -2.1740936e+02, -2.5946712e+02, -2.9956841e+02,
      -3.3765147e+02, -3.6878225e+02, -4.0090001e+02, -4.3495156e+02,
      -4.6061972e+02, -4.8952939e+02, -5.1712969e+02, -5.4061916e+02,
      -5.6432757e+02, -5.8583512e+02, -5.9997738e+02, -6.0321240e+02,
      -6.0215834e+02, -5.9411694e+02, -5.7827835e+02, -5.5352370e+02,
      -5.2218352e+02, -4.8118496e+02, -4.3153207e+02, -3.7290541e+02,
      -3.0763879e+02, -2.3045577e+02, -1.5030005e+02, -6.8343251e+01,
       1.9827505e+01,  1.0879534e+02,  2.0178224e+02,  2.8957061e+02,
       3.7779733e+02,  4.6428381e+02,  5.4088918e+02,  6.1377535e+02,
       6.8124334e+02,  7.3734864e+02,  7.7681316e+02,  8.0293170e+02,
       8.0890279e+02,  8.0620381e+02,  7.8580399e+02,  7.5204050e+02,
       7.1105755e+02,  6.5621411e+02,  5.9523050e+02,  5.2862471e+02,
       4.5293754e+02,  3.7743682e+02,  3.0188947e+02,  2.2243974e+02,
       1.4611001e+02,  6.9741588e+01,  3.0745399e+00, -6.6469812e+01,
      -1.2845539e+02, -1.8845561e+02, -2.4293629e+02, -2.9314084e+02,
      -3.3894702e+02, -3.8275365e+02, -4.2041853e+02, -4.5718992e+02,
      -4.8545538e+02, -5.1088490e+02, -5.3523663e+02, -5.5122371e+02,
      -5.7210009e+02, -5.8455365e+02, -5.9877467e+02, -6.0919241e+02,
      -6.1143839e+02, -6.0342076e+02, -5.8730343e+02, -5.6903506e+02,
      -5.3921882e+02, -5.0539402e+02, -4.6156875e+02, -4.1089541e+02,
      -3.4967700e+02, -2.8252157e+02, -2.1092501e+02, -1.3135475e+02,
      -4.8131277e+01,  3.4353741e+01,  1.2201908e+02,  2.0824054e+02,
       2.9606295e+02,  3.8043837e+02,  4.5867136e+02,  5.3182302e+02 ],

])

LTPF_C5_PREV = np.array([

    [ -9.6976885e+02, -1.0376330e+03, -1.0764010e+03, -1.0872965e+03,
      -1.0660439e+03, -1.0190108e+03, -9.4899438e+02, -8.5471701e+02,
      -7.4561708e+02, -6.2178553e+02, -4.8798080e+02, -3.4713910e+02,
      -1.9841833e+02, -5.0343781e+01,  9.6414723e+01,  2.4225179e+02,
       3.8080735e+02,  5.0947979e+02,  6.2776176e+02,  7.3826963e+02,
       8.4169803e+02,  9.3159353e+02,  1.0031138e+03,  1.0438329e+03,
       1.0480614e+03,  1.0127914e+03,  9.5385188e+02,  8.8138762e+02,
       7.9431025e+02,  7.0979394e+02,  6.2416163e+02,  5.4300509e+02,
       4.7372434e+02,  4.1360311e+02,  3.6279724e+02,  3.1673275e+02,
       2.7937838e+02,  2.4267574e+02,  2.0418215e+02,  1.7211796e+02,
       1.3914947e+02,  1.0348883e+02,  6.8276707e+01,  2.9143910e+01,
      -7.9633499e+00, -4.9829988e+01, -9.3834050e+01, -1.3941156e+02,
      -1.8501204e+02, -2.3271716e+02, -2.8327412e+02, -3.3841588e+02,
      -4.0881202e+02, -4.9188847e+02, -5.8927352e+02, -6.8905431e+02,
      -7.8967502e+02, -8.8144454e+02, -9.5326854e+02, -1.0026317e+03,
      -1.0264190e+03, -1.0268593e+03, -9.9874748e+02, -9.4795333e+02,
      -8.7252905e+02, -7.7954835e+02, -6.7411996e+02, -5.5559866e+02,
      -4.3134824e+02, -2.9558412e+02, -1.5723885e+02, -1.5143571e+01,
       1.2171974e+02,  2.6041913e+02,  3.9322892e+02,  5.1828299e+02,
       6.3629063e+02,  7.4159801e+02,  8.3990996e+02,  9.2215495e+02,
       9.7897443e+02,  1.0129522e+03,  1.0197564e+03,  9.9746243e+02,
       9.4932784e+02,  8.7962855e+02,  8.0193420e+02,  7.2094920e+02,
       6.4030858e+02,  5.6344020e+02,  4.9240203e+02,  4.2631986e+02,
       3.6714651e+02,  3.1424334e+02,  2.6363836e+02,  2.1946532e+02,
       1.7487874e+02,  1.3448581e+02,  9.0857113e+01,  5.1852661e+01,
       1.0318345e+01, -3.0189483e+01, -7.1704052e+01, -1.1290529e+02,
      -1.5806523e+02, -2.0311837e+02, -2.4946754e+02, -2.9428673e+02,
      -3.4738868e+02, -4.0117413e+02, -4.6307505e+02, -5.3393190e+02,
      -6.1557536e+02, -7.0983150e+02, -7.9873811e+02, -8.8091592e+02,
      -9.4071232e+02, -9.7639817e+02, -9.9018818e+02, -9.8005670e+02,
      -9.4511932e+02, -8.8673987e+02, -8.1023192e+02, -7.1271293e+02,
      -6.0397885e+02, -4.8247591e+02, -3.5415373e+02, -2.2240894e+02,
      -8.6345911e+01,  5.4855371e+01,  1.9247458e+02,  3.2774740e+02,
       4.5715456e+02,  5.8013934e+02,  6.9372155e+02,  7.9720836e+02,
       8.8545637e+02,  9.6285451e+02,  1.0187327e+03,  1.0521050e+03,
       1.0516389e+03,  1.0206935e+03,  9.6358799e+02,  8.8842240e+02,
       8.0838051e+02,  7.2366647e+02,  6.3870123e+02,  5.5707357e+02,
       4.7704505e+02,  4.0312802e+02,  3.3463611e+02,  2.6847649e+02,
       2.0824825e+02,  1.5901226e+02,  1.1171069e+02,  6.3026014e+01,
       1.8987746e+01, -2.5300684e+01, -6.3942070e+01, -1.0563596e+02 ],

    [ -1.4774474e+02, -1.8760166e+02, -2.2888898e+02, -2.7036124e+02,
      -3.1332685e+02, -3.5581512e+02, -3.9771599e+02, -4.4936892e+02,
      -5.0524457e+02, -5.7736164e+02, -6.5368200e+02, -7.3121324e+02,
      -8.0200835e+02, -8.5934306e+02, -8.9755526e+02, -9.1640499e+02,
      -9.1557588e+02, -8.9266707e+02, -8.5215743e+02, -7.9312106e+02,
      -7.1595184e+02, -6.2138325e+02, -5.1398723e+02, -3.9767506e+02,
      -2.7618028e+02, -1.4941349e+02, -1.8433316e+01,  1.1606930e+02,
       2.4697130e+02,  3.7323736e+02,  4.9468012e+02,  6.0791740e+02,
       7.1199453e+02,  8.0485505e+02,  8.8578958e+02,  9.5117733e+02,
       9.9264372e+02,  1.0115211e+03,  9.9864359e+02,  9.6401858e+02,
       9.1024930e+02,  8.4835436e+02,  7.7530833e+02,  7.0141455e+02,
       6.2384959e+02,  5.4721968e+02,  4.7387003e+02,  3.9972348e+02,
       3.2726546e+02,  2.6394537e+02,  2.0233246e+02,  1.4286080e+02,
       9.1074446e+01,  4.1308373e+01, -8.2303699e+00, -5.6391830e+01,
      -1.0380366e+02, -1.4867461e+02, -1.9589518e+02, -2.3948198e+02,
      -2.8179404e+02, -3.2661184e+02, -3.6849411e+02, -4.1204691e+02,
      -4.6073272e+02, -5.1299590e+02, -5.7214182e+02, -6.3765069e+02,
      -7.0266168e+02, -7.6246325e+02, -8.1462585e+02, -8.5309773e+02,
      -8.7583931e+02, -8.7977551e+02, -8.6820607e+02, -8.3598008e+02,
      -7.9127858e+02, -7.2555633e+02, -6.4513927e+02, -5.4928065e+02,
      -4.4082024e+02, -3.2665281e+02, -2.0436663e+02, -8.1707251e+01,
       4.7901921e+01,  1.7432524e+02,  2.9759562e+02,  4.1992512e+02,
       5.3317251e+02,  6.4237470e+02,  7.4113789e+02,  8.2606130e+02,
       8.9883116e+02,  9.5428904e+02,  9.8624536e+02,  9.9869004e+02,
       9.8266888e+02,  9.4925619e+02,  8.9813116e+02,  8.3623319e+02,
       7.6348823e+02,  6.8594085e+02,  6.0808870e+02,  5.2646492e+02,
       4.4600650e+02,  3.6561691e+02,  2.8872526e+02,  2.1691496e+02,
       1.4728773e+02,  8.4221474e+01,  2.6048470e+01, -3.4388207e+01,
      -8.8802146e+01, -1.4299353e+02, -1.9336320e+02, -2.3710707e+02,
      -2.8587475e+02, -3.3270152e+02, -3.7422945e+02, -4.1636455e+02,
      -4.5573665e+02, -4.9726692e+02, -5.3543762e+02, -5.8065815e+02,
      -6.2910888e+02, -6.7690028e+02, -7.2375327e+02, -7.6132305e+02,
      -7.9286869e+02, -8.0958989e+02, -8.1191872e+02, -7.9586469e+02,
      -7.7057333e+02, -7.2915594e+02, -6.7231474e+02, -6.0134288e+02,
      -5.1986758e+02, -4.2496645e+02, -3.1966315e+02, -2.0893647e+02,
      -9.4117050e+01,  2.1901801e+01,  1.4286090e+02,  2.5974483e+02,
       3.7391931e+02,  4.8475194e+02,  5.8679531e+02,  6.8253551e+02,
       7.6686124e+02,  8.3927651e+02,  9.0416036e+02,  9.4669654e+02,
       9.6938119e+02,  9.6791538e+02,  9.4724817e+02,  9.0746962e+02,
       8.5438295e+02,  7.8710681e+02,  7.1621104e+02,  6.3739148e+02 ],

])

LTPF_C5_TRANS = np.array([

       5.5469254e+02,  4.6885623e+02,  3.8726237e+02,  3.0166318e+02,
       2.2151059e+02,  1.4759060e+02,  7.6115964e+01,  1.2068315e+01,
      -5.0213437e+01, -1.1124791e+02, -1.6496412e+02, -2.1495097e+02,
      -2.6214927e+02, -3.0944212e+02, -3.5014054e+02, -3.8896389e+02,
      -4.2617800e+02, -4.6406673e+02, -4.9691186e+02, -5.3314161e+02,
      -5.6776429e+02, -6.0515795e+02, -6.4220526e+02, -6.7634237e+02,
      -7.0656015e+02, -7.2541177e+02, -7.3535446e+02, -7.3265481e+02,
      -7.1676788e+02, -6.8818630e+02, -6.4788454e+02, -5.9638237e+02,
      -5.3135114e+02, -4.5622803e+02, -3.7085252e+02, -2.7464233e+02,
      -1.7527279e+02, -6.8212512e+01,  3.6401489e+01,  1.4772605e+02,
       2.5777143e+02,  3.5986766e+02,  4.6046220e+02,  5.5572298e+02,
       6.4084254e+02,  7.2023220e+02,  7.9091946e+02,  8.4596132e+02,
       8.8561830e+02,  9.0316671e+02,  9.0353933e+02,  8.8830570e+02,
       8.5210347e+02,  8.0683786e+02,  7.4767132e+02,  6.8402115e+02,
       6.1000133e+02,  5.3637714e+02,  4.5522905e+02,  3.7796360e+02,
       2.9165202e+02,  2.1765490e+02,  1.3937214e+02,  6.9124279e+01,
       5.1484918e+00, -5.7974304e+01, -1.1532103e+02, -1.7106128e+02,
      -2.2144596e+02, -2.6763632e+02, -3.1235860e+02, -3.5424602e+02,
      -3.9420762e+02, -4.2862777e+02, -4.6393345e+02, -5.0059201e+02,
      -5.3052040e+02, -5.6462343e+02, -5.9733595e+02, -6.2614003e+02,
      -6.5395598e+02, -6.7654796e+02, -6.8956058e+02, -6.9036404e+02,
      -6.8409998e+02, -6.6772245e+02, -6.4131629e+02, -6.0423695e+02,
      -5.5819928e+02, -5.0085813e+02, -4.3386043e+02, -3.5697723e+02,
      -2.7353263e+02, -1.7946159e+02, -8.3894727e+01,  1.3870299e+01,
       1.1661147e+02,  2.1714406e+02,  3.1916911e+02,  4.1466126e+02,
       5.0807281e+02,  5.9699545e+02,  6.7493085e+02,  7.4602304e+02,
       8.0681874e+02,  8.5162090e+02,  8.7732032e+02,  8.8801241e+02,
       8.7733827e+02,  8.5711309e+02,  8.1939693e+02,  7.7020353e+02,
       7.1415915e+02,  6.4639833e+02,  5.7422282e+02,  4.9782204e+02,
       4.1400066e+02,  3.3290948e+02,  2.5272667e+02,  1.7094246e+02,
       9.4679864e+01,  1.9214845e+01, -4.7039158e+01, -1.1560544e+02,
      -1.7609914e+02, -2.3440216e+02, -2.8752910e+02, -3.3711616e+02,
      -3.8273508e+02, -4.2591674e+02, -4.6397521e+02, -5.0230060e+02,
      -5.3283216e+02, -5.6227577e+02, -5.9128189e+02, -6.1269433e+02,
      -6.3821208e+02, -6.5457809e+02, -6.7024858e+02, -6.7829939e+02,
      -6.7675317e+02, -6.6455106e+02, -6.4328023e+02, -6.1728360e+02,
      -5.7844437e+02, -5.3391997e+02, -4.7785175e+02, -4.1395668e+02,
      -3.3933775e+02, -2.5814915e+02, -1.7234111e+02, -7.9862248e+01,
       1.5711615e+01,  1.1047993e+02,  2.0926197e+02,  3.0481978e+02,
       4.0036676e+02,  4.9118016e+02,  5.7429258e+02,  6.5078515e+02,

])

### C.4.1.10  Output signal clipping

X_HAT_CLIP_10M = np.array([

    [  6.1525095e-01,  1.7117620e+00, -1.6200436e-01, -2.1107548e+00,
       9.1636911e-01,  8.7583398e+00,  1.4291828e+01,  1.4615904e+01,
       1.9312730e+01,  2.2178311e+01,  2.1178760e+01,  1.9939021e+01,
       3.2745193e+01,  3.1385007e+01,  4.5642888e+01,  3.8185668e+01,
       4.3452271e+01,  3.0174130e+01,  2.7510416e+01,  3.9048290e+00,
      -4.1911157e+00, -3.4032526e+01, -4.3089457e+01, -7.1817465e+01,
      -7.3319439e+01, -1.0231340e+02, -8.2645833e+01, -1.0207070e+02,
      -7.4253115e+01, -9.5269932e+01, -1.2210097e+02, -1.3216707e+02,
      -1.2665681e+02, -1.6033791e+02, -1.3067613e+02, -1.8796611e+01,
       6.2097263e+01,  7.2290617e+00, -1.2550979e+02, -8.9649115e+01,
       7.6135408e+02,  2.7072170e+03,  6.1558256e+03,  9.6522574e+03,
       1.2566221e+04,  1.5421574e+04,  1.8329565e+04,  2.1102710e+04,
       2.3323039e+04,  2.5451924e+04,  2.7365468e+04,  2.8891223e+04,
       3.0341813e+04,  3.1380038e+04,  3.2134454e+04,  3.2606958e+04,
       3.2683895e+04,  3.2518324e+04,  3.2036626e+04,  3.1278795e+04,
       3.0136453e+04,  2.8694957e+04,  2.7009676e+04,  2.5075921e+04,
       2.2940269e+04,  2.0564299e+04,  1.7956441e+04,  1.5202722e+04,
       1.2352834e+04,  9.3639445e+03,  6.2714619e+03,  3.1112267e+03,
      -1.0075267e+02, -3.3115389e+03, -6.4954073e+03, -9.6110111e+03,
      -1.2594144e+04, -1.5477538e+04, -1.8214406e+04, -2.0797170e+04,
      -2.3195615e+04, -2.5355124e+04, -2.7302557e+04, -2.8919494e+04,
      -3.0273227e+04, -3.1357707e+04, -3.2152224e+04, -3.2625545e+04,
      -3.2768000e+04, -3.2640407e+04, -3.2172861e+04, -3.1393517e+04,
      -3.0316009e+04, -2.8922986e+04, -2.7290775e+04, -2.5351571e+04,
      -2.3183937e+04, -2.0795771e+04, -1.8212609e+04, -1.5453610e+04,
      -1.2543978e+04, -9.4923186e+03, -6.3663932e+03, -3.1740539e+03,
       4.2407582e+01,  3.2479359e+03,  6.4369448e+03,  9.5564107e+03,
       1.2589389e+04,  1.5494437e+04,  1.8238168e+04,  2.0814647e+04,
       2.3183440e+04,  2.5340307e+04,  2.7236643e+04,  2.8879098e+04,
       3.0219987e+04,  3.1304367e+04,  3.2073455e+04,  3.2541098e+04,
       3.2660415e+04,  3.2490788e+04,  3.2004309e+04,  3.1215306e+04,
       3.0131477e+04,  2.8751611e+04,  2.7114762e+04,  2.5198099e+04,
       2.3044166e+04,  2.0655799e+04,  1.8068458e+04,  1.5312555e+04,
       1.2400688e+04,  9.3834439e+03,  6.2670869e+03,  3.1043555e+03,
      -9.0364008e+01, -3.2949280e+03, -6.4558716e+03, -9.5736656e+03,
      -1.2593429e+04, -1.5487321e+04, -1.8231003e+04, -2.0808229e+04,
      -2.3191386e+04, -2.5327915e+04, -2.7226934e+04, -2.8860268e+04,
      -3.0235350e+04, -3.1313467e+04, -3.2092397e+04, -3.2585580e+04,
      -3.2768000e+04, -3.2606534e+04, -3.2146234e+04, -3.1383325e+04,
      -3.0295384e+04, -2.8926934e+04, -2.7297985e+04, -2.5343669e+04 ],

    [ -2.3210612e+04, -2.0830109e+04, -1.8262528e+04, -1.5491202e+04,
      -1.2640992e+04, -9.6308521e+03, -6.5120997e+03, -3.2981272e+03,
      -1.0437273e+02,  3.1237705e+03,  6.3475523e+03,  9.4251090e+03,
       1.2406927e+04,  1.5369182e+04,  1.8133410e+04,  2.0696011e+04,
       2.3075625e+04,  2.5239168e+04,  2.7178122e+04,  2.8863775e+04,
       3.0239107e+04,  3.1292540e+04,  3.2055374e+04,  3.2528526e+04,
       3.2691792e+04,  3.2554032e+04,  3.2080858e+04,  3.1269701e+04,
       3.0219849e+04,  2.8872409e+04,  2.7201446e+04,  2.5283194e+04,
       2.3137397e+04,  2.0746427e+04,  1.8171258e+04,  1.5425482e+04,
       1.2542866e+04,  9.5154094e+03,  6.4038902e+03,  3.2243297e+03,
       1.8093877e+01, -3.2004786e+03, -6.3778766e+03, -9.4986572e+03,
      -1.2530644e+04, -1.5438896e+04, -1.8199553e+04, -2.0786963e+04,
      -2.3172801e+04, -2.5336869e+04, -2.7256938e+04, -2.8914212e+04,
      -3.0292811e+04, -3.1379968e+04, -3.2163160e+04, -3.2636663e+04,
      -3.2768000e+04, -3.2642190e+04, -3.2169708e+04, -3.1389768e+04,
      -3.0307591e+04, -2.8933213e+04, -2.7277887e+04, -2.5361709e+04,
      -2.3197556e+04, -2.0809583e+04, -1.8223444e+04, -1.5459240e+04,
      -1.2547850e+04, -9.5147469e+03, -6.3913071e+03, -3.2061552e+03,
       1.1474435e+01,  3.2280361e+03,  6.4146913e+03,  9.5376172e+03,
       1.2570372e+04,  1.5481142e+04,  1.8242641e+04,  2.0827867e+04,
       2.3213026e+04,  2.5375280e+04,  2.7293037e+04,  2.8947339e+04,
       3.0323330e+04,  3.1406817e+04,  3.2187891e+04,  3.2660531e+04,
       3.2767000e+04,  3.2659083e+04,  3.2187190e+04,  3.1402569e+04,
       3.0317849e+04,  2.8939620e+04,  2.7281206e+04,  2.5360297e+04,
       2.3194809e+04,  2.0805310e+04,  1.8215867e+04,  1.5450539e+04,
       1.2535402e+04,  9.5045150e+03,  6.3825434e+03,  3.1961404e+03,
      -1.9030604e+01, -3.2350097e+03, -6.4178539e+03, -9.5376996e+03,
      -1.2564818e+04, -1.5468879e+04, -1.8222757e+04, -2.0799617e+04,
      -2.3177492e+04, -2.5329860e+04, -2.7239548e+04, -2.8887388e+04,
      -3.0257022e+04, -3.1336972e+04, -3.2115659e+04, -3.2585521e+04,
      -3.2741691e+04, -3.2583544e+04, -3.2112334e+04, -3.1332786e+04,
      -3.0253039e+04, -2.8881031e+04, -2.7231757e+04, -2.5321143e+04,
      -2.3168095e+04, -2.0790131e+04, -1.8212051e+04, -1.5458479e+04,
      -1.2554090e+04, -9.5278709e+03, -6.4088050e+03, -3.2282766e+03,
      -1.5127187e+01,  3.1974595e+03,  6.3806758e+03,  9.5013370e+03,
       1.2533377e+04,  1.5443257e+04,  1.8203988e+04,  2.0793445e+04,
       2.3180292e+04,  2.5344634e+04,  2.7265053e+04,  2.8919963e+04,
       3.0294320e+04,  3.1376795e+04,  3.2154178e+04,  3.2622959e+04,
       3.2767000e+04,  3.2617286e+04,  3.2145842e+04,  3.1360660e+04,
       3.0278348e+04,  2.8901877e+04,  2.7241369e+04,  2.5301056e+04 ],

])

X_HAT_CLIP_7M5 = np.array([

     [ 9.0382948e-01,  2.8563300e+00,  2.0863167e+00,  3.2605273e+00,
       1.8231017e+00, -2.6473031e+00, -7.7420704e+00, -1.6971743e+01,
      -4.4169569e+00,  4.7473387e+00,  7.9882732e+00,  2.1090757e+00,
       6.9477046e+00,  7.6294361e+00,  4.5069158e+00,  1.1288109e+00,
       5.5301798e-01, -1.2320805e+00,  1.2696965e+01,  1.7998129e+01,
       1.9997378e+01,  2.3310802e+01,  3.4116671e+01,  3.1619222e+01,
       2.3643252e+01,  2.2595989e+01,  2.4150879e+01,  1.7561939e+01,
       2.4167995e+01,  2.1868269e+01,  1.2021561e+01,  1.0810360e+01,
      -1.1321816e+01, -1.3811836e+01, -2.7571991e+01, -3.3459505e+01,
      -2.6720233e+01, -4.0425004e+01, -4.1666697e+01, -4.8106995e+01,
      -7.1121739e+01, -8.5018856e+01, -6.4519501e+01, -6.1651047e+01,
      -6.2001672e+01, -4.9054098e+01,  5.3605147e+00, -2.7222279e+00,
      -6.3200946e+00, -2.8873822e+01, -5.6314175e+01, -5.9551902e+01,
      -2.1183627e+01, -9.5007617e+01, -6.7674879e+01,  7.6546124e+01,
       3.6355638e+02,  2.0908440e+02,  9.2290767e+01, -8.4453487e+01,
      -2.0810832e+02, -1.9235273e+02, -4.0634578e+02, -2.2011977e+02,
       6.2920459e+02,  3.1481663e+03,  6.2343351e+03,  9.4022080e+03,
       1.2520451e+04,  1.5313131e+04,  1.8128985e+04,  2.0762454e+04,
       2.3084787e+04,  2.5275848e+04,  2.7095495e+04,  2.8665301e+04,
       3.0094623e+04,  3.1202047e+04,  3.2006678e+04,  3.2461623e+04,
       3.2568832e+04,  3.2408327e+04,  3.1961953e+04,  3.1146555e+04,
       3.0073949e+04,  2.8725124e+04,  2.7099832e+04,  2.5196695e+04,
       2.3022972e+04,  2.0643354e+04,  1.8079103e+04,  1.5352852e+04,
       1.2476728e+04,  9.4135962e+03,  6.2948219e+03,  3.1010477e+03,
      -9.0897787e+01, -3.3383673e+03, -6.5093586e+03, -9.6214110e+03,
      -1.2638625e+04, -1.5564780e+04, -1.8289238e+04, -2.0877731e+04,
      -2.3274493e+04, -2.5456613e+04, -2.7372293e+04, -2.9018289e+04,
      -3.0399516e+04, -3.1474248e+04, -3.2213279e+04, -3.2686770e+04,
      -3.2768000e+04, -3.2655386e+04, -3.2161995e+04, -3.1363572e+04,
      -3.0296725e+04, -2.8893704e+04, -2.7226660e+04, -2.5303018e+04 ],

    [ -2.3111848e+04, -2.0718046e+04, -1.8145256e+04, -1.5386042e+04,
      -1.2451683e+04, -9.4561229e+03, -6.3407390e+03, -3.2078423e+03,
      -1.4231827e+01,  3.1871864e+03,  6.4028626e+03,  9.5288605e+03,
       1.2522323e+04,  1.5403074e+04,  1.8148763e+04,  2.0738239e+04,
       2.3124038e+04,  2.5248199e+04,  2.7157343e+04,  2.8821520e+04,
       3.0197933e+04,  3.1280524e+04,  3.2068994e+04,  3.2535286e+04,
       3.2674198e+04,  3.2505688e+04,  3.2021479e+04,  3.1217393e+04,
       3.0139480e+04,  2.8742712e+04,  2.7081219e+04,  2.5164753e+04,
       2.3005958e+04,  2.0625737e+04,  1.8051360e+04,  1.5314656e+04,
       1.2403135e+04,  9.3958576e+03,  6.2866076e+03,  3.1150574e+03,
      -8.3043055e+01, -3.2906309e+03, -6.4783209e+03, -9.5968770e+03,
      -1.2627329e+04, -1.5543027e+04, -1.8297188e+04, -2.0856742e+04,
      -2.3242831e+04, -2.5406524e+04, -2.7315500e+04, -2.8954115e+04,
      -3.0311322e+04, -3.1397263e+04, -3.2199564e+04, -3.2663393e+04,
      -3.2768000e+04, -3.2631820e+04, -3.2158938e+04, -3.1361864e+04,
      -3.0286591e+04, -2.8909485e+04, -2.7252911e+04, -2.5338382e+04,
      -2.3179744e+04, -2.0801866e+04, -1.8222214e+04, -1.5467629e+04,
      -1.2551908e+04, -9.5234786e+03, -6.4050183e+03, -3.2234113e+03,
      -1.0202956e+01,  3.2050827e+03,  6.3811438e+03,  9.5000550e+03,
       1.2526624e+04,  1.5433267e+04,  1.8190970e+04,  2.0775689e+04,
       2.3157406e+04,  2.5317424e+04,  2.7226647e+04,  2.8880016e+04,
       3.0250964e+04,  3.1333324e+04,  3.2108157e+04,  3.2579815e+04,
       3.2739171e+04,  3.2585501e+04,  3.2115570e+04,  3.1342610e+04,
       3.0239307e+04,  2.8881574e+04,  2.7232557e+04,  2.5325302e+04,
       2.3165560e+04,  2.0776212e+04,  1.8202819e+04,  1.5452512e+04,
       1.2552870e+04,  9.5323059e+03,  6.4141989e+03,  3.2353808e+03,
       3.2008805e+01, -3.1778835e+03, -6.3568255e+03, -9.4682388e+03,
      -1.2490842e+04, -1.5403926e+04, -1.8154186e+04, -2.0745346e+04,
      -2.3121161e+04, -2.5285375e+04, -2.7200767e+04, -2.8855100e+04,
      -3.0236188e+04, -3.1303814e+04, -3.2106304e+04, -3.2610143e+04 ],

])

X_HAT_CLIP = [ X_HAT_CLIP_7M5, X_HAT_CLIP_10M ]
